/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mailmanager.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The structure containing traffic policy conditions and actions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PolicyStatement implements SdkPojo, Serializable, ToCopyableBuilder<PolicyStatement.Builder, PolicyStatement> {
    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(PolicyStatement::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<List<PolicyCondition>> CONDITIONS_FIELD = SdkField
            .<List<PolicyCondition>> builder(MarshallingType.LIST)
            .memberName("Conditions")
            .getter(getter(PolicyStatement::conditions))
            .setter(setter(Builder::conditions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Conditions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PolicyCondition> builder(MarshallingType.SDK_POJO)
                                            .constructor(PolicyCondition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(ACTION_FIELD, CONDITIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String action;

    private final List<PolicyCondition> conditions;

    private PolicyStatement(BuilderImpl builder) {
        this.action = builder.action;
        this.conditions = builder.conditions;
    }

    /**
     * <p>
     * The action that informs a traffic policy resource to either allow or block the email if it matches a condition in
     * the policy statement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link AcceptAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action that informs a traffic policy resource to either allow or block the email if it matches a
     *         condition in the policy statement.
     * @see AcceptAction
     */
    public final AcceptAction action() {
        return AcceptAction.fromValue(action);
    }

    /**
     * <p>
     * The action that informs a traffic policy resource to either allow or block the email if it matches a condition in
     * the policy statement.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link AcceptAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action that informs a traffic policy resource to either allow or block the email if it matches a
     *         condition in the policy statement.
     * @see AcceptAction
     */
    public final String actionAsString() {
        return action;
    }

    /**
     * For responses, this returns true if the service returned a value for the Conditions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasConditions() {
        return conditions != null && !(conditions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of conditions to apply to incoming messages for filtering email traffic.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConditions} method.
     * </p>
     * 
     * @return The list of conditions to apply to incoming messages for filtering email traffic.
     */
    public final List<PolicyCondition> conditions() {
        return conditions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasConditions() ? conditions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PolicyStatement)) {
            return false;
        }
        PolicyStatement other = (PolicyStatement) obj;
        return Objects.equals(actionAsString(), other.actionAsString()) && hasConditions() == other.hasConditions()
                && Objects.equals(conditions(), other.conditions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PolicyStatement").add("Action", actionAsString())
                .add("Conditions", hasConditions() ? conditions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Conditions":
            return Optional.ofNullable(clazz.cast(conditions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PolicyStatement, T> g) {
        return obj -> g.apply((PolicyStatement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PolicyStatement> {
        /**
         * <p>
         * The action that informs a traffic policy resource to either allow or block the email if it matches a
         * condition in the policy statement.
         * </p>
         * 
         * @param action
         *        The action that informs a traffic policy resource to either allow or block the email if it matches a
         *        condition in the policy statement.
         * @see AcceptAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AcceptAction
         */
        Builder action(String action);

        /**
         * <p>
         * The action that informs a traffic policy resource to either allow or block the email if it matches a
         * condition in the policy statement.
         * </p>
         * 
         * @param action
         *        The action that informs a traffic policy resource to either allow or block the email if it matches a
         *        condition in the policy statement.
         * @see AcceptAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AcceptAction
         */
        Builder action(AcceptAction action);

        /**
         * <p>
         * The list of conditions to apply to incoming messages for filtering email traffic.
         * </p>
         * 
         * @param conditions
         *        The list of conditions to apply to incoming messages for filtering email traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditions(Collection<PolicyCondition> conditions);

        /**
         * <p>
         * The list of conditions to apply to incoming messages for filtering email traffic.
         * </p>
         * 
         * @param conditions
         *        The list of conditions to apply to incoming messages for filtering email traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditions(PolicyCondition... conditions);

        /**
         * <p>
         * The list of conditions to apply to incoming messages for filtering email traffic.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mailmanager.model.PolicyCondition.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.mailmanager.model.PolicyCondition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mailmanager.model.PolicyCondition.Builder#build()} is called
         * immediately and its result is passed to {@link #conditions(List<PolicyCondition>)}.
         * 
         * @param conditions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mailmanager.model.PolicyCondition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #conditions(java.util.Collection<PolicyCondition>)
         */
        Builder conditions(Consumer<PolicyCondition.Builder>... conditions);
    }

    static final class BuilderImpl implements Builder {
        private String action;

        private List<PolicyCondition> conditions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PolicyStatement model) {
            action(model.action);
            conditions(model.conditions);
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(AcceptAction action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final List<PolicyCondition.Builder> getConditions() {
            List<PolicyCondition.Builder> result = PolicyConditionsCopier.copyToBuilder(this.conditions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setConditions(Collection<PolicyCondition.BuilderImpl> conditions) {
            this.conditions = PolicyConditionsCopier.copyFromBuilder(conditions);
        }

        @Override
        public final Builder conditions(Collection<PolicyCondition> conditions) {
            this.conditions = PolicyConditionsCopier.copy(conditions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder conditions(PolicyCondition... conditions) {
            conditions(Arrays.asList(conditions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder conditions(Consumer<PolicyCondition.Builder>... conditions) {
            conditions(Stream.of(conditions).map(c -> PolicyCondition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public PolicyStatement build() {
            return new PolicyStatement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
