/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mailmanager.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A rule contains conditions, "unless conditions" and actions. For each envelope recipient of an email, if all
 * conditions match and none of the "unless conditions" match, then all of the actions are executed sequentially. If no
 * conditions are provided, the rule always applies and the actions are implicitly executed. If only "unless conditions"
 * are provided, the rule applies if the email does not match the evaluation of the "unless conditions".
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Rule implements SdkPojo, Serializable, ToCopyableBuilder<Rule.Builder, Rule> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Rule::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<RuleCondition>> CONDITIONS_FIELD = SdkField
            .<List<RuleCondition>> builder(MarshallingType.LIST)
            .memberName("Conditions")
            .getter(getter(Rule::conditions))
            .setter(setter(Builder::conditions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Conditions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RuleCondition> builder(MarshallingType.SDK_POJO)
                                            .constructor(RuleCondition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<RuleCondition>> UNLESS_FIELD = SdkField
            .<List<RuleCondition>> builder(MarshallingType.LIST)
            .memberName("Unless")
            .getter(getter(Rule::unless))
            .setter(setter(Builder::unless))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Unless").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RuleCondition> builder(MarshallingType.SDK_POJO)
                                            .constructor(RuleCondition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<RuleAction>> ACTIONS_FIELD = SdkField
            .<List<RuleAction>> builder(MarshallingType.LIST)
            .memberName("Actions")
            .getter(getter(Rule::actions))
            .setter(setter(Builder::actions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Actions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RuleAction> builder(MarshallingType.SDK_POJO)
                                            .constructor(RuleAction::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, CONDITIONS_FIELD,
            UNLESS_FIELD, ACTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<RuleCondition> conditions;

    private final List<RuleCondition> unless;

    private final List<RuleAction> actions;

    private Rule(BuilderImpl builder) {
        this.name = builder.name;
        this.conditions = builder.conditions;
        this.unless = builder.unless;
        this.actions = builder.actions;
    }

    /**
     * <p>
     * The user-friendly name of the rule.
     * </p>
     * 
     * @return The user-friendly name of the rule.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Conditions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasConditions() {
        return conditions != null && !(conditions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The conditions of this rule. All conditions must match the email for the actions to be executed. An empty list of
     * conditions means that all emails match, but are still subject to any "unless conditions"
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConditions} method.
     * </p>
     * 
     * @return The conditions of this rule. All conditions must match the email for the actions to be executed. An empty
     *         list of conditions means that all emails match, but are still subject to any "unless conditions"
     */
    public final List<RuleCondition> conditions() {
        return conditions;
    }

    /**
     * For responses, this returns true if the service returned a value for the Unless property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasUnless() {
        return unless != null && !(unless instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The "unless conditions" of this rule. None of the conditions can match the email for the actions to be executed.
     * If any of these conditions do match the email, then the actions are not executed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUnless} method.
     * </p>
     * 
     * @return The "unless conditions" of this rule. None of the conditions can match the email for the actions to be
     *         executed. If any of these conditions do match the email, then the actions are not executed.
     */
    public final List<RuleCondition> unless() {
        return unless;
    }

    /**
     * For responses, this returns true if the service returned a value for the Actions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasActions() {
        return actions != null && !(actions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of actions to execute when the conditions match the incoming email, and none of the "unless conditions"
     * match.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActions} method.
     * </p>
     * 
     * @return The list of actions to execute when the conditions match the incoming email, and none of the
     *         "unless conditions" match.
     */
    public final List<RuleAction> actions() {
        return actions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasConditions() ? conditions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUnless() ? unless() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasActions() ? actions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Rule)) {
            return false;
        }
        Rule other = (Rule) obj;
        return Objects.equals(name(), other.name()) && hasConditions() == other.hasConditions()
                && Objects.equals(conditions(), other.conditions()) && hasUnless() == other.hasUnless()
                && Objects.equals(unless(), other.unless()) && hasActions() == other.hasActions()
                && Objects.equals(actions(), other.actions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Rule").add("Name", name()).add("Conditions", hasConditions() ? conditions() : null)
                .add("Unless", hasUnless() ? unless() : null).add("Actions", hasActions() ? actions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Conditions":
            return Optional.ofNullable(clazz.cast(conditions()));
        case "Unless":
            return Optional.ofNullable(clazz.cast(unless()));
        case "Actions":
            return Optional.ofNullable(clazz.cast(actions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Conditions", CONDITIONS_FIELD);
        map.put("Unless", UNLESS_FIELD);
        map.put("Actions", ACTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Rule, T> g) {
        return obj -> g.apply((Rule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Rule> {
        /**
         * <p>
         * The user-friendly name of the rule.
         * </p>
         * 
         * @param name
         *        The user-friendly name of the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The conditions of this rule. All conditions must match the email for the actions to be executed. An empty
         * list of conditions means that all emails match, but are still subject to any "unless conditions"
         * </p>
         * 
         * @param conditions
         *        The conditions of this rule. All conditions must match the email for the actions to be executed. An
         *        empty list of conditions means that all emails match, but are still subject to any "unless conditions"
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditions(Collection<RuleCondition> conditions);

        /**
         * <p>
         * The conditions of this rule. All conditions must match the email for the actions to be executed. An empty
         * list of conditions means that all emails match, but are still subject to any "unless conditions"
         * </p>
         * 
         * @param conditions
         *        The conditions of this rule. All conditions must match the email for the actions to be executed. An
         *        empty list of conditions means that all emails match, but are still subject to any "unless conditions"
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditions(RuleCondition... conditions);

        /**
         * <p>
         * The conditions of this rule. All conditions must match the email for the actions to be executed. An empty
         * list of conditions means that all emails match, but are still subject to any "unless conditions"
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mailmanager.model.RuleCondition.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.mailmanager.model.RuleCondition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mailmanager.model.RuleCondition.Builder#build()} is called immediately
         * and its result is passed to {@link #conditions(List<RuleCondition>)}.
         * 
         * @param conditions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mailmanager.model.RuleCondition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #conditions(java.util.Collection<RuleCondition>)
         */
        Builder conditions(Consumer<RuleCondition.Builder>... conditions);

        /**
         * <p>
         * The "unless conditions" of this rule. None of the conditions can match the email for the actions to be
         * executed. If any of these conditions do match the email, then the actions are not executed.
         * </p>
         * 
         * @param unless
         *        The "unless conditions" of this rule. None of the conditions can match the email for the actions to be
         *        executed. If any of these conditions do match the email, then the actions are not executed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unless(Collection<RuleCondition> unless);

        /**
         * <p>
         * The "unless conditions" of this rule. None of the conditions can match the email for the actions to be
         * executed. If any of these conditions do match the email, then the actions are not executed.
         * </p>
         * 
         * @param unless
         *        The "unless conditions" of this rule. None of the conditions can match the email for the actions to be
         *        executed. If any of these conditions do match the email, then the actions are not executed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unless(RuleCondition... unless);

        /**
         * <p>
         * The "unless conditions" of this rule. None of the conditions can match the email for the actions to be
         * executed. If any of these conditions do match the email, then the actions are not executed.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mailmanager.model.RuleCondition.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.mailmanager.model.RuleCondition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mailmanager.model.RuleCondition.Builder#build()} is called immediately
         * and its result is passed to {@link #unless(List<RuleCondition>)}.
         * 
         * @param unless
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mailmanager.model.RuleCondition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #unless(java.util.Collection<RuleCondition>)
         */
        Builder unless(Consumer<RuleCondition.Builder>... unless);

        /**
         * <p>
         * The list of actions to execute when the conditions match the incoming email, and none of the
         * "unless conditions" match.
         * </p>
         * 
         * @param actions
         *        The list of actions to execute when the conditions match the incoming email, and none of the
         *        "unless conditions" match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<RuleAction> actions);

        /**
         * <p>
         * The list of actions to execute when the conditions match the incoming email, and none of the
         * "unless conditions" match.
         * </p>
         * 
         * @param actions
         *        The list of actions to execute when the conditions match the incoming email, and none of the
         *        "unless conditions" match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(RuleAction... actions);

        /**
         * <p>
         * The list of actions to execute when the conditions match the incoming email, and none of the
         * "unless conditions" match.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.mailmanager.model.RuleAction.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.mailmanager.model.RuleAction#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.mailmanager.model.RuleAction.Builder#build()} is called immediately
         * and its result is passed to {@link #actions(List<RuleAction>)}.
         * 
         * @param actions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.mailmanager.model.RuleAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #actions(java.util.Collection<RuleAction>)
         */
        Builder actions(Consumer<RuleAction.Builder>... actions);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<RuleCondition> conditions = DefaultSdkAutoConstructList.getInstance();

        private List<RuleCondition> unless = DefaultSdkAutoConstructList.getInstance();

        private List<RuleAction> actions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Rule model) {
            name(model.name);
            conditions(model.conditions);
            unless(model.unless);
            actions(model.actions);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final List<RuleCondition.Builder> getConditions() {
            List<RuleCondition.Builder> result = RuleConditionsCopier.copyToBuilder(this.conditions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setConditions(Collection<RuleCondition.BuilderImpl> conditions) {
            this.conditions = RuleConditionsCopier.copyFromBuilder(conditions);
        }

        @Override
        public final Builder conditions(Collection<RuleCondition> conditions) {
            this.conditions = RuleConditionsCopier.copy(conditions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder conditions(RuleCondition... conditions) {
            conditions(Arrays.asList(conditions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder conditions(Consumer<RuleCondition.Builder>... conditions) {
            conditions(Stream.of(conditions).map(c -> RuleCondition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<RuleCondition.Builder> getUnless() {
            List<RuleCondition.Builder> result = RuleConditionsCopier.copyToBuilder(this.unless);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUnless(Collection<RuleCondition.BuilderImpl> unless) {
            this.unless = RuleConditionsCopier.copyFromBuilder(unless);
        }

        @Override
        public final Builder unless(Collection<RuleCondition> unless) {
            this.unless = RuleConditionsCopier.copy(unless);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder unless(RuleCondition... unless) {
            unless(Arrays.asList(unless));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder unless(Consumer<RuleCondition.Builder>... unless) {
            unless(Stream.of(unless).map(c -> RuleCondition.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<RuleAction.Builder> getActions() {
            List<RuleAction.Builder> result = RuleActionsCopier.copyToBuilder(this.actions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setActions(Collection<RuleAction.BuilderImpl> actions) {
            this.actions = RuleActionsCopier.copyFromBuilder(actions);
        }

        @Override
        public final Builder actions(Collection<RuleAction> actions) {
            this.actions = RuleActionsCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(RuleAction... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(Consumer<RuleAction.Builder>... actions) {
            actions(Stream.of(actions).map(c -> RuleAction.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Rule build() {
            return new Rule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
