/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mailmanager.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A result row containing metadata for an archived email message.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Row implements SdkPojo, Serializable, ToCopyableBuilder<Row.Builder, Row> {
    private static final SdkField<String> ARCHIVED_MESSAGE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ArchivedMessageId").getter(getter(Row::archivedMessageId)).setter(setter(Builder::archivedMessageId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ArchivedMessageId").build()).build();

    private static final SdkField<Instant> RECEIVED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReceivedTimestamp").getter(getter(Row::receivedTimestamp)).setter(setter(Builder::receivedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReceivedTimestamp").build()).build();

    private static final SdkField<String> DATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Date")
            .getter(getter(Row::date)).setter(setter(Builder::date))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Date").build()).build();

    private static final SdkField<String> TO_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("To")
            .getter(getter(Row::to)).setter(setter(Builder::to))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("To").build()).build();

    private static final SdkField<String> FROM_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("From")
            .getter(getter(Row::from)).setter(setter(Builder::from))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("From").build()).build();

    private static final SdkField<String> CC_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Cc")
            .getter(getter(Row::cc)).setter(setter(Builder::cc))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Cc").build()).build();

    private static final SdkField<String> SUBJECT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Subject")
            .getter(getter(Row::subject)).setter(setter(Builder::subject))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subject").build()).build();

    private static final SdkField<String> MESSAGE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageId").getter(getter(Row::messageId)).setter(setter(Builder::messageId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageId").build()).build();

    private static final SdkField<Boolean> HAS_ATTACHMENTS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("HasAttachments").getter(getter(Row::hasAttachments)).setter(setter(Builder::hasAttachments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HasAttachments").build()).build();

    private static final SdkField<List<String>> RECEIVED_HEADERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ReceivedHeaders")
            .getter(getter(Row::receivedHeaders))
            .setter(setter(Builder::receivedHeaders))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReceivedHeaders").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IN_REPLY_TO_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InReplyTo").getter(getter(Row::inReplyTo)).setter(setter(Builder::inReplyTo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InReplyTo").build()).build();

    private static final SdkField<String> X_MAILER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("XMailer").getter(getter(Row::xMailer)).setter(setter(Builder::xMailer))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("XMailer").build()).build();

    private static final SdkField<String> X_ORIGINAL_MAILER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("XOriginalMailer").getter(getter(Row::xOriginalMailer)).setter(setter(Builder::xOriginalMailer))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("XOriginalMailer").build()).build();

    private static final SdkField<String> X_PRIORITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("XPriority").getter(getter(Row::xPriority)).setter(setter(Builder::xPriority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("XPriority").build()).build();

    private static final SdkField<String> INGRESS_POINT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IngressPointId").getter(getter(Row::ingressPointId)).setter(setter(Builder::ingressPointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IngressPointId").build()).build();

    private static final SdkField<String> SENDER_HOSTNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SenderHostname").getter(getter(Row::senderHostname)).setter(setter(Builder::senderHostname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SenderHostname").build()).build();

    private static final SdkField<String> SENDER_IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SenderIpAddress").getter(getter(Row::senderIpAddress)).setter(setter(Builder::senderIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SenderIpAddress").build()).build();

    private static final SdkField<Envelope> ENVELOPE_FIELD = SdkField.<Envelope> builder(MarshallingType.SDK_POJO)
            .memberName("Envelope").getter(getter(Row::envelope)).setter(setter(Builder::envelope))
            .constructor(Envelope::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Envelope").build()).build();

    private static final SdkField<String> SOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceArn").getter(getter(Row::sourceArn)).setter(setter(Builder::sourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARCHIVED_MESSAGE_ID_FIELD,
            RECEIVED_TIMESTAMP_FIELD, DATE_FIELD, TO_FIELD, FROM_FIELD, CC_FIELD, SUBJECT_FIELD, MESSAGE_ID_FIELD,
            HAS_ATTACHMENTS_FIELD, RECEIVED_HEADERS_FIELD, IN_REPLY_TO_FIELD, X_MAILER_FIELD, X_ORIGINAL_MAILER_FIELD,
            X_PRIORITY_FIELD, INGRESS_POINT_ID_FIELD, SENDER_HOSTNAME_FIELD, SENDER_IP_ADDRESS_FIELD, ENVELOPE_FIELD,
            SOURCE_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String archivedMessageId;

    private final Instant receivedTimestamp;

    private final String date;

    private final String to;

    private final String from;

    private final String cc;

    private final String subject;

    private final String messageId;

    private final Boolean hasAttachments;

    private final List<String> receivedHeaders;

    private final String inReplyTo;

    private final String xMailer;

    private final String xOriginalMailer;

    private final String xPriority;

    private final String ingressPointId;

    private final String senderHostname;

    private final String senderIpAddress;

    private final Envelope envelope;

    private final String sourceArn;

    private Row(BuilderImpl builder) {
        this.archivedMessageId = builder.archivedMessageId;
        this.receivedTimestamp = builder.receivedTimestamp;
        this.date = builder.date;
        this.to = builder.to;
        this.from = builder.from;
        this.cc = builder.cc;
        this.subject = builder.subject;
        this.messageId = builder.messageId;
        this.hasAttachments = builder.hasAttachments;
        this.receivedHeaders = builder.receivedHeaders;
        this.inReplyTo = builder.inReplyTo;
        this.xMailer = builder.xMailer;
        this.xOriginalMailer = builder.xOriginalMailer;
        this.xPriority = builder.xPriority;
        this.ingressPointId = builder.ingressPointId;
        this.senderHostname = builder.senderHostname;
        this.senderIpAddress = builder.senderIpAddress;
        this.envelope = builder.envelope;
        this.sourceArn = builder.sourceArn;
    }

    /**
     * <p>
     * The unique identifier of the archived message.
     * </p>
     * 
     * @return The unique identifier of the archived message.
     */
    public final String archivedMessageId() {
        return archivedMessageId;
    }

    /**
     * <p>
     * The timestamp of when the email was received.
     * </p>
     * 
     * @return The timestamp of when the email was received.
     */
    public final Instant receivedTimestamp() {
        return receivedTimestamp;
    }

    /**
     * <p>
     * The date the email was sent.
     * </p>
     * 
     * @return The date the email was sent.
     */
    public final String date() {
        return date;
    }

    /**
     * <p>
     * The email addresses in the To header.
     * </p>
     * 
     * @return The email addresses in the To header.
     */
    public final String to() {
        return to;
    }

    /**
     * <p>
     * The email address of the sender.
     * </p>
     * 
     * @return The email address of the sender.
     */
    public final String from() {
        return from;
    }

    /**
     * <p>
     * The email addresses in the CC header.
     * </p>
     * 
     * @return The email addresses in the CC header.
     */
    public final String cc() {
        return cc;
    }

    /**
     * <p>
     * The subject header value of the email.
     * </p>
     * 
     * @return The subject header value of the email.
     */
    public final String subject() {
        return subject;
    }

    /**
     * <p>
     * The unique message ID of the email.
     * </p>
     * 
     * @return The unique message ID of the email.
     */
    public final String messageId() {
        return messageId;
    }

    /**
     * <p>
     * A flag indicating if the email has attachments.
     * </p>
     * 
     * @return A flag indicating if the email has attachments.
     */
    public final Boolean hasAttachments() {
        return hasAttachments;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReceivedHeaders property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasReceivedHeaders() {
        return receivedHeaders != null && !(receivedHeaders instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The received headers from the email delivery path.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReceivedHeaders} method.
     * </p>
     * 
     * @return The received headers from the email delivery path.
     */
    public final List<String> receivedHeaders() {
        return receivedHeaders;
    }

    /**
     * <p>
     * The email message ID this is a reply to.
     * </p>
     * 
     * @return The email message ID this is a reply to.
     */
    public final String inReplyTo() {
        return inReplyTo;
    }

    /**
     * <p>
     * The user agent that sent the email.
     * </p>
     * 
     * @return The user agent that sent the email.
     */
    public final String xMailer() {
        return xMailer;
    }

    /**
     * <p>
     * The original user agent that sent the email.
     * </p>
     * 
     * @return The original user agent that sent the email.
     */
    public final String xOriginalMailer() {
        return xOriginalMailer;
    }

    /**
     * <p>
     * The priority level of the email.
     * </p>
     * 
     * @return The priority level of the email.
     */
    public final String xPriority() {
        return xPriority;
    }

    /**
     * <p>
     * The ID of the ingress endpoint through which the email was received.
     * </p>
     * 
     * @return The ID of the ingress endpoint through which the email was received.
     */
    public final String ingressPointId() {
        return ingressPointId;
    }

    /**
     * <p>
     * The name of the host from which the email was received.
     * </p>
     * 
     * @return The name of the host from which the email was received.
     */
    public final String senderHostname() {
        return senderHostname;
    }

    /**
     * <ul>
     * <li>
     * <p>
     * Mail archived with Mail Manager: The IP address of the client that connects to the ingress endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * Mail sent through a configuration set with the archiving option enabled: The IP address of the client that makes
     * the SendEmail API call.
     * </p>
     * </li>
     * </ul>
     * 
     * @return <li>
     *         <p>
     *         Mail archived with Mail Manager: The IP address of the client that connects to the ingress endpoint.
     *         </p>
     *         </li> <li>
     *         <p>
     *         Mail sent through a configuration set with the archiving option enabled: The IP address of the client
     *         that makes the SendEmail API call.
     *         </p>
     *         </li>
     */
    public final String senderIpAddress() {
        return senderIpAddress;
    }

    /**
     * <p>
     * The SMTP envelope information of the email.
     * </p>
     * 
     * @return The SMTP envelope information of the email.
     */
    public final Envelope envelope() {
        return envelope;
    }

    /**
     * <p>
     * Specifies the archived email source, identified by either a Rule Set's ARN with an Archive action, or a
     * Configuration Set's Archive ARN.
     * </p>
     * 
     * @return Specifies the archived email source, identified by either a Rule Set's ARN with an Archive action, or a
     *         Configuration Set's Archive ARN.
     */
    public final String sourceArn() {
        return sourceArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(archivedMessageId());
        hashCode = 31 * hashCode + Objects.hashCode(receivedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(date());
        hashCode = 31 * hashCode + Objects.hashCode(to());
        hashCode = 31 * hashCode + Objects.hashCode(from());
        hashCode = 31 * hashCode + Objects.hashCode(cc());
        hashCode = 31 * hashCode + Objects.hashCode(subject());
        hashCode = 31 * hashCode + Objects.hashCode(messageId());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttachments());
        hashCode = 31 * hashCode + Objects.hashCode(hasReceivedHeaders() ? receivedHeaders() : null);
        hashCode = 31 * hashCode + Objects.hashCode(inReplyTo());
        hashCode = 31 * hashCode + Objects.hashCode(xMailer());
        hashCode = 31 * hashCode + Objects.hashCode(xOriginalMailer());
        hashCode = 31 * hashCode + Objects.hashCode(xPriority());
        hashCode = 31 * hashCode + Objects.hashCode(ingressPointId());
        hashCode = 31 * hashCode + Objects.hashCode(senderHostname());
        hashCode = 31 * hashCode + Objects.hashCode(senderIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(envelope());
        hashCode = 31 * hashCode + Objects.hashCode(sourceArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Row)) {
            return false;
        }
        Row other = (Row) obj;
        return Objects.equals(archivedMessageId(), other.archivedMessageId())
                && Objects.equals(receivedTimestamp(), other.receivedTimestamp()) && Objects.equals(date(), other.date())
                && Objects.equals(to(), other.to()) && Objects.equals(from(), other.from()) && Objects.equals(cc(), other.cc())
                && Objects.equals(subject(), other.subject()) && Objects.equals(messageId(), other.messageId())
                && Objects.equals(hasAttachments(), other.hasAttachments()) && hasReceivedHeaders() == other.hasReceivedHeaders()
                && Objects.equals(receivedHeaders(), other.receivedHeaders()) && Objects.equals(inReplyTo(), other.inReplyTo())
                && Objects.equals(xMailer(), other.xMailer()) && Objects.equals(xOriginalMailer(), other.xOriginalMailer())
                && Objects.equals(xPriority(), other.xPriority()) && Objects.equals(ingressPointId(), other.ingressPointId())
                && Objects.equals(senderHostname(), other.senderHostname())
                && Objects.equals(senderIpAddress(), other.senderIpAddress()) && Objects.equals(envelope(), other.envelope())
                && Objects.equals(sourceArn(), other.sourceArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Row").add("ArchivedMessageId", archivedMessageId())
                .add("ReceivedTimestamp", receivedTimestamp()).add("Date", date()).add("To", to()).add("From", from())
                .add("Cc", cc()).add("Subject", subject()).add("MessageId", messageId()).add("HasAttachments", hasAttachments())
                .add("ReceivedHeaders", hasReceivedHeaders() ? receivedHeaders() : null).add("InReplyTo", inReplyTo())
                .add("XMailer", xMailer()).add("XOriginalMailer", xOriginalMailer()).add("XPriority", xPriority())
                .add("IngressPointId", ingressPointId()).add("SenderHostname", senderHostname())
                .add("SenderIpAddress", senderIpAddress() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Envelope", envelope()).add("SourceArn", sourceArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ArchivedMessageId":
            return Optional.ofNullable(clazz.cast(archivedMessageId()));
        case "ReceivedTimestamp":
            return Optional.ofNullable(clazz.cast(receivedTimestamp()));
        case "Date":
            return Optional.ofNullable(clazz.cast(date()));
        case "To":
            return Optional.ofNullable(clazz.cast(to()));
        case "From":
            return Optional.ofNullable(clazz.cast(from()));
        case "Cc":
            return Optional.ofNullable(clazz.cast(cc()));
        case "Subject":
            return Optional.ofNullable(clazz.cast(subject()));
        case "MessageId":
            return Optional.ofNullable(clazz.cast(messageId()));
        case "HasAttachments":
            return Optional.ofNullable(clazz.cast(hasAttachments()));
        case "ReceivedHeaders":
            return Optional.ofNullable(clazz.cast(receivedHeaders()));
        case "InReplyTo":
            return Optional.ofNullable(clazz.cast(inReplyTo()));
        case "XMailer":
            return Optional.ofNullable(clazz.cast(xMailer()));
        case "XOriginalMailer":
            return Optional.ofNullable(clazz.cast(xOriginalMailer()));
        case "XPriority":
            return Optional.ofNullable(clazz.cast(xPriority()));
        case "IngressPointId":
            return Optional.ofNullable(clazz.cast(ingressPointId()));
        case "SenderHostname":
            return Optional.ofNullable(clazz.cast(senderHostname()));
        case "SenderIpAddress":
            return Optional.ofNullable(clazz.cast(senderIpAddress()));
        case "Envelope":
            return Optional.ofNullable(clazz.cast(envelope()));
        case "SourceArn":
            return Optional.ofNullable(clazz.cast(sourceArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ArchivedMessageId", ARCHIVED_MESSAGE_ID_FIELD);
        map.put("ReceivedTimestamp", RECEIVED_TIMESTAMP_FIELD);
        map.put("Date", DATE_FIELD);
        map.put("To", TO_FIELD);
        map.put("From", FROM_FIELD);
        map.put("Cc", CC_FIELD);
        map.put("Subject", SUBJECT_FIELD);
        map.put("MessageId", MESSAGE_ID_FIELD);
        map.put("HasAttachments", HAS_ATTACHMENTS_FIELD);
        map.put("ReceivedHeaders", RECEIVED_HEADERS_FIELD);
        map.put("InReplyTo", IN_REPLY_TO_FIELD);
        map.put("XMailer", X_MAILER_FIELD);
        map.put("XOriginalMailer", X_ORIGINAL_MAILER_FIELD);
        map.put("XPriority", X_PRIORITY_FIELD);
        map.put("IngressPointId", INGRESS_POINT_ID_FIELD);
        map.put("SenderHostname", SENDER_HOSTNAME_FIELD);
        map.put("SenderIpAddress", SENDER_IP_ADDRESS_FIELD);
        map.put("Envelope", ENVELOPE_FIELD);
        map.put("SourceArn", SOURCE_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Row, T> g) {
        return obj -> g.apply((Row) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Row> {
        /**
         * <p>
         * The unique identifier of the archived message.
         * </p>
         * 
         * @param archivedMessageId
         *        The unique identifier of the archived message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder archivedMessageId(String archivedMessageId);

        /**
         * <p>
         * The timestamp of when the email was received.
         * </p>
         * 
         * @param receivedTimestamp
         *        The timestamp of when the email was received.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder receivedTimestamp(Instant receivedTimestamp);

        /**
         * <p>
         * The date the email was sent.
         * </p>
         * 
         * @param date
         *        The date the email was sent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder date(String date);

        /**
         * <p>
         * The email addresses in the To header.
         * </p>
         * 
         * @param to
         *        The email addresses in the To header.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder to(String to);

        /**
         * <p>
         * The email address of the sender.
         * </p>
         * 
         * @param from
         *        The email address of the sender.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder from(String from);

        /**
         * <p>
         * The email addresses in the CC header.
         * </p>
         * 
         * @param cc
         *        The email addresses in the CC header.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cc(String cc);

        /**
         * <p>
         * The subject header value of the email.
         * </p>
         * 
         * @param subject
         *        The subject header value of the email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subject(String subject);

        /**
         * <p>
         * The unique message ID of the email.
         * </p>
         * 
         * @param messageId
         *        The unique message ID of the email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageId(String messageId);

        /**
         * <p>
         * A flag indicating if the email has attachments.
         * </p>
         * 
         * @param hasAttachments
         *        A flag indicating if the email has attachments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hasAttachments(Boolean hasAttachments);

        /**
         * <p>
         * The received headers from the email delivery path.
         * </p>
         * 
         * @param receivedHeaders
         *        The received headers from the email delivery path.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder receivedHeaders(Collection<String> receivedHeaders);

        /**
         * <p>
         * The received headers from the email delivery path.
         * </p>
         * 
         * @param receivedHeaders
         *        The received headers from the email delivery path.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder receivedHeaders(String... receivedHeaders);

        /**
         * <p>
         * The email message ID this is a reply to.
         * </p>
         * 
         * @param inReplyTo
         *        The email message ID this is a reply to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inReplyTo(String inReplyTo);

        /**
         * <p>
         * The user agent that sent the email.
         * </p>
         * 
         * @param xMailer
         *        The user agent that sent the email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder xMailer(String xMailer);

        /**
         * <p>
         * The original user agent that sent the email.
         * </p>
         * 
         * @param xOriginalMailer
         *        The original user agent that sent the email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder xOriginalMailer(String xOriginalMailer);

        /**
         * <p>
         * The priority level of the email.
         * </p>
         * 
         * @param xPriority
         *        The priority level of the email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder xPriority(String xPriority);

        /**
         * <p>
         * The ID of the ingress endpoint through which the email was received.
         * </p>
         * 
         * @param ingressPointId
         *        The ID of the ingress endpoint through which the email was received.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ingressPointId(String ingressPointId);

        /**
         * <p>
         * The name of the host from which the email was received.
         * </p>
         * 
         * @param senderHostname
         *        The name of the host from which the email was received.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder senderHostname(String senderHostname);

        /**
         * <ul>
         * <li>
         * <p>
         * Mail archived with Mail Manager: The IP address of the client that connects to the ingress endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * Mail sent through a configuration set with the archiving option enabled: The IP address of the client that
         * makes the SendEmail API call.
         * </p>
         * </li>
         * </ul>
         * 
         * @param senderIpAddress
         *        <li>
         *        <p>
         *        Mail archived with Mail Manager: The IP address of the client that connects to the ingress endpoint.
         *        </p>
         *        </li> <li>
         *        <p>
         *        Mail sent through a configuration set with the archiving option enabled: The IP address of the client
         *        that makes the SendEmail API call.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder senderIpAddress(String senderIpAddress);

        /**
         * <p>
         * The SMTP envelope information of the email.
         * </p>
         * 
         * @param envelope
         *        The SMTP envelope information of the email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder envelope(Envelope envelope);

        /**
         * <p>
         * The SMTP envelope information of the email.
         * </p>
         * This is a convenience method that creates an instance of the {@link Envelope.Builder} avoiding the need to
         * create one manually via {@link Envelope#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Envelope.Builder#build()} is called immediately and its result is
         * passed to {@link #envelope(Envelope)}.
         * 
         * @param envelope
         *        a consumer that will call methods on {@link Envelope.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #envelope(Envelope)
         */
        default Builder envelope(Consumer<Envelope.Builder> envelope) {
            return envelope(Envelope.builder().applyMutation(envelope).build());
        }

        /**
         * <p>
         * Specifies the archived email source, identified by either a Rule Set's ARN with an Archive action, or a
         * Configuration Set's Archive ARN.
         * </p>
         * 
         * @param sourceArn
         *        Specifies the archived email source, identified by either a Rule Set's ARN with an Archive action, or
         *        a Configuration Set's Archive ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);
    }

    static final class BuilderImpl implements Builder {
        private String archivedMessageId;

        private Instant receivedTimestamp;

        private String date;

        private String to;

        private String from;

        private String cc;

        private String subject;

        private String messageId;

        private Boolean hasAttachments;

        private List<String> receivedHeaders = DefaultSdkAutoConstructList.getInstance();

        private String inReplyTo;

        private String xMailer;

        private String xOriginalMailer;

        private String xPriority;

        private String ingressPointId;

        private String senderHostname;

        private String senderIpAddress;

        private Envelope envelope;

        private String sourceArn;

        private BuilderImpl() {
        }

        private BuilderImpl(Row model) {
            archivedMessageId(model.archivedMessageId);
            receivedTimestamp(model.receivedTimestamp);
            date(model.date);
            to(model.to);
            from(model.from);
            cc(model.cc);
            subject(model.subject);
            messageId(model.messageId);
            hasAttachments(model.hasAttachments);
            receivedHeaders(model.receivedHeaders);
            inReplyTo(model.inReplyTo);
            xMailer(model.xMailer);
            xOriginalMailer(model.xOriginalMailer);
            xPriority(model.xPriority);
            ingressPointId(model.ingressPointId);
            senderHostname(model.senderHostname);
            senderIpAddress(model.senderIpAddress);
            envelope(model.envelope);
            sourceArn(model.sourceArn);
        }

        public final String getArchivedMessageId() {
            return archivedMessageId;
        }

        public final void setArchivedMessageId(String archivedMessageId) {
            this.archivedMessageId = archivedMessageId;
        }

        @Override
        public final Builder archivedMessageId(String archivedMessageId) {
            this.archivedMessageId = archivedMessageId;
            return this;
        }

        public final Instant getReceivedTimestamp() {
            return receivedTimestamp;
        }

        public final void setReceivedTimestamp(Instant receivedTimestamp) {
            this.receivedTimestamp = receivedTimestamp;
        }

        @Override
        public final Builder receivedTimestamp(Instant receivedTimestamp) {
            this.receivedTimestamp = receivedTimestamp;
            return this;
        }

        public final String getDate() {
            return date;
        }

        public final void setDate(String date) {
            this.date = date;
        }

        @Override
        public final Builder date(String date) {
            this.date = date;
            return this;
        }

        public final String getTo() {
            return to;
        }

        public final void setTo(String to) {
            this.to = to;
        }

        @Override
        public final Builder to(String to) {
            this.to = to;
            return this;
        }

        public final String getFrom() {
            return from;
        }

        public final void setFrom(String from) {
            this.from = from;
        }

        @Override
        public final Builder from(String from) {
            this.from = from;
            return this;
        }

        public final String getCc() {
            return cc;
        }

        public final void setCc(String cc) {
            this.cc = cc;
        }

        @Override
        public final Builder cc(String cc) {
            this.cc = cc;
            return this;
        }

        public final String getSubject() {
            return subject;
        }

        public final void setSubject(String subject) {
            this.subject = subject;
        }

        @Override
        public final Builder subject(String subject) {
            this.subject = subject;
            return this;
        }

        public final String getMessageId() {
            return messageId;
        }

        public final void setMessageId(String messageId) {
            this.messageId = messageId;
        }

        @Override
        public final Builder messageId(String messageId) {
            this.messageId = messageId;
            return this;
        }

        public final Boolean getHasAttachments() {
            return hasAttachments;
        }

        public final void setHasAttachments(Boolean hasAttachments) {
            this.hasAttachments = hasAttachments;
        }

        @Override
        public final Builder hasAttachments(Boolean hasAttachments) {
            this.hasAttachments = hasAttachments;
            return this;
        }

        public final Collection<String> getReceivedHeaders() {
            if (receivedHeaders instanceof SdkAutoConstructList) {
                return null;
            }
            return receivedHeaders;
        }

        public final void setReceivedHeaders(Collection<String> receivedHeaders) {
            this.receivedHeaders = EmailReceivedHeadersListCopier.copy(receivedHeaders);
        }

        @Override
        public final Builder receivedHeaders(Collection<String> receivedHeaders) {
            this.receivedHeaders = EmailReceivedHeadersListCopier.copy(receivedHeaders);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder receivedHeaders(String... receivedHeaders) {
            receivedHeaders(Arrays.asList(receivedHeaders));
            return this;
        }

        public final String getInReplyTo() {
            return inReplyTo;
        }

        public final void setInReplyTo(String inReplyTo) {
            this.inReplyTo = inReplyTo;
        }

        @Override
        public final Builder inReplyTo(String inReplyTo) {
            this.inReplyTo = inReplyTo;
            return this;
        }

        public final String getXMailer() {
            return xMailer;
        }

        public final void setXMailer(String xMailer) {
            this.xMailer = xMailer;
        }

        @Override
        public final Builder xMailer(String xMailer) {
            this.xMailer = xMailer;
            return this;
        }

        public final String getXOriginalMailer() {
            return xOriginalMailer;
        }

        public final void setXOriginalMailer(String xOriginalMailer) {
            this.xOriginalMailer = xOriginalMailer;
        }

        @Override
        public final Builder xOriginalMailer(String xOriginalMailer) {
            this.xOriginalMailer = xOriginalMailer;
            return this;
        }

        public final String getXPriority() {
            return xPriority;
        }

        public final void setXPriority(String xPriority) {
            this.xPriority = xPriority;
        }

        @Override
        public final Builder xPriority(String xPriority) {
            this.xPriority = xPriority;
            return this;
        }

        public final String getIngressPointId() {
            return ingressPointId;
        }

        public final void setIngressPointId(String ingressPointId) {
            this.ingressPointId = ingressPointId;
        }

        @Override
        public final Builder ingressPointId(String ingressPointId) {
            this.ingressPointId = ingressPointId;
            return this;
        }

        public final String getSenderHostname() {
            return senderHostname;
        }

        public final void setSenderHostname(String senderHostname) {
            this.senderHostname = senderHostname;
        }

        @Override
        public final Builder senderHostname(String senderHostname) {
            this.senderHostname = senderHostname;
            return this;
        }

        public final String getSenderIpAddress() {
            return senderIpAddress;
        }

        public final void setSenderIpAddress(String senderIpAddress) {
            this.senderIpAddress = senderIpAddress;
        }

        @Override
        public final Builder senderIpAddress(String senderIpAddress) {
            this.senderIpAddress = senderIpAddress;
            return this;
        }

        public final Envelope.Builder getEnvelope() {
            return envelope != null ? envelope.toBuilder() : null;
        }

        public final void setEnvelope(Envelope.BuilderImpl envelope) {
            this.envelope = envelope != null ? envelope.build() : null;
        }

        @Override
        public final Builder envelope(Envelope envelope) {
            this.envelope = envelope;
            return this;
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        @Override
        public Row build() {
            return new Row(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
