/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.managedblockchain;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.managedblockchain.model.AccessDeniedException;
import software.amazon.awssdk.services.managedblockchain.model.CreateMemberRequest;
import software.amazon.awssdk.services.managedblockchain.model.CreateMemberResponse;
import software.amazon.awssdk.services.managedblockchain.model.CreateNetworkRequest;
import software.amazon.awssdk.services.managedblockchain.model.CreateNetworkResponse;
import software.amazon.awssdk.services.managedblockchain.model.CreateNodeRequest;
import software.amazon.awssdk.services.managedblockchain.model.CreateNodeResponse;
import software.amazon.awssdk.services.managedblockchain.model.CreateProposalRequest;
import software.amazon.awssdk.services.managedblockchain.model.CreateProposalResponse;
import software.amazon.awssdk.services.managedblockchain.model.DeleteMemberRequest;
import software.amazon.awssdk.services.managedblockchain.model.DeleteMemberResponse;
import software.amazon.awssdk.services.managedblockchain.model.DeleteNodeRequest;
import software.amazon.awssdk.services.managedblockchain.model.DeleteNodeResponse;
import software.amazon.awssdk.services.managedblockchain.model.GetMemberRequest;
import software.amazon.awssdk.services.managedblockchain.model.GetMemberResponse;
import software.amazon.awssdk.services.managedblockchain.model.GetNetworkRequest;
import software.amazon.awssdk.services.managedblockchain.model.GetNetworkResponse;
import software.amazon.awssdk.services.managedblockchain.model.GetNodeRequest;
import software.amazon.awssdk.services.managedblockchain.model.GetNodeResponse;
import software.amazon.awssdk.services.managedblockchain.model.GetProposalRequest;
import software.amazon.awssdk.services.managedblockchain.model.GetProposalResponse;
import software.amazon.awssdk.services.managedblockchain.model.IllegalActionException;
import software.amazon.awssdk.services.managedblockchain.model.InternalServiceErrorException;
import software.amazon.awssdk.services.managedblockchain.model.InvalidRequestException;
import software.amazon.awssdk.services.managedblockchain.model.ListInvitationsRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListInvitationsResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListMembersRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListMembersResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListNetworksRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListNetworksResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListNodesRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListNodesResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListProposalVotesRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListProposalVotesResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListProposalsRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListProposalsResponse;
import software.amazon.awssdk.services.managedblockchain.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.managedblockchain.model.ManagedBlockchainException;
import software.amazon.awssdk.services.managedblockchain.model.ManagedBlockchainRequest;
import software.amazon.awssdk.services.managedblockchain.model.RejectInvitationRequest;
import software.amazon.awssdk.services.managedblockchain.model.RejectInvitationResponse;
import software.amazon.awssdk.services.managedblockchain.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.managedblockchain.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.managedblockchain.model.ResourceNotFoundException;
import software.amazon.awssdk.services.managedblockchain.model.ResourceNotReadyException;
import software.amazon.awssdk.services.managedblockchain.model.TagResourceRequest;
import software.amazon.awssdk.services.managedblockchain.model.TagResourceResponse;
import software.amazon.awssdk.services.managedblockchain.model.ThrottlingException;
import software.amazon.awssdk.services.managedblockchain.model.TooManyTagsException;
import software.amazon.awssdk.services.managedblockchain.model.UntagResourceRequest;
import software.amazon.awssdk.services.managedblockchain.model.UntagResourceResponse;
import software.amazon.awssdk.services.managedblockchain.model.UpdateMemberRequest;
import software.amazon.awssdk.services.managedblockchain.model.UpdateMemberResponse;
import software.amazon.awssdk.services.managedblockchain.model.UpdateNodeRequest;
import software.amazon.awssdk.services.managedblockchain.model.UpdateNodeResponse;
import software.amazon.awssdk.services.managedblockchain.model.VoteOnProposalRequest;
import software.amazon.awssdk.services.managedblockchain.model.VoteOnProposalResponse;
import software.amazon.awssdk.services.managedblockchain.paginators.ListInvitationsIterable;
import software.amazon.awssdk.services.managedblockchain.paginators.ListMembersIterable;
import software.amazon.awssdk.services.managedblockchain.paginators.ListNetworksIterable;
import software.amazon.awssdk.services.managedblockchain.paginators.ListNodesIterable;
import software.amazon.awssdk.services.managedblockchain.paginators.ListProposalVotesIterable;
import software.amazon.awssdk.services.managedblockchain.paginators.ListProposalsIterable;
import software.amazon.awssdk.services.managedblockchain.transform.CreateMemberRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.CreateNetworkRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.CreateNodeRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.CreateProposalRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.DeleteMemberRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.DeleteNodeRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.GetMemberRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.GetNetworkRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.GetNodeRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.GetProposalRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListInvitationsRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListMembersRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListNetworksRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListNodesRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListProposalVotesRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListProposalsRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.RejectInvitationRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.UpdateMemberRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.UpdateNodeRequestMarshaller;
import software.amazon.awssdk.services.managedblockchain.transform.VoteOnProposalRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ManagedBlockchainClient}.
 *
 * @see ManagedBlockchainClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultManagedBlockchainClient implements ManagedBlockchainClient {
    private static final Logger log = Logger.loggerFor(DefaultManagedBlockchainClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultManagedBlockchainClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a member within a Managed Blockchain network.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param createMemberRequest
     * @return Result of the CreateMember operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ResourceAlreadyExistsException
     *         A resource request is issued for a resource that already exists.
     * @throws ResourceNotReadyException
     *         The requested resource exists but is not in a status that can complete the operation.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws ResourceLimitExceededException
     *         The maximum number of resources of that type already exist. Ensure the resources requested are within the
     *         boundaries of the service edition and your account limits.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws TooManyTagsException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.CreateMember
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/CreateMember" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateMemberResponse createMember(CreateMemberRequest createMemberRequest) throws InvalidRequestException,
            AccessDeniedException, ResourceNotFoundException, ResourceAlreadyExistsException, ResourceNotReadyException,
            ThrottlingException, ResourceLimitExceededException, InternalServiceErrorException, TooManyTagsException,
            AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMemberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateMemberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMember");

            return clientHandler.execute(new ClientExecutionParams<CreateMemberRequest, CreateMemberResponse>()
                    .withOperationName("CreateMember").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createMemberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateMemberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new blockchain network using Amazon Managed Blockchain.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param createNetworkRequest
     * @return Result of the CreateNetwork operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceAlreadyExistsException
     *         A resource request is issued for a resource that already exists.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws ResourceLimitExceededException
     *         The maximum number of resources of that type already exist. Ensure the resources requested are within the
     *         boundaries of the service edition and your account limits.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws TooManyTagsException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.CreateNetwork
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/CreateNetwork"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateNetworkResponse createNetwork(CreateNetworkRequest createNetworkRequest) throws InvalidRequestException,
            AccessDeniedException, ResourceAlreadyExistsException, ThrottlingException, ResourceLimitExceededException,
            InternalServiceErrorException, TooManyTagsException, AwsServiceException, SdkClientException,
            ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateNetworkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNetwork");

            return clientHandler.execute(new ClientExecutionParams<CreateNetworkRequest, CreateNetworkResponse>()
                    .withOperationName("CreateNetwork").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createNetworkRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateNetworkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a node on the specified blockchain network.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     * <p>
     * Ethereum on Managed Blockchain is in preview release and is subject to change.
     * </p>
     *
     * @param createNodeRequest
     * @return Result of the CreateNode operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ResourceAlreadyExistsException
     *         A resource request is issued for a resource that already exists.
     * @throws ResourceNotReadyException
     *         The requested resource exists but is not in a status that can complete the operation.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws ResourceLimitExceededException
     *         The maximum number of resources of that type already exist. Ensure the resources requested are within the
     *         boundaries of the service edition and your account limits.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws TooManyTagsException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.CreateNode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/CreateNode" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateNodeResponse createNode(CreateNodeRequest createNodeRequest) throws InvalidRequestException,
            AccessDeniedException, ResourceNotFoundException, ResourceAlreadyExistsException, ResourceNotReadyException,
            ThrottlingException, ResourceLimitExceededException, InternalServiceErrorException, TooManyTagsException,
            AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateNodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNode");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateNodeRequest, CreateNodeResponse>().withOperationName("CreateNode")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createNodeRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateNodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a proposal for a change to the network that other members of the network can vote on, for example, a
     * proposal to add a new member to the network. Any member can create a proposal.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param createProposalRequest
     * @return Result of the CreateProposal operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ResourceNotReadyException
     *         The requested resource exists but is not in a status that can complete the operation.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws TooManyTagsException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.CreateProposal
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/CreateProposal"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateProposalResponse createProposal(CreateProposalRequest createProposalRequest) throws InvalidRequestException,
            AccessDeniedException, ResourceNotFoundException, ResourceNotReadyException, ThrottlingException,
            InternalServiceErrorException, TooManyTagsException, AwsServiceException, SdkClientException,
            ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProposalResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProposalResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProposalRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProposal");

            return clientHandler.execute(new ClientExecutionParams<CreateProposalRequest, CreateProposalResponse>()
                    .withOperationName("CreateProposal").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createProposalRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateProposalRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a member. Deleting a member removes the member and all associated resources from the network.
     * <code>DeleteMember</code> can only be called for a specified <code>MemberId</code> if the principal performing
     * the action is associated with the AWS account that owns the member. In all other cases, the
     * <code>DeleteMember</code> action is carried out as the result of an approved proposal to remove a member. If
     * <code>MemberId</code> is the last member in a network specified by the last AWS account, the network is deleted
     * also.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param deleteMemberRequest
     * @return Result of the DeleteMember operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ResourceNotReadyException
     *         The requested resource exists but is not in a status that can complete the operation.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.DeleteMember
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/DeleteMember" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteMemberResponse deleteMember(DeleteMemberRequest deleteMemberRequest) throws InvalidRequestException,
            AccessDeniedException, ResourceNotFoundException, ResourceNotReadyException, ThrottlingException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMemberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteMemberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMember");

            return clientHandler.execute(new ClientExecutionParams<DeleteMemberRequest, DeleteMemberResponse>()
                    .withOperationName("DeleteMember").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteMemberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteMemberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a node that your AWS account owns. All data on the node is lost and cannot be recovered.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     *
     * @param deleteNodeRequest
     * @return Result of the DeleteNode operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ResourceNotReadyException
     *         The requested resource exists but is not in a status that can complete the operation.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.DeleteNode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/DeleteNode" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteNodeResponse deleteNode(DeleteNodeRequest deleteNodeRequest) throws InvalidRequestException,
            AccessDeniedException, ResourceNotFoundException, ResourceNotReadyException, ThrottlingException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteNodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNode");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteNodeRequest, DeleteNodeResponse>().withOperationName("DeleteNode")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteNodeRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteNodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns detailed information about a member.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param getMemberRequest
     * @return Result of the GetMember operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.GetMember
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/GetMember" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMemberResponse getMember(GetMemberRequest getMemberRequest) throws InvalidRequestException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMemberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMemberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMember");

            return clientHandler.execute(new ClientExecutionParams<GetMemberRequest, GetMemberResponse>()
                    .withOperationName("GetMember").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMemberRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetMemberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns detailed information about a network.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     *
     * @param getNetworkRequest
     * @return Result of the GetNetwork operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.GetNetwork
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/GetNetwork" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetNetworkResponse getNetwork(GetNetworkRequest getNetworkRequest) throws InvalidRequestException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetNetworkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNetwork");

            return clientHandler
                    .execute(new ClientExecutionParams<GetNetworkRequest, GetNetworkResponse>().withOperationName("GetNetwork")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getNetworkRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetNetworkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns detailed information about a node.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     *
     * @param getNodeRequest
     * @return Result of the GetNode operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.GetNode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/GetNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetNodeResponse getNode(GetNodeRequest getNodeRequest) throws InvalidRequestException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetNodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNode");

            return clientHandler.execute(new ClientExecutionParams<GetNodeRequest, GetNodeResponse>()
                    .withOperationName("GetNode").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getNodeRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetNodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns detailed information about a proposal.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param getProposalRequest
     * @return Result of the GetProposal operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.GetProposal
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/GetProposal" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetProposalResponse getProposal(GetProposalRequest getProposalRequest) throws InvalidRequestException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetProposalResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetProposalResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProposalRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProposal");

            return clientHandler.execute(new ClientExecutionParams<GetProposalRequest, GetProposalResponse>()
                    .withOperationName("GetProposal").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getProposalRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetProposalRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all invitations for the current AWS account.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param listInvitationsRequest
     * @return Result of the ListInvitations operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws ResourceLimitExceededException
     *         The maximum number of resources of that type already exist. Ensure the resources requested are within the
     *         boundaries of the service edition and your account limits.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInvitationsResponse listInvitations(ListInvitationsRequest listInvitationsRequest) throws InvalidRequestException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, ResourceLimitExceededException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInvitationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvitations");

            return clientHandler.execute(new ClientExecutionParams<ListInvitationsRequest, ListInvitationsResponse>()
                    .withOperationName("ListInvitations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listInvitationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInvitationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all invitations for the current AWS account.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInvitations(software.amazon.awssdk.services.managedblockchain.model.ListInvitationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListInvitationsIterable responses = client.listInvitationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.managedblockchain.paginators.ListInvitationsIterable responses = client
     *             .listInvitationsPaginator(request);
     *     for (software.amazon.awssdk.services.managedblockchain.model.ListInvitationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListInvitationsIterable responses = client.listInvitationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInvitations(software.amazon.awssdk.services.managedblockchain.model.ListInvitationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listInvitationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws ResourceLimitExceededException
     *         The maximum number of resources of that type already exist. Ensure the resources requested are within the
     *         boundaries of the service edition and your account limits.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInvitationsIterable listInvitationsPaginator(ListInvitationsRequest listInvitationsRequest)
            throws InvalidRequestException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            ResourceLimitExceededException, InternalServiceErrorException, AwsServiceException, SdkClientException,
            ManagedBlockchainException {
        return new ListInvitationsIterable(this, applyPaginatorUserAgent(listInvitationsRequest));
    }

    /**
     * <p>
     * Returns a list of the members in a network and properties of their configurations.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param listMembersRequest
     * @return Result of the ListMembers operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListMembers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListMembersResponse listMembers(ListMembersRequest listMembersRequest) throws InvalidRequestException,
            AccessDeniedException, ThrottlingException, InternalServiceErrorException, AwsServiceException, SdkClientException,
            ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMembers");

            return clientHandler.execute(new ClientExecutionParams<ListMembersRequest, ListMembersResponse>()
                    .withOperationName("ListMembers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listMembersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMembersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the members in a network and properties of their configurations.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMembers(software.amazon.awssdk.services.managedblockchain.model.ListMembersRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListMembersIterable responses = client.listMembersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.managedblockchain.paginators.ListMembersIterable responses = client
     *             .listMembersPaginator(request);
     *     for (software.amazon.awssdk.services.managedblockchain.model.ListMembersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListMembersIterable responses = client.listMembersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMembers(software.amazon.awssdk.services.managedblockchain.model.ListMembersRequest)} operation.</b>
     * </p>
     *
     * @param listMembersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListMembers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListMembersIterable listMembersPaginator(ListMembersRequest listMembersRequest) throws InvalidRequestException,
            AccessDeniedException, ThrottlingException, InternalServiceErrorException, AwsServiceException, SdkClientException,
            ManagedBlockchainException {
        return new ListMembersIterable(this, applyPaginatorUserAgent(listMembersRequest));
    }

    /**
     * <p>
     * Returns information about the networks in which the current AWS account participates.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     *
     * @param listNetworksRequest
     * @return Result of the ListNetworks operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListNetworks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListNetworks" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListNetworksResponse listNetworks(ListNetworksRequest listNetworksRequest) throws InvalidRequestException,
            AccessDeniedException, ThrottlingException, InternalServiceErrorException, AwsServiceException, SdkClientException,
            ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNetworksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListNetworksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNetworksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNetworks");

            return clientHandler.execute(new ClientExecutionParams<ListNetworksRequest, ListNetworksResponse>()
                    .withOperationName("ListNetworks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listNetworksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListNetworksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the networks in which the current AWS account participates.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNetworks(software.amazon.awssdk.services.managedblockchain.model.ListNetworksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListNetworksIterable responses = client.listNetworksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.managedblockchain.paginators.ListNetworksIterable responses = client
     *             .listNetworksPaginator(request);
     *     for (software.amazon.awssdk.services.managedblockchain.model.ListNetworksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListNetworksIterable responses = client.listNetworksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNetworks(software.amazon.awssdk.services.managedblockchain.model.ListNetworksRequest)} operation.</b>
     * </p>
     *
     * @param listNetworksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListNetworks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListNetworks" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListNetworksIterable listNetworksPaginator(ListNetworksRequest listNetworksRequest) throws InvalidRequestException,
            AccessDeniedException, ThrottlingException, InternalServiceErrorException, AwsServiceException, SdkClientException,
            ManagedBlockchainException {
        return new ListNetworksIterable(this, applyPaginatorUserAgent(listNetworksRequest));
    }

    /**
     * <p>
     * Returns information about the nodes within a network.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     *
     * @param listNodesRequest
     * @return Result of the ListNodes operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListNodes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListNodes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListNodesResponse listNodes(ListNodesRequest listNodesRequest) throws InvalidRequestException, AccessDeniedException,
            ThrottlingException, InternalServiceErrorException, AwsServiceException, SdkClientException,
            ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNodesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListNodesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNodesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNodes");

            return clientHandler.execute(new ClientExecutionParams<ListNodesRequest, ListNodesResponse>()
                    .withOperationName("ListNodes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listNodesRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListNodesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the nodes within a network.
     * </p>
     * <p>
     * Applies to Hyperledger Fabric and Ethereum.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listNodes(software.amazon.awssdk.services.managedblockchain.model.ListNodesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListNodesIterable responses = client.listNodesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.managedblockchain.paginators.ListNodesIterable responses = client.listNodesPaginator(request);
     *     for (software.amazon.awssdk.services.managedblockchain.model.ListNodesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListNodesIterable responses = client.listNodesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNodes(software.amazon.awssdk.services.managedblockchain.model.ListNodesRequest)} operation.</b>
     * </p>
     *
     * @param listNodesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListNodes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListNodes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListNodesIterable listNodesPaginator(ListNodesRequest listNodesRequest) throws InvalidRequestException,
            AccessDeniedException, ThrottlingException, InternalServiceErrorException, AwsServiceException, SdkClientException,
            ManagedBlockchainException {
        return new ListNodesIterable(this, applyPaginatorUserAgent(listNodesRequest));
    }

    /**
     * <p>
     * Returns the list of votes for a specified proposal, including the value of each vote and the unique identifier of
     * the member that cast the vote.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param listProposalVotesRequest
     * @return Result of the ListProposalVotes operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListProposalVotes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListProposalVotes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProposalVotesResponse listProposalVotes(ListProposalVotesRequest listProposalVotesRequest)
            throws InvalidRequestException, AccessDeniedException, ThrottlingException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProposalVotesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProposalVotesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProposalVotesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProposalVotes");

            return clientHandler.execute(new ClientExecutionParams<ListProposalVotesRequest, ListProposalVotesResponse>()
                    .withOperationName("ListProposalVotes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listProposalVotesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProposalVotesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the list of votes for a specified proposal, including the value of each vote and the unique identifier of
     * the member that cast the vote.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProposalVotes(software.amazon.awssdk.services.managedblockchain.model.ListProposalVotesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListProposalVotesIterable responses = client.listProposalVotesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.managedblockchain.paginators.ListProposalVotesIterable responses = client
     *             .listProposalVotesPaginator(request);
     *     for (software.amazon.awssdk.services.managedblockchain.model.ListProposalVotesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListProposalVotesIterable responses = client.listProposalVotesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProposalVotes(software.amazon.awssdk.services.managedblockchain.model.ListProposalVotesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProposalVotesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListProposalVotes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListProposalVotes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProposalVotesIterable listProposalVotesPaginator(ListProposalVotesRequest listProposalVotesRequest)
            throws InvalidRequestException, AccessDeniedException, ThrottlingException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, ManagedBlockchainException {
        return new ListProposalVotesIterable(this, applyPaginatorUserAgent(listProposalVotesRequest));
    }

    /**
     * <p>
     * Returns a list of proposals for the network.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param listProposalsRequest
     * @return Result of the ListProposals operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListProposals
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListProposals"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProposalsResponse listProposals(ListProposalsRequest listProposalsRequest) throws InvalidRequestException,
            ResourceNotFoundException, AccessDeniedException, ThrottlingException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProposalsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProposalsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProposalsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProposals");

            return clientHandler.execute(new ClientExecutionParams<ListProposalsRequest, ListProposalsResponse>()
                    .withOperationName("ListProposals").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listProposalsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProposalsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of proposals for the network.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProposals(software.amazon.awssdk.services.managedblockchain.model.ListProposalsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListProposalsIterable responses = client.listProposalsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.managedblockchain.paginators.ListProposalsIterable responses = client
     *             .listProposalsPaginator(request);
     *     for (software.amazon.awssdk.services.managedblockchain.model.ListProposalsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.managedblockchain.paginators.ListProposalsIterable responses = client.listProposalsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProposals(software.amazon.awssdk.services.managedblockchain.model.ListProposalsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProposalsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListProposals
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListProposals"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProposalsIterable listProposalsPaginator(ListProposalsRequest listProposalsRequest)
            throws InvalidRequestException, ResourceNotFoundException, AccessDeniedException, ThrottlingException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, ManagedBlockchainException {
        return new ListProposalsIterable(this, applyPaginatorUserAgent(listProposalsRequest));
    }

    /**
     * <p>
     * Returns a list of tags for the specified resource. Each tag consists of a key and optional value.
     * </p>
     * <p>
     * For more information about tags, see <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
     * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
     * >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ResourceNotReadyException
     *         The requested resource exists but is not in a status that can complete the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServiceErrorException, InvalidRequestException, ResourceNotFoundException, ResourceNotReadyException,
            AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Rejects an invitation to join a network. This action can be called by a principal in an AWS account that has
     * received an invitation to create a member and join a network.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param rejectInvitationRequest
     * @return Result of the RejectInvitation operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws IllegalActionException
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.RejectInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/RejectInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RejectInvitationResponse rejectInvitation(RejectInvitationRequest rejectInvitationRequest)
            throws InvalidRequestException, IllegalActionException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, InternalServiceErrorException, AwsServiceException, SdkClientException,
            ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RejectInvitationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RejectInvitationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectInvitationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectInvitation");

            return clientHandler.execute(new ClientExecutionParams<RejectInvitationRequest, RejectInvitationResponse>()
                    .withOperationName("RejectInvitation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(rejectInvitationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RejectInvitationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or overwrites the specified tags for the specified Amazon Managed Blockchain resource. Each tag consists of
     * a key and optional value.
     * </p>
     * <p>
     * When you specify a tag key that already exists, the tag value is overwritten with the new value. Use
     * <code>UntagResource</code> to remove tag keys.
     * </p>
     * <p>
     * A resource can have up to 50 tags. If you try to create more than 50 tags for a resource, your request fails and
     * returns an error.
     * </p>
     * <p>
     * For more information about tags, see <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
     * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
     * >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws TooManyTagsException
     * @throws ResourceNotReadyException
     *         The requested resource exists but is not in a status that can complete the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServiceErrorException,
            InvalidRequestException, ResourceNotFoundException, TooManyTagsException, ResourceNotReadyException,
            AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the Amazon Managed Blockchain resource.
     * </p>
     * <p>
     * For more information about tags, see <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
     * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
     * >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ResourceNotReadyException
     *         The requested resource exists but is not in a status that can complete the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServiceErrorException,
            InvalidRequestException, ResourceNotFoundException, ResourceNotReadyException, AwsServiceException,
            SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a member configuration with new parameters.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param updateMemberRequest
     * @return Result of the UpdateMember operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.UpdateMember
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/UpdateMember" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateMemberResponse updateMember(UpdateMemberRequest updateMemberRequest) throws InvalidRequestException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateMemberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateMemberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMember");

            return clientHandler.execute(new ClientExecutionParams<UpdateMemberRequest, UpdateMemberResponse>()
                    .withOperationName("UpdateMember").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateMemberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateMemberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a node configuration with new parameters.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param updateNodeRequest
     * @return Result of the UpdateNode operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.UpdateNode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/UpdateNode" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateNodeResponse updateNode(UpdateNodeRequest updateNodeRequest) throws InvalidRequestException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateNodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNode");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateNodeRequest, UpdateNodeResponse>().withOperationName("UpdateNode")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateNodeRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateNodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Casts a vote for a specified <code>ProposalId</code> on behalf of a member. The member to vote as, specified by
     * <code>VoterMemberId</code>, must be in the same AWS account as the principal that calls the action.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     *
     * @param voteOnProposalRequest
     * @return Result of the VoteOnProposal operation returned by the service.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws IllegalActionException
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         A requested resource does not exist. It may have been deleted or referenced inaccurately.
     * @throws ThrottlingException
     *         The request or operation could not be performed because a service is throttling requests. The most common
     *         source of throttling errors is launching EC2 instances such that your service limit for EC2 instances is
     *         exceeded. Request a limit increase or delete unused resources if possible.
     * @throws InternalServiceErrorException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainClient.VoteOnProposal
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/VoteOnProposal"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public VoteOnProposalResponse voteOnProposal(VoteOnProposalRequest voteOnProposalRequest) throws InvalidRequestException,
            IllegalActionException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, ManagedBlockchainException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<VoteOnProposalResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                VoteOnProposalResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, voteOnProposalRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "VoteOnProposal");

            return clientHandler.execute(new ClientExecutionParams<VoteOnProposalRequest, VoteOnProposalResponse>()
                    .withOperationName("VoteOnProposal").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(voteOnProposalRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new VoteOnProposalRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ManagedBlockchainException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotReadyException")
                                .exceptionBuilderSupplier(ResourceNotReadyException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IllegalActionException")
                                .exceptionBuilderSupplier(IllegalActionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(429).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ManagedBlockchainRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
