/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.managedblockchain.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Properties of a proposal on a Managed Blockchain network.
 * </p>
 * <p>
 * Applies only to Hyperledger Fabric.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Proposal implements SdkPojo, Serializable, ToCopyableBuilder<Proposal.Builder, Proposal> {
    private static final SdkField<String> PROPOSAL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProposalId").getter(getter(Proposal::proposalId)).setter(setter(Builder::proposalId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProposalId").build()).build();

    private static final SdkField<String> NETWORK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NetworkId").getter(getter(Proposal::networkId)).setter(setter(Builder::networkId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Proposal::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<ProposalActions> ACTIONS_FIELD = SdkField.<ProposalActions> builder(MarshallingType.SDK_POJO)
            .memberName("Actions").getter(getter(Proposal::actions)).setter(setter(Builder::actions))
            .constructor(ProposalActions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Actions").build()).build();

    private static final SdkField<String> PROPOSED_BY_MEMBER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProposedByMemberId").getter(getter(Proposal::proposedByMemberId))
            .setter(setter(Builder::proposedByMemberId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProposedByMemberId").build())
            .build();

    private static final SdkField<String> PROPOSED_BY_MEMBER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProposedByMemberName").getter(getter(Proposal::proposedByMemberName))
            .setter(setter(Builder::proposedByMemberName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProposedByMemberName").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Proposal::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate")
            .getter(getter(Proposal::creationDate))
            .setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> EXPIRATION_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ExpirationDate")
            .getter(getter(Proposal::expirationDate))
            .setter(setter(Builder::expirationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpirationDate").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Integer> YES_VOTE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("YesVoteCount").getter(getter(Proposal::yesVoteCount)).setter(setter(Builder::yesVoteCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("YesVoteCount").build()).build();

    private static final SdkField<Integer> NO_VOTE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NoVoteCount").getter(getter(Proposal::noVoteCount)).setter(setter(Builder::noVoteCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NoVoteCount").build()).build();

    private static final SdkField<Integer> OUTSTANDING_VOTE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("OutstandingVoteCount").getter(getter(Proposal::outstandingVoteCount))
            .setter(setter(Builder::outstandingVoteCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutstandingVoteCount").build())
            .build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(Proposal::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Proposal::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PROPOSAL_ID_FIELD,
            NETWORK_ID_FIELD, DESCRIPTION_FIELD, ACTIONS_FIELD, PROPOSED_BY_MEMBER_ID_FIELD, PROPOSED_BY_MEMBER_NAME_FIELD,
            STATUS_FIELD, CREATION_DATE_FIELD, EXPIRATION_DATE_FIELD, YES_VOTE_COUNT_FIELD, NO_VOTE_COUNT_FIELD,
            OUTSTANDING_VOTE_COUNT_FIELD, TAGS_FIELD, ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String proposalId;

    private final String networkId;

    private final String description;

    private final ProposalActions actions;

    private final String proposedByMemberId;

    private final String proposedByMemberName;

    private final String status;

    private final Instant creationDate;

    private final Instant expirationDate;

    private final Integer yesVoteCount;

    private final Integer noVoteCount;

    private final Integer outstandingVoteCount;

    private final Map<String, String> tags;

    private final String arn;

    private Proposal(BuilderImpl builder) {
        this.proposalId = builder.proposalId;
        this.networkId = builder.networkId;
        this.description = builder.description;
        this.actions = builder.actions;
        this.proposedByMemberId = builder.proposedByMemberId;
        this.proposedByMemberName = builder.proposedByMemberName;
        this.status = builder.status;
        this.creationDate = builder.creationDate;
        this.expirationDate = builder.expirationDate;
        this.yesVoteCount = builder.yesVoteCount;
        this.noVoteCount = builder.noVoteCount;
        this.outstandingVoteCount = builder.outstandingVoteCount;
        this.tags = builder.tags;
        this.arn = builder.arn;
    }

    /**
     * <p>
     * The unique identifier of the proposal.
     * </p>
     * 
     * @return The unique identifier of the proposal.
     */
    public final String proposalId() {
        return proposalId;
    }

    /**
     * <p>
     * The unique identifier of the network for which the proposal is made.
     * </p>
     * 
     * @return The unique identifier of the network for which the proposal is made.
     */
    public final String networkId() {
        return networkId;
    }

    /**
     * <p>
     * The description of the proposal.
     * </p>
     * 
     * @return The description of the proposal.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The actions to perform on the network if the proposal is <code>APPROVED</code>.
     * </p>
     * 
     * @return The actions to perform on the network if the proposal is <code>APPROVED</code>.
     */
    public final ProposalActions actions() {
        return actions;
    }

    /**
     * <p>
     * The unique identifier of the member that created the proposal.
     * </p>
     * 
     * @return The unique identifier of the member that created the proposal.
     */
    public final String proposedByMemberId() {
        return proposedByMemberId;
    }

    /**
     * <p>
     * The name of the member that created the proposal.
     * </p>
     * 
     * @return The name of the member that created the proposal.
     */
    public final String proposedByMemberName() {
        return proposedByMemberName;
    }

    /**
     * <p>
     * The status of the proposal. Values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code> - The proposal is active and open for member voting.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>APPROVED</code> - The proposal was approved with sufficient <code>YES</code> votes among members according
     * to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified proposal actions are
     * carried out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REJECTED</code> - The proposal was rejected with insufficient <code>YES</code> votes among members
     * according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified
     * <code>ProposalActions</code> are not carried out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EXPIRED</code> - Members did not cast the number of votes required to determine the proposal outcome before
     * the proposal expired. The specified <code>ProposalActions</code> are not carried out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTION_FAILED</code> - One or more of the specified <code>ProposalActions</code> in a proposal that was
     * approved could not be completed because of an error. The <code>ACTION_FAILED</code> status occurs even if only
     * one ProposalAction fails and other actions are successful.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ProposalStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the proposal. Values are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code> - The proposal is active and open for member voting.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>APPROVED</code> - The proposal was approved with sufficient <code>YES</code> votes among members
     *         according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified proposal
     *         actions are carried out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REJECTED</code> - The proposal was rejected with insufficient <code>YES</code> votes among members
     *         according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified
     *         <code>ProposalActions</code> are not carried out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EXPIRED</code> - Members did not cast the number of votes required to determine the proposal
     *         outcome before the proposal expired. The specified <code>ProposalActions</code> are not carried out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTION_FAILED</code> - One or more of the specified <code>ProposalActions</code> in a proposal that
     *         was approved could not be completed because of an error. The <code>ACTION_FAILED</code> status occurs
     *         even if only one ProposalAction fails and other actions are successful.
     *         </p>
     *         </li>
     * @see ProposalStatus
     */
    public final ProposalStatus status() {
        return ProposalStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the proposal. Values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code> - The proposal is active and open for member voting.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>APPROVED</code> - The proposal was approved with sufficient <code>YES</code> votes among members according
     * to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified proposal actions are
     * carried out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REJECTED</code> - The proposal was rejected with insufficient <code>YES</code> votes among members
     * according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified
     * <code>ProposalActions</code> are not carried out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EXPIRED</code> - Members did not cast the number of votes required to determine the proposal outcome before
     * the proposal expired. The specified <code>ProposalActions</code> are not carried out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTION_FAILED</code> - One or more of the specified <code>ProposalActions</code> in a proposal that was
     * approved could not be completed because of an error. The <code>ACTION_FAILED</code> status occurs even if only
     * one ProposalAction fails and other actions are successful.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ProposalStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the proposal. Values are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code> - The proposal is active and open for member voting.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>APPROVED</code> - The proposal was approved with sufficient <code>YES</code> votes among members
     *         according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified proposal
     *         actions are carried out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REJECTED</code> - The proposal was rejected with insufficient <code>YES</code> votes among members
     *         according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified
     *         <code>ProposalActions</code> are not carried out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EXPIRED</code> - Members did not cast the number of votes required to determine the proposal
     *         outcome before the proposal expired. The specified <code>ProposalActions</code> are not carried out.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTION_FAILED</code> - One or more of the specified <code>ProposalActions</code> in a proposal that
     *         was approved could not be completed because of an error. The <code>ACTION_FAILED</code> status occurs
     *         even if only one ProposalAction fails and other actions are successful.
     *         </p>
     *         </li>
     * @see ProposalStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The date and time that the proposal was created.
     * </p>
     * 
     * @return The date and time that the proposal was created.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The date and time that the proposal expires. This is the <code>CreationDate</code> plus the
     * <code>ProposalDurationInHours</code> that is specified in the <code>ProposalThresholdPolicy</code>. After this
     * date and time, if members have not cast enough votes to determine the outcome according to the voting policy, the
     * proposal is <code>EXPIRED</code> and <code>Actions</code> are not carried out.
     * </p>
     * 
     * @return The date and time that the proposal expires. This is the <code>CreationDate</code> plus the
     *         <code>ProposalDurationInHours</code> that is specified in the <code>ProposalThresholdPolicy</code>. After
     *         this date and time, if members have not cast enough votes to determine the outcome according to the
     *         voting policy, the proposal is <code>EXPIRED</code> and <code>Actions</code> are not carried out.
     */
    public final Instant expirationDate() {
        return expirationDate;
    }

    /**
     * <p>
     * The current total of <code>YES</code> votes cast on the proposal by members.
     * </p>
     * 
     * @return The current total of <code>YES</code> votes cast on the proposal by members.
     */
    public final Integer yesVoteCount() {
        return yesVoteCount;
    }

    /**
     * <p>
     * The current total of <code>NO</code> votes cast on the proposal by members.
     * </p>
     * 
     * @return The current total of <code>NO</code> votes cast on the proposal by members.
     */
    public final Integer noVoteCount() {
        return noVoteCount;
    }

    /**
     * <p>
     * The number of votes remaining to be cast on the proposal by members. In other words, the number of members minus
     * the sum of <code>YES</code> votes and <code>NO</code> votes.
     * </p>
     * 
     * @return The number of votes remaining to be cast on the proposal by members. In other words, the number of
     *         members minus the sum of <code>YES</code> votes and <code>NO</code> votes.
     */
    public final Integer outstandingVoteCount() {
        return outstandingVoteCount;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Tags assigned to the proposal. Each tag consists of a key and optional value.
     * </p>
     * <p>
     * For more information about tags, see <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
     * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
     * >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Tags assigned to the proposal. Each tag consists of a key and optional value.</p>
     *         <p>
     *         For more information about tags, see <a
     *         href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
     *         Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a href=
     *         "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
     *         >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the proposal. For more information about ARNs and their format, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     * (ARNs)</a> in the <i>AWS General Reference</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the proposal. For more information about ARNs and their format, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     *         (ARNs)</a> in the <i>AWS General Reference</i>.
     */
    public final String arn() {
        return arn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(proposalId());
        hashCode = 31 * hashCode + Objects.hashCode(networkId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(actions());
        hashCode = 31 * hashCode + Objects.hashCode(proposedByMemberId());
        hashCode = 31 * hashCode + Objects.hashCode(proposedByMemberName());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(expirationDate());
        hashCode = 31 * hashCode + Objects.hashCode(yesVoteCount());
        hashCode = 31 * hashCode + Objects.hashCode(noVoteCount());
        hashCode = 31 * hashCode + Objects.hashCode(outstandingVoteCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Proposal)) {
            return false;
        }
        Proposal other = (Proposal) obj;
        return Objects.equals(proposalId(), other.proposalId()) && Objects.equals(networkId(), other.networkId())
                && Objects.equals(description(), other.description()) && Objects.equals(actions(), other.actions())
                && Objects.equals(proposedByMemberId(), other.proposedByMemberId())
                && Objects.equals(proposedByMemberName(), other.proposedByMemberName())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(creationDate(), other.creationDate())
                && Objects.equals(expirationDate(), other.expirationDate())
                && Objects.equals(yesVoteCount(), other.yesVoteCount()) && Objects.equals(noVoteCount(), other.noVoteCount())
                && Objects.equals(outstandingVoteCount(), other.outstandingVoteCount()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(arn(), other.arn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Proposal").add("ProposalId", proposalId()).add("NetworkId", networkId())
                .add("Description", description()).add("Actions", actions()).add("ProposedByMemberId", proposedByMemberId())
                .add("ProposedByMemberName", proposedByMemberName()).add("Status", statusAsString())
                .add("CreationDate", creationDate()).add("ExpirationDate", expirationDate()).add("YesVoteCount", yesVoteCount())
                .add("NoVoteCount", noVoteCount()).add("OutstandingVoteCount", outstandingVoteCount())
                .add("Tags", hasTags() ? tags() : null).add("Arn", arn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ProposalId":
            return Optional.ofNullable(clazz.cast(proposalId()));
        case "NetworkId":
            return Optional.ofNullable(clazz.cast(networkId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Actions":
            return Optional.ofNullable(clazz.cast(actions()));
        case "ProposedByMemberId":
            return Optional.ofNullable(clazz.cast(proposedByMemberId()));
        case "ProposedByMemberName":
            return Optional.ofNullable(clazz.cast(proposedByMemberName()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "ExpirationDate":
            return Optional.ofNullable(clazz.cast(expirationDate()));
        case "YesVoteCount":
            return Optional.ofNullable(clazz.cast(yesVoteCount()));
        case "NoVoteCount":
            return Optional.ofNullable(clazz.cast(noVoteCount()));
        case "OutstandingVoteCount":
            return Optional.ofNullable(clazz.cast(outstandingVoteCount()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Proposal, T> g) {
        return obj -> g.apply((Proposal) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Proposal> {
        /**
         * <p>
         * The unique identifier of the proposal.
         * </p>
         * 
         * @param proposalId
         *        The unique identifier of the proposal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder proposalId(String proposalId);

        /**
         * <p>
         * The unique identifier of the network for which the proposal is made.
         * </p>
         * 
         * @param networkId
         *        The unique identifier of the network for which the proposal is made.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkId(String networkId);

        /**
         * <p>
         * The description of the proposal.
         * </p>
         * 
         * @param description
         *        The description of the proposal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The actions to perform on the network if the proposal is <code>APPROVED</code>.
         * </p>
         * 
         * @param actions
         *        The actions to perform on the network if the proposal is <code>APPROVED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(ProposalActions actions);

        /**
         * <p>
         * The actions to perform on the network if the proposal is <code>APPROVED</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link ProposalActions.Builder} avoiding the need to
         * create one manually via {@link ProposalActions#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProposalActions.Builder#build()} is called immediately and its
         * result is passed to {@link #actions(ProposalActions)}.
         * 
         * @param actions
         *        a consumer that will call methods on {@link ProposalActions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #actions(ProposalActions)
         */
        default Builder actions(Consumer<ProposalActions.Builder> actions) {
            return actions(ProposalActions.builder().applyMutation(actions).build());
        }

        /**
         * <p>
         * The unique identifier of the member that created the proposal.
         * </p>
         * 
         * @param proposedByMemberId
         *        The unique identifier of the member that created the proposal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder proposedByMemberId(String proposedByMemberId);

        /**
         * <p>
         * The name of the member that created the proposal.
         * </p>
         * 
         * @param proposedByMemberName
         *        The name of the member that created the proposal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder proposedByMemberName(String proposedByMemberName);

        /**
         * <p>
         * The status of the proposal. Values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code> - The proposal is active and open for member voting.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>APPROVED</code> - The proposal was approved with sufficient <code>YES</code> votes among members
         * according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified proposal
         * actions are carried out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REJECTED</code> - The proposal was rejected with insufficient <code>YES</code> votes among members
         * according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified
         * <code>ProposalActions</code> are not carried out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EXPIRED</code> - Members did not cast the number of votes required to determine the proposal outcome
         * before the proposal expired. The specified <code>ProposalActions</code> are not carried out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTION_FAILED</code> - One or more of the specified <code>ProposalActions</code> in a proposal that was
         * approved could not be completed because of an error. The <code>ACTION_FAILED</code> status occurs even if
         * only one ProposalAction fails and other actions are successful.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the proposal. Values are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code> - The proposal is active and open for member voting.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>APPROVED</code> - The proposal was approved with sufficient <code>YES</code> votes among members
         *        according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified
         *        proposal actions are carried out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REJECTED</code> - The proposal was rejected with insufficient <code>YES</code> votes among
         *        members according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The
         *        specified <code>ProposalActions</code> are not carried out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EXPIRED</code> - Members did not cast the number of votes required to determine the proposal
         *        outcome before the proposal expired. The specified <code>ProposalActions</code> are not carried out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTION_FAILED</code> - One or more of the specified <code>ProposalActions</code> in a proposal
         *        that was approved could not be completed because of an error. The <code>ACTION_FAILED</code> status
         *        occurs even if only one ProposalAction fails and other actions are successful.
         *        </p>
         *        </li>
         * @see ProposalStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProposalStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the proposal. Values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code> - The proposal is active and open for member voting.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>APPROVED</code> - The proposal was approved with sufficient <code>YES</code> votes among members
         * according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified proposal
         * actions are carried out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REJECTED</code> - The proposal was rejected with insufficient <code>YES</code> votes among members
         * according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified
         * <code>ProposalActions</code> are not carried out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EXPIRED</code> - Members did not cast the number of votes required to determine the proposal outcome
         * before the proposal expired. The specified <code>ProposalActions</code> are not carried out.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTION_FAILED</code> - One or more of the specified <code>ProposalActions</code> in a proposal that was
         * approved could not be completed because of an error. The <code>ACTION_FAILED</code> status occurs even if
         * only one ProposalAction fails and other actions are successful.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the proposal. Values are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code> - The proposal is active and open for member voting.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>APPROVED</code> - The proposal was approved with sufficient <code>YES</code> votes among members
         *        according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The specified
         *        proposal actions are carried out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REJECTED</code> - The proposal was rejected with insufficient <code>YES</code> votes among
         *        members according to the <code>VotingPolicy</code> specified for the <code>Network</code>. The
         *        specified <code>ProposalActions</code> are not carried out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EXPIRED</code> - Members did not cast the number of votes required to determine the proposal
         *        outcome before the proposal expired. The specified <code>ProposalActions</code> are not carried out.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTION_FAILED</code> - One or more of the specified <code>ProposalActions</code> in a proposal
         *        that was approved could not be completed because of an error. The <code>ACTION_FAILED</code> status
         *        occurs even if only one ProposalAction fails and other actions are successful.
         *        </p>
         *        </li>
         * @see ProposalStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProposalStatus
         */
        Builder status(ProposalStatus status);

        /**
         * <p>
         * The date and time that the proposal was created.
         * </p>
         * 
         * @param creationDate
         *        The date and time that the proposal was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The date and time that the proposal expires. This is the <code>CreationDate</code> plus the
         * <code>ProposalDurationInHours</code> that is specified in the <code>ProposalThresholdPolicy</code>. After
         * this date and time, if members have not cast enough votes to determine the outcome according to the voting
         * policy, the proposal is <code>EXPIRED</code> and <code>Actions</code> are not carried out.
         * </p>
         * 
         * @param expirationDate
         *        The date and time that the proposal expires. This is the <code>CreationDate</code> plus the
         *        <code>ProposalDurationInHours</code> that is specified in the <code>ProposalThresholdPolicy</code>.
         *        After this date and time, if members have not cast enough votes to determine the outcome according to
         *        the voting policy, the proposal is <code>EXPIRED</code> and <code>Actions</code> are not carried out.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expirationDate(Instant expirationDate);

        /**
         * <p>
         * The current total of <code>YES</code> votes cast on the proposal by members.
         * </p>
         * 
         * @param yesVoteCount
         *        The current total of <code>YES</code> votes cast on the proposal by members.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder yesVoteCount(Integer yesVoteCount);

        /**
         * <p>
         * The current total of <code>NO</code> votes cast on the proposal by members.
         * </p>
         * 
         * @param noVoteCount
         *        The current total of <code>NO</code> votes cast on the proposal by members.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder noVoteCount(Integer noVoteCount);

        /**
         * <p>
         * The number of votes remaining to be cast on the proposal by members. In other words, the number of members
         * minus the sum of <code>YES</code> votes and <code>NO</code> votes.
         * </p>
         * 
         * @param outstandingVoteCount
         *        The number of votes remaining to be cast on the proposal by members. In other words, the number of
         *        members minus the sum of <code>YES</code> votes and <code>NO</code> votes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outstandingVoteCount(Integer outstandingVoteCount);

        /**
         * <p>
         * Tags assigned to the proposal. Each tag consists of a key and optional value.
         * </p>
         * <p>
         * For more information about tags, see <a
         * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
         * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a
         * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
         * >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
         * </p>
         * 
         * @param tags
         *        Tags assigned to the proposal. Each tag consists of a key and optional value.</p>
         *        <p>
         *        For more information about tags, see <a
         *        href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html"
         *        >Tagging Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a href=
         *        "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
         *        >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the proposal. For more information about ARNs and their format, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
         * (ARNs)</a> in the <i>AWS General Reference</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the proposal. For more information about ARNs and their format, see
         *        <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
         *        Names (ARNs)</a> in the <i>AWS General Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);
    }

    static final class BuilderImpl implements Builder {
        private String proposalId;

        private String networkId;

        private String description;

        private ProposalActions actions;

        private String proposedByMemberId;

        private String proposedByMemberName;

        private String status;

        private Instant creationDate;

        private Instant expirationDate;

        private Integer yesVoteCount;

        private Integer noVoteCount;

        private Integer outstandingVoteCount;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String arn;

        private BuilderImpl() {
        }

        private BuilderImpl(Proposal model) {
            proposalId(model.proposalId);
            networkId(model.networkId);
            description(model.description);
            actions(model.actions);
            proposedByMemberId(model.proposedByMemberId);
            proposedByMemberName(model.proposedByMemberName);
            status(model.status);
            creationDate(model.creationDate);
            expirationDate(model.expirationDate);
            yesVoteCount(model.yesVoteCount);
            noVoteCount(model.noVoteCount);
            outstandingVoteCount(model.outstandingVoteCount);
            tags(model.tags);
            arn(model.arn);
        }

        public final String getProposalId() {
            return proposalId;
        }

        public final void setProposalId(String proposalId) {
            this.proposalId = proposalId;
        }

        @Override
        @Transient
        public final Builder proposalId(String proposalId) {
            this.proposalId = proposalId;
            return this;
        }

        public final String getNetworkId() {
            return networkId;
        }

        public final void setNetworkId(String networkId) {
            this.networkId = networkId;
        }

        @Override
        @Transient
        public final Builder networkId(String networkId) {
            this.networkId = networkId;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        @Transient
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final ProposalActions.Builder getActions() {
            return actions != null ? actions.toBuilder() : null;
        }

        public final void setActions(ProposalActions.BuilderImpl actions) {
            this.actions = actions != null ? actions.build() : null;
        }

        @Override
        @Transient
        public final Builder actions(ProposalActions actions) {
            this.actions = actions;
            return this;
        }

        public final String getProposedByMemberId() {
            return proposedByMemberId;
        }

        public final void setProposedByMemberId(String proposedByMemberId) {
            this.proposedByMemberId = proposedByMemberId;
        }

        @Override
        @Transient
        public final Builder proposedByMemberId(String proposedByMemberId) {
            this.proposedByMemberId = proposedByMemberId;
            return this;
        }

        public final String getProposedByMemberName() {
            return proposedByMemberName;
        }

        public final void setProposedByMemberName(String proposedByMemberName) {
            this.proposedByMemberName = proposedByMemberName;
        }

        @Override
        @Transient
        public final Builder proposedByMemberName(String proposedByMemberName) {
            this.proposedByMemberName = proposedByMemberName;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        @Transient
        public final Builder status(ProposalStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        @Transient
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final Instant getExpirationDate() {
            return expirationDate;
        }

        public final void setExpirationDate(Instant expirationDate) {
            this.expirationDate = expirationDate;
        }

        @Override
        @Transient
        public final Builder expirationDate(Instant expirationDate) {
            this.expirationDate = expirationDate;
            return this;
        }

        public final Integer getYesVoteCount() {
            return yesVoteCount;
        }

        public final void setYesVoteCount(Integer yesVoteCount) {
            this.yesVoteCount = yesVoteCount;
        }

        @Override
        @Transient
        public final Builder yesVoteCount(Integer yesVoteCount) {
            this.yesVoteCount = yesVoteCount;
            return this;
        }

        public final Integer getNoVoteCount() {
            return noVoteCount;
        }

        public final void setNoVoteCount(Integer noVoteCount) {
            this.noVoteCount = noVoteCount;
        }

        @Override
        @Transient
        public final Builder noVoteCount(Integer noVoteCount) {
            this.noVoteCount = noVoteCount;
            return this;
        }

        public final Integer getOutstandingVoteCount() {
            return outstandingVoteCount;
        }

        public final void setOutstandingVoteCount(Integer outstandingVoteCount) {
            this.outstandingVoteCount = outstandingVoteCount;
        }

        @Override
        @Transient
        public final Builder outstandingVoteCount(Integer outstandingVoteCount) {
            this.outstandingVoteCount = outstandingVoteCount;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = OutputTagMapCopier.copy(tags);
        }

        @Override
        @Transient
        public final Builder tags(Map<String, String> tags) {
            this.tags = OutputTagMapCopier.copy(tags);
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        @Override
        public Proposal build() {
            return new Proposal(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
