/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.managedblockchain.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration properties of a node.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Node implements SdkPojo, Serializable, ToCopyableBuilder<Node.Builder, Node> {
    private static final SdkField<String> NETWORK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NetworkId").getter(getter(Node::networkId)).setter(setter(Builder::networkId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkId").build()).build();

    private static final SdkField<String> MEMBER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MemberId").getter(getter(Node::memberId)).setter(setter(Builder::memberId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MemberId").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Node::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(Node::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone").getter(getter(Node::availabilityZone)).setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone").build()).build();

    private static final SdkField<NodeFrameworkAttributes> FRAMEWORK_ATTRIBUTES_FIELD = SdkField
            .<NodeFrameworkAttributes> builder(MarshallingType.SDK_POJO).memberName("FrameworkAttributes")
            .getter(getter(Node::frameworkAttributes)).setter(setter(Builder::frameworkAttributes))
            .constructor(NodeFrameworkAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FrameworkAttributes").build())
            .build();

    private static final SdkField<NodeLogPublishingConfiguration> LOG_PUBLISHING_CONFIGURATION_FIELD = SdkField
            .<NodeLogPublishingConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("LogPublishingConfiguration")
            .getter(getter(Node::logPublishingConfiguration))
            .setter(setter(Builder::logPublishingConfiguration))
            .constructor(NodeLogPublishingConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogPublishingConfiguration").build())
            .build();

    private static final SdkField<String> STATE_DB_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StateDB").getter(getter(Node::stateDBAsString)).setter(setter(Builder::stateDB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateDB").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Node::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate")
            .getter(getter(Node::creationDate))
            .setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(Node::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Node::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyArn").getter(getter(Node::kmsKeyArn)).setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NETWORK_ID_FIELD,
            MEMBER_ID_FIELD, ID_FIELD, INSTANCE_TYPE_FIELD, AVAILABILITY_ZONE_FIELD, FRAMEWORK_ATTRIBUTES_FIELD,
            LOG_PUBLISHING_CONFIGURATION_FIELD, STATE_DB_FIELD, STATUS_FIELD, CREATION_DATE_FIELD, TAGS_FIELD, ARN_FIELD,
            KMS_KEY_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String networkId;

    private final String memberId;

    private final String id;

    private final String instanceType;

    private final String availabilityZone;

    private final NodeFrameworkAttributes frameworkAttributes;

    private final NodeLogPublishingConfiguration logPublishingConfiguration;

    private final String stateDB;

    private final String status;

    private final Instant creationDate;

    private final Map<String, String> tags;

    private final String arn;

    private final String kmsKeyArn;

    private Node(BuilderImpl builder) {
        this.networkId = builder.networkId;
        this.memberId = builder.memberId;
        this.id = builder.id;
        this.instanceType = builder.instanceType;
        this.availabilityZone = builder.availabilityZone;
        this.frameworkAttributes = builder.frameworkAttributes;
        this.logPublishingConfiguration = builder.logPublishingConfiguration;
        this.stateDB = builder.stateDB;
        this.status = builder.status;
        this.creationDate = builder.creationDate;
        this.tags = builder.tags;
        this.arn = builder.arn;
        this.kmsKeyArn = builder.kmsKeyArn;
    }

    /**
     * <p>
     * The unique identifier of the network that the node is on.
     * </p>
     * 
     * @return The unique identifier of the network that the node is on.
     */
    public final String networkId() {
        return networkId;
    }

    /**
     * <p>
     * The unique identifier of the member to which the node belongs.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * 
     * @return The unique identifier of the member to which the node belongs.</p>
     *         <p>
     *         Applies only to Hyperledger Fabric.
     */
    public final String memberId() {
        return memberId;
    }

    /**
     * <p>
     * The unique identifier of the node.
     * </p>
     * 
     * @return The unique identifier of the node.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The instance type of the node.
     * </p>
     * 
     * @return The instance type of the node.
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The Availability Zone in which the node exists. Required for Ethereum nodes.
     * </p>
     * 
     * @return The Availability Zone in which the node exists. Required for Ethereum nodes.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * Attributes of the blockchain framework being used.
     * </p>
     * 
     * @return Attributes of the blockchain framework being used.
     */
    public final NodeFrameworkAttributes frameworkAttributes() {
        return frameworkAttributes;
    }

    /**
     * <p>
     * Configuration properties for logging events associated with a peer node on a Hyperledger Fabric network on
     * Managed Blockchain.
     * </p>
     * 
     * @return Configuration properties for logging events associated with a peer node on a Hyperledger Fabric network
     *         on Managed Blockchain.
     */
    public final NodeLogPublishingConfiguration logPublishingConfiguration() {
        return logPublishingConfiguration;
    }

    /**
     * <p>
     * The state database that the node uses. Values are <code>LevelDB</code> or <code>CouchDB</code>.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stateDB} will
     * return {@link StateDBType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateDBAsString}.
     * </p>
     * 
     * @return The state database that the node uses. Values are <code>LevelDB</code> or <code>CouchDB</code>.</p>
     *         <p>
     *         Applies only to Hyperledger Fabric.
     * @see StateDBType
     */
    public final StateDBType stateDB() {
        return StateDBType.fromValue(stateDB);
    }

    /**
     * <p>
     * The state database that the node uses. Values are <code>LevelDB</code> or <code>CouchDB</code>.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stateDB} will
     * return {@link StateDBType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateDBAsString}.
     * </p>
     * 
     * @return The state database that the node uses. Values are <code>LevelDB</code> or <code>CouchDB</code>.</p>
     *         <p>
     *         Applies only to Hyperledger Fabric.
     * @see StateDBType
     */
    public final String stateDBAsString() {
        return stateDB;
    }

    /**
     * <p>
     * The status of the node.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The Amazon Web Services account is in the process of creating a node.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The node has been created and can participate in the network.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UNHEALTHY</code> - The node is impaired and might not function as expected. Amazon Managed Blockchain
     * automatically finds nodes in this state and tries to recover them. If a node is recoverable, it returns to
     * <code>AVAILABLE</code>. Otherwise, it moves to <code>FAILED</code> status.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a node and creation failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The node is in the process of being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The node is in the process of being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETED</code> - The node can no longer participate on the network.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The node is no longer functional, cannot be recovered, and must be deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The node is impaired and might not function as expected because it
     * cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS key was disabled
     * or deleted, or the grants on the key were revoked.
     * </p>
     * <p>
     * The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time for the
     * node resource to discover that the key is inaccessible. When a resource is in this state, we recommend deleting
     * and recreating the resource.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link NodeStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the node.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The Amazon Web Services account is in the process of creating a node.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The node has been created and can participate in the network.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNHEALTHY</code> - The node is impaired and might not function as expected. Amazon Managed
     *         Blockchain automatically finds nodes in this state and tries to recover them. If a node is recoverable,
     *         it returns to <code>AVAILABLE</code>. Otherwise, it moves to <code>FAILED</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a node and creation
     *         failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The node is in the process of being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The node is in the process of being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETED</code> - The node can no longer participate on the network.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The node is no longer functional, cannot be recovered, and must be deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The node is impaired and might not function as expected
     *         because it cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS
     *         key was disabled or deleted, or the grants on the key were revoked.
     *         </p>
     *         <p>
     *         The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time
     *         for the node resource to discover that the key is inaccessible. When a resource is in this state, we
     *         recommend deleting and recreating the resource.
     *         </p>
     *         </li>
     * @see NodeStatus
     */
    public final NodeStatus status() {
        return NodeStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the node.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The Amazon Web Services account is in the process of creating a node.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The node has been created and can participate in the network.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UNHEALTHY</code> - The node is impaired and might not function as expected. Amazon Managed Blockchain
     * automatically finds nodes in this state and tries to recover them. If a node is recoverable, it returns to
     * <code>AVAILABLE</code>. Otherwise, it moves to <code>FAILED</code> status.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a node and creation failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The node is in the process of being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The node is in the process of being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETED</code> - The node can no longer participate on the network.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The node is no longer functional, cannot be recovered, and must be deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The node is impaired and might not function as expected because it
     * cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS key was disabled
     * or deleted, or the grants on the key were revoked.
     * </p>
     * <p>
     * The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time for the
     * node resource to discover that the key is inaccessible. When a resource is in this state, we recommend deleting
     * and recreating the resource.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link NodeStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the node.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The Amazon Web Services account is in the process of creating a node.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The node has been created and can participate in the network.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNHEALTHY</code> - The node is impaired and might not function as expected. Amazon Managed
     *         Blockchain automatically finds nodes in this state and tries to recover them. If a node is recoverable,
     *         it returns to <code>AVAILABLE</code>. Otherwise, it moves to <code>FAILED</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a node and creation
     *         failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The node is in the process of being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The node is in the process of being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETED</code> - The node can no longer participate on the network.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The node is no longer functional, cannot be recovered, and must be deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The node is impaired and might not function as expected
     *         because it cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS
     *         key was disabled or deleted, or the grants on the key were revoked.
     *         </p>
     *         <p>
     *         The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time
     *         for the node resource to discover that the key is inaccessible. When a resource is in this state, we
     *         recommend deleting and recreating the resource.
     *         </p>
     *         </li>
     * @see NodeStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The date and time that the node was created.
     * </p>
     * 
     * @return The date and time that the node was created.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Tags assigned to the node. Each tag consists of a key and optional value.
     * </p>
     * <p>
     * For more information about tags, see <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
     * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a
     * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
     * >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Tags assigned to the node. Each tag consists of a key and optional value.</p>
     *         <p>
     *         For more information about tags, see <a
     *         href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
     *         Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a href=
     *         "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
     *         >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the node. For more information about ARNs and their format, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     * (ARNs)</a> in the <i>Amazon Web Services General Reference</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the node. For more information about ARNs and their format, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     *         (ARNs)</a> in the <i>Amazon Web Services General Reference</i>.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the customer managed key in Key Management Service (KMS) that the node uses for
     * encryption at rest. If the value of this parameter is <code>"AWS Owned KMS Key"</code>, the node uses an Amazon
     * Web Services owned KMS key for encryption. The node inherits this parameter from the member that it belongs to.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/managed-blockchain-encryption-at-rest.html"
     * >Encryption at Rest</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     * </p>
     * <p>
     * Applies only to Hyperledger Fabric.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the customer managed key in Key Management Service (KMS) that the node
     *         uses for encryption at rest. If the value of this parameter is <code>"AWS Owned KMS Key"</code>, the node
     *         uses an Amazon Web Services owned KMS key for encryption. The node inherits this parameter from the
     *         member that it belongs to.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/managed-blockchain-encryption-at-rest.html"
     *         >Encryption at Rest</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
     *         </p>
     *         <p>
     *         Applies only to Hyperledger Fabric.
     */
    public final String kmsKeyArn() {
        return kmsKeyArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(networkId());
        hashCode = 31 * hashCode + Objects.hashCode(memberId());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(frameworkAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(logPublishingConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(stateDBAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Node)) {
            return false;
        }
        Node other = (Node) obj;
        return Objects.equals(networkId(), other.networkId()) && Objects.equals(memberId(), other.memberId())
                && Objects.equals(id(), other.id()) && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(frameworkAttributes(), other.frameworkAttributes())
                && Objects.equals(logPublishingConfiguration(), other.logPublishingConfiguration())
                && Objects.equals(stateDBAsString(), other.stateDBAsString())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(creationDate(), other.creationDate()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(arn(), other.arn())
                && Objects.equals(kmsKeyArn(), other.kmsKeyArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Node").add("NetworkId", networkId()).add("MemberId", memberId()).add("Id", id())
                .add("InstanceType", instanceType()).add("AvailabilityZone", availabilityZone())
                .add("FrameworkAttributes", frameworkAttributes())
                .add("LogPublishingConfiguration", logPublishingConfiguration()).add("StateDB", stateDBAsString())
                .add("Status", statusAsString()).add("CreationDate", creationDate()).add("Tags", hasTags() ? tags() : null)
                .add("Arn", arn()).add("KmsKeyArn", kmsKeyArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NetworkId":
            return Optional.ofNullable(clazz.cast(networkId()));
        case "MemberId":
            return Optional.ofNullable(clazz.cast(memberId()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "FrameworkAttributes":
            return Optional.ofNullable(clazz.cast(frameworkAttributes()));
        case "LogPublishingConfiguration":
            return Optional.ofNullable(clazz.cast(logPublishingConfiguration()));
        case "StateDB":
            return Optional.ofNullable(clazz.cast(stateDBAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "KmsKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Node, T> g) {
        return obj -> g.apply((Node) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Node> {
        /**
         * <p>
         * The unique identifier of the network that the node is on.
         * </p>
         * 
         * @param networkId
         *        The unique identifier of the network that the node is on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkId(String networkId);

        /**
         * <p>
         * The unique identifier of the member to which the node belongs.
         * </p>
         * <p>
         * Applies only to Hyperledger Fabric.
         * </p>
         * 
         * @param memberId
         *        The unique identifier of the member to which the node belongs.</p>
         *        <p>
         *        Applies only to Hyperledger Fabric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memberId(String memberId);

        /**
         * <p>
         * The unique identifier of the node.
         * </p>
         * 
         * @param id
         *        The unique identifier of the node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The instance type of the node.
         * </p>
         * 
         * @param instanceType
         *        The instance type of the node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The Availability Zone in which the node exists. Required for Ethereum nodes.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone in which the node exists. Required for Ethereum nodes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * Attributes of the blockchain framework being used.
         * </p>
         * 
         * @param frameworkAttributes
         *        Attributes of the blockchain framework being used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder frameworkAttributes(NodeFrameworkAttributes frameworkAttributes);

        /**
         * <p>
         * Attributes of the blockchain framework being used.
         * </p>
         * This is a convenience method that creates an instance of the {@link NodeFrameworkAttributes.Builder} avoiding
         * the need to create one manually via {@link NodeFrameworkAttributes#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NodeFrameworkAttributes.Builder#build()} is called immediately
         * and its result is passed to {@link #frameworkAttributes(NodeFrameworkAttributes)}.
         * 
         * @param frameworkAttributes
         *        a consumer that will call methods on {@link NodeFrameworkAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #frameworkAttributes(NodeFrameworkAttributes)
         */
        default Builder frameworkAttributes(Consumer<NodeFrameworkAttributes.Builder> frameworkAttributes) {
            return frameworkAttributes(NodeFrameworkAttributes.builder().applyMutation(frameworkAttributes).build());
        }

        /**
         * <p>
         * Configuration properties for logging events associated with a peer node on a Hyperledger Fabric network on
         * Managed Blockchain.
         * </p>
         * 
         * @param logPublishingConfiguration
         *        Configuration properties for logging events associated with a peer node on a Hyperledger Fabric
         *        network on Managed Blockchain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPublishingConfiguration(NodeLogPublishingConfiguration logPublishingConfiguration);

        /**
         * <p>
         * Configuration properties for logging events associated with a peer node on a Hyperledger Fabric network on
         * Managed Blockchain.
         * </p>
         * This is a convenience method that creates an instance of the {@link NodeLogPublishingConfiguration.Builder}
         * avoiding the need to create one manually via {@link NodeLogPublishingConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NodeLogPublishingConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #logPublishingConfiguration(NodeLogPublishingConfiguration)}.
         * 
         * @param logPublishingConfiguration
         *        a consumer that will call methods on {@link NodeLogPublishingConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logPublishingConfiguration(NodeLogPublishingConfiguration)
         */
        default Builder logPublishingConfiguration(Consumer<NodeLogPublishingConfiguration.Builder> logPublishingConfiguration) {
            return logPublishingConfiguration(NodeLogPublishingConfiguration.builder().applyMutation(logPublishingConfiguration)
                    .build());
        }

        /**
         * <p>
         * The state database that the node uses. Values are <code>LevelDB</code> or <code>CouchDB</code>.
         * </p>
         * <p>
         * Applies only to Hyperledger Fabric.
         * </p>
         * 
         * @param stateDB
         *        The state database that the node uses. Values are <code>LevelDB</code> or <code>CouchDB</code>.</p>
         *        <p>
         *        Applies only to Hyperledger Fabric.
         * @see StateDBType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StateDBType
         */
        Builder stateDB(String stateDB);

        /**
         * <p>
         * The state database that the node uses. Values are <code>LevelDB</code> or <code>CouchDB</code>.
         * </p>
         * <p>
         * Applies only to Hyperledger Fabric.
         * </p>
         * 
         * @param stateDB
         *        The state database that the node uses. Values are <code>LevelDB</code> or <code>CouchDB</code>.</p>
         *        <p>
         *        Applies only to Hyperledger Fabric.
         * @see StateDBType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StateDBType
         */
        Builder stateDB(StateDBType stateDB);

        /**
         * <p>
         * The status of the node.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The Amazon Web Services account is in the process of creating a node.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The node has been created and can participate in the network.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UNHEALTHY</code> - The node is impaired and might not function as expected. Amazon Managed Blockchain
         * automatically finds nodes in this state and tries to recover them. If a node is recoverable, it returns to
         * <code>AVAILABLE</code>. Otherwise, it moves to <code>FAILED</code> status.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a node and creation failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The node is in the process of being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The node is in the process of being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETED</code> - The node can no longer participate on the network.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The node is no longer functional, cannot be recovered, and must be deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The node is impaired and might not function as expected because it
         * cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS key was
         * disabled or deleted, or the grants on the key were revoked.
         * </p>
         * <p>
         * The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time for
         * the node resource to discover that the key is inaccessible. When a resource is in this state, we recommend
         * deleting and recreating the resource.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the node.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The Amazon Web Services account is in the process of creating a node.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The node has been created and can participate in the network.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNHEALTHY</code> - The node is impaired and might not function as expected. Amazon Managed
         *        Blockchain automatically finds nodes in this state and tries to recover them. If a node is
         *        recoverable, it returns to <code>AVAILABLE</code>. Otherwise, it moves to <code>FAILED</code> status.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a node and creation
         *        failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The node is in the process of being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The node is in the process of being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETED</code> - The node can no longer participate on the network.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The node is no longer functional, cannot be recovered, and must be deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The node is impaired and might not function as expected
         *        because it cannot access the specified customer managed key in KMS for encryption at rest. Either the
         *        KMS key was disabled or deleted, or the grants on the key were revoked.
         *        </p>
         *        <p>
         *        The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some
         *        time for the node resource to discover that the key is inaccessible. When a resource is in this state,
         *        we recommend deleting and recreating the resource.
         *        </p>
         *        </li>
         * @see NodeStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NodeStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the node.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The Amazon Web Services account is in the process of creating a node.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The node has been created and can participate in the network.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UNHEALTHY</code> - The node is impaired and might not function as expected. Amazon Managed Blockchain
         * automatically finds nodes in this state and tries to recover them. If a node is recoverable, it returns to
         * <code>AVAILABLE</code>. Otherwise, it moves to <code>FAILED</code> status.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a node and creation failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The node is in the process of being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The node is in the process of being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETED</code> - The node can no longer participate on the network.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The node is no longer functional, cannot be recovered, and must be deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The node is impaired and might not function as expected because it
         * cannot access the specified customer managed key in KMS for encryption at rest. Either the KMS key was
         * disabled or deleted, or the grants on the key were revoked.
         * </p>
         * <p>
         * The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some time for
         * the node resource to discover that the key is inaccessible. When a resource is in this state, we recommend
         * deleting and recreating the resource.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the node.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The Amazon Web Services account is in the process of creating a node.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The node has been created and can participate in the network.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNHEALTHY</code> - The node is impaired and might not function as expected. Amazon Managed
         *        Blockchain automatically finds nodes in this state and tries to recover them. If a node is
         *        recoverable, it returns to <code>AVAILABLE</code>. Otherwise, it moves to <code>FAILED</code> status.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED</code> - The Amazon Web Services account attempted to create a node and creation
         *        failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The node is in the process of being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The node is in the process of being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETED</code> - The node can no longer participate on the network.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The node is no longer functional, cannot be recovered, and must be deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INACCESSIBLE_ENCRYPTION_KEY</code> - The node is impaired and might not function as expected
         *        because it cannot access the specified customer managed key in KMS for encryption at rest. Either the
         *        KMS key was disabled or deleted, or the grants on the key were revoked.
         *        </p>
         *        <p>
         *        The effect of disabling or deleting a key or of revoking a grant isn't immediate. It might take some
         *        time for the node resource to discover that the key is inaccessible. When a resource is in this state,
         *        we recommend deleting and recreating the resource.
         *        </p>
         *        </li>
         * @see NodeStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NodeStatus
         */
        Builder status(NodeStatus status);

        /**
         * <p>
         * The date and time that the node was created.
         * </p>
         * 
         * @param creationDate
         *        The date and time that the node was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * Tags assigned to the node. Each tag consists of a key and optional value.
         * </p>
         * <p>
         * For more information about tags, see <a
         * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
         * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a
         * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
         * >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
         * </p>
         * 
         * @param tags
         *        Tags assigned to the node. Each tag consists of a key and optional value.</p>
         *        <p>
         *        For more information about tags, see <a
         *        href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html"
         *        >Tagging Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>, or <a href=
         *        "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html"
         *        >Tagging Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the node. For more information about ARNs and their format, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
         * (ARNs)</a> in the <i>Amazon Web Services General Reference</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the node. For more information about ARNs and their format, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
         *        Names (ARNs)</a> in the <i>Amazon Web Services General Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the customer managed key in Key Management Service (KMS) that the node uses
         * for encryption at rest. If the value of this parameter is <code>"AWS Owned KMS Key"</code>, the node uses an
         * Amazon Web Services owned KMS key for encryption. The node inherits this parameter from the member that it
         * belongs to.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/managed-blockchain-encryption-at-rest.html"
         * >Encryption at Rest</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
         * </p>
         * <p>
         * Applies only to Hyperledger Fabric.
         * </p>
         * 
         * @param kmsKeyArn
         *        The Amazon Resource Name (ARN) of the customer managed key in Key Management Service (KMS) that the
         *        node uses for encryption at rest. If the value of this parameter is <code>"AWS Owned KMS Key"</code>,
         *        the node uses an Amazon Web Services owned KMS key for encryption. The node inherits this parameter
         *        from the member that it belongs to.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/managed-blockchain-encryption-at-rest.html"
         *        >Encryption at Rest</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer Guide</i>.
         *        </p>
         *        <p>
         *        Applies only to Hyperledger Fabric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);
    }

    static final class BuilderImpl implements Builder {
        private String networkId;

        private String memberId;

        private String id;

        private String instanceType;

        private String availabilityZone;

        private NodeFrameworkAttributes frameworkAttributes;

        private NodeLogPublishingConfiguration logPublishingConfiguration;

        private String stateDB;

        private String status;

        private Instant creationDate;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String arn;

        private String kmsKeyArn;

        private BuilderImpl() {
        }

        private BuilderImpl(Node model) {
            networkId(model.networkId);
            memberId(model.memberId);
            id(model.id);
            instanceType(model.instanceType);
            availabilityZone(model.availabilityZone);
            frameworkAttributes(model.frameworkAttributes);
            logPublishingConfiguration(model.logPublishingConfiguration);
            stateDB(model.stateDB);
            status(model.status);
            creationDate(model.creationDate);
            tags(model.tags);
            arn(model.arn);
            kmsKeyArn(model.kmsKeyArn);
        }

        public final String getNetworkId() {
            return networkId;
        }

        public final void setNetworkId(String networkId) {
            this.networkId = networkId;
        }

        @Override
        public final Builder networkId(String networkId) {
            this.networkId = networkId;
            return this;
        }

        public final String getMemberId() {
            return memberId;
        }

        public final void setMemberId(String memberId) {
            this.memberId = memberId;
        }

        @Override
        public final Builder memberId(String memberId) {
            this.memberId = memberId;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final NodeFrameworkAttributes.Builder getFrameworkAttributes() {
            return frameworkAttributes != null ? frameworkAttributes.toBuilder() : null;
        }

        public final void setFrameworkAttributes(NodeFrameworkAttributes.BuilderImpl frameworkAttributes) {
            this.frameworkAttributes = frameworkAttributes != null ? frameworkAttributes.build() : null;
        }

        @Override
        public final Builder frameworkAttributes(NodeFrameworkAttributes frameworkAttributes) {
            this.frameworkAttributes = frameworkAttributes;
            return this;
        }

        public final NodeLogPublishingConfiguration.Builder getLogPublishingConfiguration() {
            return logPublishingConfiguration != null ? logPublishingConfiguration.toBuilder() : null;
        }

        public final void setLogPublishingConfiguration(NodeLogPublishingConfiguration.BuilderImpl logPublishingConfiguration) {
            this.logPublishingConfiguration = logPublishingConfiguration != null ? logPublishingConfiguration.build() : null;
        }

        @Override
        public final Builder logPublishingConfiguration(NodeLogPublishingConfiguration logPublishingConfiguration) {
            this.logPublishingConfiguration = logPublishingConfiguration;
            return this;
        }

        public final String getStateDB() {
            return stateDB;
        }

        public final void setStateDB(String stateDB) {
            this.stateDB = stateDB;
        }

        @Override
        public final Builder stateDB(String stateDB) {
            this.stateDB = stateDB;
            return this;
        }

        @Override
        public final Builder stateDB(StateDBType stateDB) {
            this.stateDB(stateDB == null ? null : stateDB.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(NodeStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = OutputTagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = OutputTagMapCopier.copy(tags);
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        @Override
        public Node build() {
            return new Node(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
