/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.managedblockchain.paginators;

import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.managedblockchain.ManagedBlockchainClient;
import software.amazon.awssdk.services.managedblockchain.model.ListProposalsRequest;
import software.amazon.awssdk.services.managedblockchain.model.ListProposalsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.managedblockchain.ManagedBlockchainClient#listProposalsPaginator(software.amazon.awssdk.services.managedblockchain.model.ListProposalsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.managedblockchain.model.ListProposalsResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.managedblockchain.paginators.ListProposalsIterable responses = client.listProposalsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.managedblockchain.paginators.ListProposalsIterable responses = client
 *             .listProposalsPaginator(request);
 *     for (software.amazon.awssdk.services.managedblockchain.model.ListProposalsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.managedblockchain.paginators.ListProposalsIterable responses = client.listProposalsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listProposals(software.amazon.awssdk.services.managedblockchain.model.ListProposalsRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListProposalsIterable implements SdkIterable<ListProposalsResponse> {
    private final ManagedBlockchainClient client;

    private final ListProposalsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListProposalsIterable(ManagedBlockchainClient client, ListProposalsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListProposalsResponseFetcher();
    }

    @Override
    public Iterator<ListProposalsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    private class ListProposalsResponseFetcher implements SyncPageFetcher<ListProposalsResponse> {
        @Override
        public boolean hasNextPage(ListProposalsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListProposalsResponse nextPage(ListProposalsResponse previousPage) {
            if (previousPage == null) {
                return client.listProposals(firstRequest);
            }
            return client.listProposals(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
