/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.managedblockchainquery;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.managedblockchainquery.internal.ManagedBlockchainQueryServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.managedblockchainquery.model.AccessDeniedException;
import software.amazon.awssdk.services.managedblockchainquery.model.BatchGetTokenBalanceRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.BatchGetTokenBalanceResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.GetAssetContractRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.GetAssetContractResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.GetTokenBalanceRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.GetTokenBalanceResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.GetTransactionRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.GetTransactionResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.InternalServerException;
import software.amazon.awssdk.services.managedblockchainquery.model.ListAssetContractsRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.ListAssetContractsResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTokenBalancesRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTokenBalancesResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTransactionEventsRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTransactionEventsResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTransactionsRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTransactionsResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.ManagedBlockchainQueryException;
import software.amazon.awssdk.services.managedblockchainquery.model.ResourceNotFoundException;
import software.amazon.awssdk.services.managedblockchainquery.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.managedblockchainquery.model.ThrottlingException;
import software.amazon.awssdk.services.managedblockchainquery.model.ValidationException;
import software.amazon.awssdk.services.managedblockchainquery.transform.BatchGetTokenBalanceRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.GetAssetContractRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.GetTokenBalanceRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.GetTransactionRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.ListAssetContractsRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.ListTokenBalancesRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.ListTransactionEventsRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.ListTransactionsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ManagedBlockchainQueryClient}.
 *
 * @see ManagedBlockchainQueryClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultManagedBlockchainQueryClient implements ManagedBlockchainQueryClient {
    private static final Logger log = Logger.loggerFor(DefaultManagedBlockchainQueryClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ManagedBlockchainQueryServiceClientConfiguration serviceClientConfiguration;

    protected DefaultManagedBlockchainQueryClient(ManagedBlockchainQueryServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Gets the token balance for a batch of tokens by using the <code>BatchGetTokenBalance</code> action for every
     * token in the request.
     * </p>
     * <note>
     * <p>
     * Only the native tokens BTC,ETH, and the ERC-20, ERC-721, and ERC 1155 token standards are supported.
     * </p>
     * </note>
     *
     * @param batchGetTokenBalanceRequest
     * @return Result of the BatchGetTokenBalance operation returned by the service.
     * @throws ThrottlingException
     *         The request or operation couldn't be performed because a service is throttling requests. The most common
     *         source of throttling errors is when you create resources that exceed your service limit for this resource
     *         type. Request a limit increase or delete unused resources, if possible.
     * @throws ValidationException
     *         The resource passed is invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws AccessDeniedException
     *         The Amazon Web Services account doesn’t have access to this resource.
     * @throws InternalServerException
     *         The request processing has failed because of an internal error in the service.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded for this resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainQueryClient.BatchGetTokenBalance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/BatchGetTokenBalance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchGetTokenBalanceResponse batchGetTokenBalance(BatchGetTokenBalanceRequest batchGetTokenBalanceRequest)
            throws ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException,
            InternalServerException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            ManagedBlockchainQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetTokenBalanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchGetTokenBalanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetTokenBalanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetTokenBalanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetTokenBalance");

            return clientHandler.execute(new ClientExecutionParams<BatchGetTokenBalanceRequest, BatchGetTokenBalanceResponse>()
                    .withOperationName("BatchGetTokenBalance").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(batchGetTokenBalanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchGetTokenBalanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the information about a specific contract deployed on the blockchain.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * The Bitcoin blockchain networks do not support this operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * Metadata is currently only available for some <code>ERC-20</code> contracts. Metadata will be available for
     * additional contracts in the future.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param getAssetContractRequest
     * @return Result of the GetAssetContract operation returned by the service.
     * @throws ThrottlingException
     *         The request or operation couldn't be performed because a service is throttling requests. The most common
     *         source of throttling errors is when you create resources that exceed your service limit for this resource
     *         type. Request a limit increase or delete unused resources, if possible.
     * @throws ValidationException
     *         The resource passed is invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws AccessDeniedException
     *         The Amazon Web Services account doesn’t have access to this resource.
     * @throws InternalServerException
     *         The request processing has failed because of an internal error in the service.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded for this resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainQueryClient.GetAssetContract
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/GetAssetContract"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAssetContractResponse getAssetContract(GetAssetContractRequest getAssetContractRequest) throws ThrottlingException,
            ValidationException, ResourceNotFoundException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ManagedBlockchainQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAssetContractResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAssetContractResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAssetContractRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAssetContractRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAssetContract");

            return clientHandler.execute(new ClientExecutionParams<GetAssetContractRequest, GetAssetContractResponse>()
                    .withOperationName("GetAssetContract").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAssetContractRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAssetContractRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the balance of a specific token, including native tokens, for a given address (wallet or contract) on the
     * blockchain.
     * </p>
     * <note>
     * <p>
     * Only the native tokens BTC,ETH, and the ERC-20, ERC-721, and ERC 1155 token standards are supported.
     * </p>
     * </note>
     *
     * @param getTokenBalanceRequest
     * @return Result of the GetTokenBalance operation returned by the service.
     * @throws ThrottlingException
     *         The request or operation couldn't be performed because a service is throttling requests. The most common
     *         source of throttling errors is when you create resources that exceed your service limit for this resource
     *         type. Request a limit increase or delete unused resources, if possible.
     * @throws ValidationException
     *         The resource passed is invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws AccessDeniedException
     *         The Amazon Web Services account doesn’t have access to this resource.
     * @throws InternalServerException
     *         The request processing has failed because of an internal error in the service.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded for this resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainQueryClient.GetTokenBalance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/GetTokenBalance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetTokenBalanceResponse getTokenBalance(GetTokenBalanceRequest getTokenBalanceRequest) throws ThrottlingException,
            ValidationException, ResourceNotFoundException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ManagedBlockchainQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTokenBalanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTokenBalanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTokenBalanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTokenBalanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTokenBalance");

            return clientHandler.execute(new ClientExecutionParams<GetTokenBalanceRequest, GetTokenBalanceResponse>()
                    .withOperationName("GetTokenBalance").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTokenBalanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTokenBalanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get the details of a transaction.
     * </p>
     *
     * @param getTransactionRequest
     * @return Result of the GetTransaction operation returned by the service.
     * @throws ThrottlingException
     *         The request or operation couldn't be performed because a service is throttling requests. The most common
     *         source of throttling errors is when you create resources that exceed your service limit for this resource
     *         type. Request a limit increase or delete unused resources, if possible.
     * @throws ValidationException
     *         The resource passed is invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws AccessDeniedException
     *         The Amazon Web Services account doesn’t have access to this resource.
     * @throws InternalServerException
     *         The request processing has failed because of an internal error in the service.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded for this resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainQueryClient.GetTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/GetTransaction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetTransactionResponse getTransaction(GetTransactionRequest getTransactionRequest) throws ThrottlingException,
            ValidationException, ResourceNotFoundException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ManagedBlockchainQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTransactionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTransactionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTransactionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTransaction");

            return clientHandler.execute(new ClientExecutionParams<GetTransactionRequest, GetTransactionResponse>()
                    .withOperationName("GetTransaction").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTransactionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTransactionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the contracts for a given contract type deployed by an address (either a contract address or a wallet
     * address).
     * </p>
     * <p>
     * The Bitcoin blockchain networks do not support this operation.
     * </p>
     *
     * @param listAssetContractsRequest
     * @return Result of the ListAssetContracts operation returned by the service.
     * @throws ThrottlingException
     *         The request or operation couldn't be performed because a service is throttling requests. The most common
     *         source of throttling errors is when you create resources that exceed your service limit for this resource
     *         type. Request a limit increase or delete unused resources, if possible.
     * @throws ValidationException
     *         The resource passed is invalid.
     * @throws AccessDeniedException
     *         The Amazon Web Services account doesn’t have access to this resource.
     * @throws InternalServerException
     *         The request processing has failed because of an internal error in the service.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded for this resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainQueryClient.ListAssetContracts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/ListAssetContracts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAssetContractsResponse listAssetContracts(ListAssetContractsRequest listAssetContractsRequest)
            throws ThrottlingException, ValidationException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ManagedBlockchainQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAssetContractsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAssetContractsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAssetContractsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssetContractsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssetContracts");

            return clientHandler.execute(new ClientExecutionParams<ListAssetContractsRequest, ListAssetContractsResponse>()
                    .withOperationName("ListAssetContracts").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAssetContractsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAssetContractsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This action returns the following for a given blockchain network:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Lists all token balances owned by an address (either a contract address or a wallet address).
     * </p>
     * </li>
     * <li>
     * <p>
     * Lists all token balances for all tokens created by a contract.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lists all token balances for a given token.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * You must always specify the network property of the <code>tokenFilter</code> when using this operation.
     * </p>
     * </note>
     *
     * @param listTokenBalancesRequest
     * @return Result of the ListTokenBalances operation returned by the service.
     * @throws ThrottlingException
     *         The request or operation couldn't be performed because a service is throttling requests. The most common
     *         source of throttling errors is when you create resources that exceed your service limit for this resource
     *         type. Request a limit increase or delete unused resources, if possible.
     * @throws ValidationException
     *         The resource passed is invalid.
     * @throws AccessDeniedException
     *         The Amazon Web Services account doesn’t have access to this resource.
     * @throws InternalServerException
     *         The request processing has failed because of an internal error in the service.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded for this resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainQueryClient.ListTokenBalances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/ListTokenBalances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTokenBalancesResponse listTokenBalances(ListTokenBalancesRequest listTokenBalancesRequest)
            throws ThrottlingException, ValidationException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ManagedBlockchainQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTokenBalancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTokenBalancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTokenBalancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTokenBalancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTokenBalances");

            return clientHandler.execute(new ClientExecutionParams<ListTokenBalancesRequest, ListTokenBalancesResponse>()
                    .withOperationName("ListTokenBalances").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTokenBalancesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTokenBalancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * An array of <code>TransactionEvent</code> objects. Each object contains details about the transaction event.
     * </p>
     *
     * @param listTransactionEventsRequest
     * @return Result of the ListTransactionEvents operation returned by the service.
     * @throws ThrottlingException
     *         The request or operation couldn't be performed because a service is throttling requests. The most common
     *         source of throttling errors is when you create resources that exceed your service limit for this resource
     *         type. Request a limit increase or delete unused resources, if possible.
     * @throws ValidationException
     *         The resource passed is invalid.
     * @throws AccessDeniedException
     *         The Amazon Web Services account doesn’t have access to this resource.
     * @throws InternalServerException
     *         The request processing has failed because of an internal error in the service.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded for this resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainQueryClient.ListTransactionEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/ListTransactionEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTransactionEventsResponse listTransactionEvents(ListTransactionEventsRequest listTransactionEventsRequest)
            throws ThrottlingException, ValidationException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ManagedBlockchainQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTransactionEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTransactionEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTransactionEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTransactionEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTransactionEvents");

            return clientHandler.execute(new ClientExecutionParams<ListTransactionEventsRequest, ListTransactionEventsResponse>()
                    .withOperationName("ListTransactionEvents").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTransactionEventsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTransactionEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all of the transactions on a given wallet address or to a specific contract.
     * </p>
     *
     * @param listTransactionsRequest
     * @return Result of the ListTransactions operation returned by the service.
     * @throws ThrottlingException
     *         The request or operation couldn't be performed because a service is throttling requests. The most common
     *         source of throttling errors is when you create resources that exceed your service limit for this resource
     *         type. Request a limit increase or delete unused resources, if possible.
     * @throws ValidationException
     *         The resource passed is invalid.
     * @throws AccessDeniedException
     *         The Amazon Web Services account doesn’t have access to this resource.
     * @throws InternalServerException
     *         The request processing has failed because of an internal error in the service.
     * @throws ServiceQuotaExceededException
     *         The service quota has been exceeded for this resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ManagedBlockchainQueryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ManagedBlockchainQueryClient.ListTransactions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/ListTransactions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTransactionsResponse listTransactions(ListTransactionsRequest listTransactionsRequest) throws ThrottlingException,
            ValidationException, AccessDeniedException, InternalServerException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, ManagedBlockchainQueryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTransactionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTransactionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTransactionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTransactionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTransactions");

            return clientHandler.execute(new ClientExecutionParams<ListTransactionsRequest, ListTransactionsResponse>()
                    .withOperationName("ListTransactions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTransactionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTransactionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        ManagedBlockchainQueryServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = ManagedBlockchainQueryServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ManagedBlockchainQueryException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final ManagedBlockchainQueryServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
