/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.managedblockchainquery.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * There are two possible types of transactions used for this data type:
 * </p>
 * <ul>
 * <li>
 * <p>
 * A Bitcoin transaction is a movement of BTC from one address to another.
 * </p>
 * </li>
 * <li>
 * <p>
 * An Ethereum transaction refers to an action initiated by an externally owned account, which is an account managed by
 * a human, not a contract. For example, if Bob sends Alice 1 ETH, Bob's account must be debited and Alice's must be
 * credited. This state-changing action occurs within a transaction.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Transaction implements SdkPojo, Serializable, ToCopyableBuilder<Transaction.Builder, Transaction> {
    private static final SdkField<String> NETWORK_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("network")
            .getter(getter(Transaction::networkAsString)).setter(setter(Builder::network))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("network").build()).build();

    private static final SdkField<String> BLOCK_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("blockHash").getter(getter(Transaction::blockHash)).setter(setter(Builder::blockHash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("blockHash").build()).build();

    private static final SdkField<String> TRANSACTION_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("transactionHash").getter(getter(Transaction::transactionHash)).setter(setter(Builder::transactionHash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transactionHash").build()).build();

    private static final SdkField<String> BLOCK_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("blockNumber").getter(getter(Transaction::blockNumber)).setter(setter(Builder::blockNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("blockNumber").build()).build();

    private static final SdkField<Instant> TRANSACTION_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("transactionTimestamp").getter(getter(Transaction::transactionTimestamp))
            .setter(setter(Builder::transactionTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transactionTimestamp").build())
            .build();

    private static final SdkField<Long> TRANSACTION_INDEX_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("transactionIndex").getter(getter(Transaction::transactionIndex))
            .setter(setter(Builder::transactionIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transactionIndex").build()).build();

    private static final SdkField<Long> NUMBER_OF_TRANSACTIONS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("numberOfTransactions").getter(getter(Transaction::numberOfTransactions))
            .setter(setter(Builder::numberOfTransactions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numberOfTransactions").build())
            .build();

    private static final SdkField<String> TO_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("to")
            .getter(getter(Transaction::to)).setter(setter(Builder::to))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("to").build()).build();

    private static final SdkField<String> FROM_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("from")
            .getter(getter(Transaction::from)).setter(setter(Builder::from))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("from").build()).build();

    private static final SdkField<String> CONTRACT_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("contractAddress").getter(getter(Transaction::contractAddress)).setter(setter(Builder::contractAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("contractAddress").build()).build();

    private static final SdkField<String> GAS_USED_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("gasUsed").getter(getter(Transaction::gasUsed)).setter(setter(Builder::gasUsed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("gasUsed").build()).build();

    private static final SdkField<String> CUMULATIVE_GAS_USED_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("cumulativeGasUsed").getter(getter(Transaction::cumulativeGasUsed))
            .setter(setter(Builder::cumulativeGasUsed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cumulativeGasUsed").build()).build();

    private static final SdkField<String> EFFECTIVE_GAS_PRICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("effectiveGasPrice").getter(getter(Transaction::effectiveGasPrice))
            .setter(setter(Builder::effectiveGasPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("effectiveGasPrice").build()).build();

    private static final SdkField<Integer> SIGNATUREV_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("signatureV").getter(getter(Transaction::signatureV)).setter(setter(Builder::signatureV))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("signatureV").build()).build();

    private static final SdkField<String> SIGNATURER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("signatureR").getter(getter(Transaction::signatureR)).setter(setter(Builder::signatureR))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("signatureR").build()).build();

    private static final SdkField<String> SIGNATURES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("signatureS").getter(getter(Transaction::signatureS)).setter(setter(Builder::signatureS))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("signatureS").build()).build();

    private static final SdkField<String> TRANSACTION_FEE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("transactionFee").getter(getter(Transaction::transactionFee)).setter(setter(Builder::transactionFee))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transactionFee").build()).build();

    private static final SdkField<String> TRANSACTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("transactionId").getter(getter(Transaction::transactionId)).setter(setter(Builder::transactionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transactionId").build()).build();

    private static final SdkField<String> CONFIRMATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("confirmationStatus").getter(getter(Transaction::confirmationStatusAsString))
            .setter(setter(Builder::confirmationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("confirmationStatus").build())
            .build();

    private static final SdkField<String> EXECUTION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("executionStatus").getter(getter(Transaction::executionStatusAsString))
            .setter(setter(Builder::executionStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NETWORK_FIELD,
            BLOCK_HASH_FIELD, TRANSACTION_HASH_FIELD, BLOCK_NUMBER_FIELD, TRANSACTION_TIMESTAMP_FIELD, TRANSACTION_INDEX_FIELD,
            NUMBER_OF_TRANSACTIONS_FIELD, TO_FIELD, FROM_FIELD, CONTRACT_ADDRESS_FIELD, GAS_USED_FIELD,
            CUMULATIVE_GAS_USED_FIELD, EFFECTIVE_GAS_PRICE_FIELD, SIGNATUREV_FIELD, SIGNATURER_FIELD, SIGNATURES_FIELD,
            TRANSACTION_FEE_FIELD, TRANSACTION_ID_FIELD, CONFIRMATION_STATUS_FIELD, EXECUTION_STATUS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String network;

    private final String blockHash;

    private final String transactionHash;

    private final String blockNumber;

    private final Instant transactionTimestamp;

    private final Long transactionIndex;

    private final Long numberOfTransactions;

    private final String to;

    private final String from;

    private final String contractAddress;

    private final String gasUsed;

    private final String cumulativeGasUsed;

    private final String effectiveGasPrice;

    private final Integer signatureV;

    private final String signatureR;

    private final String signatureS;

    private final String transactionFee;

    private final String transactionId;

    private final String confirmationStatus;

    private final String executionStatus;

    private Transaction(BuilderImpl builder) {
        this.network = builder.network;
        this.blockHash = builder.blockHash;
        this.transactionHash = builder.transactionHash;
        this.blockNumber = builder.blockNumber;
        this.transactionTimestamp = builder.transactionTimestamp;
        this.transactionIndex = builder.transactionIndex;
        this.numberOfTransactions = builder.numberOfTransactions;
        this.to = builder.to;
        this.from = builder.from;
        this.contractAddress = builder.contractAddress;
        this.gasUsed = builder.gasUsed;
        this.cumulativeGasUsed = builder.cumulativeGasUsed;
        this.effectiveGasPrice = builder.effectiveGasPrice;
        this.signatureV = builder.signatureV;
        this.signatureR = builder.signatureR;
        this.signatureS = builder.signatureS;
        this.transactionFee = builder.transactionFee;
        this.transactionId = builder.transactionId;
        this.confirmationStatus = builder.confirmationStatus;
        this.executionStatus = builder.executionStatus;
    }

    /**
     * <p>
     * The blockchain network where the transaction occurred.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #network} will
     * return {@link QueryNetwork#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkAsString}.
     * </p>
     * 
     * @return The blockchain network where the transaction occurred.
     * @see QueryNetwork
     */
    public final QueryNetwork network() {
        return QueryNetwork.fromValue(network);
    }

    /**
     * <p>
     * The blockchain network where the transaction occurred.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #network} will
     * return {@link QueryNetwork#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkAsString}.
     * </p>
     * 
     * @return The blockchain network where the transaction occurred.
     * @see QueryNetwork
     */
    public final String networkAsString() {
        return network;
    }

    /**
     * <p>
     * The block hash is a unique identifier for a block. It is a fixed-size string that is calculated by using the
     * information in the block. The block hash is used to verify the integrity of the data in the block.
     * </p>
     * 
     * @return The block hash is a unique identifier for a block. It is a fixed-size string that is calculated by using
     *         the information in the block. The block hash is used to verify the integrity of the data in the block.
     */
    public final String blockHash() {
        return blockHash;
    }

    /**
     * <p>
     * The hash of a transaction. It is generated when a transaction is created.
     * </p>
     * 
     * @return The hash of a transaction. It is generated when a transaction is created.
     */
    public final String transactionHash() {
        return transactionHash;
    }

    /**
     * <p>
     * The block number in which the transaction is recorded.
     * </p>
     * 
     * @return The block number in which the transaction is recorded.
     */
    public final String blockNumber() {
        return blockNumber;
    }

    /**
     * <p>
     * The <code>Timestamp</code> of the transaction.
     * </p>
     * 
     * @return The <code>Timestamp</code> of the transaction.
     */
    public final Instant transactionTimestamp() {
        return transactionTimestamp;
    }

    /**
     * <p>
     * The index of the transaction within a blockchain.
     * </p>
     * 
     * @return The index of the transaction within a blockchain.
     */
    public final Long transactionIndex() {
        return transactionIndex;
    }

    /**
     * <p>
     * The number of transactions in the block.
     * </p>
     * 
     * @return The number of transactions in the block.
     */
    public final Long numberOfTransactions() {
        return numberOfTransactions;
    }

    /**
     * <p>
     * The identifier of the transaction. It is generated whenever a transaction is verified and added to the
     * blockchain.
     * </p>
     * 
     * @return The identifier of the transaction. It is generated whenever a transaction is verified and added to the
     *         blockchain.
     */
    public final String to() {
        return to;
    }

    /**
     * <p>
     * The initiator of the transaction. It is either in the form a public key or a contract address.
     * </p>
     * 
     * @return The initiator of the transaction. It is either in the form a public key or a contract address.
     */
    public final String from() {
        return from;
    }

    /**
     * <p>
     * The blockchain address for the contract.
     * </p>
     * 
     * @return The blockchain address for the contract.
     */
    public final String contractAddress() {
        return contractAddress;
    }

    /**
     * <p>
     * The amount of gas used for the transaction.
     * </p>
     * 
     * @return The amount of gas used for the transaction.
     */
    public final String gasUsed() {
        return gasUsed;
    }

    /**
     * <p>
     * The amount of gas used up to the specified point in the block.
     * </p>
     * 
     * @return The amount of gas used up to the specified point in the block.
     */
    public final String cumulativeGasUsed() {
        return cumulativeGasUsed;
    }

    /**
     * <p>
     * The effective gas price.
     * </p>
     * 
     * @return The effective gas price.
     */
    public final String effectiveGasPrice() {
        return effectiveGasPrice;
    }

    /**
     * <p>
     * The signature of the transaction. The Z coordinate of a point V.
     * </p>
     * 
     * @return The signature of the transaction. The Z coordinate of a point V.
     */
    public final Integer signatureV() {
        return signatureV;
    }

    /**
     * <p>
     * The signature of the transaction. The X coordinate of a point R.
     * </p>
     * 
     * @return The signature of the transaction. The X coordinate of a point R.
     */
    public final String signatureR() {
        return signatureR;
    }

    /**
     * <p>
     * The signature of the transaction. The Y coordinate of a point S.
     * </p>
     * 
     * @return The signature of the transaction. The Y coordinate of a point S.
     */
    public final String signatureS() {
        return signatureS;
    }

    /**
     * <p>
     * The transaction fee.
     * </p>
     * 
     * @return The transaction fee.
     */
    public final String transactionFee() {
        return transactionFee;
    }

    /**
     * <p>
     * The identifier of a Bitcoin transaction. It is generated when a transaction is created.
     * </p>
     * 
     * @return The identifier of a Bitcoin transaction. It is generated when a transaction is created.
     */
    public final String transactionId() {
        return transactionId;
    }

    /**
     * <p>
     * Specifies whether the transaction has reached Finality.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #confirmationStatus} will return {@link ConfirmationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #confirmationStatusAsString}.
     * </p>
     * 
     * @return Specifies whether the transaction has reached Finality.
     * @see ConfirmationStatus
     */
    public final ConfirmationStatus confirmationStatus() {
        return ConfirmationStatus.fromValue(confirmationStatus);
    }

    /**
     * <p>
     * Specifies whether the transaction has reached Finality.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #confirmationStatus} will return {@link ConfirmationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #confirmationStatusAsString}.
     * </p>
     * 
     * @return Specifies whether the transaction has reached Finality.
     * @see ConfirmationStatus
     */
    public final String confirmationStatusAsString() {
        return confirmationStatus;
    }

    /**
     * <p>
     * Identifies whether the transaction has succeeded or failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionStatus}
     * will return {@link ExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #executionStatusAsString}.
     * </p>
     * 
     * @return Identifies whether the transaction has succeeded or failed.
     * @see ExecutionStatus
     */
    public final ExecutionStatus executionStatus() {
        return ExecutionStatus.fromValue(executionStatus);
    }

    /**
     * <p>
     * Identifies whether the transaction has succeeded or failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionStatus}
     * will return {@link ExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #executionStatusAsString}.
     * </p>
     * 
     * @return Identifies whether the transaction has succeeded or failed.
     * @see ExecutionStatus
     */
    public final String executionStatusAsString() {
        return executionStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(networkAsString());
        hashCode = 31 * hashCode + Objects.hashCode(blockHash());
        hashCode = 31 * hashCode + Objects.hashCode(transactionHash());
        hashCode = 31 * hashCode + Objects.hashCode(blockNumber());
        hashCode = 31 * hashCode + Objects.hashCode(transactionTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(transactionIndex());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfTransactions());
        hashCode = 31 * hashCode + Objects.hashCode(to());
        hashCode = 31 * hashCode + Objects.hashCode(from());
        hashCode = 31 * hashCode + Objects.hashCode(contractAddress());
        hashCode = 31 * hashCode + Objects.hashCode(gasUsed());
        hashCode = 31 * hashCode + Objects.hashCode(cumulativeGasUsed());
        hashCode = 31 * hashCode + Objects.hashCode(effectiveGasPrice());
        hashCode = 31 * hashCode + Objects.hashCode(signatureV());
        hashCode = 31 * hashCode + Objects.hashCode(signatureR());
        hashCode = 31 * hashCode + Objects.hashCode(signatureS());
        hashCode = 31 * hashCode + Objects.hashCode(transactionFee());
        hashCode = 31 * hashCode + Objects.hashCode(transactionId());
        hashCode = 31 * hashCode + Objects.hashCode(confirmationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(executionStatusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Transaction)) {
            return false;
        }
        Transaction other = (Transaction) obj;
        return Objects.equals(networkAsString(), other.networkAsString()) && Objects.equals(blockHash(), other.blockHash())
                && Objects.equals(transactionHash(), other.transactionHash())
                && Objects.equals(blockNumber(), other.blockNumber())
                && Objects.equals(transactionTimestamp(), other.transactionTimestamp())
                && Objects.equals(transactionIndex(), other.transactionIndex())
                && Objects.equals(numberOfTransactions(), other.numberOfTransactions()) && Objects.equals(to(), other.to())
                && Objects.equals(from(), other.from()) && Objects.equals(contractAddress(), other.contractAddress())
                && Objects.equals(gasUsed(), other.gasUsed()) && Objects.equals(cumulativeGasUsed(), other.cumulativeGasUsed())
                && Objects.equals(effectiveGasPrice(), other.effectiveGasPrice())
                && Objects.equals(signatureV(), other.signatureV()) && Objects.equals(signatureR(), other.signatureR())
                && Objects.equals(signatureS(), other.signatureS()) && Objects.equals(transactionFee(), other.transactionFee())
                && Objects.equals(transactionId(), other.transactionId())
                && Objects.equals(confirmationStatusAsString(), other.confirmationStatusAsString())
                && Objects.equals(executionStatusAsString(), other.executionStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Transaction").add("Network", networkAsString()).add("BlockHash", blockHash())
                .add("TransactionHash", transactionHash()).add("BlockNumber", blockNumber())
                .add("TransactionTimestamp", transactionTimestamp()).add("TransactionIndex", transactionIndex())
                .add("NumberOfTransactions", numberOfTransactions()).add("To", to()).add("From", from())
                .add("ContractAddress", contractAddress()).add("GasUsed", gasUsed())
                .add("CumulativeGasUsed", cumulativeGasUsed()).add("EffectiveGasPrice", effectiveGasPrice())
                .add("SignatureV", signatureV()).add("SignatureR", signatureR()).add("SignatureS", signatureS())
                .add("TransactionFee", transactionFee()).add("TransactionId", transactionId())
                .add("ConfirmationStatus", confirmationStatusAsString()).add("ExecutionStatus", executionStatusAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "network":
            return Optional.ofNullable(clazz.cast(networkAsString()));
        case "blockHash":
            return Optional.ofNullable(clazz.cast(blockHash()));
        case "transactionHash":
            return Optional.ofNullable(clazz.cast(transactionHash()));
        case "blockNumber":
            return Optional.ofNullable(clazz.cast(blockNumber()));
        case "transactionTimestamp":
            return Optional.ofNullable(clazz.cast(transactionTimestamp()));
        case "transactionIndex":
            return Optional.ofNullable(clazz.cast(transactionIndex()));
        case "numberOfTransactions":
            return Optional.ofNullable(clazz.cast(numberOfTransactions()));
        case "to":
            return Optional.ofNullable(clazz.cast(to()));
        case "from":
            return Optional.ofNullable(clazz.cast(from()));
        case "contractAddress":
            return Optional.ofNullable(clazz.cast(contractAddress()));
        case "gasUsed":
            return Optional.ofNullable(clazz.cast(gasUsed()));
        case "cumulativeGasUsed":
            return Optional.ofNullable(clazz.cast(cumulativeGasUsed()));
        case "effectiveGasPrice":
            return Optional.ofNullable(clazz.cast(effectiveGasPrice()));
        case "signatureV":
            return Optional.ofNullable(clazz.cast(signatureV()));
        case "signatureR":
            return Optional.ofNullable(clazz.cast(signatureR()));
        case "signatureS":
            return Optional.ofNullable(clazz.cast(signatureS()));
        case "transactionFee":
            return Optional.ofNullable(clazz.cast(transactionFee()));
        case "transactionId":
            return Optional.ofNullable(clazz.cast(transactionId()));
        case "confirmationStatus":
            return Optional.ofNullable(clazz.cast(confirmationStatusAsString()));
        case "executionStatus":
            return Optional.ofNullable(clazz.cast(executionStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("network", NETWORK_FIELD);
        map.put("blockHash", BLOCK_HASH_FIELD);
        map.put("transactionHash", TRANSACTION_HASH_FIELD);
        map.put("blockNumber", BLOCK_NUMBER_FIELD);
        map.put("transactionTimestamp", TRANSACTION_TIMESTAMP_FIELD);
        map.put("transactionIndex", TRANSACTION_INDEX_FIELD);
        map.put("numberOfTransactions", NUMBER_OF_TRANSACTIONS_FIELD);
        map.put("to", TO_FIELD);
        map.put("from", FROM_FIELD);
        map.put("contractAddress", CONTRACT_ADDRESS_FIELD);
        map.put("gasUsed", GAS_USED_FIELD);
        map.put("cumulativeGasUsed", CUMULATIVE_GAS_USED_FIELD);
        map.put("effectiveGasPrice", EFFECTIVE_GAS_PRICE_FIELD);
        map.put("signatureV", SIGNATUREV_FIELD);
        map.put("signatureR", SIGNATURER_FIELD);
        map.put("signatureS", SIGNATURES_FIELD);
        map.put("transactionFee", TRANSACTION_FEE_FIELD);
        map.put("transactionId", TRANSACTION_ID_FIELD);
        map.put("confirmationStatus", CONFIRMATION_STATUS_FIELD);
        map.put("executionStatus", EXECUTION_STATUS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Transaction, T> g) {
        return obj -> g.apply((Transaction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Transaction> {
        /**
         * <p>
         * The blockchain network where the transaction occurred.
         * </p>
         * 
         * @param network
         *        The blockchain network where the transaction occurred.
         * @see QueryNetwork
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see QueryNetwork
         */
        Builder network(String network);

        /**
         * <p>
         * The blockchain network where the transaction occurred.
         * </p>
         * 
         * @param network
         *        The blockchain network where the transaction occurred.
         * @see QueryNetwork
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see QueryNetwork
         */
        Builder network(QueryNetwork network);

        /**
         * <p>
         * The block hash is a unique identifier for a block. It is a fixed-size string that is calculated by using the
         * information in the block. The block hash is used to verify the integrity of the data in the block.
         * </p>
         * 
         * @param blockHash
         *        The block hash is a unique identifier for a block. It is a fixed-size string that is calculated by
         *        using the information in the block. The block hash is used to verify the integrity of the data in the
         *        block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockHash(String blockHash);

        /**
         * <p>
         * The hash of a transaction. It is generated when a transaction is created.
         * </p>
         * 
         * @param transactionHash
         *        The hash of a transaction. It is generated when a transaction is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transactionHash(String transactionHash);

        /**
         * <p>
         * The block number in which the transaction is recorded.
         * </p>
         * 
         * @param blockNumber
         *        The block number in which the transaction is recorded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockNumber(String blockNumber);

        /**
         * <p>
         * The <code>Timestamp</code> of the transaction.
         * </p>
         * 
         * @param transactionTimestamp
         *        The <code>Timestamp</code> of the transaction.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transactionTimestamp(Instant transactionTimestamp);

        /**
         * <p>
         * The index of the transaction within a blockchain.
         * </p>
         * 
         * @param transactionIndex
         *        The index of the transaction within a blockchain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transactionIndex(Long transactionIndex);

        /**
         * <p>
         * The number of transactions in the block.
         * </p>
         * 
         * @param numberOfTransactions
         *        The number of transactions in the block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfTransactions(Long numberOfTransactions);

        /**
         * <p>
         * The identifier of the transaction. It is generated whenever a transaction is verified and added to the
         * blockchain.
         * </p>
         * 
         * @param to
         *        The identifier of the transaction. It is generated whenever a transaction is verified and added to the
         *        blockchain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder to(String to);

        /**
         * <p>
         * The initiator of the transaction. It is either in the form a public key or a contract address.
         * </p>
         * 
         * @param from
         *        The initiator of the transaction. It is either in the form a public key or a contract address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder from(String from);

        /**
         * <p>
         * The blockchain address for the contract.
         * </p>
         * 
         * @param contractAddress
         *        The blockchain address for the contract.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contractAddress(String contractAddress);

        /**
         * <p>
         * The amount of gas used for the transaction.
         * </p>
         * 
         * @param gasUsed
         *        The amount of gas used for the transaction.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gasUsed(String gasUsed);

        /**
         * <p>
         * The amount of gas used up to the specified point in the block.
         * </p>
         * 
         * @param cumulativeGasUsed
         *        The amount of gas used up to the specified point in the block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cumulativeGasUsed(String cumulativeGasUsed);

        /**
         * <p>
         * The effective gas price.
         * </p>
         * 
         * @param effectiveGasPrice
         *        The effective gas price.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder effectiveGasPrice(String effectiveGasPrice);

        /**
         * <p>
         * The signature of the transaction. The Z coordinate of a point V.
         * </p>
         * 
         * @param signatureV
         *        The signature of the transaction. The Z coordinate of a point V.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signatureV(Integer signatureV);

        /**
         * <p>
         * The signature of the transaction. The X coordinate of a point R.
         * </p>
         * 
         * @param signatureR
         *        The signature of the transaction. The X coordinate of a point R.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signatureR(String signatureR);

        /**
         * <p>
         * The signature of the transaction. The Y coordinate of a point S.
         * </p>
         * 
         * @param signatureS
         *        The signature of the transaction. The Y coordinate of a point S.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signatureS(String signatureS);

        /**
         * <p>
         * The transaction fee.
         * </p>
         * 
         * @param transactionFee
         *        The transaction fee.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transactionFee(String transactionFee);

        /**
         * <p>
         * The identifier of a Bitcoin transaction. It is generated when a transaction is created.
         * </p>
         * 
         * @param transactionId
         *        The identifier of a Bitcoin transaction. It is generated when a transaction is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transactionId(String transactionId);

        /**
         * <p>
         * Specifies whether the transaction has reached Finality.
         * </p>
         * 
         * @param confirmationStatus
         *        Specifies whether the transaction has reached Finality.
         * @see ConfirmationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfirmationStatus
         */
        Builder confirmationStatus(String confirmationStatus);

        /**
         * <p>
         * Specifies whether the transaction has reached Finality.
         * </p>
         * 
         * @param confirmationStatus
         *        Specifies whether the transaction has reached Finality.
         * @see ConfirmationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfirmationStatus
         */
        Builder confirmationStatus(ConfirmationStatus confirmationStatus);

        /**
         * <p>
         * Identifies whether the transaction has succeeded or failed.
         * </p>
         * 
         * @param executionStatus
         *        Identifies whether the transaction has succeeded or failed.
         * @see ExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionStatus
         */
        Builder executionStatus(String executionStatus);

        /**
         * <p>
         * Identifies whether the transaction has succeeded or failed.
         * </p>
         * 
         * @param executionStatus
         *        Identifies whether the transaction has succeeded or failed.
         * @see ExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionStatus
         */
        Builder executionStatus(ExecutionStatus executionStatus);
    }

    static final class BuilderImpl implements Builder {
        private String network;

        private String blockHash;

        private String transactionHash;

        private String blockNumber;

        private Instant transactionTimestamp;

        private Long transactionIndex;

        private Long numberOfTransactions;

        private String to;

        private String from;

        private String contractAddress;

        private String gasUsed;

        private String cumulativeGasUsed;

        private String effectiveGasPrice;

        private Integer signatureV;

        private String signatureR;

        private String signatureS;

        private String transactionFee;

        private String transactionId;

        private String confirmationStatus;

        private String executionStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(Transaction model) {
            network(model.network);
            blockHash(model.blockHash);
            transactionHash(model.transactionHash);
            blockNumber(model.blockNumber);
            transactionTimestamp(model.transactionTimestamp);
            transactionIndex(model.transactionIndex);
            numberOfTransactions(model.numberOfTransactions);
            to(model.to);
            from(model.from);
            contractAddress(model.contractAddress);
            gasUsed(model.gasUsed);
            cumulativeGasUsed(model.cumulativeGasUsed);
            effectiveGasPrice(model.effectiveGasPrice);
            signatureV(model.signatureV);
            signatureR(model.signatureR);
            signatureS(model.signatureS);
            transactionFee(model.transactionFee);
            transactionId(model.transactionId);
            confirmationStatus(model.confirmationStatus);
            executionStatus(model.executionStatus);
        }

        public final String getNetwork() {
            return network;
        }

        public final void setNetwork(String network) {
            this.network = network;
        }

        @Override
        public final Builder network(String network) {
            this.network = network;
            return this;
        }

        @Override
        public final Builder network(QueryNetwork network) {
            this.network(network == null ? null : network.toString());
            return this;
        }

        public final String getBlockHash() {
            return blockHash;
        }

        public final void setBlockHash(String blockHash) {
            this.blockHash = blockHash;
        }

        @Override
        public final Builder blockHash(String blockHash) {
            this.blockHash = blockHash;
            return this;
        }

        public final String getTransactionHash() {
            return transactionHash;
        }

        public final void setTransactionHash(String transactionHash) {
            this.transactionHash = transactionHash;
        }

        @Override
        public final Builder transactionHash(String transactionHash) {
            this.transactionHash = transactionHash;
            return this;
        }

        public final String getBlockNumber() {
            return blockNumber;
        }

        public final void setBlockNumber(String blockNumber) {
            this.blockNumber = blockNumber;
        }

        @Override
        public final Builder blockNumber(String blockNumber) {
            this.blockNumber = blockNumber;
            return this;
        }

        public final Instant getTransactionTimestamp() {
            return transactionTimestamp;
        }

        public final void setTransactionTimestamp(Instant transactionTimestamp) {
            this.transactionTimestamp = transactionTimestamp;
        }

        @Override
        public final Builder transactionTimestamp(Instant transactionTimestamp) {
            this.transactionTimestamp = transactionTimestamp;
            return this;
        }

        public final Long getTransactionIndex() {
            return transactionIndex;
        }

        public final void setTransactionIndex(Long transactionIndex) {
            this.transactionIndex = transactionIndex;
        }

        @Override
        public final Builder transactionIndex(Long transactionIndex) {
            this.transactionIndex = transactionIndex;
            return this;
        }

        public final Long getNumberOfTransactions() {
            return numberOfTransactions;
        }

        public final void setNumberOfTransactions(Long numberOfTransactions) {
            this.numberOfTransactions = numberOfTransactions;
        }

        @Override
        public final Builder numberOfTransactions(Long numberOfTransactions) {
            this.numberOfTransactions = numberOfTransactions;
            return this;
        }

        public final String getTo() {
            return to;
        }

        public final void setTo(String to) {
            this.to = to;
        }

        @Override
        public final Builder to(String to) {
            this.to = to;
            return this;
        }

        public final String getFrom() {
            return from;
        }

        public final void setFrom(String from) {
            this.from = from;
        }

        @Override
        public final Builder from(String from) {
            this.from = from;
            return this;
        }

        public final String getContractAddress() {
            return contractAddress;
        }

        public final void setContractAddress(String contractAddress) {
            this.contractAddress = contractAddress;
        }

        @Override
        public final Builder contractAddress(String contractAddress) {
            this.contractAddress = contractAddress;
            return this;
        }

        public final String getGasUsed() {
            return gasUsed;
        }

        public final void setGasUsed(String gasUsed) {
            this.gasUsed = gasUsed;
        }

        @Override
        public final Builder gasUsed(String gasUsed) {
            this.gasUsed = gasUsed;
            return this;
        }

        public final String getCumulativeGasUsed() {
            return cumulativeGasUsed;
        }

        public final void setCumulativeGasUsed(String cumulativeGasUsed) {
            this.cumulativeGasUsed = cumulativeGasUsed;
        }

        @Override
        public final Builder cumulativeGasUsed(String cumulativeGasUsed) {
            this.cumulativeGasUsed = cumulativeGasUsed;
            return this;
        }

        public final String getEffectiveGasPrice() {
            return effectiveGasPrice;
        }

        public final void setEffectiveGasPrice(String effectiveGasPrice) {
            this.effectiveGasPrice = effectiveGasPrice;
        }

        @Override
        public final Builder effectiveGasPrice(String effectiveGasPrice) {
            this.effectiveGasPrice = effectiveGasPrice;
            return this;
        }

        public final Integer getSignatureV() {
            return signatureV;
        }

        public final void setSignatureV(Integer signatureV) {
            this.signatureV = signatureV;
        }

        @Override
        public final Builder signatureV(Integer signatureV) {
            this.signatureV = signatureV;
            return this;
        }

        public final String getSignatureR() {
            return signatureR;
        }

        public final void setSignatureR(String signatureR) {
            this.signatureR = signatureR;
        }

        @Override
        public final Builder signatureR(String signatureR) {
            this.signatureR = signatureR;
            return this;
        }

        public final String getSignatureS() {
            return signatureS;
        }

        public final void setSignatureS(String signatureS) {
            this.signatureS = signatureS;
        }

        @Override
        public final Builder signatureS(String signatureS) {
            this.signatureS = signatureS;
            return this;
        }

        public final String getTransactionFee() {
            return transactionFee;
        }

        public final void setTransactionFee(String transactionFee) {
            this.transactionFee = transactionFee;
        }

        @Override
        public final Builder transactionFee(String transactionFee) {
            this.transactionFee = transactionFee;
            return this;
        }

        public final String getTransactionId() {
            return transactionId;
        }

        public final void setTransactionId(String transactionId) {
            this.transactionId = transactionId;
        }

        @Override
        public final Builder transactionId(String transactionId) {
            this.transactionId = transactionId;
            return this;
        }

        public final String getConfirmationStatus() {
            return confirmationStatus;
        }

        public final void setConfirmationStatus(String confirmationStatus) {
            this.confirmationStatus = confirmationStatus;
        }

        @Override
        public final Builder confirmationStatus(String confirmationStatus) {
            this.confirmationStatus = confirmationStatus;
            return this;
        }

        @Override
        public final Builder confirmationStatus(ConfirmationStatus confirmationStatus) {
            this.confirmationStatus(confirmationStatus == null ? null : confirmationStatus.toString());
            return this;
        }

        public final String getExecutionStatus() {
            return executionStatus;
        }

        public final void setExecutionStatus(String executionStatus) {
            this.executionStatus = executionStatus;
        }

        @Override
        public final Builder executionStatus(String executionStatus) {
            this.executionStatus = executionStatus;
            return this;
        }

        @Override
        public final Builder executionStatus(ExecutionStatus executionStatus) {
            this.executionStatus(executionStatus == null ? null : executionStatus.toString());
            return this;
        }

        @Override
        public Transaction build() {
            return new Transaction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
