/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.marketplaceagreement;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.marketplaceagreement.model.DescribeAgreementRequest;
import software.amazon.awssdk.services.marketplaceagreement.model.DescribeAgreementResponse;
import software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsRequest;
import software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsResponse;
import software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsRequest;
import software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsResponse;
import software.amazon.awssdk.services.marketplaceagreement.paginators.GetAgreementTermsPublisher;
import software.amazon.awssdk.services.marketplaceagreement.paginators.SearchAgreementsPublisher;

/**
 * Service client for accessing Agreement Service asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@link SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <p>
 * AWS Marketplace is a curated digital catalog that customers can use to find, buy, deploy, and manage third-party
 * software, data, and services to build solutions and run their businesses. The AWS Marketplace Agreement Service
 * provides an API interface that helps AWS Marketplace sellers manage their product-related agreements, including
 * listing, searching, and filtering agreements.
 * </p>
 * <p>
 * To manage agreements in AWS Marketplace, you must ensure that your AWS Identity and Access Management (IAM) policies
 * and roles are set up. The user must have the required policies/permissions that allow them to carry out the actions
 * in AWS:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>DescribeAgreement</code> – Grants permission to users to obtain detailed meta data about any of their
 * agreements.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>GetAgreementTerms</code> – Grants permission to users to obtain details about the terms of an agreement.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>SearchAgreements</code> – Grants permission to users to search through all their agreements.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface MarketplaceAgreementAsyncClient extends AwsClient {
    String SERVICE_NAME = "aws-marketplace";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "agreement-marketplace";

    /**
     * <p>
     * Provides details about an agreement, such as the proposer, acceptor, start date, and end date.
     * </p>
     *
     * @param describeAgreementRequest
     * @return A Java Future containing the result of the DescribeAgreement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceAgreementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceAgreementAsyncClient.DescribeAgreement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/DescribeAgreement"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeAgreementResponse> describeAgreement(DescribeAgreementRequest describeAgreementRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Provides details about an agreement, such as the proposer, acceptor, start date, and end date.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeAgreementRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeAgreementRequest#builder()}
     * </p>
     *
     * @param describeAgreementRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.marketplaceagreement.model.DescribeAgreementRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the DescribeAgreement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceAgreementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceAgreementAsyncClient.DescribeAgreement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/DescribeAgreement"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeAgreementResponse> describeAgreement(
            Consumer<DescribeAgreementRequest.Builder> describeAgreementRequest) {
        return describeAgreement(DescribeAgreementRequest.builder().applyMutation(describeAgreementRequest).build());
    }

    /**
     * <p>
     * Obtains details about the terms in an agreement that you participated in as proposer or acceptor.
     * </p>
     * <p>
     * The details include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TermType</code> – The type of term, such as <code>LegalTerm</code>, <code>RenewalTerm</code>, or
     * <code>ConfigurableUpfrontPricingTerm</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TermID</code> – The ID of the particular term, which is common between offer and agreement.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TermPayload</code> – The key information contained in the term, such as the EULA for <code>LegalTerm</code>
     * or pricing and dimensions for various pricing terms, such as <code>ConfigurableUpfrontPricingTerm</code> or
     * <code>UsageBasedPricingTerm</code>.
     * </p>
     * </li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * <code>Configuration</code> – The buyer/acceptor's selection at the time of agreement creation, such as the number
     * of units purchased for a dimension or setting the <code>EnableAutoRenew</code> flag.
     * </p>
     * </li>
     * </ul>
     *
     * @param getAgreementTermsRequest
     * @return A Java Future containing the result of the GetAgreementTerms operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceAgreementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceAgreementAsyncClient.GetAgreementTerms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/GetAgreementTerms"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetAgreementTermsResponse> getAgreementTerms(GetAgreementTermsRequest getAgreementTermsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Obtains details about the terms in an agreement that you participated in as proposer or acceptor.
     * </p>
     * <p>
     * The details include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TermType</code> – The type of term, such as <code>LegalTerm</code>, <code>RenewalTerm</code>, or
     * <code>ConfigurableUpfrontPricingTerm</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TermID</code> – The ID of the particular term, which is common between offer and agreement.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TermPayload</code> – The key information contained in the term, such as the EULA for <code>LegalTerm</code>
     * or pricing and dimensions for various pricing terms, such as <code>ConfigurableUpfrontPricingTerm</code> or
     * <code>UsageBasedPricingTerm</code>.
     * </p>
     * </li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * <code>Configuration</code> – The buyer/acceptor's selection at the time of agreement creation, such as the number
     * of units purchased for a dimension or setting the <code>EnableAutoRenew</code> flag.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAgreementTermsRequest.Builder} avoiding the need
     * to create one manually via {@link GetAgreementTermsRequest#builder()}
     * </p>
     *
     * @param getAgreementTermsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the GetAgreementTerms operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceAgreementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceAgreementAsyncClient.GetAgreementTerms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/GetAgreementTerms"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetAgreementTermsResponse> getAgreementTerms(
            Consumer<GetAgreementTermsRequest.Builder> getAgreementTermsRequest) {
        return getAgreementTerms(GetAgreementTermsRequest.builder().applyMutation(getAgreementTermsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #getAgreementTerms(software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.marketplaceagreement.paginators.GetAgreementTermsPublisher publisher = client.getAgreementTermsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.marketplaceagreement.paginators.GetAgreementTermsPublisher publisher = client.getAgreementTermsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getAgreementTerms(software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getAgreementTermsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceAgreementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceAgreementAsyncClient.GetAgreementTerms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/GetAgreementTerms"
     *      target="_top">AWS API Documentation</a>
     */
    default GetAgreementTermsPublisher getAgreementTermsPaginator(GetAgreementTermsRequest getAgreementTermsRequest) {
        return new GetAgreementTermsPublisher(this, getAgreementTermsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #getAgreementTerms(software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.marketplaceagreement.paginators.GetAgreementTermsPublisher publisher = client.getAgreementTermsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.marketplaceagreement.paginators.GetAgreementTermsPublisher publisher = client.getAgreementTermsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getAgreementTerms(software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAgreementTermsRequest.Builder} avoiding the need
     * to create one manually via {@link GetAgreementTermsRequest#builder()}
     * </p>
     *
     * @param getAgreementTermsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.marketplaceagreement.model.GetAgreementTermsRequest.Builder} to
     *        create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceAgreementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceAgreementAsyncClient.GetAgreementTerms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/GetAgreementTerms"
     *      target="_top">AWS API Documentation</a>
     */
    default GetAgreementTermsPublisher getAgreementTermsPaginator(
            Consumer<GetAgreementTermsRequest.Builder> getAgreementTermsRequest) {
        return getAgreementTermsPaginator(GetAgreementTermsRequest.builder().applyMutation(getAgreementTermsRequest).build());
    }

    /**
     * <p>
     * Searches across all agreements that a proposer or an acceptor has in AWS Marketplace. The search returns a list
     * of agreements with basic agreement information.
     * </p>
     * <p>
     * The following filter combinations are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>ResourceIdentifier</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>OfferId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>ResourceIdentifier</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>OfferId</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>ResourceType</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceType</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>OfferId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>OfferId</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceIdentifier</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceIdentifier</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceType</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param searchAgreementsRequest
     * @return A Java Future containing the result of the SearchAgreements operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceAgreementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceAgreementAsyncClient.SearchAgreements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/SearchAgreements"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<SearchAgreementsResponse> searchAgreements(SearchAgreementsRequest searchAgreementsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Searches across all agreements that a proposer or an acceptor has in AWS Marketplace. The search returns a list
     * of agreements with basic agreement information.
     * </p>
     * <p>
     * The following filter combinations are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>ResourceIdentifier</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>OfferId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>ResourceIdentifier</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>OfferId</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>ResourceType</code> +
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceType</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>OfferId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>OfferId</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceIdentifier</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceIdentifier</code> + <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PartyType</code> as <code>Proposer</code> + <code>AgreementType</code> + <code>AcceptorAccountId</code> +
     * <code>ResourceType</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SearchAgreementsRequest.Builder} avoiding the need
     * to create one manually via {@link SearchAgreementsRequest#builder()}
     * </p>
     *
     * @param searchAgreementsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the SearchAgreements operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceAgreementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceAgreementAsyncClient.SearchAgreements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/SearchAgreements"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<SearchAgreementsResponse> searchAgreements(
            Consumer<SearchAgreementsRequest.Builder> searchAgreementsRequest) {
        return searchAgreements(SearchAgreementsRequest.builder().applyMutation(searchAgreementsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #searchAgreements(software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.marketplaceagreement.paginators.SearchAgreementsPublisher publisher = client.searchAgreementsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.marketplaceagreement.paginators.SearchAgreementsPublisher publisher = client.searchAgreementsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchAgreements(software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsRequest)}
     * operation.</b>
     * </p>
     *
     * @param searchAgreementsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceAgreementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceAgreementAsyncClient.SearchAgreements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/SearchAgreements"
     *      target="_top">AWS API Documentation</a>
     */
    default SearchAgreementsPublisher searchAgreementsPaginator(SearchAgreementsRequest searchAgreementsRequest) {
        return new SearchAgreementsPublisher(this, searchAgreementsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #searchAgreements(software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.marketplaceagreement.paginators.SearchAgreementsPublisher publisher = client.searchAgreementsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.marketplaceagreement.paginators.SearchAgreementsPublisher publisher = client.searchAgreementsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchAgreements(software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SearchAgreementsRequest.Builder} avoiding the need
     * to create one manually via {@link SearchAgreementsRequest#builder()}
     * </p>
     *
     * @param searchAgreementsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.marketplaceagreement.model.SearchAgreementsRequest.Builder} to
     *        create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the service.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MarketplaceAgreementException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample MarketplaceAgreementAsyncClient.SearchAgreements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplace-agreement-2020-03-01/SearchAgreements"
     *      target="_top">AWS API Documentation</a>
     */
    default SearchAgreementsPublisher searchAgreementsPaginator(Consumer<SearchAgreementsRequest.Builder> searchAgreementsRequest) {
        return searchAgreementsPaginator(SearchAgreementsRequest.builder().applyMutation(searchAgreementsRequest).build());
    }

    @Override
    default MarketplaceAgreementServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link MarketplaceAgreementAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static MarketplaceAgreementAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link MarketplaceAgreementAsyncClient}.
     */
    static MarketplaceAgreementAsyncClientBuilder builder() {
        return new DefaultMarketplaceAgreementAsyncClientBuilder();
    }
}
