/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.marketplacecommerceanalytics;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.marketplacecommerceanalytics.internal.MarketplaceCommerceAnalyticsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.marketplacecommerceanalytics.model.GenerateDataSetRequest;
import software.amazon.awssdk.services.marketplacecommerceanalytics.model.GenerateDataSetResponse;
import software.amazon.awssdk.services.marketplacecommerceanalytics.model.MarketplaceCommerceAnalyticsException;
import software.amazon.awssdk.services.marketplacecommerceanalytics.model.MarketplaceCommerceAnalyticsServiceException;
import software.amazon.awssdk.services.marketplacecommerceanalytics.model.StartSupportDataExportRequest;
import software.amazon.awssdk.services.marketplacecommerceanalytics.model.StartSupportDataExportResponse;
import software.amazon.awssdk.services.marketplacecommerceanalytics.transform.GenerateDataSetRequestMarshaller;
import software.amazon.awssdk.services.marketplacecommerceanalytics.transform.StartSupportDataExportRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MarketplaceCommerceAnalyticsClient}.
 *
 * @see MarketplaceCommerceAnalyticsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMarketplaceCommerceAnalyticsClient implements MarketplaceCommerceAnalyticsClient {
    private static final Logger log = Logger.loggerFor(DefaultMarketplaceCommerceAnalyticsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMarketplaceCommerceAnalyticsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * Given a data set type and data set publication date, asynchronously publishes the requested data set to the
     * specified S3 bucket and notifies the specified SNS topic once the data is available. Returns a unique request
     * identifier that can be used to correlate requests with notifications from the SNS topic. Data sets will be
     * published in comma-separated values (CSV) format with the file name {data_set_type}_YYYY-MM-DD.csv. If a file
     * with the same name already exists (e.g. if the same data set is requested twice), the original file will be
     * overwritten by the new file. Requires a Role with an attached permissions policy providing Allow permissions for
     * the following actions: s3:PutObject, s3:GetBucketLocation, sns:GetTopicAttributes, sns:Publish,
     * iam:GetRolePolicy.
     *
     * @param generateDataSetRequest
     *        Container for the parameters to the GenerateDataSet operation.
     * @return Result of the GenerateDataSet operation returned by the service.
     * @throws MarketplaceCommerceAnalyticsServiceException
     *         This exception is thrown when an internal service error occurs.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCommerceAnalyticsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCommerceAnalyticsClient.GenerateDataSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/marketplacecommerceanalytics-2015-07-01/GenerateDataSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GenerateDataSetResponse generateDataSet(GenerateDataSetRequest generateDataSetRequest)
            throws MarketplaceCommerceAnalyticsServiceException, AwsServiceException, SdkClientException,
            MarketplaceCommerceAnalyticsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GenerateDataSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GenerateDataSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(generateDataSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, generateDataSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Commerce Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GenerateDataSet");

            return clientHandler.execute(new ClientExecutionParams<GenerateDataSetRequest, GenerateDataSetResponse>()
                    .withOperationName("GenerateDataSet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(generateDataSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GenerateDataSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <i>This target has been deprecated.</i> Given a data set type and a from date, asynchronously publishes the
     * requested customer support data to the specified S3 bucket and notifies the specified SNS topic once the data is
     * available. Returns a unique request identifier that can be used to correlate requests with notifications from the
     * SNS topic. Data sets will be published in comma-separated values (CSV) format with the file name
     * {data_set_type}_YYYY-MM-DD'T'HH-mm-ss'Z'.csv. If a file with the same name already exists (e.g. if the same data
     * set is requested twice), the original file will be overwritten by the new file. Requires a Role with an attached
     * permissions policy providing Allow permissions for the following actions: s3:PutObject, s3:GetBucketLocation,
     * sns:GetTopicAttributes, sns:Publish, iam:GetRolePolicy.
     *
     * @param startSupportDataExportRequest
     *        This target has been deprecated.
     * @return Result of the StartSupportDataExport operation returned by the service.
     * @throws MarketplaceCommerceAnalyticsServiceException
     *         This exception is thrown when an internal service error occurs.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceCommerceAnalyticsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceCommerceAnalyticsClient.StartSupportDataExport
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/marketplacecommerceanalytics-2015-07-01/StartSupportDataExport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartSupportDataExportResponse startSupportDataExport(StartSupportDataExportRequest startSupportDataExportRequest)
            throws MarketplaceCommerceAnalyticsServiceException, AwsServiceException, SdkClientException,
            MarketplaceCommerceAnalyticsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartSupportDataExportResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartSupportDataExportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startSupportDataExportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSupportDataExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Commerce Analytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSupportDataExport");

            return clientHandler
                    .execute(new ClientExecutionParams<StartSupportDataExportRequest, StartSupportDataExportResponse>()
                            .withOperationName("StartSupportDataExport").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startSupportDataExportRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartSupportDataExportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MarketplaceCommerceAnalyticsServiceClientConfigurationBuilder serviceConfigBuilder = new MarketplaceCommerceAnalyticsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MarketplaceCommerceAnalyticsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MarketplaceCommerceAnalyticsServiceException")
                                .exceptionBuilderSupplier(MarketplaceCommerceAnalyticsServiceException::builder)
                                .httpStatusCode(500).build());
    }

    @Override
    public final MarketplaceCommerceAnalyticsServiceClientConfiguration serviceClientConfiguration() {
        return new MarketplaceCommerceAnalyticsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
