/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.marketplaceentitlement.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An entitlement represents capacity in a product owned by the customer. For example, a customer might own some number
 * of users or seats in an SaaS application or some amount of data capacity in a multi-tenant database.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Entitlement implements SdkPojo, Serializable, ToCopyableBuilder<Entitlement.Builder, Entitlement> {
    private static final SdkField<String> PRODUCT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProductCode").getter(getter(Entitlement::productCode)).setter(setter(Builder::productCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProductCode").build()).build();

    private static final SdkField<String> DIMENSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Dimension").getter(getter(Entitlement::dimension)).setter(setter(Builder::dimension))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimension").build()).build();

    private static final SdkField<String> CUSTOMER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomerIdentifier").getter(getter(Entitlement::customerIdentifier))
            .setter(setter(Builder::customerIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerIdentifier").build())
            .build();

    private static final SdkField<EntitlementValue> VALUE_FIELD = SdkField.<EntitlementValue> builder(MarshallingType.SDK_POJO)
            .memberName("Value").getter(getter(Entitlement::value)).setter(setter(Builder::value))
            .constructor(EntitlementValue::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Value").build()).build();

    private static final SdkField<Instant> EXPIRATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ExpirationDate").getter(getter(Entitlement::expirationDate)).setter(setter(Builder::expirationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpirationDate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PRODUCT_CODE_FIELD,
            DIMENSION_FIELD, CUSTOMER_IDENTIFIER_FIELD, VALUE_FIELD, EXPIRATION_DATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String productCode;

    private final String dimension;

    private final String customerIdentifier;

    private final EntitlementValue value;

    private final Instant expirationDate;

    private Entitlement(BuilderImpl builder) {
        this.productCode = builder.productCode;
        this.dimension = builder.dimension;
        this.customerIdentifier = builder.customerIdentifier;
        this.value = builder.value;
        this.expirationDate = builder.expirationDate;
    }

    /**
     * <p>
     * The product code for which the given entitlement applies. Product codes are provided by AWS Marketplace when the
     * product listing is created.
     * </p>
     * 
     * @return The product code for which the given entitlement applies. Product codes are provided by AWS Marketplace
     *         when the product listing is created.
     */
    public final String productCode() {
        return productCode;
    }

    /**
     * <p>
     * The dimension for which the given entitlement applies. Dimensions represent categories of capacity in a product
     * and are specified when the product is listed in AWS Marketplace.
     * </p>
     * 
     * @return The dimension for which the given entitlement applies. Dimensions represent categories of capacity in a
     *         product and are specified when the product is listed in AWS Marketplace.
     */
    public final String dimension() {
        return dimension;
    }

    /**
     * <p>
     * The customer identifier is a handle to each unique customer in an application. Customer identifiers are obtained
     * through the ResolveCustomer operation in AWS Marketplace Metering Service.
     * </p>
     * 
     * @return The customer identifier is a handle to each unique customer in an application. Customer identifiers are
     *         obtained through the ResolveCustomer operation in AWS Marketplace Metering Service.
     */
    public final String customerIdentifier() {
        return customerIdentifier;
    }

    /**
     * <p>
     * The EntitlementValue represents the amount of capacity that the customer is entitled to for the product.
     * </p>
     * 
     * @return The EntitlementValue represents the amount of capacity that the customer is entitled to for the product.
     */
    public final EntitlementValue value() {
        return value;
    }

    /**
     * <p>
     * The expiration date represents the minimum date through which this entitlement is expected to remain valid. For
     * contractual products listed on AWS Marketplace, the expiration date is the date at which the customer will renew
     * or cancel their contract. Customers who are opting to renew their contract will still have entitlements with an
     * expiration date.
     * </p>
     * 
     * @return The expiration date represents the minimum date through which this entitlement is expected to remain
     *         valid. For contractual products listed on AWS Marketplace, the expiration date is the date at which the
     *         customer will renew or cancel their contract. Customers who are opting to renew their contract will still
     *         have entitlements with an expiration date.
     */
    public final Instant expirationDate() {
        return expirationDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(productCode());
        hashCode = 31 * hashCode + Objects.hashCode(dimension());
        hashCode = 31 * hashCode + Objects.hashCode(customerIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(expirationDate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Entitlement)) {
            return false;
        }
        Entitlement other = (Entitlement) obj;
        return Objects.equals(productCode(), other.productCode()) && Objects.equals(dimension(), other.dimension())
                && Objects.equals(customerIdentifier(), other.customerIdentifier()) && Objects.equals(value(), other.value())
                && Objects.equals(expirationDate(), other.expirationDate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Entitlement").add("ProductCode", productCode()).add("Dimension", dimension())
                .add("CustomerIdentifier", customerIdentifier()).add("Value", value()).add("ExpirationDate", expirationDate())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ProductCode":
            return Optional.ofNullable(clazz.cast(productCode()));
        case "Dimension":
            return Optional.ofNullable(clazz.cast(dimension()));
        case "CustomerIdentifier":
            return Optional.ofNullable(clazz.cast(customerIdentifier()));
        case "Value":
            return Optional.ofNullable(clazz.cast(value()));
        case "ExpirationDate":
            return Optional.ofNullable(clazz.cast(expirationDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ProductCode", PRODUCT_CODE_FIELD);
        map.put("Dimension", DIMENSION_FIELD);
        map.put("CustomerIdentifier", CUSTOMER_IDENTIFIER_FIELD);
        map.put("Value", VALUE_FIELD);
        map.put("ExpirationDate", EXPIRATION_DATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Entitlement, T> g) {
        return obj -> g.apply((Entitlement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Entitlement> {
        /**
         * <p>
         * The product code for which the given entitlement applies. Product codes are provided by AWS Marketplace when
         * the product listing is created.
         * </p>
         * 
         * @param productCode
         *        The product code for which the given entitlement applies. Product codes are provided by AWS
         *        Marketplace when the product listing is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productCode(String productCode);

        /**
         * <p>
         * The dimension for which the given entitlement applies. Dimensions represent categories of capacity in a
         * product and are specified when the product is listed in AWS Marketplace.
         * </p>
         * 
         * @param dimension
         *        The dimension for which the given entitlement applies. Dimensions represent categories of capacity in
         *        a product and are specified when the product is listed in AWS Marketplace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimension(String dimension);

        /**
         * <p>
         * The customer identifier is a handle to each unique customer in an application. Customer identifiers are
         * obtained through the ResolveCustomer operation in AWS Marketplace Metering Service.
         * </p>
         * 
         * @param customerIdentifier
         *        The customer identifier is a handle to each unique customer in an application. Customer identifiers
         *        are obtained through the ResolveCustomer operation in AWS Marketplace Metering Service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerIdentifier(String customerIdentifier);

        /**
         * <p>
         * The EntitlementValue represents the amount of capacity that the customer is entitled to for the product.
         * </p>
         * 
         * @param value
         *        The EntitlementValue represents the amount of capacity that the customer is entitled to for the
         *        product.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(EntitlementValue value);

        /**
         * <p>
         * The EntitlementValue represents the amount of capacity that the customer is entitled to for the product.
         * </p>
         * This is a convenience method that creates an instance of the {@link EntitlementValue.Builder} avoiding the
         * need to create one manually via {@link EntitlementValue#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EntitlementValue.Builder#build()} is called immediately and its
         * result is passed to {@link #value(EntitlementValue)}.
         * 
         * @param value
         *        a consumer that will call methods on {@link EntitlementValue.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #value(EntitlementValue)
         */
        default Builder value(Consumer<EntitlementValue.Builder> value) {
            return value(EntitlementValue.builder().applyMutation(value).build());
        }

        /**
         * <p>
         * The expiration date represents the minimum date through which this entitlement is expected to remain valid.
         * For contractual products listed on AWS Marketplace, the expiration date is the date at which the customer
         * will renew or cancel their contract. Customers who are opting to renew their contract will still have
         * entitlements with an expiration date.
         * </p>
         * 
         * @param expirationDate
         *        The expiration date represents the minimum date through which this entitlement is expected to remain
         *        valid. For contractual products listed on AWS Marketplace, the expiration date is the date at which
         *        the customer will renew or cancel their contract. Customers who are opting to renew their contract
         *        will still have entitlements with an expiration date.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expirationDate(Instant expirationDate);
    }

    static final class BuilderImpl implements Builder {
        private String productCode;

        private String dimension;

        private String customerIdentifier;

        private EntitlementValue value;

        private Instant expirationDate;

        private BuilderImpl() {
        }

        private BuilderImpl(Entitlement model) {
            productCode(model.productCode);
            dimension(model.dimension);
            customerIdentifier(model.customerIdentifier);
            value(model.value);
            expirationDate(model.expirationDate);
        }

        public final String getProductCode() {
            return productCode;
        }

        public final void setProductCode(String productCode) {
            this.productCode = productCode;
        }

        @Override
        public final Builder productCode(String productCode) {
            this.productCode = productCode;
            return this;
        }

        public final String getDimension() {
            return dimension;
        }

        public final void setDimension(String dimension) {
            this.dimension = dimension;
        }

        @Override
        public final Builder dimension(String dimension) {
            this.dimension = dimension;
            return this;
        }

        public final String getCustomerIdentifier() {
            return customerIdentifier;
        }

        public final void setCustomerIdentifier(String customerIdentifier) {
            this.customerIdentifier = customerIdentifier;
        }

        @Override
        public final Builder customerIdentifier(String customerIdentifier) {
            this.customerIdentifier = customerIdentifier;
            return this;
        }

        public final EntitlementValue.Builder getValue() {
            return value != null ? value.toBuilder() : null;
        }

        public final void setValue(EntitlementValue.BuilderImpl value) {
            this.value = value != null ? value.build() : null;
        }

        @Override
        public final Builder value(EntitlementValue value) {
            this.value = value;
            return this;
        }

        public final Instant getExpirationDate() {
            return expirationDate;
        }

        public final void setExpirationDate(Instant expirationDate) {
            this.expirationDate = expirationDate;
        }

        @Override
        public final Builder expirationDate(Instant expirationDate) {
            this.expirationDate = expirationDate;
            return this;
        }

        @Override
        public Entitlement build() {
            return new Entitlement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
