/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.marketplacemetering.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A <code>UsageRecord</code> indicates a quantity of usage for a given product, customer, dimension and time.
 * </p>
 * <p>
 * Multiple requests with the same <code>UsageRecords</code> as input will be de-duplicated to prevent double charges.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UsageRecord implements SdkPojo, Serializable, ToCopyableBuilder<UsageRecord.Builder, UsageRecord> {
    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("Timestamp").getter(getter(UsageRecord::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final SdkField<String> CUSTOMER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomerIdentifier").getter(getter(UsageRecord::customerIdentifier))
            .setter(setter(Builder::customerIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerIdentifier").build())
            .build();

    private static final SdkField<String> DIMENSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Dimension").getter(getter(UsageRecord::dimension)).setter(setter(Builder::dimension))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimension").build()).build();

    private static final SdkField<Integer> QUANTITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Quantity").getter(getter(UsageRecord::quantity)).setter(setter(Builder::quantity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Quantity").build()).build();

    private static final SdkField<List<UsageAllocation>> USAGE_ALLOCATIONS_FIELD = SdkField
            .<List<UsageAllocation>> builder(MarshallingType.LIST)
            .memberName("UsageAllocations")
            .getter(getter(UsageRecord::usageAllocations))
            .setter(setter(Builder::usageAllocations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsageAllocations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UsageAllocation> builder(MarshallingType.SDK_POJO)
                                            .constructor(UsageAllocation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TIMESTAMP_FIELD,
            CUSTOMER_IDENTIFIER_FIELD, DIMENSION_FIELD, QUANTITY_FIELD, USAGE_ALLOCATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant timestamp;

    private final String customerIdentifier;

    private final String dimension;

    private final Integer quantity;

    private final List<UsageAllocation> usageAllocations;

    private UsageRecord(BuilderImpl builder) {
        this.timestamp = builder.timestamp;
        this.customerIdentifier = builder.customerIdentifier;
        this.dimension = builder.dimension;
        this.quantity = builder.quantity;
        this.usageAllocations = builder.usageAllocations;
    }

    /**
     * <p>
     * Timestamp, in UTC, for which the usage is being reported.
     * </p>
     * <p>
     * Your application can meter usage for up to one hour in the past. Make sure the <code>timestamp</code> value is
     * not before the start of the software usage.
     * </p>
     * 
     * @return Timestamp, in UTC, for which the usage is being reported.</p>
     *         <p>
     *         Your application can meter usage for up to one hour in the past. Make sure the <code>timestamp</code>
     *         value is not before the start of the software usage.
     */
    public final Instant timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * The <code>CustomerIdentifier</code> is obtained through the <code>ResolveCustomer</code> operation and represents
     * an individual buyer in your application.
     * </p>
     * 
     * @return The <code>CustomerIdentifier</code> is obtained through the <code>ResolveCustomer</code> operation and
     *         represents an individual buyer in your application.
     */
    public final String customerIdentifier() {
        return customerIdentifier;
    }

    /**
     * <p>
     * During the process of registering a product on AWS Marketplace, dimensions are specified. These represent
     * different units of value in your application.
     * </p>
     * 
     * @return During the process of registering a product on AWS Marketplace, dimensions are specified. These represent
     *         different units of value in your application.
     */
    public final String dimension() {
        return dimension;
    }

    /**
     * <p>
     * The quantity of usage consumed by the customer for the given dimension and time. Defaults to <code>0</code> if
     * not specified.
     * </p>
     * 
     * @return The quantity of usage consumed by the customer for the given dimension and time. Defaults to
     *         <code>0</code> if not specified.
     */
    public final Integer quantity() {
        return quantity;
    }

    /**
     * For responses, this returns true if the service returned a value for the UsageAllocations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUsageAllocations() {
        return usageAllocations != null && !(usageAllocations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of <code>UsageAllocations</code> to submit. The sum of all <code>UsageAllocation</code> quantities must
     * equal the Quantity of the <code>UsageRecord</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUsageAllocations} method.
     * </p>
     * 
     * @return The set of <code>UsageAllocations</code> to submit. The sum of all <code>UsageAllocation</code>
     *         quantities must equal the Quantity of the <code>UsageRecord</code>.
     */
    public final List<UsageAllocation> usageAllocations() {
        return usageAllocations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(customerIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(dimension());
        hashCode = 31 * hashCode + Objects.hashCode(quantity());
        hashCode = 31 * hashCode + Objects.hashCode(hasUsageAllocations() ? usageAllocations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UsageRecord)) {
            return false;
        }
        UsageRecord other = (UsageRecord) obj;
        return Objects.equals(timestamp(), other.timestamp()) && Objects.equals(customerIdentifier(), other.customerIdentifier())
                && Objects.equals(dimension(), other.dimension()) && Objects.equals(quantity(), other.quantity())
                && hasUsageAllocations() == other.hasUsageAllocations()
                && Objects.equals(usageAllocations(), other.usageAllocations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UsageRecord").add("Timestamp", timestamp()).add("CustomerIdentifier", customerIdentifier())
                .add("Dimension", dimension()).add("Quantity", quantity())
                .add("UsageAllocations", hasUsageAllocations() ? usageAllocations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "CustomerIdentifier":
            return Optional.ofNullable(clazz.cast(customerIdentifier()));
        case "Dimension":
            return Optional.ofNullable(clazz.cast(dimension()));
        case "Quantity":
            return Optional.ofNullable(clazz.cast(quantity()));
        case "UsageAllocations":
            return Optional.ofNullable(clazz.cast(usageAllocations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UsageRecord, T> g) {
        return obj -> g.apply((UsageRecord) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UsageRecord> {
        /**
         * <p>
         * Timestamp, in UTC, for which the usage is being reported.
         * </p>
         * <p>
         * Your application can meter usage for up to one hour in the past. Make sure the <code>timestamp</code> value
         * is not before the start of the software usage.
         * </p>
         * 
         * @param timestamp
         *        Timestamp, in UTC, for which the usage is being reported.</p>
         *        <p>
         *        Your application can meter usage for up to one hour in the past. Make sure the <code>timestamp</code>
         *        value is not before the start of the software usage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * The <code>CustomerIdentifier</code> is obtained through the <code>ResolveCustomer</code> operation and
         * represents an individual buyer in your application.
         * </p>
         * 
         * @param customerIdentifier
         *        The <code>CustomerIdentifier</code> is obtained through the <code>ResolveCustomer</code> operation and
         *        represents an individual buyer in your application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerIdentifier(String customerIdentifier);

        /**
         * <p>
         * During the process of registering a product on AWS Marketplace, dimensions are specified. These represent
         * different units of value in your application.
         * </p>
         * 
         * @param dimension
         *        During the process of registering a product on AWS Marketplace, dimensions are specified. These
         *        represent different units of value in your application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimension(String dimension);

        /**
         * <p>
         * The quantity of usage consumed by the customer for the given dimension and time. Defaults to <code>0</code>
         * if not specified.
         * </p>
         * 
         * @param quantity
         *        The quantity of usage consumed by the customer for the given dimension and time. Defaults to
         *        <code>0</code> if not specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * The set of <code>UsageAllocations</code> to submit. The sum of all <code>UsageAllocation</code> quantities
         * must equal the Quantity of the <code>UsageRecord</code>.
         * </p>
         * 
         * @param usageAllocations
         *        The set of <code>UsageAllocations</code> to submit. The sum of all <code>UsageAllocation</code>
         *        quantities must equal the Quantity of the <code>UsageRecord</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usageAllocations(Collection<UsageAllocation> usageAllocations);

        /**
         * <p>
         * The set of <code>UsageAllocations</code> to submit. The sum of all <code>UsageAllocation</code> quantities
         * must equal the Quantity of the <code>UsageRecord</code>.
         * </p>
         * 
         * @param usageAllocations
         *        The set of <code>UsageAllocations</code> to submit. The sum of all <code>UsageAllocation</code>
         *        quantities must equal the Quantity of the <code>UsageRecord</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usageAllocations(UsageAllocation... usageAllocations);

        /**
         * <p>
         * The set of <code>UsageAllocations</code> to submit. The sum of all <code>UsageAllocation</code> quantities
         * must equal the Quantity of the <code>UsageRecord</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<UsageAllocation>.Builder} avoiding
         * the need to create one manually via {@link List<UsageAllocation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<UsageAllocation>.Builder#build()} is called immediately and
         * its result is passed to {@link #usageAllocations(List<UsageAllocation>)}.
         * 
         * @param usageAllocations
         *        a consumer that will call methods on {@link List<UsageAllocation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #usageAllocations(List<UsageAllocation>)
         */
        Builder usageAllocations(Consumer<UsageAllocation.Builder>... usageAllocations);
    }

    static final class BuilderImpl implements Builder {
        private Instant timestamp;

        private String customerIdentifier;

        private String dimension;

        private Integer quantity;

        private List<UsageAllocation> usageAllocations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UsageRecord model) {
            timestamp(model.timestamp);
            customerIdentifier(model.customerIdentifier);
            dimension(model.dimension);
            quantity(model.quantity);
            usageAllocations(model.usageAllocations);
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final String getCustomerIdentifier() {
            return customerIdentifier;
        }

        public final void setCustomerIdentifier(String customerIdentifier) {
            this.customerIdentifier = customerIdentifier;
        }

        @Override
        public final Builder customerIdentifier(String customerIdentifier) {
            this.customerIdentifier = customerIdentifier;
            return this;
        }

        public final String getDimension() {
            return dimension;
        }

        public final void setDimension(String dimension) {
            this.dimension = dimension;
        }

        @Override
        public final Builder dimension(String dimension) {
            this.dimension = dimension;
            return this;
        }

        public final Integer getQuantity() {
            return quantity;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final List<UsageAllocation.Builder> getUsageAllocations() {
            List<UsageAllocation.Builder> result = UsageAllocationsCopier.copyToBuilder(this.usageAllocations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUsageAllocations(Collection<UsageAllocation.BuilderImpl> usageAllocations) {
            this.usageAllocations = UsageAllocationsCopier.copyFromBuilder(usageAllocations);
        }

        @Override
        public final Builder usageAllocations(Collection<UsageAllocation> usageAllocations) {
            this.usageAllocations = UsageAllocationsCopier.copy(usageAllocations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder usageAllocations(UsageAllocation... usageAllocations) {
            usageAllocations(Arrays.asList(usageAllocations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder usageAllocations(Consumer<UsageAllocation.Builder>... usageAllocations) {
            usageAllocations(Stream.of(usageAllocations).map(c -> UsageAllocation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public UsageRecord build() {
            return new UsageRecord(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
