/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.marketplacemetering;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.marketplacemetering.internal.MarketplaceMeteringServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.marketplacemetering.model.BatchMeterUsageRequest;
import software.amazon.awssdk.services.marketplacemetering.model.BatchMeterUsageResponse;
import software.amazon.awssdk.services.marketplacemetering.model.CustomerNotEntitledException;
import software.amazon.awssdk.services.marketplacemetering.model.DisabledApiException;
import software.amazon.awssdk.services.marketplacemetering.model.DuplicateRequestException;
import software.amazon.awssdk.services.marketplacemetering.model.ExpiredTokenException;
import software.amazon.awssdk.services.marketplacemetering.model.InternalServiceErrorException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidCustomerIdentifierException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidEndpointRegionException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidProductCodeException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidPublicKeyVersionException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidRegionException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidTagException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidTokenException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidUsageAllocationsException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidUsageDimensionException;
import software.amazon.awssdk.services.marketplacemetering.model.MarketplaceMeteringException;
import software.amazon.awssdk.services.marketplacemetering.model.MeterUsageRequest;
import software.amazon.awssdk.services.marketplacemetering.model.MeterUsageResponse;
import software.amazon.awssdk.services.marketplacemetering.model.PlatformNotSupportedException;
import software.amazon.awssdk.services.marketplacemetering.model.RegisterUsageRequest;
import software.amazon.awssdk.services.marketplacemetering.model.RegisterUsageResponse;
import software.amazon.awssdk.services.marketplacemetering.model.ResolveCustomerRequest;
import software.amazon.awssdk.services.marketplacemetering.model.ResolveCustomerResponse;
import software.amazon.awssdk.services.marketplacemetering.model.ThrottlingException;
import software.amazon.awssdk.services.marketplacemetering.model.TimestampOutOfBoundsException;
import software.amazon.awssdk.services.marketplacemetering.transform.BatchMeterUsageRequestMarshaller;
import software.amazon.awssdk.services.marketplacemetering.transform.MeterUsageRequestMarshaller;
import software.amazon.awssdk.services.marketplacemetering.transform.RegisterUsageRequestMarshaller;
import software.amazon.awssdk.services.marketplacemetering.transform.ResolveCustomerRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MarketplaceMeteringClient}.
 *
 * @see MarketplaceMeteringClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMarketplaceMeteringClient implements MarketplaceMeteringClient {
    private static final Logger log = Logger.loggerFor(DefaultMarketplaceMeteringClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMarketplaceMeteringClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * <code>BatchMeterUsage</code> is called from a SaaS application listed on AWS Marketplace to post metering records
     * for a set of customers.
     * </p>
     * <p>
     * For identical requests, the API is idempotent; requests can be retried with the same records or a subset of the
     * input records.
     * </p>
     * <p>
     * Every request to <code>BatchMeterUsage</code> is for one product. If you need to meter usage for multiple
     * products, you must make multiple calls to <code>BatchMeterUsage</code>.
     * </p>
     * <p>
     * Usage records are expected to be submitted as quickly as possible after the event that is being recorded, and are
     * not accepted more than 6 hours after the event.
     * </p>
     * <p>
     * <code>BatchMeterUsage</code> can process up to 25 <code>UsageRecords</code> at a time.
     * </p>
     * <p>
     * A <code>UsageRecord</code> can optionally include multiple usage allocations, to provide customers with usage
     * data split into buckets by tags that you define (or allow the customer to define).
     * </p>
     * <p>
     * <code>BatchMeterUsage</code> returns a list of <code>UsageRecordResult</code> objects, showing the result for
     * each <code>UsageRecord</code>, as well as a list of <code>UnprocessedRecords</code>, indicating errors in the
     * service side that you should retry.
     * </p>
     * <p>
     * <code>BatchMeterUsage</code> requests must be less than 1MB in size.
     * </p>
     * <note>
     * <p>
     * For an example of using <code>BatchMeterUsage</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/userguide/saas-code-examples.html#saas-batchmeterusage-example">
     * BatchMeterUsage code example</a> in the <i>AWS Marketplace Seller Guide</i>.
     * </p>
     * </note>
     *
     * @param batchMeterUsageRequest
     *        A <code>BatchMeterUsageRequest</code> contains <code>UsageRecords</code>, which indicate quantities of
     *        usage within your application.
     * @return Result of the BatchMeterUsage operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the AWS forums.
     * @throws InvalidProductCodeException
     *         The product code passed does not match the product code used for publishing the product.
     * @throws InvalidUsageDimensionException
     *         The usage dimension does not match one of the <code>UsageDimensions</code> associated with products.
     * @throws InvalidTagException
     *         The tag is invalid, or the number of tags is greater than 5.
     * @throws InvalidUsageAllocationsException
     *         The usage allocation objects are invalid, or the number of allocations is greater than 500 for a single
     *         usage record.
     * @throws InvalidCustomerIdentifierException
     *         You have metered usage for a <code>CustomerIdentifier</code> that does not exist.
     * @throws TimestampOutOfBoundsException
     *         The <code>timestamp</code> value passed in the <code>UsageRecord</code> is out of allowed range.</p>
     *         <p>
     *         For <code>BatchMeterUsage</code>, if any of the records are outside of the allowed range, the entire
     *         batch is not processed. You must remove invalid records and try again.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws DisabledApiException
     *         The API is disabled in the Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.BatchMeterUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/BatchMeterUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchMeterUsageResponse batchMeterUsage(BatchMeterUsageRequest batchMeterUsageRequest)
            throws InternalServiceErrorException, InvalidProductCodeException, InvalidUsageDimensionException,
            InvalidTagException, InvalidUsageAllocationsException, InvalidCustomerIdentifierException,
            TimestampOutOfBoundsException, ThrottlingException, DisabledApiException, AwsServiceException, SdkClientException,
            MarketplaceMeteringException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchMeterUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchMeterUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchMeterUsageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchMeterUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Metering");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchMeterUsage");

            return clientHandler.execute(new ClientExecutionParams<BatchMeterUsageRequest, BatchMeterUsageResponse>()
                    .withOperationName("BatchMeterUsage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(batchMeterUsageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchMeterUsageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * API to emit metering records. For identical requests, the API is idempotent. It simply returns the metering
     * record ID.
     * </p>
     * <p>
     * <code>MeterUsage</code> is authenticated on the buyer's AWS account using credentials from the EC2 instance, ECS
     * task, or EKS pod.
     * </p>
     * <p>
     * <code>MeterUsage</code> can optionally include multiple usage allocations, to provide customers with usage data
     * split into buckets by tags that you define (or allow the customer to define).
     * </p>
     * <p>
     * Usage records are expected to be submitted as quickly as possible after the event that is being recorded, and are
     * not accepted more than 6 hours after the event.
     * </p>
     *
     * @param meterUsageRequest
     * @return Result of the MeterUsage operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the AWS forums.
     * @throws InvalidProductCodeException
     *         The product code passed does not match the product code used for publishing the product.
     * @throws InvalidUsageDimensionException
     *         The usage dimension does not match one of the <code>UsageDimensions</code> associated with products.
     * @throws InvalidTagException
     *         The tag is invalid, or the number of tags is greater than 5.
     * @throws InvalidUsageAllocationsException
     *         The usage allocation objects are invalid, or the number of allocations is greater than 500 for a single
     *         usage record.
     * @throws InvalidEndpointRegionException
     *         The endpoint being called is in a AWS Region different from your EC2 instance, ECS task, or EKS pod. The
     *         Region of the Metering Service endpoint and the AWS Region of the resource must match.
     * @throws TimestampOutOfBoundsException
     *         The <code>timestamp</code> value passed in the <code>UsageRecord</code> is out of allowed range.</p>
     *         <p>
     *         For <code>BatchMeterUsage</code>, if any of the records are outside of the allowed range, the entire
     *         batch is not processed. You must remove invalid records and try again.
     * @throws DuplicateRequestException
     *         A metering record has already been emitted by the same EC2 instance, ECS task, or EKS pod for the given {
     *         <code>usageDimension</code>, <code>timestamp</code> with a different <code>usageQuantity</code>.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws CustomerNotEntitledException
     *         Exception thrown when the customer does not have a valid subscription for the product.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.MeterUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/MeterUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public MeterUsageResponse meterUsage(MeterUsageRequest meterUsageRequest) throws InternalServiceErrorException,
            InvalidProductCodeException, InvalidUsageDimensionException, InvalidTagException, InvalidUsageAllocationsException,
            InvalidEndpointRegionException, TimestampOutOfBoundsException, DuplicateRequestException, ThrottlingException,
            CustomerNotEntitledException, AwsServiceException, SdkClientException, MarketplaceMeteringException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<MeterUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                MeterUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(meterUsageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, meterUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Metering");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MeterUsage");

            return clientHandler.execute(new ClientExecutionParams<MeterUsageRequest, MeterUsageResponse>()
                    .withOperationName("MeterUsage").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(meterUsageRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new MeterUsageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Paid container software products sold through AWS Marketplace must integrate with the AWS Marketplace Metering
     * Service and call the <code>RegisterUsage</code> operation for software entitlement and metering. Free and BYOL
     * products for Amazon ECS or Amazon EKS aren't required to call <code>RegisterUsage</code>, but you may choose to
     * do so if you would like to receive usage data in your seller reports. The sections below explain the behavior of
     * <code>RegisterUsage</code>. <code>RegisterUsage</code> performs two primary functions: metering and entitlement.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Entitlement</i>: <code>RegisterUsage</code> allows you to verify that the customer running your paid software
     * is subscribed to your product on AWS Marketplace, enabling you to guard against unauthorized use. Your container
     * image that integrates with <code>RegisterUsage</code> is only required to guard against unauthorized use at
     * container startup, as such a <code>CustomerNotSubscribedException</code> or
     * <code>PlatformNotSupportedException</code> will only be thrown on the initial call to <code>RegisterUsage</code>.
     * Subsequent calls from the same Amazon ECS task instance (e.g. task-id) or Amazon EKS pod will not throw a
     * <code>CustomerNotSubscribedException</code>, even if the customer unsubscribes while the Amazon ECS task or
     * Amazon EKS pod is still running.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Metering</i>: <code>RegisterUsage</code> meters software use per ECS task, per hour, or per pod for Amazon EKS
     * with usage prorated to the second. A minimum of 1 minute of usage applies to tasks that are short lived. For
     * example, if a customer has a 10 node Amazon ECS or Amazon EKS cluster and a service configured as a Daemon Set,
     * then Amazon ECS or Amazon EKS will launch a task on all 10 cluster nodes and the customer will be charged: (10 *
     * hourly_rate). Metering for software use is automatically handled by the AWS Marketplace Metering Control Plane --
     * your software is not required to perform any metering specific actions, other than call
     * <code>RegisterUsage</code> once for metering of software use to commence. The AWS Marketplace Metering Control
     * Plane will also continue to bill customers for running ECS tasks and Amazon EKS pods, regardless of the customers
     * subscription state, removing the need for your software to perform entitlement checks at runtime.
     * </p>
     * </li>
     * </ul>
     *
     * @param registerUsageRequest
     * @return Result of the RegisterUsage operation returned by the service.
     * @throws InvalidProductCodeException
     *         The product code passed does not match the product code used for publishing the product.
     * @throws InvalidRegionException
     *         <code>RegisterUsage</code> must be called in the same AWS Region the ECS task was launched in. This
     *         prevents a container from hardcoding a Region (e.g. withRegion(“us-east-1”) when calling
     *         <code>RegisterUsage</code>.
     * @throws InvalidPublicKeyVersionException
     *         Public Key version is invalid.
     * @throws PlatformNotSupportedException
     *         AWS Marketplace does not support metering usage from the underlying platform. Currently, Amazon ECS,
     *         Amazon EKS, and AWS Fargate are supported.
     * @throws CustomerNotEntitledException
     *         Exception thrown when the customer does not have a valid subscription for the product.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the AWS forums.
     * @throws DisabledApiException
     *         The API is disabled in the Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.RegisterUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/RegisterUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RegisterUsageResponse registerUsage(RegisterUsageRequest registerUsageRequest) throws InvalidProductCodeException,
            InvalidRegionException, InvalidPublicKeyVersionException, PlatformNotSupportedException,
            CustomerNotEntitledException, ThrottlingException, InternalServiceErrorException, DisabledApiException,
            AwsServiceException, SdkClientException, MarketplaceMeteringException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RegisterUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerUsageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Metering");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterUsage");

            return clientHandler.execute(new ClientExecutionParams<RegisterUsageRequest, RegisterUsageResponse>()
                    .withOperationName("RegisterUsage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(registerUsageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RegisterUsageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>ResolveCustomer</code> is called by a SaaS application during the registration process. When a buyer visits
     * your website during the registration process, the buyer submits a registration token through their browser. The
     * registration token is resolved through this API to obtain a <code>CustomerIdentifier</code> along with the
     * <code>CustomerAWSAccountId</code> and <code>ProductCode</code>.
     * </p>
     * <note>
     * <p>
     * The API needs to called from the seller account id used to publish the SaaS application to successfully resolve
     * the token.
     * </p>
     * <p>
     * For an example of using <code>ResolveCustomer</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/userguide/saas-code-examples.html#saas-resolvecustomer-example">
     * ResolveCustomer code example</a> in the <i>AWS Marketplace Seller Guide</i>.
     * </p>
     * </note>
     *
     * @param resolveCustomerRequest
     *        Contains input to the <code>ResolveCustomer</code> operation.
     * @return Result of the ResolveCustomer operation returned by the service.
     * @throws InvalidTokenException
     *         Registration token is invalid.
     * @throws ExpiredTokenException
     *         The submitted registration token has expired. This can happen if the buyer's browser takes too long to
     *         redirect to your page, the buyer has resubmitted the registration token, or your application has held on
     *         to the registration token for too long. Your SaaS registration website should redeem this token as soon
     *         as it is submitted by the buyer's browser.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the AWS forums.
     * @throws DisabledApiException
     *         The API is disabled in the Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.ResolveCustomer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/ResolveCustomer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ResolveCustomerResponse resolveCustomer(ResolveCustomerRequest resolveCustomerRequest) throws InvalidTokenException,
            ExpiredTokenException, ThrottlingException, InternalServiceErrorException, DisabledApiException, AwsServiceException,
            SdkClientException, MarketplaceMeteringException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResolveCustomerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ResolveCustomerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resolveCustomerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resolveCustomerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Marketplace Metering");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResolveCustomer");

            return clientHandler.execute(new ClientExecutionParams<ResolveCustomerRequest, ResolveCustomerResponse>()
                    .withOperationName("ResolveCustomer").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(resolveCustomerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ResolveCustomerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MarketplaceMeteringServiceClientConfigurationBuilder serviceConfigBuilder = new MarketplaceMeteringServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MarketplaceMeteringException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidUsageAllocationsException")
                                .exceptionBuilderSupplier(InvalidUsageAllocationsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagException")
                                .exceptionBuilderSupplier(InvalidTagException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTokenException")
                                .exceptionBuilderSupplier(InvalidTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExpiredTokenException")
                                .exceptionBuilderSupplier(ExpiredTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEndpointRegionException")
                                .exceptionBuilderSupplier(InvalidEndpointRegionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PlatformNotSupportedException")
                                .exceptionBuilderSupplier(PlatformNotSupportedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TimestampOutOfBoundsException")
                                .exceptionBuilderSupplier(TimestampOutOfBoundsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidUsageDimensionException")
                                .exceptionBuilderSupplier(InvalidUsageDimensionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRequestException")
                                .exceptionBuilderSupplier(DuplicateRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPublicKeyVersionException")
                                .exceptionBuilderSupplier(InvalidPublicKeyVersionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRegionException")
                                .exceptionBuilderSupplier(InvalidRegionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomerNotEntitledException")
                                .exceptionBuilderSupplier(CustomerNotEntitledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DisabledApiException")
                                .exceptionBuilderSupplier(DisabledApiException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCustomerIdentifierException")
                                .exceptionBuilderSupplier(InvalidCustomerIdentifierException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidProductCodeException")
                                .exceptionBuilderSupplier(InvalidProductCodeException::builder).httpStatusCode(400).build());
    }

    @Override
    public final MarketplaceMeteringServiceClientConfiguration serviceClientConfiguration() {
        return new MarketplaceMeteringServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
