/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.marketplacemetering;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.marketplacemetering.model.BatchMeterUsageRequest;
import software.amazon.awssdk.services.marketplacemetering.model.BatchMeterUsageResponse;
import software.amazon.awssdk.services.marketplacemetering.model.CustomerNotEntitledException;
import software.amazon.awssdk.services.marketplacemetering.model.DisabledApiException;
import software.amazon.awssdk.services.marketplacemetering.model.DuplicateRequestException;
import software.amazon.awssdk.services.marketplacemetering.model.ExpiredTokenException;
import software.amazon.awssdk.services.marketplacemetering.model.InternalServiceErrorException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidCustomerIdentifierException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidEndpointRegionException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidProductCodeException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidPublicKeyVersionException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidRegionException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidTagException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidTokenException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidUsageAllocationsException;
import software.amazon.awssdk.services.marketplacemetering.model.InvalidUsageDimensionException;
import software.amazon.awssdk.services.marketplacemetering.model.MarketplaceMeteringException;
import software.amazon.awssdk.services.marketplacemetering.model.MeterUsageRequest;
import software.amazon.awssdk.services.marketplacemetering.model.MeterUsageResponse;
import software.amazon.awssdk.services.marketplacemetering.model.PlatformNotSupportedException;
import software.amazon.awssdk.services.marketplacemetering.model.RegisterUsageRequest;
import software.amazon.awssdk.services.marketplacemetering.model.RegisterUsageResponse;
import software.amazon.awssdk.services.marketplacemetering.model.ResolveCustomerRequest;
import software.amazon.awssdk.services.marketplacemetering.model.ResolveCustomerResponse;
import software.amazon.awssdk.services.marketplacemetering.model.ThrottlingException;
import software.amazon.awssdk.services.marketplacemetering.model.TimestampOutOfBoundsException;

/**
 * Service client for accessing AWSMarketplace Metering. This can be created using the static {@link #builder()} method.
 *
 * <fullname>Amazon Web Services Marketplace Metering Service</fullname>
 * <p>
 * This reference provides descriptions of the low-level Marketplace Metering Service API.
 * </p>
 * <p>
 * Amazon Web Services Marketplace sellers can use this API to submit usage data for custom usage dimensions.
 * </p>
 * <p>
 * For information about the permissions that you need to use this API, see <a
 * href="https://docs.aws.amazon.com/marketplace/latest/userguide/iam-user-policy-for-aws-marketplace-actions.html"
 * >Amazon Web Services Marketplace metering and entitlement API permissions</a> in the <i>Amazon Web Services
 * Marketplace Seller Guide.</i>
 * </p>
 * <p>
 * <b>Submitting metering records</b>
 * </p>
 * <p>
 * <i>MeterUsage</i>
 * </p>
 * <ul>
 * <li>
 * <p>
 * Submits the metering record for an Amazon Web Services Marketplace product.
 * </p>
 * </li>
 * <li>
 * <p>
 * Called from: Amazon Elastic Compute Cloud (Amazon EC2) instance or a container running on either Amazon Elastic
 * Kubernetes Service (Amazon EKS) or Amazon Elastic Container Service (Amazon ECS)
 * </p>
 * </li>
 * <li>
 * <p>
 * Supported product types: Amazon Machine Images (AMIs) and containers
 * </p>
 * </li>
 * <li>
 * <p>
 * Vendor-metered tagging: Supported allocation tagging
 * </p>
 * </li>
 * </ul>
 * <p>
 * <i>BatchMeterUsage</i>
 * </p>
 * <ul>
 * <li>
 * <p>
 * Submits the metering record for a set of customers. <code>BatchMeterUsage</code> API calls are captured by
 * CloudTrail. You can use CloudTrail to verify that the software as a subscription (SaaS) metering records that you
 * sent are accurate by searching for records with the <code>eventName</code> of <code>BatchMeterUsage</code>. You can
 * also use CloudTrail to audit records over time. For more information, see the <a
 * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html">CloudTrail User Guide</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Called from: SaaS applications
 * </p>
 * </li>
 * <li>
 * <p>
 * Supported product type: SaaS
 * </p>
 * </li>
 * <li>
 * <p>
 * Vendor-metered tagging: Supports allocation tagging
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Accepting new customers</b>
 * </p>
 * <p>
 * <i>ResolveCustomer</i>
 * </p>
 * <ul>
 * <li>
 * <p>
 * Resolves the registration token that the buyer submits through the browser during the registration process. Obtains a
 * <code>CustomerIdentifier</code> along with the <code>CustomerAWSAccountId</code> and <code>ProductCode</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Called from: SaaS application during the registration process
 * </p>
 * </li>
 * <li>
 * <p>
 * Supported product type: SaaS
 * </p>
 * </li>
 * <li>
 * <p>
 * Vendor-metered tagging: Not applicable
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Entitlement and metering for paid container products</b>
 * </p>
 * <p>
 * <i>RegisteredUsage</i>
 * </p>
 * <ul>
 * <li>
 * <p>
 * Provides software entitlement and metering. Paid container software products sold through Amazon Web Services
 * Marketplace must integrate with the Marketplace Metering Service and call the <code>RegisterUsage</code> operation.
 * Free and Bring Your Own License model (BYOL) products for Amazon ECS or Amazon EKS aren't required to call
 * <code>RegisterUsage</code>. However, you can do so if you want to receive usage data in your seller reports. For more
 * information about using the <code>RegisterUsage</code> operation, see <a
 * href="https://docs.aws.amazon.com/marketplace/latest/userguide/container-based-products.html">Container-based
 * products</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Called from: Paid container software products
 * </p>
 * </li>
 * <li>
 * <p>
 * Supported product type: Containers
 * </p>
 * </li>
 * <li>
 * <p>
 * Vendor-metered tagging: Not applicable
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Entitlement custom metering for container products</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * MeterUsage API is available in GovCloud Regions but only supports AMI FCP products in GovCloud Regions. Flexible
 * Consumption Pricing (FCP) Container products aren’t supported in GovCloud Regions: us-gov-west-1 and us-gov-east-1.
 * For more information, see <a
 * href="https://docs.aws.amazon.com/marketplace/latest/userguide/container-based-products.html">Container-based
 * products</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Custom metering for container products are called using the MeterUsage API. The API is used for FCP AMI and FCP
 * Container product metering.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Custom metering for Amazon EKS is available in 17 Amazon Web Services Regions</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * The metering service supports Amazon ECS and EKS for Flexible Consumption Pricing (FCP) products using MeterUsage
 * API. Amazon ECS is supported in all Amazon Web Services Regions that MeterUsage API is available except for GovCloud.
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon EKS is supported in the following: us-east-1, us-east-2, us-west-1, us-west-2, eu-west-1, eu-central-1,
 * eu-west-2, eu-west-3, eu-north-1, ap-east-1, ap-southeast-1, ap-northeast-1, ap-southeast-2, ap-northeast-2,
 * ap-south-1, ca-central-1, sa-east-1.
 * </p>
 * <note>
 * <p>
 * For questions about adding Amazon Web Services Regions for metering, contact <a
 * href="mailto://aws.amazon.com/marketplace/management/contact-us/">Amazon Web Services Marketplace Seller
 * Operations</a>.
 * </p>
 * </note></li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface MarketplaceMeteringClient extends AwsClient {
    String SERVICE_NAME = "aws-marketplace";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "metering.marketplace";

    /**
     * <important>
     * <p>
     * The <code>CustomerIdentifier</code> parameter is scheduled for deprecation. Use <code>CustomerAWSAccountID</code>
     * instead.
     * </p>
     * <p>
     * These parameters are mutually exclusive. You can't specify both <code>CustomerIdentifier</code> and
     * <code>CustomerAWSAccountID</code> in the same request.
     * </p>
     * </important>
     * <p>
     * To post metering records for customers, SaaS applications call <code>BatchMeterUsage</code>, which is used for
     * metering SaaS flexible consumption pricing (FCP). Identical requests are idempotent and can be retried with the
     * same records or a subset of records. Each <code>BatchMeterUsage</code> request is for only one product. If you
     * want to meter usage for multiple products, you must make multiple <code>BatchMeterUsage</code> calls.
     * </p>
     * <p>
     * Usage records should be submitted in quick succession following a recorded event. Usage records aren't accepted 6
     * hours or more after an event.
     * </p>
     * <p>
     * <code>BatchMeterUsage</code> can process up to 25 <code>UsageRecords</code> at a time, and each request must be
     * less than 1 MB in size. Optionally, you can have multiple usage allocations for usage data that's split into
     * buckets according to predefined tags.
     * </p>
     * <p>
     * <code>BatchMeterUsage</code> returns a list of <code>UsageRecordResult</code> objects, which have each
     * <code>UsageRecord</code>. It also returns a list of <code>UnprocessedRecords</code>, which indicate errors on the
     * service side that should be retried.
     * </p>
     * <p>
     * For Amazon Web Services Regions that support <code>BatchMeterUsage</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/APIReference/metering-regions.html#batchmeterusage-region-support"
     * >BatchMeterUsage Region support</a>.
     * </p>
     * <note>
     * <p>
     * For an example of <code>BatchMeterUsage</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/userguide/saas-code-examples.html#saas-batchmeterusage-example">
     * BatchMeterUsage code example</a> in the <i>Amazon Web Services Marketplace Seller Guide</i>.
     * </p>
     * </note>
     *
     * @param batchMeterUsageRequest
     *        A <code>BatchMeterUsageRequest</code> contains <code>UsageRecords</code>, which indicate quantities of
     *        usage within your application.
     * @return Result of the BatchMeterUsage operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the Amazon Web Services forums.
     * @throws InvalidProductCodeException
     *         The product code passed does not match the product code used for publishing the product.
     * @throws InvalidUsageDimensionException
     *         The usage dimension does not match one of the <code>UsageDimensions</code> associated with products.
     * @throws InvalidTagException
     *         The tag is invalid, or the number of tags is greater than 5.
     * @throws InvalidUsageAllocationsException
     *         Sum of allocated usage quantities is not equal to the usage quantity.
     * @throws InvalidCustomerIdentifierException
     *         You have metered usage for a <code>CustomerIdentifier</code> that does not exist.
     * @throws TimestampOutOfBoundsException
     *         The <code>timestamp</code> value passed in the <code>UsageRecord</code> is out of allowed range.</p>
     *         <p>
     *         For <code>BatchMeterUsage</code>, if any of the records are outside of the allowed range, the entire
     *         batch is not processed. You must remove invalid records and try again.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws DisabledApiException
     *         The API is disabled in the Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.BatchMeterUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/BatchMeterUsage"
     *      target="_top">AWS API Documentation</a>
     */
    default BatchMeterUsageResponse batchMeterUsage(BatchMeterUsageRequest batchMeterUsageRequest)
            throws InternalServiceErrorException, InvalidProductCodeException, InvalidUsageDimensionException,
            InvalidTagException, InvalidUsageAllocationsException, InvalidCustomerIdentifierException,
            TimestampOutOfBoundsException, ThrottlingException, DisabledApiException, AwsServiceException, SdkClientException,
            MarketplaceMeteringException {
        throw new UnsupportedOperationException();
    }

    /**
     * <important>
     * <p>
     * The <code>CustomerIdentifier</code> parameter is scheduled for deprecation. Use <code>CustomerAWSAccountID</code>
     * instead.
     * </p>
     * <p>
     * These parameters are mutually exclusive. You can't specify both <code>CustomerIdentifier</code> and
     * <code>CustomerAWSAccountID</code> in the same request.
     * </p>
     * </important>
     * <p>
     * To post metering records for customers, SaaS applications call <code>BatchMeterUsage</code>, which is used for
     * metering SaaS flexible consumption pricing (FCP). Identical requests are idempotent and can be retried with the
     * same records or a subset of records. Each <code>BatchMeterUsage</code> request is for only one product. If you
     * want to meter usage for multiple products, you must make multiple <code>BatchMeterUsage</code> calls.
     * </p>
     * <p>
     * Usage records should be submitted in quick succession following a recorded event. Usage records aren't accepted 6
     * hours or more after an event.
     * </p>
     * <p>
     * <code>BatchMeterUsage</code> can process up to 25 <code>UsageRecords</code> at a time, and each request must be
     * less than 1 MB in size. Optionally, you can have multiple usage allocations for usage data that's split into
     * buckets according to predefined tags.
     * </p>
     * <p>
     * <code>BatchMeterUsage</code> returns a list of <code>UsageRecordResult</code> objects, which have each
     * <code>UsageRecord</code>. It also returns a list of <code>UnprocessedRecords</code>, which indicate errors on the
     * service side that should be retried.
     * </p>
     * <p>
     * For Amazon Web Services Regions that support <code>BatchMeterUsage</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/APIReference/metering-regions.html#batchmeterusage-region-support"
     * >BatchMeterUsage Region support</a>.
     * </p>
     * <note>
     * <p>
     * For an example of <code>BatchMeterUsage</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/userguide/saas-code-examples.html#saas-batchmeterusage-example">
     * BatchMeterUsage code example</a> in the <i>Amazon Web Services Marketplace Seller Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link BatchMeterUsageRequest.Builder} avoiding the need
     * to create one manually via {@link BatchMeterUsageRequest#builder()}
     * </p>
     *
     * @param batchMeterUsageRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.marketplacemetering.model.BatchMeterUsageRequest.Builder} to create
     *        a request. A <code>BatchMeterUsageRequest</code> contains <code>UsageRecords</code>, which indicate
     *        quantities of usage within your application.
     * @return Result of the BatchMeterUsage operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the Amazon Web Services forums.
     * @throws InvalidProductCodeException
     *         The product code passed does not match the product code used for publishing the product.
     * @throws InvalidUsageDimensionException
     *         The usage dimension does not match one of the <code>UsageDimensions</code> associated with products.
     * @throws InvalidTagException
     *         The tag is invalid, or the number of tags is greater than 5.
     * @throws InvalidUsageAllocationsException
     *         Sum of allocated usage quantities is not equal to the usage quantity.
     * @throws InvalidCustomerIdentifierException
     *         You have metered usage for a <code>CustomerIdentifier</code> that does not exist.
     * @throws TimestampOutOfBoundsException
     *         The <code>timestamp</code> value passed in the <code>UsageRecord</code> is out of allowed range.</p>
     *         <p>
     *         For <code>BatchMeterUsage</code>, if any of the records are outside of the allowed range, the entire
     *         batch is not processed. You must remove invalid records and try again.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws DisabledApiException
     *         The API is disabled in the Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.BatchMeterUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/BatchMeterUsage"
     *      target="_top">AWS API Documentation</a>
     */
    default BatchMeterUsageResponse batchMeterUsage(Consumer<BatchMeterUsageRequest.Builder> batchMeterUsageRequest)
            throws InternalServiceErrorException, InvalidProductCodeException, InvalidUsageDimensionException,
            InvalidTagException, InvalidUsageAllocationsException, InvalidCustomerIdentifierException,
            TimestampOutOfBoundsException, ThrottlingException, DisabledApiException, AwsServiceException, SdkClientException,
            MarketplaceMeteringException {
        return batchMeterUsage(BatchMeterUsageRequest.builder().applyMutation(batchMeterUsageRequest).build());
    }

    /**
     * <p>
     * API to emit metering records. For identical requests, the API is idempotent and returns the metering record ID.
     * This is used for metering flexible consumption pricing (FCP) Amazon Machine Images (AMI) and container products.
     * </p>
     * <p>
     * <code>MeterUsage</code> is authenticated on the buyer's Amazon Web Services account using credentials from the
     * Amazon EC2 instance, Amazon ECS task, or Amazon EKS pod.
     * </p>
     * <p>
     * <code>MeterUsage</code> can optionally include multiple usage allocations, to provide customers with usage data
     * split into buckets by tags that you define (or allow the customer to define).
     * </p>
     * <p>
     * Usage records are expected to be submitted as quickly as possible after the event that is being recorded, and are
     * not accepted more than 6 hours after the event.
     * </p>
     * <p>
     * For Amazon Web Services Regions that support <code>MeterUsage</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/APIReference/metering-regions.html#meterusage-region-support-ec2"
     * >MeterUsage Region support for Amazon EC2</a> and <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/APIReference/metering-regions.html#meterusage-region-support-ecs-eks"
     * >MeterUsage Region support for Amazon ECS and Amazon EKS</a>.
     * </p>
     *
     * @param meterUsageRequest
     * @return Result of the MeterUsage operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the Amazon Web Services forums.
     * @throws InvalidProductCodeException
     *         The product code passed does not match the product code used for publishing the product.
     * @throws InvalidUsageDimensionException
     *         The usage dimension does not match one of the <code>UsageDimensions</code> associated with products.
     * @throws InvalidTagException
     *         The tag is invalid, or the number of tags is greater than 5.
     * @throws InvalidUsageAllocationsException
     *         Sum of allocated usage quantities is not equal to the usage quantity.
     * @throws InvalidEndpointRegionException
     *         The endpoint being called is in a Amazon Web Services Region different from your EC2 instance, ECS task,
     *         or EKS pod. The Region of the Metering Service endpoint and the Amazon Web Services Region of the
     *         resource must match.
     * @throws TimestampOutOfBoundsException
     *         The <code>timestamp</code> value passed in the <code>UsageRecord</code> is out of allowed range.</p>
     *         <p>
     *         For <code>BatchMeterUsage</code>, if any of the records are outside of the allowed range, the entire
     *         batch is not processed. You must remove invalid records and try again.
     * @throws DuplicateRequestException
     *         A metering record has already been emitted by the same EC2 instance, ECS task, or EKS pod for the given {
     *         <code>usageDimension</code>, <code>timestamp</code> with a different <code>usageQuantity</code>.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws CustomerNotEntitledException
     *         Exception thrown when the customer does not have a valid subscription for the product.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.MeterUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/MeterUsage"
     *      target="_top">AWS API Documentation</a>
     */
    default MeterUsageResponse meterUsage(MeterUsageRequest meterUsageRequest) throws InternalServiceErrorException,
            InvalidProductCodeException, InvalidUsageDimensionException, InvalidTagException, InvalidUsageAllocationsException,
            InvalidEndpointRegionException, TimestampOutOfBoundsException, DuplicateRequestException, ThrottlingException,
            CustomerNotEntitledException, AwsServiceException, SdkClientException, MarketplaceMeteringException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * API to emit metering records. For identical requests, the API is idempotent and returns the metering record ID.
     * This is used for metering flexible consumption pricing (FCP) Amazon Machine Images (AMI) and container products.
     * </p>
     * <p>
     * <code>MeterUsage</code> is authenticated on the buyer's Amazon Web Services account using credentials from the
     * Amazon EC2 instance, Amazon ECS task, or Amazon EKS pod.
     * </p>
     * <p>
     * <code>MeterUsage</code> can optionally include multiple usage allocations, to provide customers with usage data
     * split into buckets by tags that you define (or allow the customer to define).
     * </p>
     * <p>
     * Usage records are expected to be submitted as quickly as possible after the event that is being recorded, and are
     * not accepted more than 6 hours after the event.
     * </p>
     * <p>
     * For Amazon Web Services Regions that support <code>MeterUsage</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/APIReference/metering-regions.html#meterusage-region-support-ec2"
     * >MeterUsage Region support for Amazon EC2</a> and <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/APIReference/metering-regions.html#meterusage-region-support-ecs-eks"
     * >MeterUsage Region support for Amazon ECS and Amazon EKS</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link MeterUsageRequest.Builder} avoiding the need to
     * create one manually via {@link MeterUsageRequest#builder()}
     * </p>
     *
     * @param meterUsageRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.marketplacemetering.model.MeterUsageRequest.Builder} to create a
     *        request.
     * @return Result of the MeterUsage operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the Amazon Web Services forums.
     * @throws InvalidProductCodeException
     *         The product code passed does not match the product code used for publishing the product.
     * @throws InvalidUsageDimensionException
     *         The usage dimension does not match one of the <code>UsageDimensions</code> associated with products.
     * @throws InvalidTagException
     *         The tag is invalid, or the number of tags is greater than 5.
     * @throws InvalidUsageAllocationsException
     *         Sum of allocated usage quantities is not equal to the usage quantity.
     * @throws InvalidEndpointRegionException
     *         The endpoint being called is in a Amazon Web Services Region different from your EC2 instance, ECS task,
     *         or EKS pod. The Region of the Metering Service endpoint and the Amazon Web Services Region of the
     *         resource must match.
     * @throws TimestampOutOfBoundsException
     *         The <code>timestamp</code> value passed in the <code>UsageRecord</code> is out of allowed range.</p>
     *         <p>
     *         For <code>BatchMeterUsage</code>, if any of the records are outside of the allowed range, the entire
     *         batch is not processed. You must remove invalid records and try again.
     * @throws DuplicateRequestException
     *         A metering record has already been emitted by the same EC2 instance, ECS task, or EKS pod for the given {
     *         <code>usageDimension</code>, <code>timestamp</code> with a different <code>usageQuantity</code>.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws CustomerNotEntitledException
     *         Exception thrown when the customer does not have a valid subscription for the product.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.MeterUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/MeterUsage"
     *      target="_top">AWS API Documentation</a>
     */
    default MeterUsageResponse meterUsage(Consumer<MeterUsageRequest.Builder> meterUsageRequest)
            throws InternalServiceErrorException, InvalidProductCodeException, InvalidUsageDimensionException,
            InvalidTagException, InvalidUsageAllocationsException, InvalidEndpointRegionException, TimestampOutOfBoundsException,
            DuplicateRequestException, ThrottlingException, CustomerNotEntitledException, AwsServiceException,
            SdkClientException, MarketplaceMeteringException {
        return meterUsage(MeterUsageRequest.builder().applyMutation(meterUsageRequest).build());
    }

    /**
     * <p>
     * Paid container software products sold through Amazon Web Services Marketplace must integrate with the Amazon Web
     * Services Marketplace Metering Service and call the <code>RegisterUsage</code> operation for software entitlement
     * and metering. Free and BYOL products for Amazon ECS or Amazon EKS aren't required to call
     * <code>RegisterUsage</code>, but you may choose to do so if you would like to receive usage data in your seller
     * reports. The sections below explain the behavior of <code>RegisterUsage</code>. <code>RegisterUsage</code>
     * performs two primary functions: metering and entitlement.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Entitlement</i>: <code>RegisterUsage</code> allows you to verify that the customer running your paid software
     * is subscribed to your product on Amazon Web Services Marketplace, enabling you to guard against unauthorized use.
     * Your container image that integrates with <code>RegisterUsage</code> is only required to guard against
     * unauthorized use at container startup, as such a <code>CustomerNotSubscribedException</code> or
     * <code>PlatformNotSupportedException</code> will only be thrown on the initial call to <code>RegisterUsage</code>.
     * Subsequent calls from the same Amazon ECS task instance (e.g. task-id) or Amazon EKS pod will not throw a
     * <code>CustomerNotSubscribedException</code>, even if the customer unsubscribes while the Amazon ECS task or
     * Amazon EKS pod is still running.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Metering</i>: <code>RegisterUsage</code> meters software use per ECS task, per hour, or per pod for Amazon EKS
     * with usage prorated to the second. A minimum of 1 minute of usage applies to tasks that are short lived. For
     * example, if a customer has a 10 node Amazon ECS or Amazon EKS cluster and a service configured as a Daemon Set,
     * then Amazon ECS or Amazon EKS will launch a task on all 10 cluster nodes and the customer will be charged for 10
     * tasks. Software metering is handled by the Amazon Web Services Marketplace metering control plane—your software
     * is not required to perform metering-specific actions other than to call <code>RegisterUsage</code> to commence
     * metering. The Amazon Web Services Marketplace metering control plane will also bill customers for running ECS
     * tasks and Amazon EKS pods, regardless of the customer's subscription state, which removes the need for your
     * software to run entitlement checks at runtime. For containers, <code>RegisterUsage</code> should be called
     * immediately at launch. If you don’t register the container within the first 6 hours of the launch, Amazon Web
     * Services Marketplace Metering Service doesn’t provide any metering guarantees for previous months. Metering will
     * continue, however, for the current month forward until the container ends. <code>RegisterUsage</code> is for
     * metering paid hourly container products.
     * </p>
     * <p>
     * For Amazon Web Services Regions that support <code>RegisterUsage</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/APIReference/metering-regions.html#registerusage-region-support"
     * >RegisterUsage Region support</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param registerUsageRequest
     * @return Result of the RegisterUsage operation returned by the service.
     * @throws InvalidProductCodeException
     *         The product code passed does not match the product code used for publishing the product.
     * @throws InvalidRegionException
     *         <code>RegisterUsage</code> must be called in the same Amazon Web Services Region the ECS task was
     *         launched in. This prevents a container from hardcoding a Region (e.g. withRegion(“us-east-1”) when
     *         calling <code>RegisterUsage</code>.
     * @throws InvalidPublicKeyVersionException
     *         Public Key version is invalid.
     * @throws PlatformNotSupportedException
     *         Amazon Web Services Marketplace does not support metering usage from the underlying platform. Currently,
     *         Amazon ECS, Amazon EKS, and Fargate are supported.
     * @throws CustomerNotEntitledException
     *         Exception thrown when the customer does not have a valid subscription for the product.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the Amazon Web Services forums.
     * @throws DisabledApiException
     *         The API is disabled in the Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.RegisterUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/RegisterUsage"
     *      target="_top">AWS API Documentation</a>
     */
    default RegisterUsageResponse registerUsage(RegisterUsageRequest registerUsageRequest) throws InvalidProductCodeException,
            InvalidRegionException, InvalidPublicKeyVersionException, PlatformNotSupportedException,
            CustomerNotEntitledException, ThrottlingException, InternalServiceErrorException, DisabledApiException,
            AwsServiceException, SdkClientException, MarketplaceMeteringException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Paid container software products sold through Amazon Web Services Marketplace must integrate with the Amazon Web
     * Services Marketplace Metering Service and call the <code>RegisterUsage</code> operation for software entitlement
     * and metering. Free and BYOL products for Amazon ECS or Amazon EKS aren't required to call
     * <code>RegisterUsage</code>, but you may choose to do so if you would like to receive usage data in your seller
     * reports. The sections below explain the behavior of <code>RegisterUsage</code>. <code>RegisterUsage</code>
     * performs two primary functions: metering and entitlement.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Entitlement</i>: <code>RegisterUsage</code> allows you to verify that the customer running your paid software
     * is subscribed to your product on Amazon Web Services Marketplace, enabling you to guard against unauthorized use.
     * Your container image that integrates with <code>RegisterUsage</code> is only required to guard against
     * unauthorized use at container startup, as such a <code>CustomerNotSubscribedException</code> or
     * <code>PlatformNotSupportedException</code> will only be thrown on the initial call to <code>RegisterUsage</code>.
     * Subsequent calls from the same Amazon ECS task instance (e.g. task-id) or Amazon EKS pod will not throw a
     * <code>CustomerNotSubscribedException</code>, even if the customer unsubscribes while the Amazon ECS task or
     * Amazon EKS pod is still running.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Metering</i>: <code>RegisterUsage</code> meters software use per ECS task, per hour, or per pod for Amazon EKS
     * with usage prorated to the second. A minimum of 1 minute of usage applies to tasks that are short lived. For
     * example, if a customer has a 10 node Amazon ECS or Amazon EKS cluster and a service configured as a Daemon Set,
     * then Amazon ECS or Amazon EKS will launch a task on all 10 cluster nodes and the customer will be charged for 10
     * tasks. Software metering is handled by the Amazon Web Services Marketplace metering control plane—your software
     * is not required to perform metering-specific actions other than to call <code>RegisterUsage</code> to commence
     * metering. The Amazon Web Services Marketplace metering control plane will also bill customers for running ECS
     * tasks and Amazon EKS pods, regardless of the customer's subscription state, which removes the need for your
     * software to run entitlement checks at runtime. For containers, <code>RegisterUsage</code> should be called
     * immediately at launch. If you don’t register the container within the first 6 hours of the launch, Amazon Web
     * Services Marketplace Metering Service doesn’t provide any metering guarantees for previous months. Metering will
     * continue, however, for the current month forward until the container ends. <code>RegisterUsage</code> is for
     * metering paid hourly container products.
     * </p>
     * <p>
     * For Amazon Web Services Regions that support <code>RegisterUsage</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/APIReference/metering-regions.html#registerusage-region-support"
     * >RegisterUsage Region support</a>.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RegisterUsageRequest.Builder} avoiding the need to
     * create one manually via {@link RegisterUsageRequest#builder()}
     * </p>
     *
     * @param registerUsageRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.marketplacemetering.model.RegisterUsageRequest.Builder} to create a
     *        request.
     * @return Result of the RegisterUsage operation returned by the service.
     * @throws InvalidProductCodeException
     *         The product code passed does not match the product code used for publishing the product.
     * @throws InvalidRegionException
     *         <code>RegisterUsage</code> must be called in the same Amazon Web Services Region the ECS task was
     *         launched in. This prevents a container from hardcoding a Region (e.g. withRegion(“us-east-1”) when
     *         calling <code>RegisterUsage</code>.
     * @throws InvalidPublicKeyVersionException
     *         Public Key version is invalid.
     * @throws PlatformNotSupportedException
     *         Amazon Web Services Marketplace does not support metering usage from the underlying platform. Currently,
     *         Amazon ECS, Amazon EKS, and Fargate are supported.
     * @throws CustomerNotEntitledException
     *         Exception thrown when the customer does not have a valid subscription for the product.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the Amazon Web Services forums.
     * @throws DisabledApiException
     *         The API is disabled in the Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.RegisterUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/RegisterUsage"
     *      target="_top">AWS API Documentation</a>
     */
    default RegisterUsageResponse registerUsage(Consumer<RegisterUsageRequest.Builder> registerUsageRequest)
            throws InvalidProductCodeException, InvalidRegionException, InvalidPublicKeyVersionException,
            PlatformNotSupportedException, CustomerNotEntitledException, ThrottlingException, InternalServiceErrorException,
            DisabledApiException, AwsServiceException, SdkClientException, MarketplaceMeteringException {
        return registerUsage(RegisterUsageRequest.builder().applyMutation(registerUsageRequest).build());
    }

    /**
     * <p>
     * <code>ResolveCustomer</code> is called by a SaaS application during the registration process. When a buyer visits
     * your website during the registration process, the buyer submits a registration token through their browser. The
     * registration token is resolved through this API to obtain a <code>CustomerIdentifier</code> along with the
     * <code>CustomerAWSAccountId</code> and <code>ProductCode</code>.
     * </p>
     * <note>
     * <p>
     * To successfully resolve the token, the API must be called from the account that was used to publish the SaaS
     * application. For an example of using <code>ResolveCustomer</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/userguide/saas-code-examples.html#saas-resolvecustomer-example">
     * ResolveCustomer code example</a> in the <i>Amazon Web Services Marketplace Seller Guide</i>.
     * </p>
     * </note>
     * <p>
     * Permission is required for this operation. Your IAM role or user performing this operation requires a policy to
     * allow the <code>aws-marketplace:ResolveCustomer</code> action. For more information, see <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplacemeteringservice.html"
     * >Actions, resources, and condition keys for Amazon Web Services Marketplace Metering Service</a> in the
     * <i>Service Authorization Reference</i>.
     * </p>
     * <p>
     * For Amazon Web Services Regions that support <code>ResolveCustomer</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/APIReference/metering-regions.html#resolvecustomer-region-support"
     * >ResolveCustomer Region support</a>.
     * </p>
     *
     * @param resolveCustomerRequest
     *        Contains input to the <code>ResolveCustomer</code> operation.
     * @return Result of the ResolveCustomer operation returned by the service.
     * @throws InvalidTokenException
     *         Registration token is invalid.
     * @throws ExpiredTokenException
     *         The submitted registration token has expired. This can happen if the buyer's browser takes too long to
     *         redirect to your page, the buyer has resubmitted the registration token, or your application has held on
     *         to the registration token for too long. Your SaaS registration website should redeem this token as soon
     *         as it is submitted by the buyer's browser.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the Amazon Web Services forums.
     * @throws DisabledApiException
     *         The API is disabled in the Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.ResolveCustomer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/ResolveCustomer"
     *      target="_top">AWS API Documentation</a>
     */
    default ResolveCustomerResponse resolveCustomer(ResolveCustomerRequest resolveCustomerRequest) throws InvalidTokenException,
            ExpiredTokenException, ThrottlingException, InternalServiceErrorException, DisabledApiException, AwsServiceException,
            SdkClientException, MarketplaceMeteringException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * <code>ResolveCustomer</code> is called by a SaaS application during the registration process. When a buyer visits
     * your website during the registration process, the buyer submits a registration token through their browser. The
     * registration token is resolved through this API to obtain a <code>CustomerIdentifier</code> along with the
     * <code>CustomerAWSAccountId</code> and <code>ProductCode</code>.
     * </p>
     * <note>
     * <p>
     * To successfully resolve the token, the API must be called from the account that was used to publish the SaaS
     * application. For an example of using <code>ResolveCustomer</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/userguide/saas-code-examples.html#saas-resolvecustomer-example">
     * ResolveCustomer code example</a> in the <i>Amazon Web Services Marketplace Seller Guide</i>.
     * </p>
     * </note>
     * <p>
     * Permission is required for this operation. Your IAM role or user performing this operation requires a policy to
     * allow the <code>aws-marketplace:ResolveCustomer</code> action. For more information, see <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplacemeteringservice.html"
     * >Actions, resources, and condition keys for Amazon Web Services Marketplace Metering Service</a> in the
     * <i>Service Authorization Reference</i>.
     * </p>
     * <p>
     * For Amazon Web Services Regions that support <code>ResolveCustomer</code>, see <a href=
     * "https://docs.aws.amazon.com/marketplace/latest/APIReference/metering-regions.html#resolvecustomer-region-support"
     * >ResolveCustomer Region support</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ResolveCustomerRequest.Builder} avoiding the need
     * to create one manually via {@link ResolveCustomerRequest#builder()}
     * </p>
     *
     * @param resolveCustomerRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.marketplacemetering.model.ResolveCustomerRequest.Builder} to create
     *        a request. Contains input to the <code>ResolveCustomer</code> operation.
     * @return Result of the ResolveCustomer operation returned by the service.
     * @throws InvalidTokenException
     *         Registration token is invalid.
     * @throws ExpiredTokenException
     *         The submitted registration token has expired. This can happen if the buyer's browser takes too long to
     *         redirect to your page, the buyer has resubmitted the registration token, or your application has held on
     *         to the registration token for too long. Your SaaS registration website should redeem this token as soon
     *         as it is submitted by the buyer's browser.
     * @throws ThrottlingException
     *         The calls to the API are throttled.
     * @throws InternalServiceErrorException
     *         An internal error has occurred. Retry your request. If the problem persists, post a message with details
     *         on the Amazon Web Services forums.
     * @throws DisabledApiException
     *         The API is disabled in the Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MarketplaceMeteringException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MarketplaceMeteringClient.ResolveCustomer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/meteringmarketplace-2016-01-14/ResolveCustomer"
     *      target="_top">AWS API Documentation</a>
     */
    default ResolveCustomerResponse resolveCustomer(Consumer<ResolveCustomerRequest.Builder> resolveCustomerRequest)
            throws InvalidTokenException, ExpiredTokenException, ThrottlingException, InternalServiceErrorException,
            DisabledApiException, AwsServiceException, SdkClientException, MarketplaceMeteringException {
        return resolveCustomer(ResolveCustomerRequest.builder().applyMutation(resolveCustomerRequest).build());
    }

    /**
     * Create a {@link MarketplaceMeteringClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static MarketplaceMeteringClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link MarketplaceMeteringClient}.
     */
    static MarketplaceMeteringClientBuilder builder() {
        return new DefaultMarketplaceMeteringClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default MarketplaceMeteringServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
