/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconnect;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputs420Exception;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputsRequest;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputsResponse;
import software.amazon.awssdk.services.mediaconnect.model.BadRequestException;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlow420Exception;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.DeleteFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.DeleteFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.ForbiddenException;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlements420Exception;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlementsRequest;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlementsResponse;
import software.amazon.awssdk.services.mediaconnect.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsResponse;
import software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListFlowsResponse;
import software.amazon.awssdk.services.mediaconnect.model.MediaConnectException;
import software.amazon.awssdk.services.mediaconnect.model.MediaConnectRequest;
import software.amazon.awssdk.services.mediaconnect.model.NotFoundException;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowOutputRequest;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowOutputResponse;
import software.amazon.awssdk.services.mediaconnect.model.RevokeFlowEntitlementRequest;
import software.amazon.awssdk.services.mediaconnect.model.RevokeFlowEntitlementResponse;
import software.amazon.awssdk.services.mediaconnect.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mediaconnect.model.StartFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.StartFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.StopFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.StopFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowEntitlementRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowEntitlementResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowOutputRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowOutputResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowSourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowSourceResponse;
import software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsPublisher;
import software.amazon.awssdk.services.mediaconnect.transform.AddFlowOutputsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.CreateFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DeleteFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DescribeFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.GrantFlowEntitlementsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListEntitlementsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListFlowsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RemoveFlowOutputRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RevokeFlowEntitlementRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.StartFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.StopFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowEntitlementRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowOutputRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowSourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MediaConnectAsyncClient}.
 *
 * @see MediaConnectAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaConnectAsyncClient implements MediaConnectAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMediaConnectAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaConnectAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Adds outputs to an existing flow. You can create up to 20 outputs per flow.
     *
     * @param addFlowOutputsRequest
     *        A request to add outputs to the specified flow.
     * @return A Java Future containing the result of the AddFlowOutputs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AddFlowOutputs420Exception AWS Elemental MediaConnect can't complete this request because this flow
     *         already has the maximum number of allowed outputs (20). For more information, contact AWS Customer
     *         Support.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.AddFlowOutputs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/AddFlowOutputs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AddFlowOutputsResponse> addFlowOutputs(AddFlowOutputsRequest addFlowOutputsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddFlowOutputsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddFlowOutputsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<AddFlowOutputsRequest, AddFlowOutputsResponse>()
                    .withMarshaller(new AddFlowOutputsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(addFlowOutputsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a new flow. The request must include one source. The request optionally can include outputs (up to 20)
     * and entitlements (up to 50).
     *
     * @param createFlowRequest
     *        Creates a new flow. The request must include one source. The request optionally can include outputs (up to
     *        20) and entitlements (up to 50).
     * @return A Java Future containing the result of the CreateFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CreateFlow420Exception Your account already contains the maximum number of 20 flows per account, per
     *         Region. For more information, contact AWS Customer Support.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.CreateFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/CreateFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFlowResponse> createFlow(CreateFlowRequest createFlowRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<CreateFlowRequest, CreateFlowResponse>()
                    .withMarshaller(new CreateFlowRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createFlowRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes a flow. Before you can delete a flow, you must stop the flow.
     *
     * @param deleteFlowRequest
     * @return A Java Future containing the result of the DeleteFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.DeleteFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DeleteFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFlowResponse> deleteFlow(DeleteFlowRequest deleteFlowRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DeleteFlowRequest, DeleteFlowResponse>()
                    .withMarshaller(new DeleteFlowRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteFlowRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays the details of a flow. The response includes the flow ARN, name, and Availability Zone, as well as
     * details about the source, outputs, and entitlements.
     *
     * @param describeFlowRequest
     * @return A Java Future containing the result of the DescribeFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.DescribeFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DescribeFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFlowResponse> describeFlow(DescribeFlowRequest describeFlowRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DescribeFlowRequest, DescribeFlowResponse>()
                    .withMarshaller(new DescribeFlowRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeFlowRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Grants entitlements to an existing flow.
     *
     * @param grantFlowEntitlementsRequest
     *        A request to grant entitlements on a flow.
     * @return A Java Future containing the result of the GrantFlowEntitlements operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>GrantFlowEntitlements420Exception AWS Elemental MediaConnect can't complete this request because this
     *         flow already has the maximum number of allowed entitlements (50). For more information, contact AWS
     *         Customer Support.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.GrantFlowEntitlements
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/GrantFlowEntitlements"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GrantFlowEntitlementsResponse> grantFlowEntitlements(
            GrantFlowEntitlementsRequest grantFlowEntitlementsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GrantFlowEntitlementsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GrantFlowEntitlementsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GrantFlowEntitlementsRequest, GrantFlowEntitlementsResponse>()
                    .withMarshaller(new GrantFlowEntitlementsRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(grantFlowEntitlementsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays a list of all entitlements that have been granted to this account. This request returns 20 results per
     * page.
     *
     * @param listEntitlementsRequest
     * @return A Java Future containing the result of the ListEntitlements operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListEntitlements
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListEntitlements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEntitlementsResponse> listEntitlements(ListEntitlementsRequest listEntitlementsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEntitlementsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEntitlementsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListEntitlementsRequest, ListEntitlementsResponse>()
                    .withMarshaller(new ListEntitlementsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listEntitlementsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays a list of flows that are associated with this account. This request returns a paginated result.
     *
     * @param listFlowsRequest
     * @return A Java Future containing the result of the ListFlows operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListFlows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListFlows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFlowsResponse> listFlows(ListFlowsRequest listFlowsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFlowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFlowsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListFlowsRequest, ListFlowsResponse>()
                    .withMarshaller(new ListFlowsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listFlowsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays a list of flows that are associated with this account. This request returns a paginated result.<br/>
     * <p>
     * This is a variant of {@link #listFlows(software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsPublisher publisher = client.listFlowsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsPublisher publisher = client.listFlowsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediaconnect.model.ListFlowsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediaconnect.model.ListFlowsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFlows(software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest)} operation.</b>
     * </p>
     *
     * @param listFlowsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListFlows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListFlows" target="_top">AWS API
     *      Documentation</a>
     */
    public ListFlowsPublisher listFlowsPaginator(ListFlowsRequest listFlowsRequest) {
        return new ListFlowsPublisher(this, applyPaginatorUserAgent(listFlowsRequest));
    }

    /**
     * Removes an output from an existing flow. This request can be made only on an output that does not have an
     * entitlement associated with it. If the output has an entitlement, you must revoke the entitlement instead. When
     * an entitlement is revoked from a flow, the service automatically removes the associated output.
     *
     * @param removeFlowOutputRequest
     * @return A Java Future containing the result of the RemoveFlowOutput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.RemoveFlowOutput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RemoveFlowOutput" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveFlowOutputResponse> removeFlowOutput(RemoveFlowOutputRequest removeFlowOutputRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveFlowOutputResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveFlowOutputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<RemoveFlowOutputRequest, RemoveFlowOutputResponse>()
                    .withMarshaller(new RemoveFlowOutputRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(removeFlowOutputRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Revokes an entitlement from a flow. Once an entitlement is revoked, the content becomes unavailable to the
     * subscriber and the associated output is removed.
     *
     * @param revokeFlowEntitlementRequest
     * @return A Java Future containing the result of the RevokeFlowEntitlement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.RevokeFlowEntitlement
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RevokeFlowEntitlement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RevokeFlowEntitlementResponse> revokeFlowEntitlement(
            RevokeFlowEntitlementRequest revokeFlowEntitlementRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RevokeFlowEntitlementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RevokeFlowEntitlementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<RevokeFlowEntitlementRequest, RevokeFlowEntitlementResponse>()
                    .withMarshaller(new RevokeFlowEntitlementRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(revokeFlowEntitlementRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Starts a flow.
     *
     * @param startFlowRequest
     * @return A Java Future containing the result of the StartFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.StartFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/StartFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartFlowResponse> startFlow(StartFlowRequest startFlowRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<StartFlowRequest, StartFlowResponse>()
                    .withMarshaller(new StartFlowRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startFlowRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Stops a flow.
     *
     * @param stopFlowRequest
     * @return A Java Future containing the result of the StopFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.StopFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/StopFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopFlowResponse> stopFlow(StopFlowRequest stopFlowRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<StopFlowRequest, StopFlowResponse>()
                    .withMarshaller(new StopFlowRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopFlowRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * You can change an entitlement's description, subscribers, and encryption. If you change the subscribers, the
     * service will remove the outputs that are are used by the subscribers that are removed.
     *
     * @param updateFlowEntitlementRequest
     *        The entitlement fields that you want to update.
     * @return A Java Future containing the result of the UpdateFlowEntitlement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.UpdateFlowEntitlement
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowEntitlement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFlowEntitlementResponse> updateFlowEntitlement(
            UpdateFlowEntitlementRequest updateFlowEntitlementRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFlowEntitlementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFlowEntitlementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<UpdateFlowEntitlementRequest, UpdateFlowEntitlementResponse>()
                    .withMarshaller(new UpdateFlowEntitlementRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(updateFlowEntitlementRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates an existing flow output.
     *
     * @param updateFlowOutputRequest
     *        The fields that you want to update in the output.
     * @return A Java Future containing the result of the UpdateFlowOutput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.UpdateFlowOutput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowOutput" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFlowOutputResponse> updateFlowOutput(UpdateFlowOutputRequest updateFlowOutputRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFlowOutputResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFlowOutputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<UpdateFlowOutputRequest, UpdateFlowOutputResponse>()
                    .withMarshaller(new UpdateFlowOutputRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateFlowOutputRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates the source of a flow.
     *
     * @param updateFlowSourceRequest
     *        A request to update the source of a flow.
     * @return A Java Future containing the result of the UpdateFlowSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.UpdateFlowSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFlowSourceResponse> updateFlowSource(UpdateFlowSourceRequest updateFlowSourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFlowSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFlowSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<UpdateFlowSourceRequest, UpdateFlowSourceResponse>()
                    .withMarshaller(new UpdateFlowSourceRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateFlowSourceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaConnectException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CreateFlow420Exception")
                                .exceptionBuilderSupplier(CreateFlow420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GrantFlowEntitlements420Exception")
                                .exceptionBuilderSupplier(GrantFlowEntitlements420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AddFlowOutputs420Exception")
                                .exceptionBuilderSupplier(AddFlowOutputs420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private <T extends MediaConnectRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
