/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconnect;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputs420Exception;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputsRequest;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputsResponse;
import software.amazon.awssdk.services.mediaconnect.model.BadRequestException;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlow420Exception;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.DeleteFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.DeleteFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.ForbiddenException;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlements420Exception;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlementsRequest;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlementsResponse;
import software.amazon.awssdk.services.mediaconnect.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsResponse;
import software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListFlowsResponse;
import software.amazon.awssdk.services.mediaconnect.model.MediaConnectException;
import software.amazon.awssdk.services.mediaconnect.model.MediaConnectRequest;
import software.amazon.awssdk.services.mediaconnect.model.NotFoundException;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowOutputRequest;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowOutputResponse;
import software.amazon.awssdk.services.mediaconnect.model.RevokeFlowEntitlementRequest;
import software.amazon.awssdk.services.mediaconnect.model.RevokeFlowEntitlementResponse;
import software.amazon.awssdk.services.mediaconnect.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mediaconnect.model.StartFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.StartFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.StopFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.StopFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowEntitlementRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowEntitlementResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowOutputRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowOutputResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowSourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowSourceResponse;
import software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsIterable;
import software.amazon.awssdk.services.mediaconnect.transform.AddFlowOutputsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.CreateFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DeleteFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DescribeFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.GrantFlowEntitlementsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListEntitlementsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListFlowsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RemoveFlowOutputRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RevokeFlowEntitlementRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.StartFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.StopFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowEntitlementRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowOutputRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowSourceRequestMarshaller;

/**
 * Internal implementation of {@link MediaConnectClient}.
 *
 * @see MediaConnectClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaConnectClient implements MediaConnectClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaConnectClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Adds outputs to an existing flow. You can create up to 20 outputs per flow.
     *
     * @param addFlowOutputsRequest
     *        A request to add outputs to the specified flow.
     * @return Result of the AddFlowOutputs operation returned by the service.
     * @throws AddFlowOutputs420Exception
     *         AWS Elemental MediaConnect can't complete this request because this flow already has the maximum number
     *         of allowed outputs (20). For more information, contact AWS Customer Support.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.AddFlowOutputs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/AddFlowOutputs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AddFlowOutputsResponse addFlowOutputs(AddFlowOutputsRequest addFlowOutputsRequest) throws AddFlowOutputs420Exception,
            BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddFlowOutputsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddFlowOutputsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AddFlowOutputsRequest, AddFlowOutputsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(addFlowOutputsRequest).withMarshaller(new AddFlowOutputsRequestMarshaller(protocolFactory)));
    }

    /**
     * Creates a new flow. The request must include one source. The request optionally can include outputs (up to 20)
     * and entitlements (up to 50).
     *
     * @param createFlowRequest
     *        Creates a new flow. The request must include one source. The request optionally can include outputs (up to
     *        20) and entitlements (up to 50).
     * @return Result of the CreateFlow operation returned by the service.
     * @throws CreateFlow420Exception
     *         Your account already contains the maximum number of 20 flows per account, per Region. For more
     *         information, contact AWS Customer Support.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.CreateFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/CreateFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFlowResponse createFlow(CreateFlowRequest createFlowRequest) throws CreateFlow420Exception, BadRequestException,
            InternalServerErrorException, ForbiddenException, ServiceUnavailableException, TooManyRequestsException,
            AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateFlowRequest, CreateFlowResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(createFlowRequest)
                .withMarshaller(new CreateFlowRequestMarshaller(protocolFactory)));
    }

    /**
     * Deletes a flow. Before you can delete a flow, you must stop the flow.
     *
     * @param deleteFlowRequest
     * @return Result of the DeleteFlow operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.DeleteFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DeleteFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFlowResponse deleteFlow(DeleteFlowRequest deleteFlowRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteFlowRequest, DeleteFlowResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(deleteFlowRequest)
                .withMarshaller(new DeleteFlowRequestMarshaller(protocolFactory)));
    }

    /**
     * Displays the details of a flow. The response includes the flow ARN, name, and Availability Zone, as well as
     * details about the source, outputs, and entitlements.
     *
     * @param describeFlowRequest
     * @return Result of the DescribeFlow operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.DescribeFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DescribeFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeFlowResponse describeFlow(DescribeFlowRequest describeFlowRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeFlowRequest, DescribeFlowResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeFlowRequest).withMarshaller(new DescribeFlowRequestMarshaller(protocolFactory)));
    }

    /**
     * Grants entitlements to an existing flow.
     *
     * @param grantFlowEntitlementsRequest
     *        A request to grant entitlements on a flow.
     * @return Result of the GrantFlowEntitlements operation returned by the service.
     * @throws GrantFlowEntitlements420Exception
     *         AWS Elemental MediaConnect can't complete this request because this flow already has the maximum number
     *         of allowed entitlements (50). For more information, contact AWS Customer Support.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.GrantFlowEntitlements
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/GrantFlowEntitlements"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GrantFlowEntitlementsResponse grantFlowEntitlements(GrantFlowEntitlementsRequest grantFlowEntitlementsRequest)
            throws GrantFlowEntitlements420Exception, BadRequestException, InternalServerErrorException, ForbiddenException,
            NotFoundException, ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GrantFlowEntitlementsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GrantFlowEntitlementsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GrantFlowEntitlementsRequest, GrantFlowEntitlementsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(grantFlowEntitlementsRequest)
                .withMarshaller(new GrantFlowEntitlementsRequestMarshaller(protocolFactory)));
    }

    /**
     * Displays a list of all entitlements that have been granted to this account. This request returns 20 results per
     * page.
     *
     * @param listEntitlementsRequest
     * @return Result of the ListEntitlements operation returned by the service.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListEntitlements
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListEntitlements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListEntitlementsResponse listEntitlements(ListEntitlementsRequest listEntitlementsRequest)
            throws ServiceUnavailableException, TooManyRequestsException, BadRequestException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEntitlementsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEntitlementsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListEntitlementsRequest, ListEntitlementsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listEntitlementsRequest).withMarshaller(new ListEntitlementsRequestMarshaller(protocolFactory)));
    }

    /**
     * Displays a list of flows that are associated with this account. This request returns a paginated result.
     *
     * @param listFlowsRequest
     * @return Result of the ListFlows operation returned by the service.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListFlows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListFlows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFlowsResponse listFlows(ListFlowsRequest listFlowsRequest) throws ServiceUnavailableException,
            TooManyRequestsException, BadRequestException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFlowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListFlowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListFlowsRequest, ListFlowsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listFlowsRequest)
                .withMarshaller(new ListFlowsRequestMarshaller(protocolFactory)));
    }

    /**
     * Displays a list of flows that are associated with this account. This request returns a paginated result.<br/>
     * <p>
     * This is a variant of {@link #listFlows(software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsIterable responses = client.listFlowsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsIterable responses = client.listFlowsPaginator(request);
     *     for (software.amazon.awssdk.services.mediaconnect.model.ListFlowsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsIterable responses = client.listFlowsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFlows(software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest)} operation.</b>
     * </p>
     *
     * @param listFlowsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListFlows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListFlows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFlowsIterable listFlowsPaginator(ListFlowsRequest listFlowsRequest) throws ServiceUnavailableException,
            TooManyRequestsException, BadRequestException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaConnectException {
        return new ListFlowsIterable(this, applyPaginatorUserAgent(listFlowsRequest));
    }

    /**
     * Removes an output from an existing flow. This request can be made only on an output that does not have an
     * entitlement associated with it. If the output has an entitlement, you must revoke the entitlement instead. When
     * an entitlement is revoked from a flow, the service automatically removes the associated output.
     *
     * @param removeFlowOutputRequest
     * @return Result of the RemoveFlowOutput operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.RemoveFlowOutput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RemoveFlowOutput" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RemoveFlowOutputResponse removeFlowOutput(RemoveFlowOutputRequest removeFlowOutputRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveFlowOutputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RemoveFlowOutputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RemoveFlowOutputRequest, RemoveFlowOutputResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(removeFlowOutputRequest).withMarshaller(new RemoveFlowOutputRequestMarshaller(protocolFactory)));
    }

    /**
     * Revokes an entitlement from a flow. Once an entitlement is revoked, the content becomes unavailable to the
     * subscriber and the associated output is removed.
     *
     * @param revokeFlowEntitlementRequest
     * @return Result of the RevokeFlowEntitlement operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.RevokeFlowEntitlement
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RevokeFlowEntitlement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RevokeFlowEntitlementResponse revokeFlowEntitlement(RevokeFlowEntitlementRequest revokeFlowEntitlementRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RevokeFlowEntitlementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RevokeFlowEntitlementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RevokeFlowEntitlementRequest, RevokeFlowEntitlementResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(revokeFlowEntitlementRequest)
                .withMarshaller(new RevokeFlowEntitlementRequestMarshaller(protocolFactory)));
    }

    /**
     * Starts a flow.
     *
     * @param startFlowRequest
     * @return Result of the StartFlow operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.StartFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/StartFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartFlowResponse startFlow(StartFlowRequest startFlowRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StartFlowRequest, StartFlowResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(startFlowRequest)
                .withMarshaller(new StartFlowRequestMarshaller(protocolFactory)));
    }

    /**
     * Stops a flow.
     *
     * @param stopFlowRequest
     * @return Result of the StopFlow operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.StopFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/StopFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopFlowResponse stopFlow(StopFlowRequest stopFlowRequest) throws BadRequestException, InternalServerErrorException,
            ForbiddenException, NotFoundException, ServiceUnavailableException, TooManyRequestsException, AwsServiceException,
            SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StopFlowRequest, StopFlowResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(stopFlowRequest)
                .withMarshaller(new StopFlowRequestMarshaller(protocolFactory)));
    }

    /**
     * You can change an entitlement's description, subscribers, and encryption. If you change the subscribers, the
     * service will remove the outputs that are are used by the subscribers that are removed.
     *
     * @param updateFlowEntitlementRequest
     *        The entitlement fields that you want to update.
     * @return Result of the UpdateFlowEntitlement operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.UpdateFlowEntitlement
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowEntitlement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateFlowEntitlementResponse updateFlowEntitlement(UpdateFlowEntitlementRequest updateFlowEntitlementRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFlowEntitlementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateFlowEntitlementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateFlowEntitlementRequest, UpdateFlowEntitlementResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(updateFlowEntitlementRequest)
                .withMarshaller(new UpdateFlowEntitlementRequestMarshaller(protocolFactory)));
    }

    /**
     * Updates an existing flow output.
     *
     * @param updateFlowOutputRequest
     *        The fields that you want to update in the output.
     * @return Result of the UpdateFlowOutput operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.UpdateFlowOutput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowOutput" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateFlowOutputResponse updateFlowOutput(UpdateFlowOutputRequest updateFlowOutputRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFlowOutputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFlowOutputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateFlowOutputRequest, UpdateFlowOutputResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(updateFlowOutputRequest).withMarshaller(new UpdateFlowOutputRequestMarshaller(protocolFactory)));
    }

    /**
     * Updates the source of a flow.
     *
     * @param updateFlowSourceRequest
     *        A request to update the source of a flow.
     * @return Result of the UpdateFlowSource operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.UpdateFlowSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateFlowSourceResponse updateFlowSource(UpdateFlowSourceRequest updateFlowSourceRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFlowSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFlowSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateFlowSourceRequest, UpdateFlowSourceResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(updateFlowSourceRequest).withMarshaller(new UpdateFlowSourceRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaConnectException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CreateFlow420Exception")
                                .exceptionBuilderSupplier(CreateFlow420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GrantFlowEntitlements420Exception")
                                .exceptionBuilderSupplier(GrantFlowEntitlements420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AddFlowOutputs420Exception")
                                .exceptionBuilderSupplier(AddFlowOutputs420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MediaConnectRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
