/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconnect;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowMediaStreamsRequest;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowMediaStreamsResponse;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputs420Exception;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputsRequest;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputsResponse;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowSourcesRequest;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowSourcesResponse;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowVpcInterfacesRequest;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowVpcInterfacesResponse;
import software.amazon.awssdk.services.mediaconnect.model.BadRequestException;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlow420Exception;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.DeleteFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.DeleteFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.DescribeOfferingRequest;
import software.amazon.awssdk.services.mediaconnect.model.DescribeOfferingResponse;
import software.amazon.awssdk.services.mediaconnect.model.DescribeReservationRequest;
import software.amazon.awssdk.services.mediaconnect.model.DescribeReservationResponse;
import software.amazon.awssdk.services.mediaconnect.model.ForbiddenException;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlements420Exception;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlementsRequest;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlementsResponse;
import software.amazon.awssdk.services.mediaconnect.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsResponse;
import software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListFlowsResponse;
import software.amazon.awssdk.services.mediaconnect.model.ListOfferingsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListOfferingsResponse;
import software.amazon.awssdk.services.mediaconnect.model.ListReservationsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListReservationsResponse;
import software.amazon.awssdk.services.mediaconnect.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediaconnect.model.MediaConnectException;
import software.amazon.awssdk.services.mediaconnect.model.MediaConnectRequest;
import software.amazon.awssdk.services.mediaconnect.model.NotFoundException;
import software.amazon.awssdk.services.mediaconnect.model.PurchaseOfferingRequest;
import software.amazon.awssdk.services.mediaconnect.model.PurchaseOfferingResponse;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowMediaStreamRequest;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowMediaStreamResponse;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowOutputRequest;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowOutputResponse;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowSourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowSourceResponse;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowVpcInterfaceRequest;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowVpcInterfaceResponse;
import software.amazon.awssdk.services.mediaconnect.model.RevokeFlowEntitlementRequest;
import software.amazon.awssdk.services.mediaconnect.model.RevokeFlowEntitlementResponse;
import software.amazon.awssdk.services.mediaconnect.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mediaconnect.model.StartFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.StartFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.StopFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.StopFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.TagResourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.TagResourceResponse;
import software.amazon.awssdk.services.mediaconnect.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediaconnect.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowEntitlementRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowEntitlementResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowMediaStreamRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowMediaStreamResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowOutputRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowOutputResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowSourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowSourceResponse;
import software.amazon.awssdk.services.mediaconnect.paginators.ListEntitlementsPublisher;
import software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsPublisher;
import software.amazon.awssdk.services.mediaconnect.paginators.ListOfferingsPublisher;
import software.amazon.awssdk.services.mediaconnect.paginators.ListReservationsPublisher;
import software.amazon.awssdk.services.mediaconnect.transform.AddFlowMediaStreamsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.AddFlowOutputsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.AddFlowSourcesRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.AddFlowVpcInterfacesRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.CreateFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DeleteFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DescribeFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DescribeOfferingRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DescribeReservationRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.GrantFlowEntitlementsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListEntitlementsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListFlowsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListOfferingsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListReservationsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.PurchaseOfferingRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RemoveFlowMediaStreamRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RemoveFlowOutputRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RemoveFlowSourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RemoveFlowVpcInterfaceRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RevokeFlowEntitlementRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.StartFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.StopFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowEntitlementRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowMediaStreamRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowOutputRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowSourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.waiters.MediaConnectAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MediaConnectAsyncClient}.
 *
 * @see MediaConnectAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaConnectAsyncClient implements MediaConnectAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMediaConnectAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultMediaConnectAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Adds media streams to an existing flow. After you add a media stream to a flow, you can associate it with a
     * source and/or an output that uses the ST 2110 JPEG XS or CDI protocol.
     *
     * @param addFlowMediaStreamsRequest
     *        A request to add media streams to the flow.
     * @return A Java Future containing the result of the AddFlowMediaStreams operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.AddFlowMediaStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/AddFlowMediaStreams"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddFlowMediaStreamsResponse> addFlowMediaStreams(
            AddFlowMediaStreamsRequest addFlowMediaStreamsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addFlowMediaStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddFlowMediaStreams");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddFlowMediaStreamsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddFlowMediaStreamsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddFlowMediaStreamsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddFlowMediaStreamsRequest, AddFlowMediaStreamsResponse>()
                            .withOperationName("AddFlowMediaStreams")
                            .withMarshaller(new AddFlowMediaStreamsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addFlowMediaStreamsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addFlowMediaStreamsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<AddFlowMediaStreamsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Adds outputs to an existing flow. You can create up to 50 outputs per flow.
     *
     * @param addFlowOutputsRequest
     *        A request to add outputs to the specified flow.
     * @return A Java Future containing the result of the AddFlowOutputs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AddFlowOutputs420Exception AWS Elemental MediaConnect can't complete this request because this flow
     *         already has the maximum number of allowed outputs (50). For more information, contact AWS Customer
     *         Support.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.AddFlowOutputs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/AddFlowOutputs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AddFlowOutputsResponse> addFlowOutputs(AddFlowOutputsRequest addFlowOutputsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addFlowOutputsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddFlowOutputs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddFlowOutputsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddFlowOutputsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddFlowOutputsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddFlowOutputsRequest, AddFlowOutputsResponse>()
                            .withOperationName("AddFlowOutputs")
                            .withMarshaller(new AddFlowOutputsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addFlowOutputsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addFlowOutputsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<AddFlowOutputsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Adds Sources to flow
     *
     * @param addFlowSourcesRequest
     *        A request to add sources to the flow.
     * @return A Java Future containing the result of the AddFlowSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.AddFlowSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/AddFlowSources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AddFlowSourcesResponse> addFlowSources(AddFlowSourcesRequest addFlowSourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addFlowSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddFlowSources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddFlowSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddFlowSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddFlowSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddFlowSourcesRequest, AddFlowSourcesResponse>()
                            .withOperationName("AddFlowSources")
                            .withMarshaller(new AddFlowSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addFlowSourcesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addFlowSourcesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<AddFlowSourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Adds VPC interfaces to flow
     *
     * @param addFlowVpcInterfacesRequest
     *        A request to add VPC interfaces to the flow.
     * @return A Java Future containing the result of the AddFlowVpcInterfaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.AddFlowVpcInterfaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/AddFlowVpcInterfaces"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddFlowVpcInterfacesResponse> addFlowVpcInterfaces(
            AddFlowVpcInterfacesRequest addFlowVpcInterfacesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addFlowVpcInterfacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddFlowVpcInterfaces");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddFlowVpcInterfacesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddFlowVpcInterfacesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddFlowVpcInterfacesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddFlowVpcInterfacesRequest, AddFlowVpcInterfacesResponse>()
                            .withOperationName("AddFlowVpcInterfaces")
                            .withMarshaller(new AddFlowVpcInterfacesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addFlowVpcInterfacesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addFlowVpcInterfacesRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<AddFlowVpcInterfacesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a new flow. The request must include one source. The request optionally can include outputs (up to 50)
     * and entitlements (up to 50).
     *
     * @param createFlowRequest
     *        Creates a new flow. The request must include one source. The request optionally can include outputs (up to
     *        50) and entitlements (up to 50).
     * @return A Java Future containing the result of the CreateFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CreateFlow420Exception Your account already contains the maximum number of 20 flows per account, per
     *         Region. For more information, contact AWS Customer Support.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.CreateFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/CreateFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFlowResponse> createFlow(CreateFlowRequest createFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFlowRequest, CreateFlowResponse>().withOperationName("CreateFlow")
                            .withMarshaller(new CreateFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createFlowRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createFlowRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes a flow. Before you can delete a flow, you must stop the flow.
     *
     * @param deleteFlowRequest
     * @return A Java Future containing the result of the DeleteFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.DeleteFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DeleteFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFlowResponse> deleteFlow(DeleteFlowRequest deleteFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFlowRequest, DeleteFlowResponse>().withOperationName("DeleteFlow")
                            .withMarshaller(new DeleteFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFlowRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteFlowRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays the details of a flow. The response includes the flow ARN, name, and Availability Zone, as well as
     * details about the source, outputs, and entitlements.
     *
     * @param describeFlowRequest
     * @return A Java Future containing the result of the DescribeFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.DescribeFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DescribeFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFlowResponse> describeFlow(DescribeFlowRequest describeFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFlowRequest, DescribeFlowResponse>()
                            .withOperationName("DescribeFlow").withMarshaller(new DescribeFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFlowRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeFlowRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays the details of an offering. The response includes the offering description, duration, outbound
     * bandwidth, price, and Amazon Resource Name (ARN).
     *
     * @param describeOfferingRequest
     * @return A Java Future containing the result of the DescribeOffering operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.DescribeOffering
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DescribeOffering" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeOfferingResponse> describeOffering(DescribeOfferingRequest describeOfferingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeOfferingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOffering");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeOfferingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeOfferingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeOfferingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeOfferingRequest, DescribeOfferingResponse>()
                            .withOperationName("DescribeOffering")
                            .withMarshaller(new DescribeOfferingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeOfferingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeOfferingRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeOfferingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays the details of a reservation. The response includes the reservation name, state, start date and time,
     * and the details of the offering that make up the rest of the reservation (such as price, duration, and outbound
     * bandwidth).
     *
     * @param describeReservationRequest
     * @return A Java Future containing the result of the DescribeReservation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.DescribeReservation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DescribeReservation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeReservationResponse> describeReservation(
            DescribeReservationRequest describeReservationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeReservationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReservation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeReservationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeReservationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeReservationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeReservationRequest, DescribeReservationResponse>()
                            .withOperationName("DescribeReservation")
                            .withMarshaller(new DescribeReservationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeReservationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeReservationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeReservationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Grants entitlements to an existing flow.
     *
     * @param grantFlowEntitlementsRequest
     *        A request to grant entitlements on a flow.
     * @return A Java Future containing the result of the GrantFlowEntitlements operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>GrantFlowEntitlements420Exception AWS Elemental MediaConnect can't complete this request because this
     *         flow already has the maximum number of allowed entitlements (50). For more information, contact AWS
     *         Customer Support.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.GrantFlowEntitlements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/GrantFlowEntitlements"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GrantFlowEntitlementsResponse> grantFlowEntitlements(
            GrantFlowEntitlementsRequest grantFlowEntitlementsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, grantFlowEntitlementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GrantFlowEntitlements");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GrantFlowEntitlementsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GrantFlowEntitlementsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GrantFlowEntitlementsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GrantFlowEntitlementsRequest, GrantFlowEntitlementsResponse>()
                            .withOperationName("GrantFlowEntitlements")
                            .withMarshaller(new GrantFlowEntitlementsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(grantFlowEntitlementsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = grantFlowEntitlementsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GrantFlowEntitlementsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays a list of all entitlements that have been granted to this account. This request returns 20 results per
     * page.
     *
     * @param listEntitlementsRequest
     * @return A Java Future containing the result of the ListEntitlements operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListEntitlements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListEntitlements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEntitlementsResponse> listEntitlements(ListEntitlementsRequest listEntitlementsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEntitlementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEntitlements");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEntitlementsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEntitlementsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEntitlementsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEntitlementsRequest, ListEntitlementsResponse>()
                            .withOperationName("ListEntitlements")
                            .withMarshaller(new ListEntitlementsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEntitlementsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listEntitlementsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListEntitlementsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays a list of all entitlements that have been granted to this account. This request returns 20 results per
     * page.<br/>
     * <p>
     * This is a variant of
     * {@link #listEntitlements(software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListEntitlementsPublisher publisher = client.listEntitlementsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListEntitlementsPublisher publisher = client.listEntitlementsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEntitlements(software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEntitlementsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListEntitlements
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListEntitlements" target="_top">AWS
     *      API Documentation</a>
     */
    public ListEntitlementsPublisher listEntitlementsPaginator(ListEntitlementsRequest listEntitlementsRequest) {
        return new ListEntitlementsPublisher(this, applyPaginatorUserAgent(listEntitlementsRequest));
    }

    /**
     * Displays a list of flows that are associated with this account. This request returns a paginated result.
     *
     * @param listFlowsRequest
     * @return A Java Future containing the result of the ListFlows operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListFlows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListFlows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFlowsResponse> listFlows(ListFlowsRequest listFlowsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFlowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFlows");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFlowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFlowsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFlowsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFlowsRequest, ListFlowsResponse>().withOperationName("ListFlows")
                            .withMarshaller(new ListFlowsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listFlowsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listFlowsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListFlowsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays a list of flows that are associated with this account. This request returns a paginated result.<br/>
     * <p>
     * This is a variant of {@link #listFlows(software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsPublisher publisher = client.listFlowsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsPublisher publisher = client.listFlowsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediaconnect.model.ListFlowsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediaconnect.model.ListFlowsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFlows(software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest)} operation.</b>
     * </p>
     *
     * @param listFlowsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListFlows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListFlows" target="_top">AWS API
     *      Documentation</a>
     */
    public ListFlowsPublisher listFlowsPaginator(ListFlowsRequest listFlowsRequest) {
        return new ListFlowsPublisher(this, applyPaginatorUserAgent(listFlowsRequest));
    }

    /**
     * Displays a list of all offerings that are available to this account in the current AWS Region. If you have an
     * active reservation (which means you've purchased an offering that has already started and hasn't expired yet),
     * your account isn't eligible for other offerings.
     *
     * @param listOfferingsRequest
     * @return A Java Future containing the result of the ListOfferings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListOfferings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListOfferings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOfferingsResponse> listOfferings(ListOfferingsRequest listOfferingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOfferingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOfferings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOfferingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListOfferingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOfferingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOfferingsRequest, ListOfferingsResponse>()
                            .withOperationName("ListOfferings")
                            .withMarshaller(new ListOfferingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listOfferingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listOfferingsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListOfferingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays a list of all offerings that are available to this account in the current AWS Region. If you have an
     * active reservation (which means you've purchased an offering that has already started and hasn't expired yet),
     * your account isn't eligible for other offerings.<br/>
     * <p>
     * This is a variant of
     * {@link #listOfferings(software.amazon.awssdk.services.mediaconnect.model.ListOfferingsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListOfferingsPublisher publisher = client.listOfferingsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListOfferingsPublisher publisher = client.listOfferingsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediaconnect.model.ListOfferingsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediaconnect.model.ListOfferingsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOfferings(software.amazon.awssdk.services.mediaconnect.model.ListOfferingsRequest)} operation.</b>
     * </p>
     *
     * @param listOfferingsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListOfferings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListOfferings" target="_top">AWS
     *      API Documentation</a>
     */
    public ListOfferingsPublisher listOfferingsPaginator(ListOfferingsRequest listOfferingsRequest) {
        return new ListOfferingsPublisher(this, applyPaginatorUserAgent(listOfferingsRequest));
    }

    /**
     * Displays a list of all reservations that have been purchased by this account in the current AWS Region. This list
     * includes all reservations in all states (such as active and expired).
     *
     * @param listReservationsRequest
     * @return A Java Future containing the result of the ListReservations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListReservations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListReservations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReservationsResponse> listReservations(ListReservationsRequest listReservationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReservationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReservations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReservationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReservationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReservationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReservationsRequest, ListReservationsResponse>()
                            .withOperationName("ListReservations")
                            .withMarshaller(new ListReservationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listReservationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listReservationsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListReservationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Displays a list of all reservations that have been purchased by this account in the current AWS Region. This list
     * includes all reservations in all states (such as active and expired).<br/>
     * <p>
     * This is a variant of
     * {@link #listReservations(software.amazon.awssdk.services.mediaconnect.model.ListReservationsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListReservationsPublisher publisher = client.listReservationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListReservationsPublisher publisher = client.listReservationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediaconnect.model.ListReservationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediaconnect.model.ListReservationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReservations(software.amazon.awssdk.services.mediaconnect.model.ListReservationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listReservationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListReservations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListReservations" target="_top">AWS
     *      API Documentation</a>
     */
    public ListReservationsPublisher listReservationsPaginator(ListReservationsRequest listReservationsRequest) {
        return new ListReservationsPublisher(this, applyPaginatorUserAgent(listReservationsRequest));
    }

    /**
     * List all tags on an AWS Elemental MediaConnect resource
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The requested resource was not found</li>
     *         <li>BadRequestException The client performed an invalid request</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Submits a request to purchase an offering. If you already have an active reservation, you can't purchase another
     * offering.
     *
     * @param purchaseOfferingRequest
     *        A request to purchase a offering.
     * @return A Java Future containing the result of the PurchaseOffering operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.PurchaseOffering
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/PurchaseOffering" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PurchaseOfferingResponse> purchaseOffering(PurchaseOfferingRequest purchaseOfferingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, purchaseOfferingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PurchaseOffering");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PurchaseOfferingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PurchaseOfferingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PurchaseOfferingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PurchaseOfferingRequest, PurchaseOfferingResponse>()
                            .withOperationName("PurchaseOffering")
                            .withMarshaller(new PurchaseOfferingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(purchaseOfferingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = purchaseOfferingRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PurchaseOfferingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Removes a media stream from a flow. This action is only available if the media stream is not associated with a
     * source or output.
     *
     * @param removeFlowMediaStreamRequest
     * @return A Java Future containing the result of the RemoveFlowMediaStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.RemoveFlowMediaStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RemoveFlowMediaStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveFlowMediaStreamResponse> removeFlowMediaStream(
            RemoveFlowMediaStreamRequest removeFlowMediaStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeFlowMediaStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveFlowMediaStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveFlowMediaStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveFlowMediaStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveFlowMediaStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveFlowMediaStreamRequest, RemoveFlowMediaStreamResponse>()
                            .withOperationName("RemoveFlowMediaStream")
                            .withMarshaller(new RemoveFlowMediaStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeFlowMediaStreamRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = removeFlowMediaStreamRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<RemoveFlowMediaStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Removes an output from an existing flow. This request can be made only on an output that does not have an
     * entitlement associated with it. If the output has an entitlement, you must revoke the entitlement instead. When
     * an entitlement is revoked from a flow, the service automatically removes the associated output.
     *
     * @param removeFlowOutputRequest
     * @return A Java Future containing the result of the RemoveFlowOutput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.RemoveFlowOutput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RemoveFlowOutput" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveFlowOutputResponse> removeFlowOutput(RemoveFlowOutputRequest removeFlowOutputRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeFlowOutputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveFlowOutput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveFlowOutputResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveFlowOutputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveFlowOutputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveFlowOutputRequest, RemoveFlowOutputResponse>()
                            .withOperationName("RemoveFlowOutput")
                            .withMarshaller(new RemoveFlowOutputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeFlowOutputRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = removeFlowOutputRequest.overrideConfiguration().orElse(null);
            CompletableFuture<RemoveFlowOutputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Removes a source from an existing flow. This request can be made only if there is more than one source on the
     * flow.
     *
     * @param removeFlowSourceRequest
     * @return A Java Future containing the result of the RemoveFlowSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.RemoveFlowSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RemoveFlowSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveFlowSourceResponse> removeFlowSource(RemoveFlowSourceRequest removeFlowSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeFlowSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveFlowSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveFlowSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveFlowSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveFlowSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveFlowSourceRequest, RemoveFlowSourceResponse>()
                            .withOperationName("RemoveFlowSource")
                            .withMarshaller(new RemoveFlowSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeFlowSourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = removeFlowSourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<RemoveFlowSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Removes a VPC Interface from an existing flow. This request can be made only on a VPC interface that does not
     * have a Source or Output associated with it. If the VPC interface is referenced by a Source or Output, you must
     * first delete or update the Source or Output to no longer reference the VPC interface.
     *
     * @param removeFlowVpcInterfaceRequest
     * @return A Java Future containing the result of the RemoveFlowVpcInterface operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.RemoveFlowVpcInterface
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RemoveFlowVpcInterface"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveFlowVpcInterfaceResponse> removeFlowVpcInterface(
            RemoveFlowVpcInterfaceRequest removeFlowVpcInterfaceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeFlowVpcInterfaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveFlowVpcInterface");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveFlowVpcInterfaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveFlowVpcInterfaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveFlowVpcInterfaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveFlowVpcInterfaceRequest, RemoveFlowVpcInterfaceResponse>()
                            .withOperationName("RemoveFlowVpcInterface")
                            .withMarshaller(new RemoveFlowVpcInterfaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeFlowVpcInterfaceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = removeFlowVpcInterfaceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<RemoveFlowVpcInterfaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Revokes an entitlement from a flow. Once an entitlement is revoked, the content becomes unavailable to the
     * subscriber and the associated output is removed.
     *
     * @param revokeFlowEntitlementRequest
     * @return A Java Future containing the result of the RevokeFlowEntitlement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.RevokeFlowEntitlement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RevokeFlowEntitlement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RevokeFlowEntitlementResponse> revokeFlowEntitlement(
            RevokeFlowEntitlementRequest revokeFlowEntitlementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, revokeFlowEntitlementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RevokeFlowEntitlement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RevokeFlowEntitlementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RevokeFlowEntitlementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RevokeFlowEntitlementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RevokeFlowEntitlementRequest, RevokeFlowEntitlementResponse>()
                            .withOperationName("RevokeFlowEntitlement")
                            .withMarshaller(new RevokeFlowEntitlementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(revokeFlowEntitlementRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = revokeFlowEntitlementRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<RevokeFlowEntitlementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Starts a flow.
     *
     * @param startFlowRequest
     * @return A Java Future containing the result of the StartFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.StartFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/StartFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartFlowResponse> startFlow(StartFlowRequest startFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartFlowRequest, StartFlowResponse>().withOperationName("StartFlow")
                            .withMarshaller(new StartFlowRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(startFlowRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startFlowRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StartFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Stops a flow.
     *
     * @param stopFlowRequest
     * @return A Java Future containing the result of the StopFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.StopFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/StopFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopFlowResponse> stopFlow(StopFlowRequest stopFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopFlowRequest, StopFlowResponse>().withOperationName("StopFlow")
                            .withMarshaller(new StopFlowRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopFlowRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = stopFlowRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StopFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Associates the specified tags to a resource with the specified resourceArn. If existing tags on a resource are
     * not specified in the request parameters, they are not changed. When a resource is deleted, the tags associated
     * with that resource are deleted as well.
     *
     * @param tagResourceRequest
     *        The tags to add to the resource. A tag is an array of key-value pairs. Tag keys can have a maximum
     *        character length of 128 characters, and tag values can have a maximum length of 256 characters.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The requested resource was not found</li>
     *         <li>BadRequestException The client performed an invalid request</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes specified tags from a resource.
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The requested resource was not found</li>
     *         <li>BadRequestException The client performed an invalid request</li>
     *         <li>InternalServerErrorException Internal service error</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates flow
     *
     * @param updateFlowRequest
     *        A request to update flow.
     * @return A Java Future containing the result of the UpdateFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.UpdateFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFlowResponse> updateFlow(UpdateFlowRequest updateFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFlowRequest, UpdateFlowResponse>().withOperationName("UpdateFlow")
                            .withMarshaller(new UpdateFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFlowRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateFlowRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * You can change an entitlement's description, subscribers, and encryption. If you change the subscribers, the
     * service will remove the outputs that are are used by the subscribers that are removed.
     *
     * @param updateFlowEntitlementRequest
     *        The entitlement fields that you want to update.
     * @return A Java Future containing the result of the UpdateFlowEntitlement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.UpdateFlowEntitlement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowEntitlement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFlowEntitlementResponse> updateFlowEntitlement(
            UpdateFlowEntitlementRequest updateFlowEntitlementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFlowEntitlementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFlowEntitlement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFlowEntitlementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFlowEntitlementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFlowEntitlementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFlowEntitlementRequest, UpdateFlowEntitlementResponse>()
                            .withOperationName("UpdateFlowEntitlement")
                            .withMarshaller(new UpdateFlowEntitlementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFlowEntitlementRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateFlowEntitlementRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateFlowEntitlementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates an existing media stream.
     *
     * @param updateFlowMediaStreamRequest
     *        The fields that you want to update in the media stream.
     * @return A Java Future containing the result of the UpdateFlowMediaStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.UpdateFlowMediaStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowMediaStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFlowMediaStreamResponse> updateFlowMediaStream(
            UpdateFlowMediaStreamRequest updateFlowMediaStreamRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFlowMediaStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFlowMediaStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFlowMediaStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFlowMediaStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFlowMediaStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFlowMediaStreamRequest, UpdateFlowMediaStreamResponse>()
                            .withOperationName("UpdateFlowMediaStream")
                            .withMarshaller(new UpdateFlowMediaStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFlowMediaStreamRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateFlowMediaStreamRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateFlowMediaStreamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates an existing flow output.
     *
     * @param updateFlowOutputRequest
     *        The fields that you want to update in the output.
     * @return A Java Future containing the result of the UpdateFlowOutput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.UpdateFlowOutput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowOutput" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFlowOutputResponse> updateFlowOutput(UpdateFlowOutputRequest updateFlowOutputRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFlowOutputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFlowOutput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFlowOutputResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFlowOutputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFlowOutputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFlowOutputRequest, UpdateFlowOutputResponse>()
                            .withOperationName("UpdateFlowOutput")
                            .withMarshaller(new UpdateFlowOutputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFlowOutputRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateFlowOutputRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateFlowOutputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates the source of a flow.
     *
     * @param updateFlowSourceRequest
     *        A request to update the source of a flow.
     * @return A Java Future containing the result of the UpdateFlowSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The request that you submitted is not valid.</li>
     *         <li>InternalServerErrorException AWS Elemental MediaConnect can't fulfill your request because it
     *         encountered an unexpected condition.</li>
     *         <li>ForbiddenException You don't have the required permissions to perform this operation.</li>
     *         <li>NotFoundException AWS Elemental MediaConnect did not find the resource that you specified in the
     *         request.</li>
     *         <li>ServiceUnavailableException AWS Elemental MediaConnect is currently unavailable. Try again later.</li>
     *         <li>TooManyRequestsException You have exceeded the service request rate limit for your AWS Elemental
     *         MediaConnect account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConnectAsyncClient.UpdateFlowSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFlowSourceResponse> updateFlowSource(UpdateFlowSourceRequest updateFlowSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFlowSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFlowSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFlowSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFlowSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFlowSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFlowSourceRequest, UpdateFlowSourceResponse>()
                            .withOperationName("UpdateFlowSource")
                            .withMarshaller(new UpdateFlowSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFlowSourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateFlowSourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateFlowSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaConnectException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CreateFlow420Exception")
                                .exceptionBuilderSupplier(CreateFlow420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GrantFlowEntitlements420Exception")
                                .exceptionBuilderSupplier(GrantFlowEntitlements420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AddFlowOutputs420Exception")
                                .exceptionBuilderSupplier(AddFlowOutputs420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends MediaConnectRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public MediaConnectAsyncWaiter waiter() {
        return MediaConnectAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }
}
