/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Settings related to WebVTT captions. WebVTT is a sidecar format that holds captions in a file that is separate from
 * the video container. Set up sidecar captions in the same output group, but different output from your video. For more
 * information, see https://docs.aws.amazon.com/mediaconvert/latest/ug/ttml-and-webvtt-output-captions.html. When you
 * work directly in your JSON job specification, include this object and any required children when you set
 * destinationType to WebVTT.
 */
@Generated("software.amazon.awssdk:codegen")
public final class WebvttDestinationSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<WebvttDestinationSettings.Builder, WebvttDestinationSettings> {
    private static final SdkField<String> ACCESSIBILITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Accessibility").getter(getter(WebvttDestinationSettings::accessibilityAsString))
            .setter(setter(Builder::accessibility))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessibility").build()).build();

    private static final SdkField<String> STYLE_PASSTHROUGH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StylePassthrough").getter(getter(WebvttDestinationSettings::stylePassthroughAsString))
            .setter(setter(Builder::stylePassthrough))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stylePassthrough").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCESSIBILITY_FIELD,
            STYLE_PASSTHROUGH_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accessibility;

    private final String stylePassthrough;

    private WebvttDestinationSettings(BuilderImpl builder) {
        this.accessibility = builder.accessibility;
        this.stylePassthrough = builder.stylePassthrough;
    }

    /**
     * Set Accessibility subtitles (Accessibility) to Enabled (ENABLED) if the ISMC or WebVTT captions track is intended
     * to provide accessibility for people who are deaf or hard of hearing. When you enable this feature, MediaConvert
     * adds the following attributes under EXT-X-MEDIA in the HLS or CMAF manifest for this track:
     * CHARACTERISTICS="public.accessibility.describes-spoken-dialog,public.accessibility.describes-music-and-sound" and
     * AUTOSELECT="YES". Keep the default value, Disabled (DISABLED), if the captions track is not intended to provide
     * such accessibility. MediaConvert will not add the above attributes.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessibility}
     * will return {@link WebvttAccessibilitySubs#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #accessibilityAsString}.
     * </p>
     * 
     * @return Set Accessibility subtitles (Accessibility) to Enabled (ENABLED) if the ISMC or WebVTT captions track is
     *         intended to provide accessibility for people who are deaf or hard of hearing. When you enable this
     *         feature, MediaConvert adds the following attributes under EXT-X-MEDIA in the HLS or CMAF manifest for
     *         this track: CHARACTERISTICS=
     *         "public.accessibility.describes-spoken-dialog,public.accessibility.describes-music-and-sound" and
     *         AUTOSELECT="YES". Keep the default value, Disabled (DISABLED), if the captions track is not intended to
     *         provide such accessibility. MediaConvert will not add the above attributes.
     * @see WebvttAccessibilitySubs
     */
    public final WebvttAccessibilitySubs accessibility() {
        return WebvttAccessibilitySubs.fromValue(accessibility);
    }

    /**
     * Set Accessibility subtitles (Accessibility) to Enabled (ENABLED) if the ISMC or WebVTT captions track is intended
     * to provide accessibility for people who are deaf or hard of hearing. When you enable this feature, MediaConvert
     * adds the following attributes under EXT-X-MEDIA in the HLS or CMAF manifest for this track:
     * CHARACTERISTICS="public.accessibility.describes-spoken-dialog,public.accessibility.describes-music-and-sound" and
     * AUTOSELECT="YES". Keep the default value, Disabled (DISABLED), if the captions track is not intended to provide
     * such accessibility. MediaConvert will not add the above attributes.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessibility}
     * will return {@link WebvttAccessibilitySubs#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #accessibilityAsString}.
     * </p>
     * 
     * @return Set Accessibility subtitles (Accessibility) to Enabled (ENABLED) if the ISMC or WebVTT captions track is
     *         intended to provide accessibility for people who are deaf or hard of hearing. When you enable this
     *         feature, MediaConvert adds the following attributes under EXT-X-MEDIA in the HLS or CMAF manifest for
     *         this track: CHARACTERISTICS=
     *         "public.accessibility.describes-spoken-dialog,public.accessibility.describes-music-and-sound" and
     *         AUTOSELECT="YES". Keep the default value, Disabled (DISABLED), if the captions track is not intended to
     *         provide such accessibility. MediaConvert will not add the above attributes.
     * @see WebvttAccessibilitySubs
     */
    public final String accessibilityAsString() {
        return accessibility;
    }

    /**
     * To use the available style, color, and position information from your input captions: Set Style passthrough
     * (stylePassthrough) to Enabled (ENABLED). MediaConvert uses default settings when style and position information
     * is missing from your input captions. To recreate the input captions exactly: Set Style passthrough to Strict
     * (STRICT). MediaConvert automatically applies timing adjustments, including adjustments for frame rate conversion,
     * ad avails, and input clipping. Your input captions format must be WebVTT. To ignore the style and position
     * information from your input captions and use simplified output captions: Set Style passthrough to Disabled
     * (DISABLED), or leave blank.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stylePassthrough}
     * will return {@link WebvttStylePassthrough#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stylePassthroughAsString}.
     * </p>
     * 
     * @return To use the available style, color, and position information from your input captions: Set Style
     *         passthrough (stylePassthrough) to Enabled (ENABLED). MediaConvert uses default settings when style and
     *         position information is missing from your input captions. To recreate the input captions exactly: Set
     *         Style passthrough to Strict (STRICT). MediaConvert automatically applies timing adjustments, including
     *         adjustments for frame rate conversion, ad avails, and input clipping. Your input captions format must be
     *         WebVTT. To ignore the style and position information from your input captions and use simplified output
     *         captions: Set Style passthrough to Disabled (DISABLED), or leave blank.
     * @see WebvttStylePassthrough
     */
    public final WebvttStylePassthrough stylePassthrough() {
        return WebvttStylePassthrough.fromValue(stylePassthrough);
    }

    /**
     * To use the available style, color, and position information from your input captions: Set Style passthrough
     * (stylePassthrough) to Enabled (ENABLED). MediaConvert uses default settings when style and position information
     * is missing from your input captions. To recreate the input captions exactly: Set Style passthrough to Strict
     * (STRICT). MediaConvert automatically applies timing adjustments, including adjustments for frame rate conversion,
     * ad avails, and input clipping. Your input captions format must be WebVTT. To ignore the style and position
     * information from your input captions and use simplified output captions: Set Style passthrough to Disabled
     * (DISABLED), or leave blank.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stylePassthrough}
     * will return {@link WebvttStylePassthrough#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stylePassthroughAsString}.
     * </p>
     * 
     * @return To use the available style, color, and position information from your input captions: Set Style
     *         passthrough (stylePassthrough) to Enabled (ENABLED). MediaConvert uses default settings when style and
     *         position information is missing from your input captions. To recreate the input captions exactly: Set
     *         Style passthrough to Strict (STRICT). MediaConvert automatically applies timing adjustments, including
     *         adjustments for frame rate conversion, ad avails, and input clipping. Your input captions format must be
     *         WebVTT. To ignore the style and position information from your input captions and use simplified output
     *         captions: Set Style passthrough to Disabled (DISABLED), or leave blank.
     * @see WebvttStylePassthrough
     */
    public final String stylePassthroughAsString() {
        return stylePassthrough;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accessibilityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stylePassthroughAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WebvttDestinationSettings)) {
            return false;
        }
        WebvttDestinationSettings other = (WebvttDestinationSettings) obj;
        return Objects.equals(accessibilityAsString(), other.accessibilityAsString())
                && Objects.equals(stylePassthroughAsString(), other.stylePassthroughAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WebvttDestinationSettings").add("Accessibility", accessibilityAsString())
                .add("StylePassthrough", stylePassthroughAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Accessibility":
            return Optional.ofNullable(clazz.cast(accessibilityAsString()));
        case "StylePassthrough":
            return Optional.ofNullable(clazz.cast(stylePassthroughAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WebvttDestinationSettings, T> g) {
        return obj -> g.apply((WebvttDestinationSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WebvttDestinationSettings> {
        /**
         * Set Accessibility subtitles (Accessibility) to Enabled (ENABLED) if the ISMC or WebVTT captions track is
         * intended to provide accessibility for people who are deaf or hard of hearing. When you enable this feature,
         * MediaConvert adds the following attributes under EXT-X-MEDIA in the HLS or CMAF manifest for this track:
         * CHARACTERISTICS="public.accessibility.describes-spoken-dialog,public.accessibility.describes-music-and-sound"
         * and AUTOSELECT="YES". Keep the default value, Disabled (DISABLED), if the captions track is not intended to
         * provide such accessibility. MediaConvert will not add the above attributes.
         * 
         * @param accessibility
         *        Set Accessibility subtitles (Accessibility) to Enabled (ENABLED) if the ISMC or WebVTT captions track
         *        is intended to provide accessibility for people who are deaf or hard of hearing. When you enable this
         *        feature, MediaConvert adds the following attributes under EXT-X-MEDIA in the HLS or CMAF manifest for
         *        this track: CHARACTERISTICS=
         *        "public.accessibility.describes-spoken-dialog,public.accessibility.describes-music-and-sound" and
         *        AUTOSELECT="YES". Keep the default value, Disabled (DISABLED), if the captions track is not intended
         *        to provide such accessibility. MediaConvert will not add the above attributes.
         * @see WebvttAccessibilitySubs
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebvttAccessibilitySubs
         */
        Builder accessibility(String accessibility);

        /**
         * Set Accessibility subtitles (Accessibility) to Enabled (ENABLED) if the ISMC or WebVTT captions track is
         * intended to provide accessibility for people who are deaf or hard of hearing. When you enable this feature,
         * MediaConvert adds the following attributes under EXT-X-MEDIA in the HLS or CMAF manifest for this track:
         * CHARACTERISTICS="public.accessibility.describes-spoken-dialog,public.accessibility.describes-music-and-sound"
         * and AUTOSELECT="YES". Keep the default value, Disabled (DISABLED), if the captions track is not intended to
         * provide such accessibility. MediaConvert will not add the above attributes.
         * 
         * @param accessibility
         *        Set Accessibility subtitles (Accessibility) to Enabled (ENABLED) if the ISMC or WebVTT captions track
         *        is intended to provide accessibility for people who are deaf or hard of hearing. When you enable this
         *        feature, MediaConvert adds the following attributes under EXT-X-MEDIA in the HLS or CMAF manifest for
         *        this track: CHARACTERISTICS=
         *        "public.accessibility.describes-spoken-dialog,public.accessibility.describes-music-and-sound" and
         *        AUTOSELECT="YES". Keep the default value, Disabled (DISABLED), if the captions track is not intended
         *        to provide such accessibility. MediaConvert will not add the above attributes.
         * @see WebvttAccessibilitySubs
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebvttAccessibilitySubs
         */
        Builder accessibility(WebvttAccessibilitySubs accessibility);

        /**
         * To use the available style, color, and position information from your input captions: Set Style passthrough
         * (stylePassthrough) to Enabled (ENABLED). MediaConvert uses default settings when style and position
         * information is missing from your input captions. To recreate the input captions exactly: Set Style
         * passthrough to Strict (STRICT). MediaConvert automatically applies timing adjustments, including adjustments
         * for frame rate conversion, ad avails, and input clipping. Your input captions format must be WebVTT. To
         * ignore the style and position information from your input captions and use simplified output captions: Set
         * Style passthrough to Disabled (DISABLED), or leave blank.
         * 
         * @param stylePassthrough
         *        To use the available style, color, and position information from your input captions: Set Style
         *        passthrough (stylePassthrough) to Enabled (ENABLED). MediaConvert uses default settings when style and
         *        position information is missing from your input captions. To recreate the input captions exactly: Set
         *        Style passthrough to Strict (STRICT). MediaConvert automatically applies timing adjustments, including
         *        adjustments for frame rate conversion, ad avails, and input clipping. Your input captions format must
         *        be WebVTT. To ignore the style and position information from your input captions and use simplified
         *        output captions: Set Style passthrough to Disabled (DISABLED), or leave blank.
         * @see WebvttStylePassthrough
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebvttStylePassthrough
         */
        Builder stylePassthrough(String stylePassthrough);

        /**
         * To use the available style, color, and position information from your input captions: Set Style passthrough
         * (stylePassthrough) to Enabled (ENABLED). MediaConvert uses default settings when style and position
         * information is missing from your input captions. To recreate the input captions exactly: Set Style
         * passthrough to Strict (STRICT). MediaConvert automatically applies timing adjustments, including adjustments
         * for frame rate conversion, ad avails, and input clipping. Your input captions format must be WebVTT. To
         * ignore the style and position information from your input captions and use simplified output captions: Set
         * Style passthrough to Disabled (DISABLED), or leave blank.
         * 
         * @param stylePassthrough
         *        To use the available style, color, and position information from your input captions: Set Style
         *        passthrough (stylePassthrough) to Enabled (ENABLED). MediaConvert uses default settings when style and
         *        position information is missing from your input captions. To recreate the input captions exactly: Set
         *        Style passthrough to Strict (STRICT). MediaConvert automatically applies timing adjustments, including
         *        adjustments for frame rate conversion, ad avails, and input clipping. Your input captions format must
         *        be WebVTT. To ignore the style and position information from your input captions and use simplified
         *        output captions: Set Style passthrough to Disabled (DISABLED), or leave blank.
         * @see WebvttStylePassthrough
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebvttStylePassthrough
         */
        Builder stylePassthrough(WebvttStylePassthrough stylePassthrough);
    }

    static final class BuilderImpl implements Builder {
        private String accessibility;

        private String stylePassthrough;

        private BuilderImpl() {
        }

        private BuilderImpl(WebvttDestinationSettings model) {
            accessibility(model.accessibility);
            stylePassthrough(model.stylePassthrough);
        }

        public final String getAccessibility() {
            return accessibility;
        }

        public final void setAccessibility(String accessibility) {
            this.accessibility = accessibility;
        }

        @Override
        public final Builder accessibility(String accessibility) {
            this.accessibility = accessibility;
            return this;
        }

        @Override
        public final Builder accessibility(WebvttAccessibilitySubs accessibility) {
            this.accessibility(accessibility == null ? null : accessibility.toString());
            return this;
        }

        public final String getStylePassthrough() {
            return stylePassthrough;
        }

        public final void setStylePassthrough(String stylePassthrough) {
            this.stylePassthrough = stylePassthrough;
        }

        @Override
        public final Builder stylePassthrough(String stylePassthrough) {
            this.stylePassthrough = stylePassthrough;
            return this;
        }

        @Override
        public final Builder stylePassthrough(WebvttStylePassthrough stylePassthrough) {
            this.stylePassthrough(stylePassthrough == null ? null : stylePassthrough.toString());
            return this;
        }

        @Override
        public WebvttDestinationSettings build() {
            return new WebvttDestinationSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
