/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Advanced audio normalization settings. Ignore these settings unless you need to comply with a loudness standard.
 */
@Generated("software.amazon.awssdk:codegen")
public final class AudioNormalizationSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<AudioNormalizationSettings.Builder, AudioNormalizationSettings> {
    private static final SdkField<String> ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Algorithm").getter(getter(AudioNormalizationSettings::algorithmAsString))
            .setter(setter(Builder::algorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("algorithm").build()).build();

    private static final SdkField<String> ALGORITHM_CONTROL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlgorithmControl").getter(getter(AudioNormalizationSettings::algorithmControlAsString))
            .setter(setter(Builder::algorithmControl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("algorithmControl").build()).build();

    private static final SdkField<Integer> CORRECTION_GATE_LEVEL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("CorrectionGateLevel").getter(getter(AudioNormalizationSettings::correctionGateLevel))
            .setter(setter(Builder::correctionGateLevel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("correctionGateLevel").build())
            .build();

    private static final SdkField<String> LOUDNESS_LOGGING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LoudnessLogging").getter(getter(AudioNormalizationSettings::loudnessLoggingAsString))
            .setter(setter(Builder::loudnessLogging))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("loudnessLogging").build()).build();

    private static final SdkField<String> PEAK_CALCULATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PeakCalculation").getter(getter(AudioNormalizationSettings::peakCalculationAsString))
            .setter(setter(Builder::peakCalculation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("peakCalculation").build()).build();

    private static final SdkField<Double> TARGET_LKFS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("TargetLkfs").getter(getter(AudioNormalizationSettings::targetLkfs)).setter(setter(Builder::targetLkfs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetLkfs").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALGORITHM_FIELD,
            ALGORITHM_CONTROL_FIELD, CORRECTION_GATE_LEVEL_FIELD, LOUDNESS_LOGGING_FIELD, PEAK_CALCULATION_FIELD,
            TARGET_LKFS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String algorithm;

    private final String algorithmControl;

    private final Integer correctionGateLevel;

    private final String loudnessLogging;

    private final String peakCalculation;

    private final Double targetLkfs;

    private AudioNormalizationSettings(BuilderImpl builder) {
        this.algorithm = builder.algorithm;
        this.algorithmControl = builder.algorithmControl;
        this.correctionGateLevel = builder.correctionGateLevel;
        this.loudnessLogging = builder.loudnessLogging;
        this.peakCalculation = builder.peakCalculation;
        this.targetLkfs = builder.targetLkfs;
    }

    /**
     * Choose one of the following audio normalization algorithms: ITU-R BS.1770-1: Ungated loudness. A measurement of
     * ungated average loudness for an entire piece of content, suitable for measurement of short-form content under
     * ATSC recommendation A/85. Supports up to 5.1 audio channels. ITU-R BS.1770-2: Gated loudness. A measurement of
     * gated average loudness compliant with the requirements of EBU-R128. Supports up to 5.1 audio channels. ITU-R
     * BS.1770-3: Modified peak. The same loudness measurement algorithm as 1770-2, with an updated true peak
     * measurement. ITU-R BS.1770-4: Higher channel count. Allows for more audio channels than the other algorithms,
     * including configurations such as 7.1.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithm} will
     * return {@link AudioNormalizationAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #algorithmAsString}.
     * </p>
     * 
     * @return Choose one of the following audio normalization algorithms: ITU-R BS.1770-1: Ungated loudness. A
     *         measurement of ungated average loudness for an entire piece of content, suitable for measurement of
     *         short-form content under ATSC recommendation A/85. Supports up to 5.1 audio channels. ITU-R BS.1770-2:
     *         Gated loudness. A measurement of gated average loudness compliant with the requirements of EBU-R128.
     *         Supports up to 5.1 audio channels. ITU-R BS.1770-3: Modified peak. The same loudness measurement
     *         algorithm as 1770-2, with an updated true peak measurement. ITU-R BS.1770-4: Higher channel count. Allows
     *         for more audio channels than the other algorithms, including configurations such as 7.1.
     * @see AudioNormalizationAlgorithm
     */
    public final AudioNormalizationAlgorithm algorithm() {
        return AudioNormalizationAlgorithm.fromValue(algorithm);
    }

    /**
     * Choose one of the following audio normalization algorithms: ITU-R BS.1770-1: Ungated loudness. A measurement of
     * ungated average loudness for an entire piece of content, suitable for measurement of short-form content under
     * ATSC recommendation A/85. Supports up to 5.1 audio channels. ITU-R BS.1770-2: Gated loudness. A measurement of
     * gated average loudness compliant with the requirements of EBU-R128. Supports up to 5.1 audio channels. ITU-R
     * BS.1770-3: Modified peak. The same loudness measurement algorithm as 1770-2, with an updated true peak
     * measurement. ITU-R BS.1770-4: Higher channel count. Allows for more audio channels than the other algorithms,
     * including configurations such as 7.1.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithm} will
     * return {@link AudioNormalizationAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #algorithmAsString}.
     * </p>
     * 
     * @return Choose one of the following audio normalization algorithms: ITU-R BS.1770-1: Ungated loudness. A
     *         measurement of ungated average loudness for an entire piece of content, suitable for measurement of
     *         short-form content under ATSC recommendation A/85. Supports up to 5.1 audio channels. ITU-R BS.1770-2:
     *         Gated loudness. A measurement of gated average loudness compliant with the requirements of EBU-R128.
     *         Supports up to 5.1 audio channels. ITU-R BS.1770-3: Modified peak. The same loudness measurement
     *         algorithm as 1770-2, with an updated true peak measurement. ITU-R BS.1770-4: Higher channel count. Allows
     *         for more audio channels than the other algorithms, including configurations such as 7.1.
     * @see AudioNormalizationAlgorithm
     */
    public final String algorithmAsString() {
        return algorithm;
    }

    /**
     * When enabled the output audio is corrected using the chosen algorithm. If disabled, the audio will be measured
     * but not adjusted.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithmControl}
     * will return {@link AudioNormalizationAlgorithmControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #algorithmControlAsString}.
     * </p>
     * 
     * @return When enabled the output audio is corrected using the chosen algorithm. If disabled, the audio will be
     *         measured but not adjusted.
     * @see AudioNormalizationAlgorithmControl
     */
    public final AudioNormalizationAlgorithmControl algorithmControl() {
        return AudioNormalizationAlgorithmControl.fromValue(algorithmControl);
    }

    /**
     * When enabled the output audio is corrected using the chosen algorithm. If disabled, the audio will be measured
     * but not adjusted.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithmControl}
     * will return {@link AudioNormalizationAlgorithmControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #algorithmControlAsString}.
     * </p>
     * 
     * @return When enabled the output audio is corrected using the chosen algorithm. If disabled, the audio will be
     *         measured but not adjusted.
     * @see AudioNormalizationAlgorithmControl
     */
    public final String algorithmControlAsString() {
        return algorithmControl;
    }

    /**
     * Content measuring above this level will be corrected to the target level. Content measuring below this level will
     * not be corrected.
     * 
     * @return Content measuring above this level will be corrected to the target level. Content measuring below this
     *         level will not be corrected.
     */
    public final Integer correctionGateLevel() {
        return correctionGateLevel;
    }

    /**
     * If set to LOG, log each output's audio track loudness to a CSV file.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #loudnessLogging}
     * will return {@link AudioNormalizationLoudnessLogging#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #loudnessLoggingAsString}.
     * </p>
     * 
     * @return If set to LOG, log each output's audio track loudness to a CSV file.
     * @see AudioNormalizationLoudnessLogging
     */
    public final AudioNormalizationLoudnessLogging loudnessLogging() {
        return AudioNormalizationLoudnessLogging.fromValue(loudnessLogging);
    }

    /**
     * If set to LOG, log each output's audio track loudness to a CSV file.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #loudnessLogging}
     * will return {@link AudioNormalizationLoudnessLogging#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #loudnessLoggingAsString}.
     * </p>
     * 
     * @return If set to LOG, log each output's audio track loudness to a CSV file.
     * @see AudioNormalizationLoudnessLogging
     */
    public final String loudnessLoggingAsString() {
        return loudnessLogging;
    }

    /**
     * If set to TRUE_PEAK, calculate and log the TruePeak for each output's audio track loudness.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #peakCalculation}
     * will return {@link AudioNormalizationPeakCalculation#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #peakCalculationAsString}.
     * </p>
     * 
     * @return If set to TRUE_PEAK, calculate and log the TruePeak for each output's audio track loudness.
     * @see AudioNormalizationPeakCalculation
     */
    public final AudioNormalizationPeakCalculation peakCalculation() {
        return AudioNormalizationPeakCalculation.fromValue(peakCalculation);
    }

    /**
     * If set to TRUE_PEAK, calculate and log the TruePeak for each output's audio track loudness.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #peakCalculation}
     * will return {@link AudioNormalizationPeakCalculation#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #peakCalculationAsString}.
     * </p>
     * 
     * @return If set to TRUE_PEAK, calculate and log the TruePeak for each output's audio track loudness.
     * @see AudioNormalizationPeakCalculation
     */
    public final String peakCalculationAsString() {
        return peakCalculation;
    }

    /**
     * When you use Audio normalization (AudioNormalizationSettings), optionally use this setting to specify a target
     * loudness. If you don't specify a value here, the encoder chooses a value for you, based on the algorithm that you
     * choose for Algorithm (algorithm). If you choose algorithm 1770-1, the encoder will choose -24 LKFS; otherwise,
     * the encoder will choose -23 LKFS.
     * 
     * @return When you use Audio normalization (AudioNormalizationSettings), optionally use this setting to specify a
     *         target loudness. If you don't specify a value here, the encoder chooses a value for you, based on the
     *         algorithm that you choose for Algorithm (algorithm). If you choose algorithm 1770-1, the encoder will
     *         choose -24 LKFS; otherwise, the encoder will choose -23 LKFS.
     */
    public final Double targetLkfs() {
        return targetLkfs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(algorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(algorithmControlAsString());
        hashCode = 31 * hashCode + Objects.hashCode(correctionGateLevel());
        hashCode = 31 * hashCode + Objects.hashCode(loudnessLoggingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(peakCalculationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetLkfs());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AudioNormalizationSettings)) {
            return false;
        }
        AudioNormalizationSettings other = (AudioNormalizationSettings) obj;
        return Objects.equals(algorithmAsString(), other.algorithmAsString())
                && Objects.equals(algorithmControlAsString(), other.algorithmControlAsString())
                && Objects.equals(correctionGateLevel(), other.correctionGateLevel())
                && Objects.equals(loudnessLoggingAsString(), other.loudnessLoggingAsString())
                && Objects.equals(peakCalculationAsString(), other.peakCalculationAsString())
                && Objects.equals(targetLkfs(), other.targetLkfs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AudioNormalizationSettings").add("Algorithm", algorithmAsString())
                .add("AlgorithmControl", algorithmControlAsString()).add("CorrectionGateLevel", correctionGateLevel())
                .add("LoudnessLogging", loudnessLoggingAsString()).add("PeakCalculation", peakCalculationAsString())
                .add("TargetLkfs", targetLkfs()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Algorithm":
            return Optional.ofNullable(clazz.cast(algorithmAsString()));
        case "AlgorithmControl":
            return Optional.ofNullable(clazz.cast(algorithmControlAsString()));
        case "CorrectionGateLevel":
            return Optional.ofNullable(clazz.cast(correctionGateLevel()));
        case "LoudnessLogging":
            return Optional.ofNullable(clazz.cast(loudnessLoggingAsString()));
        case "PeakCalculation":
            return Optional.ofNullable(clazz.cast(peakCalculationAsString()));
        case "TargetLkfs":
            return Optional.ofNullable(clazz.cast(targetLkfs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AudioNormalizationSettings, T> g) {
        return obj -> g.apply((AudioNormalizationSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AudioNormalizationSettings> {
        /**
         * Choose one of the following audio normalization algorithms: ITU-R BS.1770-1: Ungated loudness. A measurement
         * of ungated average loudness for an entire piece of content, suitable for measurement of short-form content
         * under ATSC recommendation A/85. Supports up to 5.1 audio channels. ITU-R BS.1770-2: Gated loudness. A
         * measurement of gated average loudness compliant with the requirements of EBU-R128. Supports up to 5.1 audio
         * channels. ITU-R BS.1770-3: Modified peak. The same loudness measurement algorithm as 1770-2, with an updated
         * true peak measurement. ITU-R BS.1770-4: Higher channel count. Allows for more audio channels than the other
         * algorithms, including configurations such as 7.1.
         * 
         * @param algorithm
         *        Choose one of the following audio normalization algorithms: ITU-R BS.1770-1: Ungated loudness. A
         *        measurement of ungated average loudness for an entire piece of content, suitable for measurement of
         *        short-form content under ATSC recommendation A/85. Supports up to 5.1 audio channels. ITU-R BS.1770-2:
         *        Gated loudness. A measurement of gated average loudness compliant with the requirements of EBU-R128.
         *        Supports up to 5.1 audio channels. ITU-R BS.1770-3: Modified peak. The same loudness measurement
         *        algorithm as 1770-2, with an updated true peak measurement. ITU-R BS.1770-4: Higher channel count.
         *        Allows for more audio channels than the other algorithms, including configurations such as 7.1.
         * @see AudioNormalizationAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationAlgorithm
         */
        Builder algorithm(String algorithm);

        /**
         * Choose one of the following audio normalization algorithms: ITU-R BS.1770-1: Ungated loudness. A measurement
         * of ungated average loudness for an entire piece of content, suitable for measurement of short-form content
         * under ATSC recommendation A/85. Supports up to 5.1 audio channels. ITU-R BS.1770-2: Gated loudness. A
         * measurement of gated average loudness compliant with the requirements of EBU-R128. Supports up to 5.1 audio
         * channels. ITU-R BS.1770-3: Modified peak. The same loudness measurement algorithm as 1770-2, with an updated
         * true peak measurement. ITU-R BS.1770-4: Higher channel count. Allows for more audio channels than the other
         * algorithms, including configurations such as 7.1.
         * 
         * @param algorithm
         *        Choose one of the following audio normalization algorithms: ITU-R BS.1770-1: Ungated loudness. A
         *        measurement of ungated average loudness for an entire piece of content, suitable for measurement of
         *        short-form content under ATSC recommendation A/85. Supports up to 5.1 audio channels. ITU-R BS.1770-2:
         *        Gated loudness. A measurement of gated average loudness compliant with the requirements of EBU-R128.
         *        Supports up to 5.1 audio channels. ITU-R BS.1770-3: Modified peak. The same loudness measurement
         *        algorithm as 1770-2, with an updated true peak measurement. ITU-R BS.1770-4: Higher channel count.
         *        Allows for more audio channels than the other algorithms, including configurations such as 7.1.
         * @see AudioNormalizationAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationAlgorithm
         */
        Builder algorithm(AudioNormalizationAlgorithm algorithm);

        /**
         * When enabled the output audio is corrected using the chosen algorithm. If disabled, the audio will be
         * measured but not adjusted.
         * 
         * @param algorithmControl
         *        When enabled the output audio is corrected using the chosen algorithm. If disabled, the audio will be
         *        measured but not adjusted.
         * @see AudioNormalizationAlgorithmControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationAlgorithmControl
         */
        Builder algorithmControl(String algorithmControl);

        /**
         * When enabled the output audio is corrected using the chosen algorithm. If disabled, the audio will be
         * measured but not adjusted.
         * 
         * @param algorithmControl
         *        When enabled the output audio is corrected using the chosen algorithm. If disabled, the audio will be
         *        measured but not adjusted.
         * @see AudioNormalizationAlgorithmControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationAlgorithmControl
         */
        Builder algorithmControl(AudioNormalizationAlgorithmControl algorithmControl);

        /**
         * Content measuring above this level will be corrected to the target level. Content measuring below this level
         * will not be corrected.
         * 
         * @param correctionGateLevel
         *        Content measuring above this level will be corrected to the target level. Content measuring below this
         *        level will not be corrected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder correctionGateLevel(Integer correctionGateLevel);

        /**
         * If set to LOG, log each output's audio track loudness to a CSV file.
         * 
         * @param loudnessLogging
         *        If set to LOG, log each output's audio track loudness to a CSV file.
         * @see AudioNormalizationLoudnessLogging
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationLoudnessLogging
         */
        Builder loudnessLogging(String loudnessLogging);

        /**
         * If set to LOG, log each output's audio track loudness to a CSV file.
         * 
         * @param loudnessLogging
         *        If set to LOG, log each output's audio track loudness to a CSV file.
         * @see AudioNormalizationLoudnessLogging
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationLoudnessLogging
         */
        Builder loudnessLogging(AudioNormalizationLoudnessLogging loudnessLogging);

        /**
         * If set to TRUE_PEAK, calculate and log the TruePeak for each output's audio track loudness.
         * 
         * @param peakCalculation
         *        If set to TRUE_PEAK, calculate and log the TruePeak for each output's audio track loudness.
         * @see AudioNormalizationPeakCalculation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationPeakCalculation
         */
        Builder peakCalculation(String peakCalculation);

        /**
         * If set to TRUE_PEAK, calculate and log the TruePeak for each output's audio track loudness.
         * 
         * @param peakCalculation
         *        If set to TRUE_PEAK, calculate and log the TruePeak for each output's audio track loudness.
         * @see AudioNormalizationPeakCalculation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationPeakCalculation
         */
        Builder peakCalculation(AudioNormalizationPeakCalculation peakCalculation);

        /**
         * When you use Audio normalization (AudioNormalizationSettings), optionally use this setting to specify a
         * target loudness. If you don't specify a value here, the encoder chooses a value for you, based on the
         * algorithm that you choose for Algorithm (algorithm). If you choose algorithm 1770-1, the encoder will choose
         * -24 LKFS; otherwise, the encoder will choose -23 LKFS.
         * 
         * @param targetLkfs
         *        When you use Audio normalization (AudioNormalizationSettings), optionally use this setting to specify
         *        a target loudness. If you don't specify a value here, the encoder chooses a value for you, based on
         *        the algorithm that you choose for Algorithm (algorithm). If you choose algorithm 1770-1, the encoder
         *        will choose -24 LKFS; otherwise, the encoder will choose -23 LKFS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLkfs(Double targetLkfs);
    }

    static final class BuilderImpl implements Builder {
        private String algorithm;

        private String algorithmControl;

        private Integer correctionGateLevel;

        private String loudnessLogging;

        private String peakCalculation;

        private Double targetLkfs;

        private BuilderImpl() {
        }

        private BuilderImpl(AudioNormalizationSettings model) {
            algorithm(model.algorithm);
            algorithmControl(model.algorithmControl);
            correctionGateLevel(model.correctionGateLevel);
            loudnessLogging(model.loudnessLogging);
            peakCalculation(model.peakCalculation);
            targetLkfs(model.targetLkfs);
        }

        public final String getAlgorithm() {
            return algorithm;
        }

        public final void setAlgorithm(String algorithm) {
            this.algorithm = algorithm;
        }

        @Override
        public final Builder algorithm(String algorithm) {
            this.algorithm = algorithm;
            return this;
        }

        @Override
        public final Builder algorithm(AudioNormalizationAlgorithm algorithm) {
            this.algorithm(algorithm == null ? null : algorithm.toString());
            return this;
        }

        public final String getAlgorithmControl() {
            return algorithmControl;
        }

        public final void setAlgorithmControl(String algorithmControl) {
            this.algorithmControl = algorithmControl;
        }

        @Override
        public final Builder algorithmControl(String algorithmControl) {
            this.algorithmControl = algorithmControl;
            return this;
        }

        @Override
        public final Builder algorithmControl(AudioNormalizationAlgorithmControl algorithmControl) {
            this.algorithmControl(algorithmControl == null ? null : algorithmControl.toString());
            return this;
        }

        public final Integer getCorrectionGateLevel() {
            return correctionGateLevel;
        }

        public final void setCorrectionGateLevel(Integer correctionGateLevel) {
            this.correctionGateLevel = correctionGateLevel;
        }

        @Override
        public final Builder correctionGateLevel(Integer correctionGateLevel) {
            this.correctionGateLevel = correctionGateLevel;
            return this;
        }

        public final String getLoudnessLogging() {
            return loudnessLogging;
        }

        public final void setLoudnessLogging(String loudnessLogging) {
            this.loudnessLogging = loudnessLogging;
        }

        @Override
        public final Builder loudnessLogging(String loudnessLogging) {
            this.loudnessLogging = loudnessLogging;
            return this;
        }

        @Override
        public final Builder loudnessLogging(AudioNormalizationLoudnessLogging loudnessLogging) {
            this.loudnessLogging(loudnessLogging == null ? null : loudnessLogging.toString());
            return this;
        }

        public final String getPeakCalculation() {
            return peakCalculation;
        }

        public final void setPeakCalculation(String peakCalculation) {
            this.peakCalculation = peakCalculation;
        }

        @Override
        public final Builder peakCalculation(String peakCalculation) {
            this.peakCalculation = peakCalculation;
            return this;
        }

        @Override
        public final Builder peakCalculation(AudioNormalizationPeakCalculation peakCalculation) {
            this.peakCalculation(peakCalculation == null ? null : peakCalculation.toString());
            return this;
        }

        public final Double getTargetLkfs() {
            return targetLkfs;
        }

        public final void setTargetLkfs(Double targetLkfs) {
            this.targetLkfs = targetLkfs;
        }

        @Override
        public final Builder targetLkfs(Double targetLkfs) {
            this.targetLkfs = targetLkfs;
            return this;
        }

        @Override
        public AudioNormalizationSettings build() {
            return new AudioNormalizationSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
