/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * If your output group type is HLS, DASH, or Microsoft Smooth, use these settings when doing DRM encryption with a
 * SPEKE-compliant key provider. If your output group type is CMAF, use the SpekeKeyProviderCmaf settings instead.
 */
@Generated("software.amazon.awssdk:codegen")
public final class SpekeKeyProvider implements SdkPojo, Serializable,
        ToCopyableBuilder<SpekeKeyProvider.Builder, SpekeKeyProvider> {
    private static final SdkField<String> CERTIFICATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CertificateArn").getter(getter(SpekeKeyProvider::certificateArn))
            .setter(setter(Builder::certificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateArn").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceId").getter(getter(SpekeKeyProvider::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceId").build()).build();

    private static final SdkField<List<String>> SYSTEM_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SystemIds")
            .getter(getter(SpekeKeyProvider::systemIds))
            .setter(setter(Builder::systemIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("systemIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Url")
            .getter(getter(SpekeKeyProvider::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("url").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CERTIFICATE_ARN_FIELD,
            RESOURCE_ID_FIELD, SYSTEM_IDS_FIELD, URL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String certificateArn;

    private final String resourceId;

    private final List<String> systemIds;

    private final String url;

    private SpekeKeyProvider(BuilderImpl builder) {
        this.certificateArn = builder.certificateArn;
        this.resourceId = builder.resourceId;
        this.systemIds = builder.systemIds;
        this.url = builder.url;
    }

    /**
     * If you want your key provider to encrypt the content keys that it provides to MediaConvert, set up a certificate
     * with a master key using AWS Certificate Manager. Specify the certificate's Amazon Resource Name (ARN) here.
     * 
     * @return If you want your key provider to encrypt the content keys that it provides to MediaConvert, set up a
     *         certificate with a master key using AWS Certificate Manager. Specify the certificate's Amazon Resource
     *         Name (ARN) here.
     */
    public final String certificateArn() {
        return certificateArn;
    }

    /**
     * Specify the resource ID that your SPEKE-compliant key provider uses to identify this content.
     * 
     * @return Specify the resource ID that your SPEKE-compliant key provider uses to identify this content.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * For responses, this returns true if the service returned a value for the SystemIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSystemIds() {
        return systemIds != null && !(systemIds instanceof SdkAutoConstructList);
    }

    /**
     * Relates to SPEKE implementation. DRM system identifiers. DASH output groups support a max of two system ids.
     * Other group types support one system id. See https://dashif.org/identifiers/content_protection/ for more details.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSystemIds} method.
     * </p>
     * 
     * @return Relates to SPEKE implementation. DRM system identifiers. DASH output groups support a max of two system
     *         ids. Other group types support one system id. See https://dashif.org/identifiers/content_protection/ for
     *         more details.
     */
    public final List<String> systemIds() {
        return systemIds;
    }

    /**
     * Specify the URL to the key server that your SPEKE-compliant DRM key provider uses to provide keys for encrypting
     * your content.
     * 
     * @return Specify the URL to the key server that your SPEKE-compliant DRM key provider uses to provide keys for
     *         encrypting your content.
     */
    public final String url() {
        return url;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(certificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSystemIds() ? systemIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(url());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SpekeKeyProvider)) {
            return false;
        }
        SpekeKeyProvider other = (SpekeKeyProvider) obj;
        return Objects.equals(certificateArn(), other.certificateArn()) && Objects.equals(resourceId(), other.resourceId())
                && hasSystemIds() == other.hasSystemIds() && Objects.equals(systemIds(), other.systemIds())
                && Objects.equals(url(), other.url());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SpekeKeyProvider").add("CertificateArn", certificateArn()).add("ResourceId", resourceId())
                .add("SystemIds", hasSystemIds() ? systemIds() : null).add("Url", url()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CertificateArn":
            return Optional.ofNullable(clazz.cast(certificateArn()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "SystemIds":
            return Optional.ofNullable(clazz.cast(systemIds()));
        case "Url":
            return Optional.ofNullable(clazz.cast(url()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SpekeKeyProvider, T> g) {
        return obj -> g.apply((SpekeKeyProvider) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SpekeKeyProvider> {
        /**
         * If you want your key provider to encrypt the content keys that it provides to MediaConvert, set up a
         * certificate with a master key using AWS Certificate Manager. Specify the certificate's Amazon Resource Name
         * (ARN) here.
         * 
         * @param certificateArn
         *        If you want your key provider to encrypt the content keys that it provides to MediaConvert, set up a
         *        certificate with a master key using AWS Certificate Manager. Specify the certificate's Amazon Resource
         *        Name (ARN) here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateArn(String certificateArn);

        /**
         * Specify the resource ID that your SPEKE-compliant key provider uses to identify this content.
         * 
         * @param resourceId
         *        Specify the resource ID that your SPEKE-compliant key provider uses to identify this content.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * Relates to SPEKE implementation. DRM system identifiers. DASH output groups support a max of two system ids.
         * Other group types support one system id. See https://dashif.org/identifiers/content_protection/ for more
         * details.
         * 
         * @param systemIds
         *        Relates to SPEKE implementation. DRM system identifiers. DASH output groups support a max of two
         *        system ids. Other group types support one system id. See
         *        https://dashif.org/identifiers/content_protection/ for more details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder systemIds(Collection<String> systemIds);

        /**
         * Relates to SPEKE implementation. DRM system identifiers. DASH output groups support a max of two system ids.
         * Other group types support one system id. See https://dashif.org/identifiers/content_protection/ for more
         * details.
         * 
         * @param systemIds
         *        Relates to SPEKE implementation. DRM system identifiers. DASH output groups support a max of two
         *        system ids. Other group types support one system id. See
         *        https://dashif.org/identifiers/content_protection/ for more details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder systemIds(String... systemIds);

        /**
         * Specify the URL to the key server that your SPEKE-compliant DRM key provider uses to provide keys for
         * encrypting your content.
         * 
         * @param url
         *        Specify the URL to the key server that your SPEKE-compliant DRM key provider uses to provide keys for
         *        encrypting your content.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);
    }

    static final class BuilderImpl implements Builder {
        private String certificateArn;

        private String resourceId;

        private List<String> systemIds = DefaultSdkAutoConstructList.getInstance();

        private String url;

        private BuilderImpl() {
        }

        private BuilderImpl(SpekeKeyProvider model) {
            certificateArn(model.certificateArn);
            resourceId(model.resourceId);
            systemIds(model.systemIds);
            url(model.url);
        }

        public final String getCertificateArn() {
            return certificateArn;
        }

        public final void setCertificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
        }

        @Override
        @Transient
        public final Builder certificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        @Transient
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final Collection<String> getSystemIds() {
            if (systemIds instanceof SdkAutoConstructList) {
                return null;
            }
            return systemIds;
        }

        public final void setSystemIds(Collection<String> systemIds) {
            this.systemIds = ___listOf__stringPattern09aFAF809aFAF409aFAF409aFAF409aFAF12Copier.copy(systemIds);
        }

        @Override
        @Transient
        public final Builder systemIds(Collection<String> systemIds) {
            this.systemIds = ___listOf__stringPattern09aFAF809aFAF409aFAF409aFAF409aFAF12Copier.copy(systemIds);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder systemIds(String... systemIds) {
            systemIds(Arrays.asList(systemIds));
            return this;
        }

        public final String getUrl() {
            return url;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        @Transient
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        @Override
        public SpekeKeyProvider build() {
            return new SpekeKeyProvider(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
