/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediaconvert.model.AssociateCertificateRequest;
import software.amazon.awssdk.services.mediaconvert.model.AssociateCertificateResponse;
import software.amazon.awssdk.services.mediaconvert.model.BadRequestException;
import software.amazon.awssdk.services.mediaconvert.model.CancelJobRequest;
import software.amazon.awssdk.services.mediaconvert.model.CancelJobResponse;
import software.amazon.awssdk.services.mediaconvert.model.ConflictException;
import software.amazon.awssdk.services.mediaconvert.model.CreateJobRequest;
import software.amazon.awssdk.services.mediaconvert.model.CreateJobResponse;
import software.amazon.awssdk.services.mediaconvert.model.CreateJobTemplateRequest;
import software.amazon.awssdk.services.mediaconvert.model.CreateJobTemplateResponse;
import software.amazon.awssdk.services.mediaconvert.model.CreatePresetRequest;
import software.amazon.awssdk.services.mediaconvert.model.CreatePresetResponse;
import software.amazon.awssdk.services.mediaconvert.model.CreateQueueRequest;
import software.amazon.awssdk.services.mediaconvert.model.CreateQueueResponse;
import software.amazon.awssdk.services.mediaconvert.model.DeleteJobTemplateRequest;
import software.amazon.awssdk.services.mediaconvert.model.DeleteJobTemplateResponse;
import software.amazon.awssdk.services.mediaconvert.model.DeletePolicyRequest;
import software.amazon.awssdk.services.mediaconvert.model.DeletePolicyResponse;
import software.amazon.awssdk.services.mediaconvert.model.DeletePresetRequest;
import software.amazon.awssdk.services.mediaconvert.model.DeletePresetResponse;
import software.amazon.awssdk.services.mediaconvert.model.DeleteQueueRequest;
import software.amazon.awssdk.services.mediaconvert.model.DeleteQueueResponse;
import software.amazon.awssdk.services.mediaconvert.model.DescribeEndpointsRequest;
import software.amazon.awssdk.services.mediaconvert.model.DescribeEndpointsResponse;
import software.amazon.awssdk.services.mediaconvert.model.DisassociateCertificateRequest;
import software.amazon.awssdk.services.mediaconvert.model.DisassociateCertificateResponse;
import software.amazon.awssdk.services.mediaconvert.model.ForbiddenException;
import software.amazon.awssdk.services.mediaconvert.model.GetJobRequest;
import software.amazon.awssdk.services.mediaconvert.model.GetJobResponse;
import software.amazon.awssdk.services.mediaconvert.model.GetJobTemplateRequest;
import software.amazon.awssdk.services.mediaconvert.model.GetJobTemplateResponse;
import software.amazon.awssdk.services.mediaconvert.model.GetPolicyRequest;
import software.amazon.awssdk.services.mediaconvert.model.GetPolicyResponse;
import software.amazon.awssdk.services.mediaconvert.model.GetPresetRequest;
import software.amazon.awssdk.services.mediaconvert.model.GetPresetResponse;
import software.amazon.awssdk.services.mediaconvert.model.GetQueueRequest;
import software.amazon.awssdk.services.mediaconvert.model.GetQueueResponse;
import software.amazon.awssdk.services.mediaconvert.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediaconvert.model.ListJobTemplatesRequest;
import software.amazon.awssdk.services.mediaconvert.model.ListJobTemplatesResponse;
import software.amazon.awssdk.services.mediaconvert.model.ListJobsRequest;
import software.amazon.awssdk.services.mediaconvert.model.ListJobsResponse;
import software.amazon.awssdk.services.mediaconvert.model.ListPresetsRequest;
import software.amazon.awssdk.services.mediaconvert.model.ListPresetsResponse;
import software.amazon.awssdk.services.mediaconvert.model.ListQueuesRequest;
import software.amazon.awssdk.services.mediaconvert.model.ListQueuesResponse;
import software.amazon.awssdk.services.mediaconvert.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediaconvert.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediaconvert.model.MediaConvertException;
import software.amazon.awssdk.services.mediaconvert.model.MediaConvertRequest;
import software.amazon.awssdk.services.mediaconvert.model.NotFoundException;
import software.amazon.awssdk.services.mediaconvert.model.PutPolicyRequest;
import software.amazon.awssdk.services.mediaconvert.model.PutPolicyResponse;
import software.amazon.awssdk.services.mediaconvert.model.TagResourceRequest;
import software.amazon.awssdk.services.mediaconvert.model.TagResourceResponse;
import software.amazon.awssdk.services.mediaconvert.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediaconvert.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediaconvert.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediaconvert.model.UpdateJobTemplateRequest;
import software.amazon.awssdk.services.mediaconvert.model.UpdateJobTemplateResponse;
import software.amazon.awssdk.services.mediaconvert.model.UpdatePresetRequest;
import software.amazon.awssdk.services.mediaconvert.model.UpdatePresetResponse;
import software.amazon.awssdk.services.mediaconvert.model.UpdateQueueRequest;
import software.amazon.awssdk.services.mediaconvert.model.UpdateQueueResponse;
import software.amazon.awssdk.services.mediaconvert.paginators.DescribeEndpointsIterable;
import software.amazon.awssdk.services.mediaconvert.paginators.ListJobTemplatesIterable;
import software.amazon.awssdk.services.mediaconvert.paginators.ListJobsIterable;
import software.amazon.awssdk.services.mediaconvert.paginators.ListPresetsIterable;
import software.amazon.awssdk.services.mediaconvert.paginators.ListQueuesIterable;
import software.amazon.awssdk.services.mediaconvert.transform.AssociateCertificateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.CreateJobTemplateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.CreatePresetRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.CreateQueueRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DeleteJobTemplateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DeletePolicyRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DeletePresetRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DeleteQueueRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DescribeEndpointsRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DisassociateCertificateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.GetJobRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.GetJobTemplateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.GetPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.GetPresetRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.GetQueueRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.ListJobTemplatesRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.ListPresetsRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.ListQueuesRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.PutPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.UpdateJobTemplateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.UpdatePresetRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.UpdateQueueRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MediaConvertClient}.
 *
 * @see MediaConvertClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaConvertClient implements MediaConvertClient {
    private static final Logger log = Logger.loggerFor(DefaultMediaConvertClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final MediaConvertServiceClientConfiguration serviceClientConfiguration;

    protected DefaultMediaConvertClient(MediaConvertServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * Associates an AWS Certificate Manager (ACM) Amazon Resource Name (ARN) with AWS Elemental MediaConvert.
     *
     * @param associateCertificateRequest
     * @return Result of the AssociateCertificate operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and cannot fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested does not exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service could not complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.AssociateCertificate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/AssociateCertificate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateCertificateResponse associateCertificate(AssociateCertificateRequest associateCertificateRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            TooManyRequestsException, ConflictException, AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateCertificateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateCertificate");

            return clientHandler.execute(new ClientExecutionParams<AssociateCertificateRequest, AssociateCertificateResponse>()
                    .withOperationName("AssociateCertificate").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateCertificateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateCertificateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Permanently cancel a job. Once you have canceled a job, you can't start it again.
     *
     * @param cancelJobRequest
     * @return Result of the CancelJob operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.CancelJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelJobResponse cancelJob(CancelJobRequest cancelJobRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJob");

            return clientHandler.execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>()
                    .withOperationName("CancelJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CancelJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create a new transcoding job. For information about jobs and job settings, see the User Guide at
     * http://docs.aws.amazon.com/mediaconvert/latest/ug/what-is.html
     *
     * @param createJobRequest
     * @return Result of the CreateJob operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.CreateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateJobResponse createJob(CreateJobRequest createJobRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJob");

            return clientHandler.execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>()
                    .withOperationName("CreateJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CreateJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create a new job template. For information about job templates see the User Guide at
     * http://docs.aws.amazon.com/mediaconvert/latest/ug/what-is.html
     *
     * @param createJobTemplateRequest
     * @return Result of the CreateJobTemplate operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.CreateJobTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/CreateJobTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateJobTemplateResponse createJobTemplate(CreateJobTemplateRequest createJobTemplateRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            TooManyRequestsException, ConflictException, AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateJobTemplateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateJobTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJobTemplate");

            return clientHandler.execute(new ClientExecutionParams<CreateJobTemplateRequest, CreateJobTemplateResponse>()
                    .withOperationName("CreateJobTemplate").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createJobTemplateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateJobTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create a new preset. For information about job templates see the User Guide at
     * http://docs.aws.amazon.com/mediaconvert/latest/ug/what-is.html
     *
     * @param createPresetRequest
     * @return Result of the CreatePreset operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.CreatePreset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/CreatePreset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreatePresetResponse createPreset(CreatePresetRequest createPresetRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreatePresetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPresetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePreset");

            return clientHandler.execute(new ClientExecutionParams<CreatePresetRequest, CreatePresetResponse>()
                    .withOperationName("CreatePreset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createPresetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreatePresetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create a new transcoding queue. For information about queues, see Working With Queues in the User Guide at
     * https://docs.aws.amazon.com/mediaconvert/latest/ug/working-with-queues.html
     *
     * @param createQueueRequest
     * @return Result of the CreateQueue operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.CreateQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/CreateQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateQueueResponse createQueue(CreateQueueRequest createQueueRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateQueueResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateQueue");

            return clientHandler.execute(new ClientExecutionParams<CreateQueueRequest, CreateQueueResponse>()
                    .withOperationName("CreateQueue").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createQueueRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateQueueRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Permanently delete a job template you have created.
     *
     * @param deleteJobTemplateRequest
     * @return Result of the DeleteJobTemplate operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.DeleteJobTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DeleteJobTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteJobTemplateResponse deleteJobTemplate(DeleteJobTemplateRequest deleteJobTemplateRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            TooManyRequestsException, ConflictException, AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteJobTemplateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteJobTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJobTemplate");

            return clientHandler.execute(new ClientExecutionParams<DeleteJobTemplateRequest, DeleteJobTemplateResponse>()
                    .withOperationName("DeleteJobTemplate").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteJobTemplateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteJobTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Permanently delete a policy that you created.
     *
     * @param deletePolicyRequest
     * @return Result of the DeletePolicy operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.DeletePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DeletePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePolicyResponse deletePolicy(DeletePolicyRequest deletePolicyRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePolicy");

            return clientHandler.execute(new ClientExecutionParams<DeletePolicyRequest, DeletePolicyResponse>()
                    .withOperationName("DeletePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deletePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Permanently delete a preset you have created.
     *
     * @param deletePresetRequest
     * @return Result of the DeletePreset operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.DeletePreset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DeletePreset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePresetResponse deletePreset(DeletePresetRequest deletePresetRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePresetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePresetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePreset");

            return clientHandler.execute(new ClientExecutionParams<DeletePresetRequest, DeletePresetResponse>()
                    .withOperationName("DeletePreset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deletePresetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePresetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Permanently delete a queue you have created.
     *
     * @param deleteQueueRequest
     * @return Result of the DeleteQueue operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.DeleteQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DeleteQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteQueueResponse deleteQueue(DeleteQueueRequest deleteQueueRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteQueueResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteQueue");

            return clientHandler.execute(new ClientExecutionParams<DeleteQueueRequest, DeleteQueueResponse>()
                    .withOperationName("DeleteQueue").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteQueueRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteQueueRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Send an request with an empty body to the regional API endpoint to get your account API endpoint.
     *
     * @param describeEndpointsRequest
     *        DescribeEndpointsRequest
     * @return Result of the DescribeEndpoints operation returned by the service.
     * @throws BadRequestException
     *         BadRequestException
     * @throws InternalServerErrorException
     *         InternalServiceException
     * @throws ForbiddenException
     *         AccessDeniedException
     * @throws NotFoundException
     *         ResourceNotFoundException
     * @throws TooManyRequestsException
     *         LimitExceededException
     * @throws ConflictException
     *         ResourceInUseException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.DescribeEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DescribeEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEndpointsResponse describeEndpoints(DescribeEndpointsRequest describeEndpointsRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            TooManyRequestsException, ConflictException, AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeEndpointsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEndpoints");

            return clientHandler.execute(new ClientExecutionParams<DescribeEndpointsRequest, DescribeEndpointsResponse>()
                    .withOperationName("DescribeEndpoints").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeEndpointsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEndpointsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Send an request with an empty body to the regional API endpoint to get your account API endpoint.<br/>
     * <p>
     * This is a variant of
     * {@link #describeEndpoints(software.amazon.awssdk.services.mediaconvert.model.DescribeEndpointsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconvert.paginators.DescribeEndpointsIterable responses = client.describeEndpointsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediaconvert.paginators.DescribeEndpointsIterable responses = client
     *             .describeEndpointsPaginator(request);
     *     for (software.amazon.awssdk.services.mediaconvert.model.DescribeEndpointsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconvert.paginators.DescribeEndpointsIterable responses = client.describeEndpointsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEndpoints(software.amazon.awssdk.services.mediaconvert.model.DescribeEndpointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEndpointsRequest
     *        DescribeEndpointsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         BadRequestException
     * @throws InternalServerErrorException
     *         InternalServiceException
     * @throws ForbiddenException
     *         AccessDeniedException
     * @throws NotFoundException
     *         ResourceNotFoundException
     * @throws TooManyRequestsException
     *         LimitExceededException
     * @throws ConflictException
     *         ResourceInUseException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.DescribeEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DescribeEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEndpointsIterable describeEndpointsPaginator(DescribeEndpointsRequest describeEndpointsRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            TooManyRequestsException, ConflictException, AwsServiceException, SdkClientException, MediaConvertException {
        return new DescribeEndpointsIterable(this, applyPaginatorUserAgent(describeEndpointsRequest));
    }

    /**
     * Removes an association between the Amazon Resource Name (ARN) of an AWS Certificate Manager (ACM) certificate and
     * an AWS Elemental MediaConvert resource.
     *
     * @param disassociateCertificateRequest
     * @return Result of the DisassociateCertificate operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and cannot fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested does not exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service could not complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.DisassociateCertificate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DisassociateCertificate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateCertificateResponse disassociateCertificate(DisassociateCertificateRequest disassociateCertificateRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            TooManyRequestsException, ConflictException, AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateCertificateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateCertificate");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateCertificateRequest, DisassociateCertificateResponse>()
                            .withOperationName("DisassociateCertificate").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateCertificateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateCertificateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Retrieve the JSON for a specific completed transcoding job.
     *
     * @param getJobRequest
     * @return Result of the GetJob operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.GetJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/GetJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetJobResponse getJob(GetJobRequest getJobRequest) throws BadRequestException, InternalServerErrorException,
            ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException, AwsServiceException,
            SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJob");

            return clientHandler.execute(new ClientExecutionParams<GetJobRequest, GetJobResponse>().withOperationName("GetJob")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Retrieve the JSON for a specific job template.
     *
     * @param getJobTemplateRequest
     * @return Result of the GetJobTemplate operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.GetJobTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/GetJobTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetJobTemplateResponse getJobTemplate(GetJobTemplateRequest getJobTemplateRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetJobTemplateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetJobTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJobTemplate");

            return clientHandler.execute(new ClientExecutionParams<GetJobTemplateRequest, GetJobTemplateResponse>()
                    .withOperationName("GetJobTemplate").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getJobTemplateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetJobTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Retrieve the JSON for your policy.
     *
     * @param getPolicyRequest
     * @return Result of the GetPolicy operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.GetPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/GetPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPolicyResponse getPolicy(GetPolicyRequest getPolicyRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetPolicyRequest, GetPolicyResponse>()
                    .withOperationName("GetPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Retrieve the JSON for a specific preset.
     *
     * @param getPresetRequest
     * @return Result of the GetPreset operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.GetPreset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/GetPreset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPresetResponse getPreset(GetPresetRequest getPresetRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPresetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPresetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPreset");

            return clientHandler.execute(new ClientExecutionParams<GetPresetRequest, GetPresetResponse>()
                    .withOperationName("GetPreset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getPresetRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetPresetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Retrieve the JSON for a specific queue.
     *
     * @param getQueueRequest
     * @return Result of the GetQueue operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.GetQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/GetQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetQueueResponse getQueue(GetQueueRequest getQueueRequest) throws BadRequestException, InternalServerErrorException,
            ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException, AwsServiceException,
            SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetQueueResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueue");

            return clientHandler.execute(new ClientExecutionParams<GetQueueRequest, GetQueueResponse>()
                    .withOperationName("GetQueue").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getQueueRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetQueueRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Retrieve a JSON array of up to twenty of your job templates. This will return the templates themselves, not just
     * a list of them. To retrieve the next twenty templates, use the nextToken string returned with the array
     *
     * @param listJobTemplatesRequest
     * @return Result of the ListJobTemplates operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.ListJobTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListJobTemplates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListJobTemplatesResponse listJobTemplates(ListJobTemplatesRequest listJobTemplatesRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListJobTemplatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobTemplatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobTemplates");

            return clientHandler.execute(new ClientExecutionParams<ListJobTemplatesRequest, ListJobTemplatesResponse>()
                    .withOperationName("ListJobTemplates").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listJobTemplatesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListJobTemplatesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Retrieve a JSON array of up to twenty of your job templates. This will return the templates themselves, not just
     * a list of them. To retrieve the next twenty templates, use the nextToken string returned with the array<br/>
     * <p>
     * This is a variant of
     * {@link #listJobTemplates(software.amazon.awssdk.services.mediaconvert.model.ListJobTemplatesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconvert.paginators.ListJobTemplatesIterable responses = client.listJobTemplatesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediaconvert.paginators.ListJobTemplatesIterable responses = client
     *             .listJobTemplatesPaginator(request);
     *     for (software.amazon.awssdk.services.mediaconvert.model.ListJobTemplatesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconvert.paginators.ListJobTemplatesIterable responses = client.listJobTemplatesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobTemplates(software.amazon.awssdk.services.mediaconvert.model.ListJobTemplatesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listJobTemplatesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.ListJobTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListJobTemplates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListJobTemplatesIterable listJobTemplatesPaginator(ListJobTemplatesRequest listJobTemplatesRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            TooManyRequestsException, ConflictException, AwsServiceException, SdkClientException, MediaConvertException {
        return new ListJobTemplatesIterable(this, applyPaginatorUserAgent(listJobTemplatesRequest));
    }

    /**
     * Retrieve a JSON array of up to twenty of your most recently created jobs. This array includes in-process,
     * completed, and errored jobs. This will return the jobs themselves, not just a list of the jobs. To retrieve the
     * twenty next most recent jobs, use the nextToken string returned with the array.
     *
     * @param listJobsRequest
     * @return Result of the ListJobs operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsResponse listJobs(ListJobsRequest listJobsRequest) throws BadRequestException, InternalServerErrorException,
            ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException, AwsServiceException,
            SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");

            return clientHandler.execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>()
                    .withOperationName("ListJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listJobsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Retrieve a JSON array of up to twenty of your most recently created jobs. This array includes in-process,
     * completed, and errored jobs. This will return the jobs themselves, not just a list of the jobs. To retrieve the
     * twenty next most recent jobs, use the nextToken string returned with the array.<br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.mediaconvert.model.ListJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconvert.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediaconvert.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     *     for (software.amazon.awssdk.services.mediaconvert.model.ListJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconvert.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.mediaconvert.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsIterable listJobsPaginator(ListJobsRequest listJobsRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        return new ListJobsIterable(this, applyPaginatorUserAgent(listJobsRequest));
    }

    /**
     * Retrieve a JSON array of up to twenty of your presets. This will return the presets themselves, not just a list
     * of them. To retrieve the next twenty presets, use the nextToken string returned with the array.
     *
     * @param listPresetsRequest
     * @return Result of the ListPresets operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.ListPresets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListPresets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPresetsResponse listPresets(ListPresetsRequest listPresetsRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPresetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPresetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPresetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPresets");

            return clientHandler.execute(new ClientExecutionParams<ListPresetsRequest, ListPresetsResponse>()
                    .withOperationName("ListPresets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPresetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPresetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Retrieve a JSON array of up to twenty of your presets. This will return the presets themselves, not just a list
     * of them. To retrieve the next twenty presets, use the nextToken string returned with the array.<br/>
     * <p>
     * This is a variant of {@link #listPresets(software.amazon.awssdk.services.mediaconvert.model.ListPresetsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconvert.paginators.ListPresetsIterable responses = client.listPresetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediaconvert.paginators.ListPresetsIterable responses = client.listPresetsPaginator(request);
     *     for (software.amazon.awssdk.services.mediaconvert.model.ListPresetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconvert.paginators.ListPresetsIterable responses = client.listPresetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPresets(software.amazon.awssdk.services.mediaconvert.model.ListPresetsRequest)} operation.</b>
     * </p>
     *
     * @param listPresetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.ListPresets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListPresets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPresetsIterable listPresetsPaginator(ListPresetsRequest listPresetsRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        return new ListPresetsIterable(this, applyPaginatorUserAgent(listPresetsRequest));
    }

    /**
     * Retrieve a JSON array of up to twenty of your queues. This will return the queues themselves, not just a list of
     * them. To retrieve the next twenty queues, use the nextToken string returned with the array.
     *
     * @param listQueuesRequest
     * @return Result of the ListQueues operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.ListQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListQueues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListQueuesResponse listQueues(ListQueuesRequest listQueuesRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListQueuesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListQueuesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQueuesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQueues");

            return clientHandler
                    .execute(new ClientExecutionParams<ListQueuesRequest, ListQueuesResponse>().withOperationName("ListQueues")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listQueuesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListQueuesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Retrieve a JSON array of up to twenty of your queues. This will return the queues themselves, not just a list of
     * them. To retrieve the next twenty queues, use the nextToken string returned with the array.<br/>
     * <p>
     * This is a variant of {@link #listQueues(software.amazon.awssdk.services.mediaconvert.model.ListQueuesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconvert.paginators.ListQueuesIterable responses = client.listQueuesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediaconvert.paginators.ListQueuesIterable responses = client.listQueuesPaginator(request);
     *     for (software.amazon.awssdk.services.mediaconvert.model.ListQueuesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconvert.paginators.ListQueuesIterable responses = client.listQueuesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listQueues(software.amazon.awssdk.services.mediaconvert.model.ListQueuesRequest)} operation.</b>
     * </p>
     *
     * @param listQueuesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.ListQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListQueues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListQueuesIterable listQueuesPaginator(ListQueuesRequest listQueuesRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        return new ListQueuesIterable(this, applyPaginatorUserAgent(listQueuesRequest));
    }

    /**
     * Retrieve the tags for a MediaConvert resource.
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            TooManyRequestsException, ConflictException, AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create or change your policy. For more information about policies, see the user guide at
     * http://docs.aws.amazon.com/mediaconvert/latest/ug/what-is.html
     *
     * @param putPolicyRequest
     * @return Result of the PutPolicy operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.PutPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/PutPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutPolicyResponse putPolicy(PutPolicyRequest putPolicyRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutPolicyRequest, PutPolicyResponse>()
                    .withOperationName("PutPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new PutPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Add tags to a MediaConvert queue, preset, or job template. For information about tagging, see the User Guide at
     * https://docs.aws.amazon.com/mediaconvert/latest/ug/tagging-resources.html
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Remove tags from a MediaConvert queue, preset, or job template. For information about tagging, see the User Guide
     * at https://docs.aws.amazon.com/mediaconvert/latest/ug/tagging-resources.html
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Modify one of your existing job templates.
     *
     * @param updateJobTemplateRequest
     * @return Result of the UpdateJobTemplate operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.UpdateJobTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/UpdateJobTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateJobTemplateResponse updateJobTemplate(UpdateJobTemplateRequest updateJobTemplateRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            TooManyRequestsException, ConflictException, AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateJobTemplateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateJobTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJobTemplate");

            return clientHandler.execute(new ClientExecutionParams<UpdateJobTemplateRequest, UpdateJobTemplateResponse>()
                    .withOperationName("UpdateJobTemplate").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateJobTemplateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateJobTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Modify one of your existing presets.
     *
     * @param updatePresetRequest
     * @return Result of the UpdatePreset operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.UpdatePreset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/UpdatePreset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdatePresetResponse updatePreset(UpdatePresetRequest updatePresetRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdatePresetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePresetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePreset");

            return clientHandler.execute(new ClientExecutionParams<UpdatePresetRequest, UpdatePresetResponse>()
                    .withOperationName("UpdatePreset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updatePresetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdatePresetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Modify one of your existing queues.
     *
     * @param updateQueueRequest
     * @return Result of the UpdateQueue operation returned by the service.
     * @throws BadRequestException
     *         The service can't process your request because of a problem in the request. Please check your request
     *         form and syntax.
     * @throws InternalServerErrorException
     *         The service encountered an unexpected condition and can't fulfill your request.
     * @throws ForbiddenException
     *         You don't have permissions for this action with the credentials you sent.
     * @throws NotFoundException
     *         The resource you requested doesn't exist.
     * @throws TooManyRequestsException
     *         Too many requests have been sent in too short of a time. The service limits the rate at which it will
     *         accept requests.
     * @throws ConflictException
     *         The service couldn't complete your request because there is a conflict with the current state of the
     *         resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConvertException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConvertClient.UpdateQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/UpdateQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateQueueResponse updateQueue(UpdateQueueRequest updateQueueRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, TooManyRequestsException, ConflictException,
            AwsServiceException, SdkClientException, MediaConvertException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateQueueResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateQueue");

            return clientHandler.execute(new ClientExecutionParams<UpdateQueueRequest, UpdateQueueResponse>()
                    .withOperationName("UpdateQueue").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateQueueRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateQueueRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends MediaConvertRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaConvertException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public final MediaConvertServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
