/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert.endpoints.internal;

import java.net.URI;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.mediaconvert.endpoints.MediaConvertEndpointParams;
import software.amazon.awssdk.services.mediaconvert.endpoints.MediaConvertEndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultMediaConvertEndpointProvider implements MediaConvertEndpointProvider {
    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(MediaConvertEndpointParams params) {
        Validate.notNull(params.useDualStack(), "Parameter 'UseDualStack' must not be null");
        Validate.notNull(params.useFips(), "Parameter 'UseFIPS' must not be null");
        try {
            RuleResult result = endpointRule0(params, new LocalState(params.region()));
            if (result.canContinue()) {
                throw SdkClientException.create("Rule engine did not reach an error or endpoint result");
            }
            if (result.isError()) {
                String errorMsg = result.error();
                if (errorMsg.contains("Invalid ARN") && errorMsg.contains(":s3:::")) {
                    errorMsg += ". Use the bucket name instead of simple bucket ARNs in GetBucketLocationRequest.";
                }
                throw SdkClientException.create(errorMsg);
            }
            return CompletableFuture.completedFuture(result.endpoint());
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static RuleResult endpointRule0(MediaConvertEndpointParams params, LocalState locals) {
        RuleResult result = endpointRule1(params, locals);
        if (result.isResolved()) {
            return result;
        }
        result = endpointRule5(params, locals);
        if (result.isResolved()) {
            return result;
        }
        return endpointRule22(params, locals);
    }

    private static RuleResult endpointRule1(MediaConvertEndpointParams params, LocalState locals) {
        if (params.endpoint() != null) {
            RuleResult result = endpointRule2(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule3(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule4(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule2(MediaConvertEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.error("Invalid Configuration: FIPS and custom endpoint are not supported");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule3(MediaConvertEndpointParams params, LocalState locals) {
        if (params.useDualStack()) {
            return RuleResult.error("Invalid Configuration: Dualstack and custom endpoint are not supported");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule4(MediaConvertEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint.builder().url(URI.create(params.endpoint())).build());
    }

    private static RuleResult endpointRule5(MediaConvertEndpointParams params, LocalState locals) {
        if (locals.region() != null) {
            return endpointRule6(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule6(MediaConvertEndpointParams params, LocalState locals) {
        RulePartition partitionResult = null;
        if ((partitionResult = RulesFunctions.awsPartition(locals.region())) != null) {
            locals = locals.toBuilder().partitionResult(partitionResult).build();
            RuleResult result = endpointRule7(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule11(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule16(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule20(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule21(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule7(MediaConvertEndpointParams params, LocalState locals) {
        if (params.useFips() && params.useDualStack()) {
            RuleResult result = endpointRule8(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule10(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule8(MediaConvertEndpointParams params, LocalState locals) {
        if (locals.partitionResult().supportsFIPS() && locals.partitionResult().supportsDualStack()) {
            return endpointRule9(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule9(MediaConvertEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create("https://mediaconvert-fips." + locals.region() + "."
                        + locals.partitionResult().dualStackDnsSuffix())).build());
    }

    private static RuleResult endpointRule10(MediaConvertEndpointParams params, LocalState locals) {
        return RuleResult.error("FIPS and DualStack are enabled, but this partition does not support one or both");
    }

    private static RuleResult endpointRule11(MediaConvertEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            RuleResult result = endpointRule12(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule15(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule12(MediaConvertEndpointParams params, LocalState locals) {
        if (locals.partitionResult().supportsFIPS()) {
            RuleResult result = endpointRule13(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule14(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule13(MediaConvertEndpointParams params, LocalState locals) {
        if ("aws-us-gov".equals(locals.partitionResult().name())) {
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://mediaconvert." + locals.region() + ".amazonaws.com")).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule14(MediaConvertEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint.builder()
                .url(URI.create("https://mediaconvert-fips." + locals.region() + "." + locals.partitionResult().dnsSuffix()))
                .build());
    }

    private static RuleResult endpointRule15(MediaConvertEndpointParams params, LocalState locals) {
        return RuleResult.error("FIPS is enabled but this partition does not support FIPS");
    }

    private static RuleResult endpointRule16(MediaConvertEndpointParams params, LocalState locals) {
        if (params.useDualStack()) {
            RuleResult result = endpointRule17(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule19(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule17(MediaConvertEndpointParams params, LocalState locals) {
        if (locals.partitionResult().supportsDualStack()) {
            return endpointRule18(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule18(MediaConvertEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint.builder()
                .url(URI.create("https://mediaconvert." + locals.region() + "." + locals.partitionResult().dualStackDnsSuffix()))
                .build());
    }

    private static RuleResult endpointRule19(MediaConvertEndpointParams params, LocalState locals) {
        return RuleResult.error("DualStack is enabled but this partition does not support DualStack");
    }

    private static RuleResult endpointRule20(MediaConvertEndpointParams params, LocalState locals) {
        if ("cn-northwest-1".equals(locals.region())) {
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://subscribe.mediaconvert.cn-northwest-1.amazonaws.com.cn")).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule21(MediaConvertEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint.builder()
                .url(URI.create("https://mediaconvert." + locals.region() + "." + locals.partitionResult().dnsSuffix())).build());
    }

    private static RuleResult endpointRule22(MediaConvertEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid Configuration: Missing Region");
    }

    @Override
    public boolean equals(Object rhs) {
        return rhs != null && getClass().equals(rhs.getClass());
    }

    @Override
    public int hashCode() {
        return getClass().hashCode();
    }

    private static final class LocalState {
        private final String region;

        private final RulePartition partitionResult;

        LocalState() {
            this.region = null;
            this.partitionResult = null;
        }

        LocalState(Region region) {
            if (region != null) {
                this.region = region.id();
            } else {
                this.region = null;
            }
            this.partitionResult = null;
        }

        LocalState(LocalStateBuilder builder) {
            this.region = builder.region;
            this.partitionResult = builder.partitionResult;
        }

        public String region() {
            return this.region;
        }

        public RulePartition partitionResult() {
            return this.partitionResult;
        }

        public LocalStateBuilder toBuilder() {
            return new LocalStateBuilder(this);
        }
    }

    private static final class LocalStateBuilder {
        private String region;

        private RulePartition partitionResult;

        LocalStateBuilder() {
            this.region = null;
            this.partitionResult = null;
        }

        LocalStateBuilder(LocalState locals) {
            this.region = locals.region;
            this.partitionResult = locals.partitionResult;
        }

        public LocalStateBuilder region(String value) {
            this.region = value;
            return this;
        }

        public LocalStateBuilder partitionResult(RulePartition value) {
            this.partitionResult = value;
            return this;
        }

        LocalState build() {
            return new LocalState(this);
        }
    }
}
