/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * If your output group type is CMAF, use these settings when doing DRM encryption with a SPEKE-compliant key provider.
 * If your output group type is HLS, DASH, or Microsoft Smooth, use the SpekeKeyProvider settings instead.
 */
@Generated("software.amazon.awssdk:codegen")
public final class SpekeKeyProviderCmaf implements SdkPojo, Serializable,
        ToCopyableBuilder<SpekeKeyProviderCmaf.Builder, SpekeKeyProviderCmaf> {
    private static final SdkField<String> CERTIFICATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CertificateArn").getter(getter(SpekeKeyProviderCmaf::certificateArn))
            .setter(setter(Builder::certificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateArn").build()).build();

    private static final SdkField<List<String>> DASH_SIGNALED_SYSTEM_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DashSignaledSystemIds")
            .getter(getter(SpekeKeyProviderCmaf::dashSignaledSystemIds))
            .setter(setter(Builder::dashSignaledSystemIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dashSignaledSystemIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EncryptionContractConfiguration> ENCRYPTION_CONTRACT_CONFIGURATION_FIELD = SdkField
            .<EncryptionContractConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("EncryptionContractConfiguration")
            .getter(getter(SpekeKeyProviderCmaf::encryptionContractConfiguration))
            .setter(setter(Builder::encryptionContractConfiguration))
            .constructor(EncryptionContractConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionContractConfiguration")
                    .build()).build();

    private static final SdkField<List<String>> HLS_SIGNALED_SYSTEM_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("HlsSignaledSystemIds")
            .getter(getter(SpekeKeyProviderCmaf::hlsSignaledSystemIds))
            .setter(setter(Builder::hlsSignaledSystemIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hlsSignaledSystemIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceId").getter(getter(SpekeKeyProviderCmaf::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceId").build()).build();

    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Url")
            .getter(getter(SpekeKeyProviderCmaf::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("url").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CERTIFICATE_ARN_FIELD,
            DASH_SIGNALED_SYSTEM_IDS_FIELD, ENCRYPTION_CONTRACT_CONFIGURATION_FIELD, HLS_SIGNALED_SYSTEM_IDS_FIELD,
            RESOURCE_ID_FIELD, URL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String certificateArn;

    private final List<String> dashSignaledSystemIds;

    private final EncryptionContractConfiguration encryptionContractConfiguration;

    private final List<String> hlsSignaledSystemIds;

    private final String resourceId;

    private final String url;

    private SpekeKeyProviderCmaf(BuilderImpl builder) {
        this.certificateArn = builder.certificateArn;
        this.dashSignaledSystemIds = builder.dashSignaledSystemIds;
        this.encryptionContractConfiguration = builder.encryptionContractConfiguration;
        this.hlsSignaledSystemIds = builder.hlsSignaledSystemIds;
        this.resourceId = builder.resourceId;
        this.url = builder.url;
    }

    /**
     * If you want your key provider to encrypt the content keys that it provides to MediaConvert, set up a certificate
     * with a master key using AWS Certificate Manager. Specify the certificate's Amazon Resource Name (ARN) here.
     * 
     * @return If you want your key provider to encrypt the content keys that it provides to MediaConvert, set up a
     *         certificate with a master key using AWS Certificate Manager. Specify the certificate's Amazon Resource
     *         Name (ARN) here.
     */
    public final String certificateArn() {
        return certificateArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the DashSignaledSystemIds property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDashSignaledSystemIds() {
        return dashSignaledSystemIds != null && !(dashSignaledSystemIds instanceof SdkAutoConstructList);
    }

    /**
     * Specify the DRM system IDs that you want signaled in the DASH manifest that MediaConvert creates as part of this
     * CMAF package. The DASH manifest can currently signal up to three system IDs. For more information, see
     * https://dashif.org/identifiers/content_protection/.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDashSignaledSystemIds} method.
     * </p>
     * 
     * @return Specify the DRM system IDs that you want signaled in the DASH manifest that MediaConvert creates as part
     *         of this CMAF package. The DASH manifest can currently signal up to three system IDs. For more
     *         information, see https://dashif.org/identifiers/content_protection/.
     */
    public final List<String> dashSignaledSystemIds() {
        return dashSignaledSystemIds;
    }

    /**
     * Specify the SPEKE version, either v1.0 or v2.0, that MediaConvert uses when encrypting your output. For more
     * information, see: https://docs.aws.amazon.com/speke/latest/documentation/speke-api-specification.html To use
     * SPEKE v1.0: Leave blank. To use SPEKE v2.0: Specify a SPEKE v2.0 video preset and a SPEKE v2.0 audio preset.
     * 
     * @return Specify the SPEKE version, either v1.0 or v2.0, that MediaConvert uses when encrypting your output. For
     *         more information, see:
     *         https://docs.aws.amazon.com/speke/latest/documentation/speke-api-specification.html To use SPEKE v1.0:
     *         Leave blank. To use SPEKE v2.0: Specify a SPEKE v2.0 video preset and a SPEKE v2.0 audio preset.
     */
    public final EncryptionContractConfiguration encryptionContractConfiguration() {
        return encryptionContractConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the HlsSignaledSystemIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasHlsSignaledSystemIds() {
        return hlsSignaledSystemIds != null && !(hlsSignaledSystemIds instanceof SdkAutoConstructList);
    }

    /**
     * Specify the DRM system ID that you want signaled in the HLS manifest that MediaConvert creates as part of this
     * CMAF package. The HLS manifest can currently signal only one system ID. For more information, see
     * https://dashif.org/identifiers/content_protection/.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHlsSignaledSystemIds} method.
     * </p>
     * 
     * @return Specify the DRM system ID that you want signaled in the HLS manifest that MediaConvert creates as part of
     *         this CMAF package. The HLS manifest can currently signal only one system ID. For more information, see
     *         https://dashif.org/identifiers/content_protection/.
     */
    public final List<String> hlsSignaledSystemIds() {
        return hlsSignaledSystemIds;
    }

    /**
     * Specify the resource ID that your SPEKE-compliant key provider uses to identify this content.
     * 
     * @return Specify the resource ID that your SPEKE-compliant key provider uses to identify this content.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * Specify the URL to the key server that your SPEKE-compliant DRM key provider uses to provide keys for encrypting
     * your content.
     * 
     * @return Specify the URL to the key server that your SPEKE-compliant DRM key provider uses to provide keys for
     *         encrypting your content.
     */
    public final String url() {
        return url;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(certificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasDashSignaledSystemIds() ? dashSignaledSystemIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(encryptionContractConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasHlsSignaledSystemIds() ? hlsSignaledSystemIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(url());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SpekeKeyProviderCmaf)) {
            return false;
        }
        SpekeKeyProviderCmaf other = (SpekeKeyProviderCmaf) obj;
        return Objects.equals(certificateArn(), other.certificateArn())
                && hasDashSignaledSystemIds() == other.hasDashSignaledSystemIds()
                && Objects.equals(dashSignaledSystemIds(), other.dashSignaledSystemIds())
                && Objects.equals(encryptionContractConfiguration(), other.encryptionContractConfiguration())
                && hasHlsSignaledSystemIds() == other.hasHlsSignaledSystemIds()
                && Objects.equals(hlsSignaledSystemIds(), other.hlsSignaledSystemIds())
                && Objects.equals(resourceId(), other.resourceId()) && Objects.equals(url(), other.url());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SpekeKeyProviderCmaf").add("CertificateArn", certificateArn())
                .add("DashSignaledSystemIds", hasDashSignaledSystemIds() ? dashSignaledSystemIds() : null)
                .add("EncryptionContractConfiguration", encryptionContractConfiguration())
                .add("HlsSignaledSystemIds", hasHlsSignaledSystemIds() ? hlsSignaledSystemIds() : null)
                .add("ResourceId", resourceId()).add("Url", url()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CertificateArn":
            return Optional.ofNullable(clazz.cast(certificateArn()));
        case "DashSignaledSystemIds":
            return Optional.ofNullable(clazz.cast(dashSignaledSystemIds()));
        case "EncryptionContractConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionContractConfiguration()));
        case "HlsSignaledSystemIds":
            return Optional.ofNullable(clazz.cast(hlsSignaledSystemIds()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "Url":
            return Optional.ofNullable(clazz.cast(url()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SpekeKeyProviderCmaf, T> g) {
        return obj -> g.apply((SpekeKeyProviderCmaf) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SpekeKeyProviderCmaf> {
        /**
         * If you want your key provider to encrypt the content keys that it provides to MediaConvert, set up a
         * certificate with a master key using AWS Certificate Manager. Specify the certificate's Amazon Resource Name
         * (ARN) here.
         * 
         * @param certificateArn
         *        If you want your key provider to encrypt the content keys that it provides to MediaConvert, set up a
         *        certificate with a master key using AWS Certificate Manager. Specify the certificate's Amazon Resource
         *        Name (ARN) here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateArn(String certificateArn);

        /**
         * Specify the DRM system IDs that you want signaled in the DASH manifest that MediaConvert creates as part of
         * this CMAF package. The DASH manifest can currently signal up to three system IDs. For more information, see
         * https://dashif.org/identifiers/content_protection/.
         * 
         * @param dashSignaledSystemIds
         *        Specify the DRM system IDs that you want signaled in the DASH manifest that MediaConvert creates as
         *        part of this CMAF package. The DASH manifest can currently signal up to three system IDs. For more
         *        information, see https://dashif.org/identifiers/content_protection/.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dashSignaledSystemIds(Collection<String> dashSignaledSystemIds);

        /**
         * Specify the DRM system IDs that you want signaled in the DASH manifest that MediaConvert creates as part of
         * this CMAF package. The DASH manifest can currently signal up to three system IDs. For more information, see
         * https://dashif.org/identifiers/content_protection/.
         * 
         * @param dashSignaledSystemIds
         *        Specify the DRM system IDs that you want signaled in the DASH manifest that MediaConvert creates as
         *        part of this CMAF package. The DASH manifest can currently signal up to three system IDs. For more
         *        information, see https://dashif.org/identifiers/content_protection/.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dashSignaledSystemIds(String... dashSignaledSystemIds);

        /**
         * Specify the SPEKE version, either v1.0 or v2.0, that MediaConvert uses when encrypting your output. For more
         * information, see: https://docs.aws.amazon.com/speke/latest/documentation/speke-api-specification.html To use
         * SPEKE v1.0: Leave blank. To use SPEKE v2.0: Specify a SPEKE v2.0 video preset and a SPEKE v2.0 audio preset.
         * 
         * @param encryptionContractConfiguration
         *        Specify the SPEKE version, either v1.0 or v2.0, that MediaConvert uses when encrypting your output.
         *        For more information, see:
         *        https://docs.aws.amazon.com/speke/latest/documentation/speke-api-specification.html To use SPEKE v1.0:
         *        Leave blank. To use SPEKE v2.0: Specify a SPEKE v2.0 video preset and a SPEKE v2.0 audio preset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionContractConfiguration(EncryptionContractConfiguration encryptionContractConfiguration);

        /**
         * Specify the SPEKE version, either v1.0 or v2.0, that MediaConvert uses when encrypting your output. For more
         * information, see: https://docs.aws.amazon.com/speke/latest/documentation/speke-api-specification.html To use
         * SPEKE v1.0: Leave blank. To use SPEKE v2.0: Specify a SPEKE v2.0 video preset and a SPEKE v2.0 audio preset.
         * This is a convenience method that creates an instance of the {@link EncryptionContractConfiguration.Builder}
         * avoiding the need to create one manually via {@link EncryptionContractConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionContractConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #encryptionContractConfiguration(EncryptionContractConfiguration)}.
         * 
         * @param encryptionContractConfiguration
         *        a consumer that will call methods on {@link EncryptionContractConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionContractConfiguration(EncryptionContractConfiguration)
         */
        default Builder encryptionContractConfiguration(
                Consumer<EncryptionContractConfiguration.Builder> encryptionContractConfiguration) {
            return encryptionContractConfiguration(EncryptionContractConfiguration.builder()
                    .applyMutation(encryptionContractConfiguration).build());
        }

        /**
         * Specify the DRM system ID that you want signaled in the HLS manifest that MediaConvert creates as part of
         * this CMAF package. The HLS manifest can currently signal only one system ID. For more information, see
         * https://dashif.org/identifiers/content_protection/.
         * 
         * @param hlsSignaledSystemIds
         *        Specify the DRM system ID that you want signaled in the HLS manifest that MediaConvert creates as part
         *        of this CMAF package. The HLS manifest can currently signal only one system ID. For more information,
         *        see https://dashif.org/identifiers/content_protection/.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hlsSignaledSystemIds(Collection<String> hlsSignaledSystemIds);

        /**
         * Specify the DRM system ID that you want signaled in the HLS manifest that MediaConvert creates as part of
         * this CMAF package. The HLS manifest can currently signal only one system ID. For more information, see
         * https://dashif.org/identifiers/content_protection/.
         * 
         * @param hlsSignaledSystemIds
         *        Specify the DRM system ID that you want signaled in the HLS manifest that MediaConvert creates as part
         *        of this CMAF package. The HLS manifest can currently signal only one system ID. For more information,
         *        see https://dashif.org/identifiers/content_protection/.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hlsSignaledSystemIds(String... hlsSignaledSystemIds);

        /**
         * Specify the resource ID that your SPEKE-compliant key provider uses to identify this content.
         * 
         * @param resourceId
         *        Specify the resource ID that your SPEKE-compliant key provider uses to identify this content.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * Specify the URL to the key server that your SPEKE-compliant DRM key provider uses to provide keys for
         * encrypting your content.
         * 
         * @param url
         *        Specify the URL to the key server that your SPEKE-compliant DRM key provider uses to provide keys for
         *        encrypting your content.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);
    }

    static final class BuilderImpl implements Builder {
        private String certificateArn;

        private List<String> dashSignaledSystemIds = DefaultSdkAutoConstructList.getInstance();

        private EncryptionContractConfiguration encryptionContractConfiguration;

        private List<String> hlsSignaledSystemIds = DefaultSdkAutoConstructList.getInstance();

        private String resourceId;

        private String url;

        private BuilderImpl() {
        }

        private BuilderImpl(SpekeKeyProviderCmaf model) {
            certificateArn(model.certificateArn);
            dashSignaledSystemIds(model.dashSignaledSystemIds);
            encryptionContractConfiguration(model.encryptionContractConfiguration);
            hlsSignaledSystemIds(model.hlsSignaledSystemIds);
            resourceId(model.resourceId);
            url(model.url);
        }

        public final String getCertificateArn() {
            return certificateArn;
        }

        public final void setCertificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
        }

        @Override
        public final Builder certificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
            return this;
        }

        public final Collection<String> getDashSignaledSystemIds() {
            if (dashSignaledSystemIds instanceof SdkAutoConstructList) {
                return null;
            }
            return dashSignaledSystemIds;
        }

        public final void setDashSignaledSystemIds(Collection<String> dashSignaledSystemIds) {
            this.dashSignaledSystemIds = ___listOf__stringMin36Max36Pattern09aFAF809aFAF409aFAF409aFAF409aFAF12Copier
                    .copy(dashSignaledSystemIds);
        }

        @Override
        public final Builder dashSignaledSystemIds(Collection<String> dashSignaledSystemIds) {
            this.dashSignaledSystemIds = ___listOf__stringMin36Max36Pattern09aFAF809aFAF409aFAF409aFAF409aFAF12Copier
                    .copy(dashSignaledSystemIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dashSignaledSystemIds(String... dashSignaledSystemIds) {
            dashSignaledSystemIds(Arrays.asList(dashSignaledSystemIds));
            return this;
        }

        public final EncryptionContractConfiguration.Builder getEncryptionContractConfiguration() {
            return encryptionContractConfiguration != null ? encryptionContractConfiguration.toBuilder() : null;
        }

        public final void setEncryptionContractConfiguration(
                EncryptionContractConfiguration.BuilderImpl encryptionContractConfiguration) {
            this.encryptionContractConfiguration = encryptionContractConfiguration != null ? encryptionContractConfiguration
                    .build() : null;
        }

        @Override
        public final Builder encryptionContractConfiguration(EncryptionContractConfiguration encryptionContractConfiguration) {
            this.encryptionContractConfiguration = encryptionContractConfiguration;
            return this;
        }

        public final Collection<String> getHlsSignaledSystemIds() {
            if (hlsSignaledSystemIds instanceof SdkAutoConstructList) {
                return null;
            }
            return hlsSignaledSystemIds;
        }

        public final void setHlsSignaledSystemIds(Collection<String> hlsSignaledSystemIds) {
            this.hlsSignaledSystemIds = ___listOf__stringMin36Max36Pattern09aFAF809aFAF409aFAF409aFAF409aFAF12Copier
                    .copy(hlsSignaledSystemIds);
        }

        @Override
        public final Builder hlsSignaledSystemIds(Collection<String> hlsSignaledSystemIds) {
            this.hlsSignaledSystemIds = ___listOf__stringMin36Max36Pattern09aFAF809aFAF409aFAF409aFAF409aFAF12Copier
                    .copy(hlsSignaledSystemIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder hlsSignaledSystemIds(String... hlsSignaledSystemIds) {
            hlsSignaledSystemIds(Arrays.asList(hlsSignaledSystemIds));
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getUrl() {
            return url;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        @Override
        public SpekeKeyProviderCmaf build() {
            return new SpekeKeyProviderCmaf(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
