/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medialive.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Udp Output Settings
 */
@Generated("software.amazon.awssdk:codegen")
public final class UdpOutputSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<UdpOutputSettings.Builder, UdpOutputSettings> {
    private static final SdkField<Integer> BUFFER_MSEC_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("BufferMsec").getter(getter(UdpOutputSettings::bufferMsec)).setter(setter(Builder::bufferMsec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bufferMsec").build()).build();

    private static final SdkField<UdpContainerSettings> CONTAINER_SETTINGS_FIELD = SdkField
            .<UdpContainerSettings> builder(MarshallingType.SDK_POJO).memberName("ContainerSettings")
            .getter(getter(UdpOutputSettings::containerSettings)).setter(setter(Builder::containerSettings))
            .constructor(UdpContainerSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerSettings").build()).build();

    private static final SdkField<OutputLocationRef> DESTINATION_FIELD = SdkField
            .<OutputLocationRef> builder(MarshallingType.SDK_POJO).memberName("Destination")
            .getter(getter(UdpOutputSettings::destination)).setter(setter(Builder::destination))
            .constructor(OutputLocationRef::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("destination").build()).build();

    private static final SdkField<FecOutputSettings> FEC_OUTPUT_SETTINGS_FIELD = SdkField
            .<FecOutputSettings> builder(MarshallingType.SDK_POJO).memberName("FecOutputSettings")
            .getter(getter(UdpOutputSettings::fecOutputSettings)).setter(setter(Builder::fecOutputSettings))
            .constructor(FecOutputSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fecOutputSettings").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUFFER_MSEC_FIELD,
            CONTAINER_SETTINGS_FIELD, DESTINATION_FIELD, FEC_OUTPUT_SETTINGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer bufferMsec;

    private final UdpContainerSettings containerSettings;

    private final OutputLocationRef destination;

    private final FecOutputSettings fecOutputSettings;

    private UdpOutputSettings(BuilderImpl builder) {
        this.bufferMsec = builder.bufferMsec;
        this.containerSettings = builder.containerSettings;
        this.destination = builder.destination;
        this.fecOutputSettings = builder.fecOutputSettings;
    }

    /**
     * UDP output buffering in milliseconds. Larger values increase latency through the transcoder but simultaneously
     * assist the transcoder in maintaining a constant, low-jitter UDP/RTP output while accommodating clock recovery,
     * input switching, input disruptions, picture reordering, etc.
     * 
     * @return UDP output buffering in milliseconds. Larger values increase latency through the transcoder but
     *         simultaneously assist the transcoder in maintaining a constant, low-jitter UDP/RTP output while
     *         accommodating clock recovery, input switching, input disruptions, picture reordering, etc.
     */
    public final Integer bufferMsec() {
        return bufferMsec;
    }

    /**
     * Returns the value of the ContainerSettings property for this object.
     * 
     * @return The value of the ContainerSettings property for this object.
     */
    public final UdpContainerSettings containerSettings() {
        return containerSettings;
    }

    /**
     * Destination address and port number for RTP or UDP packets. Can be unicast or multicast RTP or UDP (eg.
     * rtp://239.10.10.10:5001 or udp://10.100.100.100:5002).
     * 
     * @return Destination address and port number for RTP or UDP packets. Can be unicast or multicast RTP or UDP (eg.
     *         rtp://239.10.10.10:5001 or udp://10.100.100.100:5002).
     */
    public final OutputLocationRef destination() {
        return destination;
    }

    /**
     * Settings for enabling and adjusting Forward Error Correction on UDP outputs.
     * 
     * @return Settings for enabling and adjusting Forward Error Correction on UDP outputs.
     */
    public final FecOutputSettings fecOutputSettings() {
        return fecOutputSettings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bufferMsec());
        hashCode = 31 * hashCode + Objects.hashCode(containerSettings());
        hashCode = 31 * hashCode + Objects.hashCode(destination());
        hashCode = 31 * hashCode + Objects.hashCode(fecOutputSettings());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UdpOutputSettings)) {
            return false;
        }
        UdpOutputSettings other = (UdpOutputSettings) obj;
        return Objects.equals(bufferMsec(), other.bufferMsec()) && Objects.equals(containerSettings(), other.containerSettings())
                && Objects.equals(destination(), other.destination())
                && Objects.equals(fecOutputSettings(), other.fecOutputSettings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UdpOutputSettings").add("BufferMsec", bufferMsec())
                .add("ContainerSettings", containerSettings()).add("Destination", destination())
                .add("FecOutputSettings", fecOutputSettings()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BufferMsec":
            return Optional.ofNullable(clazz.cast(bufferMsec()));
        case "ContainerSettings":
            return Optional.ofNullable(clazz.cast(containerSettings()));
        case "Destination":
            return Optional.ofNullable(clazz.cast(destination()));
        case "FecOutputSettings":
            return Optional.ofNullable(clazz.cast(fecOutputSettings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UdpOutputSettings, T> g) {
        return obj -> g.apply((UdpOutputSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UdpOutputSettings> {
        /**
         * UDP output buffering in milliseconds. Larger values increase latency through the transcoder but
         * simultaneously assist the transcoder in maintaining a constant, low-jitter UDP/RTP output while accommodating
         * clock recovery, input switching, input disruptions, picture reordering, etc.
         * 
         * @param bufferMsec
         *        UDP output buffering in milliseconds. Larger values increase latency through the transcoder but
         *        simultaneously assist the transcoder in maintaining a constant, low-jitter UDP/RTP output while
         *        accommodating clock recovery, input switching, input disruptions, picture reordering, etc.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bufferMsec(Integer bufferMsec);

        /**
         * Sets the value of the ContainerSettings property for this object.
         *
         * @param containerSettings
         *        The new value for the ContainerSettings property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerSettings(UdpContainerSettings containerSettings);

        /**
         * Sets the value of the ContainerSettings property for this object.
         *
         * This is a convenience that creates an instance of the {@link UdpContainerSettings.Builder} avoiding the need
         * to create one manually via {@link UdpContainerSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link UdpContainerSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #containerSettings(UdpContainerSettings)}.
         * 
         * @param containerSettings
         *        a consumer that will call methods on {@link UdpContainerSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #containerSettings(UdpContainerSettings)
         */
        default Builder containerSettings(Consumer<UdpContainerSettings.Builder> containerSettings) {
            return containerSettings(UdpContainerSettings.builder().applyMutation(containerSettings).build());
        }

        /**
         * Destination address and port number for RTP or UDP packets. Can be unicast or multicast RTP or UDP (eg.
         * rtp://239.10.10.10:5001 or udp://10.100.100.100:5002).
         * 
         * @param destination
         *        Destination address and port number for RTP or UDP packets. Can be unicast or multicast RTP or UDP
         *        (eg. rtp://239.10.10.10:5001 or udp://10.100.100.100:5002).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(OutputLocationRef destination);

        /**
         * Destination address and port number for RTP or UDP packets. Can be unicast or multicast RTP or UDP (eg.
         * rtp://239.10.10.10:5001 or udp://10.100.100.100:5002). This is a convenience that creates an instance of the
         * {@link OutputLocationRef.Builder} avoiding the need to create one manually via
         * {@link OutputLocationRef#builder()}.
         *
         * When the {@link Consumer} completes, {@link OutputLocationRef.Builder#build()} is called immediately and its
         * result is passed to {@link #destination(OutputLocationRef)}.
         * 
         * @param destination
         *        a consumer that will call methods on {@link OutputLocationRef.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destination(OutputLocationRef)
         */
        default Builder destination(Consumer<OutputLocationRef.Builder> destination) {
            return destination(OutputLocationRef.builder().applyMutation(destination).build());
        }

        /**
         * Settings for enabling and adjusting Forward Error Correction on UDP outputs.
         * 
         * @param fecOutputSettings
         *        Settings for enabling and adjusting Forward Error Correction on UDP outputs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fecOutputSettings(FecOutputSettings fecOutputSettings);

        /**
         * Settings for enabling and adjusting Forward Error Correction on UDP outputs. This is a convenience that
         * creates an instance of the {@link FecOutputSettings.Builder} avoiding the need to create one manually via
         * {@link FecOutputSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link FecOutputSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #fecOutputSettings(FecOutputSettings)}.
         * 
         * @param fecOutputSettings
         *        a consumer that will call methods on {@link FecOutputSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fecOutputSettings(FecOutputSettings)
         */
        default Builder fecOutputSettings(Consumer<FecOutputSettings.Builder> fecOutputSettings) {
            return fecOutputSettings(FecOutputSettings.builder().applyMutation(fecOutputSettings).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Integer bufferMsec;

        private UdpContainerSettings containerSettings;

        private OutputLocationRef destination;

        private FecOutputSettings fecOutputSettings;

        private BuilderImpl() {
        }

        private BuilderImpl(UdpOutputSettings model) {
            bufferMsec(model.bufferMsec);
            containerSettings(model.containerSettings);
            destination(model.destination);
            fecOutputSettings(model.fecOutputSettings);
        }

        public final Integer getBufferMsec() {
            return bufferMsec;
        }

        @Override
        public final Builder bufferMsec(Integer bufferMsec) {
            this.bufferMsec = bufferMsec;
            return this;
        }

        public final void setBufferMsec(Integer bufferMsec) {
            this.bufferMsec = bufferMsec;
        }

        public final UdpContainerSettings.Builder getContainerSettings() {
            return containerSettings != null ? containerSettings.toBuilder() : null;
        }

        @Override
        public final Builder containerSettings(UdpContainerSettings containerSettings) {
            this.containerSettings = containerSettings;
            return this;
        }

        public final void setContainerSettings(UdpContainerSettings.BuilderImpl containerSettings) {
            this.containerSettings = containerSettings != null ? containerSettings.build() : null;
        }

        public final OutputLocationRef.Builder getDestination() {
            return destination != null ? destination.toBuilder() : null;
        }

        @Override
        public final Builder destination(OutputLocationRef destination) {
            this.destination = destination;
            return this;
        }

        public final void setDestination(OutputLocationRef.BuilderImpl destination) {
            this.destination = destination != null ? destination.build() : null;
        }

        public final FecOutputSettings.Builder getFecOutputSettings() {
            return fecOutputSettings != null ? fecOutputSettings.toBuilder() : null;
        }

        @Override
        public final Builder fecOutputSettings(FecOutputSettings fecOutputSettings) {
            this.fecOutputSettings = fecOutputSettings;
            return this;
        }

        public final void setFecOutputSettings(FecOutputSettings.BuilderImpl fecOutputSettings) {
            this.fecOutputSettings = fecOutputSettings != null ? fecOutputSettings.build() : null;
        }

        @Override
        public UdpOutputSettings build() {
            return new UdpOutputSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
