/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medialive.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Audio Normalization Settings
 */
@Generated("software.amazon.awssdk:codegen")
public final class AudioNormalizationSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<AudioNormalizationSettings.Builder, AudioNormalizationSettings> {
    private static final SdkField<String> ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Algorithm").getter(getter(AudioNormalizationSettings::algorithmAsString))
            .setter(setter(Builder::algorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("algorithm").build()).build();

    private static final SdkField<String> ALGORITHM_CONTROL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlgorithmControl").getter(getter(AudioNormalizationSettings::algorithmControlAsString))
            .setter(setter(Builder::algorithmControl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("algorithmControl").build()).build();

    private static final SdkField<Double> TARGET_LKFS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("TargetLkfs").getter(getter(AudioNormalizationSettings::targetLkfs)).setter(setter(Builder::targetLkfs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetLkfs").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALGORITHM_FIELD,
            ALGORITHM_CONTROL_FIELD, TARGET_LKFS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String algorithm;

    private final String algorithmControl;

    private final Double targetLkfs;

    private AudioNormalizationSettings(BuilderImpl builder) {
        this.algorithm = builder.algorithm;
        this.algorithmControl = builder.algorithmControl;
        this.targetLkfs = builder.targetLkfs;
    }

    /**
     * Audio normalization algorithm to use. itu17701 conforms to the CALM Act specification, itu17702 conforms to the
     * EBU R-128 specification.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithm} will
     * return {@link AudioNormalizationAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #algorithmAsString}.
     * </p>
     * 
     * @return Audio normalization algorithm to use. itu17701 conforms to the CALM Act specification, itu17702 conforms
     *         to the EBU R-128 specification.
     * @see AudioNormalizationAlgorithm
     */
    public final AudioNormalizationAlgorithm algorithm() {
        return AudioNormalizationAlgorithm.fromValue(algorithm);
    }

    /**
     * Audio normalization algorithm to use. itu17701 conforms to the CALM Act specification, itu17702 conforms to the
     * EBU R-128 specification.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithm} will
     * return {@link AudioNormalizationAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #algorithmAsString}.
     * </p>
     * 
     * @return Audio normalization algorithm to use. itu17701 conforms to the CALM Act specification, itu17702 conforms
     *         to the EBU R-128 specification.
     * @see AudioNormalizationAlgorithm
     */
    public final String algorithmAsString() {
        return algorithm;
    }

    /**
     * When set to correctAudio the output audio is corrected using the chosen algorithm. If set to measureOnly, the
     * audio will be measured but not adjusted.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithmControl}
     * will return {@link AudioNormalizationAlgorithmControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #algorithmControlAsString}.
     * </p>
     * 
     * @return When set to correctAudio the output audio is corrected using the chosen algorithm. If set to measureOnly,
     *         the audio will be measured but not adjusted.
     * @see AudioNormalizationAlgorithmControl
     */
    public final AudioNormalizationAlgorithmControl algorithmControl() {
        return AudioNormalizationAlgorithmControl.fromValue(algorithmControl);
    }

    /**
     * When set to correctAudio the output audio is corrected using the chosen algorithm. If set to measureOnly, the
     * audio will be measured but not adjusted.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithmControl}
     * will return {@link AudioNormalizationAlgorithmControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #algorithmControlAsString}.
     * </p>
     * 
     * @return When set to correctAudio the output audio is corrected using the chosen algorithm. If set to measureOnly,
     *         the audio will be measured but not adjusted.
     * @see AudioNormalizationAlgorithmControl
     */
    public final String algorithmControlAsString() {
        return algorithmControl;
    }

    /**
     * Target LKFS(loudness) to adjust volume to. If no value is entered, a default value will be used according to the
     * chosen algorithm. The CALM Act (1770-1) recommends a target of -24 LKFS. The EBU R-128 specification (1770-2)
     * recommends a target of -23 LKFS.
     * 
     * @return Target LKFS(loudness) to adjust volume to. If no value is entered, a default value will be used according
     *         to the chosen algorithm. The CALM Act (1770-1) recommends a target of -24 LKFS. The EBU R-128
     *         specification (1770-2) recommends a target of -23 LKFS.
     */
    public final Double targetLkfs() {
        return targetLkfs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(algorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(algorithmControlAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetLkfs());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AudioNormalizationSettings)) {
            return false;
        }
        AudioNormalizationSettings other = (AudioNormalizationSettings) obj;
        return Objects.equals(algorithmAsString(), other.algorithmAsString())
                && Objects.equals(algorithmControlAsString(), other.algorithmControlAsString())
                && Objects.equals(targetLkfs(), other.targetLkfs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AudioNormalizationSettings").add("Algorithm", algorithmAsString())
                .add("AlgorithmControl", algorithmControlAsString()).add("TargetLkfs", targetLkfs()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Algorithm":
            return Optional.ofNullable(clazz.cast(algorithmAsString()));
        case "AlgorithmControl":
            return Optional.ofNullable(clazz.cast(algorithmControlAsString()));
        case "TargetLkfs":
            return Optional.ofNullable(clazz.cast(targetLkfs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AudioNormalizationSettings, T> g) {
        return obj -> g.apply((AudioNormalizationSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AudioNormalizationSettings> {
        /**
         * Audio normalization algorithm to use. itu17701 conforms to the CALM Act specification, itu17702 conforms to
         * the EBU R-128 specification.
         * 
         * @param algorithm
         *        Audio normalization algorithm to use. itu17701 conforms to the CALM Act specification, itu17702
         *        conforms to the EBU R-128 specification.
         * @see AudioNormalizationAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationAlgorithm
         */
        Builder algorithm(String algorithm);

        /**
         * Audio normalization algorithm to use. itu17701 conforms to the CALM Act specification, itu17702 conforms to
         * the EBU R-128 specification.
         * 
         * @param algorithm
         *        Audio normalization algorithm to use. itu17701 conforms to the CALM Act specification, itu17702
         *        conforms to the EBU R-128 specification.
         * @see AudioNormalizationAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationAlgorithm
         */
        Builder algorithm(AudioNormalizationAlgorithm algorithm);

        /**
         * When set to correctAudio the output audio is corrected using the chosen algorithm. If set to measureOnly, the
         * audio will be measured but not adjusted.
         * 
         * @param algorithmControl
         *        When set to correctAudio the output audio is corrected using the chosen algorithm. If set to
         *        measureOnly, the audio will be measured but not adjusted.
         * @see AudioNormalizationAlgorithmControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationAlgorithmControl
         */
        Builder algorithmControl(String algorithmControl);

        /**
         * When set to correctAudio the output audio is corrected using the chosen algorithm. If set to measureOnly, the
         * audio will be measured but not adjusted.
         * 
         * @param algorithmControl
         *        When set to correctAudio the output audio is corrected using the chosen algorithm. If set to
         *        measureOnly, the audio will be measured but not adjusted.
         * @see AudioNormalizationAlgorithmControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioNormalizationAlgorithmControl
         */
        Builder algorithmControl(AudioNormalizationAlgorithmControl algorithmControl);

        /**
         * Target LKFS(loudness) to adjust volume to. If no value is entered, a default value will be used according to
         * the chosen algorithm. The CALM Act (1770-1) recommends a target of -24 LKFS. The EBU R-128 specification
         * (1770-2) recommends a target of -23 LKFS.
         * 
         * @param targetLkfs
         *        Target LKFS(loudness) to adjust volume to. If no value is entered, a default value will be used
         *        according to the chosen algorithm. The CALM Act (1770-1) recommends a target of -24 LKFS. The EBU
         *        R-128 specification (1770-2) recommends a target of -23 LKFS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLkfs(Double targetLkfs);
    }

    static final class BuilderImpl implements Builder {
        private String algorithm;

        private String algorithmControl;

        private Double targetLkfs;

        private BuilderImpl() {
        }

        private BuilderImpl(AudioNormalizationSettings model) {
            algorithm(model.algorithm);
            algorithmControl(model.algorithmControl);
            targetLkfs(model.targetLkfs);
        }

        public final String getAlgorithm() {
            return algorithm;
        }

        public final void setAlgorithm(String algorithm) {
            this.algorithm = algorithm;
        }

        @Override
        @Transient
        public final Builder algorithm(String algorithm) {
            this.algorithm = algorithm;
            return this;
        }

        @Override
        @Transient
        public final Builder algorithm(AudioNormalizationAlgorithm algorithm) {
            this.algorithm(algorithm == null ? null : algorithm.toString());
            return this;
        }

        public final String getAlgorithmControl() {
            return algorithmControl;
        }

        public final void setAlgorithmControl(String algorithmControl) {
            this.algorithmControl = algorithmControl;
        }

        @Override
        @Transient
        public final Builder algorithmControl(String algorithmControl) {
            this.algorithmControl = algorithmControl;
            return this;
        }

        @Override
        @Transient
        public final Builder algorithmControl(AudioNormalizationAlgorithmControl algorithmControl) {
            this.algorithmControl(algorithmControl == null ? null : algorithmControl.toString());
            return this;
        }

        public final Double getTargetLkfs() {
            return targetLkfs;
        }

        public final void setTargetLkfs(Double targetLkfs) {
            this.targetLkfs = targetLkfs;
        }

        @Override
        @Transient
        public final Builder targetLkfs(Double targetLkfs) {
            this.targetLkfs = targetLkfs;
            return this;
        }

        @Override
        public AudioNormalizationSettings build() {
            return new AudioNormalizationSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
