/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medialive.waiters;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.medialive.MediaLiveClient;
import software.amazon.awssdk.services.medialive.model.DescribeChannelRequest;
import software.amazon.awssdk.services.medialive.model.DescribeChannelResponse;
import software.amazon.awssdk.services.medialive.model.DescribeInputRequest;
import software.amazon.awssdk.services.medialive.model.DescribeInputResponse;
import software.amazon.awssdk.services.medialive.model.DescribeMultiplexRequest;
import software.amazon.awssdk.services.medialive.model.DescribeMultiplexResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface MediaLiveWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelCreated} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeChannelRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelCreated(DescribeChannelRequest describeChannelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelCreated} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeChannelRequest#builder()}
     *
     * @param describeChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelCreated(
            Consumer<DescribeChannelRequest.Builder> describeChannelRequest) {
        return waitUntilChannelCreated(DescribeChannelRequest.builder().applyMutation(describeChannelRequest).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelCreated} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeChannelRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelCreated(DescribeChannelRequest describeChannelRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelCreated} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelCreated(
            Consumer<DescribeChannelRequest.Builder> describeChannelRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilChannelCreated(DescribeChannelRequest.builder().applyMutation(describeChannelRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeChannelRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelDeleted(DescribeChannelRequest describeChannelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeChannelRequest#builder()}
     *
     * @param describeChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelDeleted(
            Consumer<DescribeChannelRequest.Builder> describeChannelRequest) {
        return waitUntilChannelDeleted(DescribeChannelRequest.builder().applyMutation(describeChannelRequest).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeChannelRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelDeleted(DescribeChannelRequest describeChannelRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelDeleted(
            Consumer<DescribeChannelRequest.Builder> describeChannelRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilChannelDeleted(DescribeChannelRequest.builder().applyMutation(describeChannelRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelRunning} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeChannelRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelRunning(DescribeChannelRequest describeChannelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelRunning} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeChannelRequest#builder()}
     *
     * @param describeChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelRunning(
            Consumer<DescribeChannelRequest.Builder> describeChannelRequest) {
        return waitUntilChannelRunning(DescribeChannelRequest.builder().applyMutation(describeChannelRequest).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelRunning} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeChannelRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelRunning(DescribeChannelRequest describeChannelRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelRunning} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelRunning(
            Consumer<DescribeChannelRequest.Builder> describeChannelRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilChannelRunning(DescribeChannelRequest.builder().applyMutation(describeChannelRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelStopped} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeChannelRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelStopped(DescribeChannelRequest describeChannelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelStopped} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeChannelRequest#builder()}
     *
     * @param describeChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelStopped(
            Consumer<DescribeChannelRequest.Builder> describeChannelRequest) {
        return waitUntilChannelStopped(DescribeChannelRequest.builder().applyMutation(describeChannelRequest).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelStopped} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeChannelRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelStopped(DescribeChannelRequest describeChannelRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeChannel} API until the desired condition {@code ChannelStopped} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeChannelRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeChannelResponse> waitUntilChannelStopped(
            Consumer<DescribeChannelRequest.Builder> describeChannelRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilChannelStopped(DescribeChannelRequest.builder().applyMutation(describeChannelRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputAttached} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeInputRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputAttached(DescribeInputRequest describeInputRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputAttached} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeInputRequest#builder()}
     *
     * @param describeInputRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputAttached(
            Consumer<DescribeInputRequest.Builder> describeInputRequest) {
        return waitUntilInputAttached(DescribeInputRequest.builder().applyMutation(describeInputRequest).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputAttached} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeInputRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputAttached(DescribeInputRequest describeInputRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputAttached} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeInputRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputAttached(
            Consumer<DescribeInputRequest.Builder> describeInputRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilInputAttached(DescribeInputRequest.builder().applyMutation(describeInputRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state
     *
     * @param describeInputRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputDeleted(DescribeInputRequest describeInputRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeInputRequest#builder()}
     *
     * @param describeInputRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputDeleted(
            Consumer<DescribeInputRequest.Builder> describeInputRequest) {
        return waitUntilInputDeleted(DescribeInputRequest.builder().applyMutation(describeInputRequest).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state
     *
     * @param describeInputRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputDeleted(DescribeInputRequest describeInputRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeInputRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputDeleted(
            Consumer<DescribeInputRequest.Builder> describeInputRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilInputDeleted(DescribeInputRequest.builder().applyMutation(describeInputRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputDetached} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeInputRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputDetached(DescribeInputRequest describeInputRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputDetached} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeInputRequest#builder()}
     *
     * @param describeInputRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputDetached(
            Consumer<DescribeInputRequest.Builder> describeInputRequest) {
        return waitUntilInputDetached(DescribeInputRequest.builder().applyMutation(describeInputRequest).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputDetached} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeInputRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputDetached(DescribeInputRequest describeInputRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeInput} API until the desired condition {@code InputDetached} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeInputRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeInputResponse> waitUntilInputDetached(
            Consumer<DescribeInputRequest.Builder> describeInputRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilInputDetached(DescribeInputRequest.builder().applyMutation(describeInputRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexCreated} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeMultiplexRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexCreated(DescribeMultiplexRequest describeMultiplexRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexCreated} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeMultiplexRequest#builder()}
     *
     * @param describeMultiplexRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexCreated(
            Consumer<DescribeMultiplexRequest.Builder> describeMultiplexRequest) {
        return waitUntilMultiplexCreated(DescribeMultiplexRequest.builder().applyMutation(describeMultiplexRequest).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexCreated} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeMultiplexRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexCreated(
            DescribeMultiplexRequest describeMultiplexRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexCreated} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeMultiplexRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexCreated(
            Consumer<DescribeMultiplexRequest.Builder> describeMultiplexRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilMultiplexCreated(DescribeMultiplexRequest.builder().applyMutation(describeMultiplexRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeMultiplexRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexDeleted(DescribeMultiplexRequest describeMultiplexRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeMultiplexRequest#builder()}
     *
     * @param describeMultiplexRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexDeleted(
            Consumer<DescribeMultiplexRequest.Builder> describeMultiplexRequest) {
        return waitUntilMultiplexDeleted(DescribeMultiplexRequest.builder().applyMutation(describeMultiplexRequest).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeMultiplexRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexDeleted(
            DescribeMultiplexRequest describeMultiplexRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeMultiplexRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexDeleted(
            Consumer<DescribeMultiplexRequest.Builder> describeMultiplexRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilMultiplexDeleted(DescribeMultiplexRequest.builder().applyMutation(describeMultiplexRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexRunning} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeMultiplexRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexRunning(DescribeMultiplexRequest describeMultiplexRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexRunning} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeMultiplexRequest#builder()}
     *
     * @param describeMultiplexRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexRunning(
            Consumer<DescribeMultiplexRequest.Builder> describeMultiplexRequest) {
        return waitUntilMultiplexRunning(DescribeMultiplexRequest.builder().applyMutation(describeMultiplexRequest).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexRunning} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeMultiplexRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexRunning(
            DescribeMultiplexRequest describeMultiplexRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexRunning} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeMultiplexRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexRunning(
            Consumer<DescribeMultiplexRequest.Builder> describeMultiplexRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilMultiplexRunning(DescribeMultiplexRequest.builder().applyMutation(describeMultiplexRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexStopped} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeMultiplexRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexStopped(DescribeMultiplexRequest describeMultiplexRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexStopped} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeMultiplexRequest#builder()}
     *
     * @param describeMultiplexRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexStopped(
            Consumer<DescribeMultiplexRequest.Builder> describeMultiplexRequest) {
        return waitUntilMultiplexStopped(DescribeMultiplexRequest.builder().applyMutation(describeMultiplexRequest).build());
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexStopped} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeMultiplexRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexStopped(
            DescribeMultiplexRequest describeMultiplexRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaLiveClient#describeMultiplex} API until the desired condition {@code MultiplexStopped} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeMultiplexRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexStopped(
            Consumer<DescribeMultiplexRequest.Builder> describeMultiplexRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilMultiplexStopped(DescribeMultiplexRequest.builder().applyMutation(describeMultiplexRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link MediaLiveWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultMediaLiveWaiter.builder();
    }

    /**
     * Create an instance of {@link MediaLiveWaiter} with the default configuration.
     * <p>
     * <b>A default {@link MediaLiveClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link MediaLiveWaiter}
     */
    static MediaLiveWaiter create() {
        return DefaultMediaLiveWaiter.builder().build();
    }

    interface Builder {
        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link MediaLiveClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(MediaLiveClient client);

        /**
         * Builds an instance of {@link MediaLiveWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link MediaLiveWaiter}
         */
        MediaLiveWaiter build();
    }
}
