/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medialive.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Fec Output Settings
 */
@Generated("software.amazon.awssdk:codegen")
public final class FecOutputSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<FecOutputSettings.Builder, FecOutputSettings> {
    private static final SdkField<Integer> COLUMN_DEPTH_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ColumnDepth").getter(getter(FecOutputSettings::columnDepth)).setter(setter(Builder::columnDepth))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("columnDepth").build()).build();

    private static final SdkField<String> INCLUDE_FEC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IncludeFec").getter(getter(FecOutputSettings::includeFecAsString)).setter(setter(Builder::includeFec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("includeFec").build()).build();

    private static final SdkField<Integer> ROW_LENGTH_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RowLength").getter(getter(FecOutputSettings::rowLength)).setter(setter(Builder::rowLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rowLength").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COLUMN_DEPTH_FIELD,
            INCLUDE_FEC_FIELD, ROW_LENGTH_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer columnDepth;

    private final String includeFec;

    private final Integer rowLength;

    private FecOutputSettings(BuilderImpl builder) {
        this.columnDepth = builder.columnDepth;
        this.includeFec = builder.includeFec;
        this.rowLength = builder.rowLength;
    }

    /**
     * Parameter D from SMPTE 2022-1. The height of the FEC protection matrix. The number of transport stream packets
     * per column error correction packet. Must be between 4 and 20, inclusive.
     * 
     * @return Parameter D from SMPTE 2022-1. The height of the FEC protection matrix. The number of transport stream
     *         packets per column error correction packet. Must be between 4 and 20, inclusive.
     */
    public final Integer columnDepth() {
        return columnDepth;
    }

    /**
     * Enables column only or column and row based FEC
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #includeFec} will
     * return {@link FecOutputIncludeFec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #includeFecAsString}.
     * </p>
     * 
     * @return Enables column only or column and row based FEC
     * @see FecOutputIncludeFec
     */
    public final FecOutputIncludeFec includeFec() {
        return FecOutputIncludeFec.fromValue(includeFec);
    }

    /**
     * Enables column only or column and row based FEC
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #includeFec} will
     * return {@link FecOutputIncludeFec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #includeFecAsString}.
     * </p>
     * 
     * @return Enables column only or column and row based FEC
     * @see FecOutputIncludeFec
     */
    public final String includeFecAsString() {
        return includeFec;
    }

    /**
     * Parameter L from SMPTE 2022-1. The width of the FEC protection matrix. Must be between 1 and 20, inclusive. If
     * only Column FEC is used, then larger values increase robustness. If Row FEC is used, then this is the number of
     * transport stream packets per row error correction packet, and the value must be between 4 and 20, inclusive, if
     * includeFec is columnAndRow. If includeFec is column, this value must be 1 to 20, inclusive.
     * 
     * @return Parameter L from SMPTE 2022-1. The width of the FEC protection matrix. Must be between 1 and 20,
     *         inclusive. If only Column FEC is used, then larger values increase robustness. If Row FEC is used, then
     *         this is the number of transport stream packets per row error correction packet, and the value must be
     *         between 4 and 20, inclusive, if includeFec is columnAndRow. If includeFec is column, this value must be 1
     *         to 20, inclusive.
     */
    public final Integer rowLength() {
        return rowLength;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(columnDepth());
        hashCode = 31 * hashCode + Objects.hashCode(includeFecAsString());
        hashCode = 31 * hashCode + Objects.hashCode(rowLength());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FecOutputSettings)) {
            return false;
        }
        FecOutputSettings other = (FecOutputSettings) obj;
        return Objects.equals(columnDepth(), other.columnDepth())
                && Objects.equals(includeFecAsString(), other.includeFecAsString())
                && Objects.equals(rowLength(), other.rowLength());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FecOutputSettings").add("ColumnDepth", columnDepth()).add("IncludeFec", includeFecAsString())
                .add("RowLength", rowLength()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ColumnDepth":
            return Optional.ofNullable(clazz.cast(columnDepth()));
        case "IncludeFec":
            return Optional.ofNullable(clazz.cast(includeFecAsString()));
        case "RowLength":
            return Optional.ofNullable(clazz.cast(rowLength()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FecOutputSettings, T> g) {
        return obj -> g.apply((FecOutputSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FecOutputSettings> {
        /**
         * Parameter D from SMPTE 2022-1. The height of the FEC protection matrix. The number of transport stream
         * packets per column error correction packet. Must be between 4 and 20, inclusive.
         * 
         * @param columnDepth
         *        Parameter D from SMPTE 2022-1. The height of the FEC protection matrix. The number of transport stream
         *        packets per column error correction packet. Must be between 4 and 20, inclusive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnDepth(Integer columnDepth);

        /**
         * Enables column only or column and row based FEC
         * 
         * @param includeFec
         *        Enables column only or column and row based FEC
         * @see FecOutputIncludeFec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FecOutputIncludeFec
         */
        Builder includeFec(String includeFec);

        /**
         * Enables column only or column and row based FEC
         * 
         * @param includeFec
         *        Enables column only or column and row based FEC
         * @see FecOutputIncludeFec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FecOutputIncludeFec
         */
        Builder includeFec(FecOutputIncludeFec includeFec);

        /**
         * Parameter L from SMPTE 2022-1. The width of the FEC protection matrix. Must be between 1 and 20, inclusive.
         * If only Column FEC is used, then larger values increase robustness. If Row FEC is used, then this is the
         * number of transport stream packets per row error correction packet, and the value must be between 4 and 20,
         * inclusive, if includeFec is columnAndRow. If includeFec is column, this value must be 1 to 20, inclusive.
         * 
         * @param rowLength
         *        Parameter L from SMPTE 2022-1. The width of the FEC protection matrix. Must be between 1 and 20,
         *        inclusive. If only Column FEC is used, then larger values increase robustness. If Row FEC is used,
         *        then this is the number of transport stream packets per row error correction packet, and the value
         *        must be between 4 and 20, inclusive, if includeFec is columnAndRow. If includeFec is column, this
         *        value must be 1 to 20, inclusive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rowLength(Integer rowLength);
    }

    static final class BuilderImpl implements Builder {
        private Integer columnDepth;

        private String includeFec;

        private Integer rowLength;

        private BuilderImpl() {
        }

        private BuilderImpl(FecOutputSettings model) {
            columnDepth(model.columnDepth);
            includeFec(model.includeFec);
            rowLength(model.rowLength);
        }

        public final Integer getColumnDepth() {
            return columnDepth;
        }

        public final void setColumnDepth(Integer columnDepth) {
            this.columnDepth = columnDepth;
        }

        @Override
        public final Builder columnDepth(Integer columnDepth) {
            this.columnDepth = columnDepth;
            return this;
        }

        public final String getIncludeFec() {
            return includeFec;
        }

        public final void setIncludeFec(String includeFec) {
            this.includeFec = includeFec;
        }

        @Override
        public final Builder includeFec(String includeFec) {
            this.includeFec = includeFec;
            return this;
        }

        @Override
        public final Builder includeFec(FecOutputIncludeFec includeFec) {
            this.includeFec(includeFec == null ? null : includeFec.toString());
            return this;
        }

        public final Integer getRowLength() {
            return rowLength;
        }

        public final void setRowLength(Integer rowLength) {
            this.rowLength = rowLength;
        }

        @Override
        public final Builder rowLength(Integer rowLength) {
            this.rowLength = rowLength;
            return this;
        }

        @Override
        public FecOutputSettings build() {
            return new FecOutputSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
