/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medialive.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Srt Output Settings
 */
@Generated("software.amazon.awssdk:codegen")
public final class SrtOutputSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<SrtOutputSettings.Builder, SrtOutputSettings> {
    private static final SdkField<Integer> BUFFER_MSEC_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("BufferMsec").getter(getter(SrtOutputSettings::bufferMsec)).setter(setter(Builder::bufferMsec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bufferMsec").build()).build();

    private static final SdkField<UdpContainerSettings> CONTAINER_SETTINGS_FIELD = SdkField
            .<UdpContainerSettings> builder(MarshallingType.SDK_POJO).memberName("ContainerSettings")
            .getter(getter(SrtOutputSettings::containerSettings)).setter(setter(Builder::containerSettings))
            .constructor(UdpContainerSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerSettings").build()).build();

    private static final SdkField<OutputLocationRef> DESTINATION_FIELD = SdkField
            .<OutputLocationRef> builder(MarshallingType.SDK_POJO).memberName("Destination")
            .getter(getter(SrtOutputSettings::destination)).setter(setter(Builder::destination))
            .constructor(OutputLocationRef::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("destination").build()).build();

    private static final SdkField<String> ENCRYPTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionType").getter(getter(SrtOutputSettings::encryptionTypeAsString))
            .setter(setter(Builder::encryptionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionType").build()).build();

    private static final SdkField<Integer> LATENCY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Latency").getter(getter(SrtOutputSettings::latency)).setter(setter(Builder::latency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("latency").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUFFER_MSEC_FIELD,
            CONTAINER_SETTINGS_FIELD, DESTINATION_FIELD, ENCRYPTION_TYPE_FIELD, LATENCY_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer bufferMsec;

    private final UdpContainerSettings containerSettings;

    private final OutputLocationRef destination;

    private final String encryptionType;

    private final Integer latency;

    private SrtOutputSettings(BuilderImpl builder) {
        this.bufferMsec = builder.bufferMsec;
        this.containerSettings = builder.containerSettings;
        this.destination = builder.destination;
        this.encryptionType = builder.encryptionType;
        this.latency = builder.latency;
    }

    /**
     * SRT output buffering in milliseconds. A higher value increases latency through the encoder. But the benefits are
     * that it helps to maintain a constant, low-jitter SRT output, and it accommodates clock recovery, input switching,
     * input disruptions, picture reordering, and so on. Range: 0-10000 milliseconds.
     * 
     * @return SRT output buffering in milliseconds. A higher value increases latency through the encoder. But the
     *         benefits are that it helps to maintain a constant, low-jitter SRT output, and it accommodates clock
     *         recovery, input switching, input disruptions, picture reordering, and so on. Range: 0-10000 milliseconds.
     */
    public final Integer bufferMsec() {
        return bufferMsec;
    }

    /**
     * Returns the value of the ContainerSettings property for this object.
     * 
     * @return The value of the ContainerSettings property for this object.
     */
    public final UdpContainerSettings containerSettings() {
        return containerSettings;
    }

    /**
     * Returns the value of the Destination property for this object.
     * 
     * @return The value of the Destination property for this object.
     */
    public final OutputLocationRef destination() {
        return destination;
    }

    /**
     * The encryption level for the content. Valid values are AES128, AES192, AES256. You and the downstream system
     * should plan how to set this field because the values must not conflict with each other.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionType}
     * will return {@link SrtEncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionTypeAsString}.
     * </p>
     * 
     * @return The encryption level for the content. Valid values are AES128, AES192, AES256. You and the downstream
     *         system should plan how to set this field because the values must not conflict with each other.
     * @see SrtEncryptionType
     */
    public final SrtEncryptionType encryptionType() {
        return SrtEncryptionType.fromValue(encryptionType);
    }

    /**
     * The encryption level for the content. Valid values are AES128, AES192, AES256. You and the downstream system
     * should plan how to set this field because the values must not conflict with each other.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionType}
     * will return {@link SrtEncryptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionTypeAsString}.
     * </p>
     * 
     * @return The encryption level for the content. Valid values are AES128, AES192, AES256. You and the downstream
     *         system should plan how to set this field because the values must not conflict with each other.
     * @see SrtEncryptionType
     */
    public final String encryptionTypeAsString() {
        return encryptionType;
    }

    /**
     * The latency value, in milliseconds, that is proposed during the SRT connection handshake. SRT will choose the
     * maximum of the values proposed by the sender and receiver. On the sender side, latency is the amount of time a
     * packet is held to give it a chance to be delivered successfully. On the receiver side, latency is the amount of
     * time the packet is held before delivering to the application, aiding in packet recovery and matching as closely
     * as possible the packet timing of the sender. Range: 40-16000 milliseconds.
     * 
     * @return The latency value, in milliseconds, that is proposed during the SRT connection handshake. SRT will choose
     *         the maximum of the values proposed by the sender and receiver. On the sender side, latency is the amount
     *         of time a packet is held to give it a chance to be delivered successfully. On the receiver side, latency
     *         is the amount of time the packet is held before delivering to the application, aiding in packet recovery
     *         and matching as closely as possible the packet timing of the sender. Range: 40-16000 milliseconds.
     */
    public final Integer latency() {
        return latency;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bufferMsec());
        hashCode = 31 * hashCode + Objects.hashCode(containerSettings());
        hashCode = 31 * hashCode + Objects.hashCode(destination());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(latency());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SrtOutputSettings)) {
            return false;
        }
        SrtOutputSettings other = (SrtOutputSettings) obj;
        return Objects.equals(bufferMsec(), other.bufferMsec()) && Objects.equals(containerSettings(), other.containerSettings())
                && Objects.equals(destination(), other.destination())
                && Objects.equals(encryptionTypeAsString(), other.encryptionTypeAsString())
                && Objects.equals(latency(), other.latency());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SrtOutputSettings").add("BufferMsec", bufferMsec())
                .add("ContainerSettings", containerSettings()).add("Destination", destination())
                .add("EncryptionType", encryptionTypeAsString()).add("Latency", latency()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BufferMsec":
            return Optional.ofNullable(clazz.cast(bufferMsec()));
        case "ContainerSettings":
            return Optional.ofNullable(clazz.cast(containerSettings()));
        case "Destination":
            return Optional.ofNullable(clazz.cast(destination()));
        case "EncryptionType":
            return Optional.ofNullable(clazz.cast(encryptionTypeAsString()));
        case "Latency":
            return Optional.ofNullable(clazz.cast(latency()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SrtOutputSettings, T> g) {
        return obj -> g.apply((SrtOutputSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SrtOutputSettings> {
        /**
         * SRT output buffering in milliseconds. A higher value increases latency through the encoder. But the benefits
         * are that it helps to maintain a constant, low-jitter SRT output, and it accommodates clock recovery, input
         * switching, input disruptions, picture reordering, and so on. Range: 0-10000 milliseconds.
         * 
         * @param bufferMsec
         *        SRT output buffering in milliseconds. A higher value increases latency through the encoder. But the
         *        benefits are that it helps to maintain a constant, low-jitter SRT output, and it accommodates clock
         *        recovery, input switching, input disruptions, picture reordering, and so on. Range: 0-10000
         *        milliseconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bufferMsec(Integer bufferMsec);

        /**
         * Sets the value of the ContainerSettings property for this object.
         *
         * @param containerSettings
         *        The new value for the ContainerSettings property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerSettings(UdpContainerSettings containerSettings);

        /**
         * Sets the value of the ContainerSettings property for this object.
         *
         * This is a convenience method that creates an instance of the {@link UdpContainerSettings.Builder} avoiding
         * the need to create one manually via {@link UdpContainerSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UdpContainerSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #containerSettings(UdpContainerSettings)}.
         * 
         * @param containerSettings
         *        a consumer that will call methods on {@link UdpContainerSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #containerSettings(UdpContainerSettings)
         */
        default Builder containerSettings(Consumer<UdpContainerSettings.Builder> containerSettings) {
            return containerSettings(UdpContainerSettings.builder().applyMutation(containerSettings).build());
        }

        /**
         * Sets the value of the Destination property for this object.
         *
         * @param destination
         *        The new value for the Destination property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(OutputLocationRef destination);

        /**
         * Sets the value of the Destination property for this object.
         *
         * This is a convenience method that creates an instance of the {@link OutputLocationRef.Builder} avoiding the
         * need to create one manually via {@link OutputLocationRef#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OutputLocationRef.Builder#build()} is called immediately and its
         * result is passed to {@link #destination(OutputLocationRef)}.
         * 
         * @param destination
         *        a consumer that will call methods on {@link OutputLocationRef.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destination(OutputLocationRef)
         */
        default Builder destination(Consumer<OutputLocationRef.Builder> destination) {
            return destination(OutputLocationRef.builder().applyMutation(destination).build());
        }

        /**
         * The encryption level for the content. Valid values are AES128, AES192, AES256. You and the downstream system
         * should plan how to set this field because the values must not conflict with each other.
         * 
         * @param encryptionType
         *        The encryption level for the content. Valid values are AES128, AES192, AES256. You and the downstream
         *        system should plan how to set this field because the values must not conflict with each other.
         * @see SrtEncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SrtEncryptionType
         */
        Builder encryptionType(String encryptionType);

        /**
         * The encryption level for the content. Valid values are AES128, AES192, AES256. You and the downstream system
         * should plan how to set this field because the values must not conflict with each other.
         * 
         * @param encryptionType
         *        The encryption level for the content. Valid values are AES128, AES192, AES256. You and the downstream
         *        system should plan how to set this field because the values must not conflict with each other.
         * @see SrtEncryptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SrtEncryptionType
         */
        Builder encryptionType(SrtEncryptionType encryptionType);

        /**
         * The latency value, in milliseconds, that is proposed during the SRT connection handshake. SRT will choose the
         * maximum of the values proposed by the sender and receiver. On the sender side, latency is the amount of time
         * a packet is held to give it a chance to be delivered successfully. On the receiver side, latency is the
         * amount of time the packet is held before delivering to the application, aiding in packet recovery and
         * matching as closely as possible the packet timing of the sender. Range: 40-16000 milliseconds.
         * 
         * @param latency
         *        The latency value, in milliseconds, that is proposed during the SRT connection handshake. SRT will
         *        choose the maximum of the values proposed by the sender and receiver. On the sender side, latency is
         *        the amount of time a packet is held to give it a chance to be delivered successfully. On the receiver
         *        side, latency is the amount of time the packet is held before delivering to the application, aiding in
         *        packet recovery and matching as closely as possible the packet timing of the sender. Range: 40-16000
         *        milliseconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latency(Integer latency);
    }

    static final class BuilderImpl implements Builder {
        private Integer bufferMsec;

        private UdpContainerSettings containerSettings;

        private OutputLocationRef destination;

        private String encryptionType;

        private Integer latency;

        private BuilderImpl() {
        }

        private BuilderImpl(SrtOutputSettings model) {
            bufferMsec(model.bufferMsec);
            containerSettings(model.containerSettings);
            destination(model.destination);
            encryptionType(model.encryptionType);
            latency(model.latency);
        }

        public final Integer getBufferMsec() {
            return bufferMsec;
        }

        public final void setBufferMsec(Integer bufferMsec) {
            this.bufferMsec = bufferMsec;
        }

        @Override
        public final Builder bufferMsec(Integer bufferMsec) {
            this.bufferMsec = bufferMsec;
            return this;
        }

        public final UdpContainerSettings.Builder getContainerSettings() {
            return containerSettings != null ? containerSettings.toBuilder() : null;
        }

        public final void setContainerSettings(UdpContainerSettings.BuilderImpl containerSettings) {
            this.containerSettings = containerSettings != null ? containerSettings.build() : null;
        }

        @Override
        public final Builder containerSettings(UdpContainerSettings containerSettings) {
            this.containerSettings = containerSettings;
            return this;
        }

        public final OutputLocationRef.Builder getDestination() {
            return destination != null ? destination.toBuilder() : null;
        }

        public final void setDestination(OutputLocationRef.BuilderImpl destination) {
            this.destination = destination != null ? destination.build() : null;
        }

        @Override
        public final Builder destination(OutputLocationRef destination) {
            this.destination = destination;
            return this;
        }

        public final String getEncryptionType() {
            return encryptionType;
        }

        public final void setEncryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
        }

        @Override
        public final Builder encryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
            return this;
        }

        @Override
        public final Builder encryptionType(SrtEncryptionType encryptionType) {
            this.encryptionType(encryptionType == null ? null : encryptionType.toString());
            return this;
        }

        public final Integer getLatency() {
            return latency;
        }

        public final void setLatency(Integer latency) {
            this.latency = latency;
        }

        @Override
        public final Builder latency(Integer latency) {
            this.latency = latency;
            return this;
        }

        @Override
        public SrtOutputSettings build() {
            return new SrtOutputSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
