/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medialive.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.medialive.MediaLiveAsyncClient;
import software.amazon.awssdk.services.medialive.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.medialive.model.DescribeChannelPlacementGroupRequest;
import software.amazon.awssdk.services.medialive.model.DescribeChannelPlacementGroupResponse;
import software.amazon.awssdk.services.medialive.model.DescribeChannelRequest;
import software.amazon.awssdk.services.medialive.model.DescribeChannelResponse;
import software.amazon.awssdk.services.medialive.model.DescribeClusterRequest;
import software.amazon.awssdk.services.medialive.model.DescribeClusterResponse;
import software.amazon.awssdk.services.medialive.model.DescribeInputRequest;
import software.amazon.awssdk.services.medialive.model.DescribeInputResponse;
import software.amazon.awssdk.services.medialive.model.DescribeMultiplexRequest;
import software.amazon.awssdk.services.medialive.model.DescribeMultiplexResponse;
import software.amazon.awssdk.services.medialive.model.DescribeNodeRequest;
import software.amazon.awssdk.services.medialive.model.DescribeNodeResponse;
import software.amazon.awssdk.services.medialive.model.GetSignalMapRequest;
import software.amazon.awssdk.services.medialive.model.GetSignalMapResponse;
import software.amazon.awssdk.services.medialive.model.MediaLiveRequest;
import software.amazon.awssdk.services.medialive.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultMediaLiveAsyncWaiter implements MediaLiveAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final MediaLiveAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeChannelResponse> channelCreatedWaiter;

    private final AsyncWaiter<DescribeChannelResponse> channelRunningWaiter;

    private final AsyncWaiter<DescribeChannelResponse> channelStoppedWaiter;

    private final AsyncWaiter<DescribeChannelResponse> channelDeletedWaiter;

    private final AsyncWaiter<DescribeInputResponse> inputAttachedWaiter;

    private final AsyncWaiter<DescribeInputResponse> inputDetachedWaiter;

    private final AsyncWaiter<DescribeInputResponse> inputDeletedWaiter;

    private final AsyncWaiter<DescribeMultiplexResponse> multiplexCreatedWaiter;

    private final AsyncWaiter<DescribeMultiplexResponse> multiplexRunningWaiter;

    private final AsyncWaiter<DescribeMultiplexResponse> multiplexStoppedWaiter;

    private final AsyncWaiter<DescribeMultiplexResponse> multiplexDeletedWaiter;

    private final AsyncWaiter<GetSignalMapResponse> signalMapCreatedWaiter;

    private final AsyncWaiter<GetSignalMapResponse> signalMapMonitorDeletedWaiter;

    private final AsyncWaiter<GetSignalMapResponse> signalMapMonitorDeployedWaiter;

    private final AsyncWaiter<GetSignalMapResponse> signalMapUpdatedWaiter;

    private final AsyncWaiter<DescribeClusterResponse> clusterCreatedWaiter;

    private final AsyncWaiter<DescribeClusterResponse> clusterDeletedWaiter;

    private final AsyncWaiter<DescribeNodeResponse> nodeRegisteredWaiter;

    private final AsyncWaiter<DescribeNodeResponse> nodeDeregisteredWaiter;

    private final AsyncWaiter<DescribeChannelPlacementGroupResponse> channelPlacementGroupAssignedWaiter;

    private final AsyncWaiter<DescribeChannelPlacementGroupResponse> channelPlacementGroupUnassignedWaiter;

    private final AsyncWaiter<DescribeChannelPlacementGroupResponse> channelPlacementGroupDeletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultMediaLiveAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = MediaLiveAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.channelCreatedWaiter = AsyncWaiter.builder(DescribeChannelResponse.class).acceptors(channelCreatedWaiterAcceptors())
                .overrideConfiguration(channelCreatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.channelRunningWaiter = AsyncWaiter.builder(DescribeChannelResponse.class).acceptors(channelRunningWaiterAcceptors())
                .overrideConfiguration(channelRunningWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.channelStoppedWaiter = AsyncWaiter.builder(DescribeChannelResponse.class).acceptors(channelStoppedWaiterAcceptors())
                .overrideConfiguration(channelStoppedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.channelDeletedWaiter = AsyncWaiter.builder(DescribeChannelResponse.class).acceptors(channelDeletedWaiterAcceptors())
                .overrideConfiguration(channelDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.inputAttachedWaiter = AsyncWaiter.builder(DescribeInputResponse.class).acceptors(inputAttachedWaiterAcceptors())
                .overrideConfiguration(inputAttachedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.inputDetachedWaiter = AsyncWaiter.builder(DescribeInputResponse.class).acceptors(inputDetachedWaiterAcceptors())
                .overrideConfiguration(inputDetachedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.inputDeletedWaiter = AsyncWaiter.builder(DescribeInputResponse.class).acceptors(inputDeletedWaiterAcceptors())
                .overrideConfiguration(inputDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.multiplexCreatedWaiter = AsyncWaiter.builder(DescribeMultiplexResponse.class)
                .acceptors(multiplexCreatedWaiterAcceptors())
                .overrideConfiguration(multiplexCreatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.multiplexRunningWaiter = AsyncWaiter.builder(DescribeMultiplexResponse.class)
                .acceptors(multiplexRunningWaiterAcceptors())
                .overrideConfiguration(multiplexRunningWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.multiplexStoppedWaiter = AsyncWaiter.builder(DescribeMultiplexResponse.class)
                .acceptors(multiplexStoppedWaiterAcceptors())
                .overrideConfiguration(multiplexStoppedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.multiplexDeletedWaiter = AsyncWaiter.builder(DescribeMultiplexResponse.class)
                .acceptors(multiplexDeletedWaiterAcceptors())
                .overrideConfiguration(multiplexDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.signalMapCreatedWaiter = AsyncWaiter.builder(GetSignalMapResponse.class)
                .acceptors(signalMapCreatedWaiterAcceptors())
                .overrideConfiguration(signalMapCreatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.signalMapMonitorDeletedWaiter = AsyncWaiter.builder(GetSignalMapResponse.class)
                .acceptors(signalMapMonitorDeletedWaiterAcceptors())
                .overrideConfiguration(signalMapMonitorDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.signalMapMonitorDeployedWaiter = AsyncWaiter.builder(GetSignalMapResponse.class)
                .acceptors(signalMapMonitorDeployedWaiterAcceptors())
                .overrideConfiguration(signalMapMonitorDeployedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.signalMapUpdatedWaiter = AsyncWaiter.builder(GetSignalMapResponse.class)
                .acceptors(signalMapUpdatedWaiterAcceptors())
                .overrideConfiguration(signalMapUpdatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.clusterCreatedWaiter = AsyncWaiter.builder(DescribeClusterResponse.class).acceptors(clusterCreatedWaiterAcceptors())
                .overrideConfiguration(clusterCreatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.clusterDeletedWaiter = AsyncWaiter.builder(DescribeClusterResponse.class).acceptors(clusterDeletedWaiterAcceptors())
                .overrideConfiguration(clusterDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.nodeRegisteredWaiter = AsyncWaiter.builder(DescribeNodeResponse.class).acceptors(nodeRegisteredWaiterAcceptors())
                .overrideConfiguration(nodeRegisteredWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.nodeDeregisteredWaiter = AsyncWaiter.builder(DescribeNodeResponse.class)
                .acceptors(nodeDeregisteredWaiterAcceptors())
                .overrideConfiguration(nodeDeregisteredWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.channelPlacementGroupAssignedWaiter = AsyncWaiter.builder(DescribeChannelPlacementGroupResponse.class)
                .acceptors(channelPlacementGroupAssignedWaiterAcceptors())
                .overrideConfiguration(channelPlacementGroupAssignedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.channelPlacementGroupUnassignedWaiter = AsyncWaiter.builder(DescribeChannelPlacementGroupResponse.class)
                .acceptors(channelPlacementGroupUnassignedWaiterAcceptors())
                .overrideConfiguration(channelPlacementGroupUnassignedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.channelPlacementGroupDeletedWaiter = AsyncWaiter.builder(DescribeChannelPlacementGroupResponse.class)
                .acceptors(channelPlacementGroupDeletedWaiterAcceptors())
                .overrideConfiguration(channelPlacementGroupDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelResponse>> waitUntilChannelCreated(
            DescribeChannelRequest describeChannelRequest) {
        return channelCreatedWaiter.runAsync(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelResponse>> waitUntilChannelCreated(
            DescribeChannelRequest describeChannelRequest, WaiterOverrideConfiguration overrideConfig) {
        return channelCreatedWaiter.runAsync(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)),
                channelCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelResponse>> waitUntilChannelDeleted(
            DescribeChannelRequest describeChannelRequest) {
        return channelDeletedWaiter.runAsync(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelResponse>> waitUntilChannelDeleted(
            DescribeChannelRequest describeChannelRequest, WaiterOverrideConfiguration overrideConfig) {
        return channelDeletedWaiter.runAsync(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)),
                channelDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelPlacementGroupResponse>> waitUntilChannelPlacementGroupAssigned(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest) {
        return channelPlacementGroupAssignedWaiter.runAsync(() -> client
                .describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelPlacementGroupResponse>> waitUntilChannelPlacementGroupAssigned(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest, WaiterOverrideConfiguration overrideConfig) {
        return channelPlacementGroupAssignedWaiter.runAsync(
                () -> client.describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)),
                channelPlacementGroupAssignedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelPlacementGroupResponse>> waitUntilChannelPlacementGroupDeleted(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest) {
        return channelPlacementGroupDeletedWaiter.runAsync(() -> client
                .describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelPlacementGroupResponse>> waitUntilChannelPlacementGroupDeleted(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest, WaiterOverrideConfiguration overrideConfig) {
        return channelPlacementGroupDeletedWaiter.runAsync(
                () -> client.describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)),
                channelPlacementGroupDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelPlacementGroupResponse>> waitUntilChannelPlacementGroupUnassigned(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest) {
        return channelPlacementGroupUnassignedWaiter.runAsync(() -> client
                .describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelPlacementGroupResponse>> waitUntilChannelPlacementGroupUnassigned(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest, WaiterOverrideConfiguration overrideConfig) {
        return channelPlacementGroupUnassignedWaiter.runAsync(
                () -> client.describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)),
                channelPlacementGroupUnassignedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelResponse>> waitUntilChannelRunning(
            DescribeChannelRequest describeChannelRequest) {
        return channelRunningWaiter.runAsync(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelResponse>> waitUntilChannelRunning(
            DescribeChannelRequest describeChannelRequest, WaiterOverrideConfiguration overrideConfig) {
        return channelRunningWaiter.runAsync(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)),
                channelRunningWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelResponse>> waitUntilChannelStopped(
            DescribeChannelRequest describeChannelRequest) {
        return channelStoppedWaiter.runAsync(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChannelResponse>> waitUntilChannelStopped(
            DescribeChannelRequest describeChannelRequest, WaiterOverrideConfiguration overrideConfig) {
        return channelStoppedWaiter.runAsync(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)),
                channelStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterCreated(
            DescribeClusterRequest describeClusterRequest) {
        return clusterCreatedWaiter.runAsync(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterCreated(
            DescribeClusterRequest describeClusterRequest, WaiterOverrideConfiguration overrideConfig) {
        return clusterCreatedWaiter.runAsync(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            DescribeClusterRequest describeClusterRequest) {
        return clusterDeletedWaiter.runAsync(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            DescribeClusterRequest describeClusterRequest, WaiterOverrideConfiguration overrideConfig) {
        return clusterDeletedWaiter.runAsync(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInputResponse>> waitUntilInputAttached(
            DescribeInputRequest describeInputRequest) {
        return inputAttachedWaiter.runAsync(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInputResponse>> waitUntilInputAttached(
            DescribeInputRequest describeInputRequest, WaiterOverrideConfiguration overrideConfig) {
        return inputAttachedWaiter.runAsync(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)),
                inputAttachedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInputResponse>> waitUntilInputDeleted(
            DescribeInputRequest describeInputRequest) {
        return inputDeletedWaiter.runAsync(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInputResponse>> waitUntilInputDeleted(
            DescribeInputRequest describeInputRequest, WaiterOverrideConfiguration overrideConfig) {
        return inputDeletedWaiter.runAsync(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)),
                inputDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInputResponse>> waitUntilInputDetached(
            DescribeInputRequest describeInputRequest) {
        return inputDetachedWaiter.runAsync(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInputResponse>> waitUntilInputDetached(
            DescribeInputRequest describeInputRequest, WaiterOverrideConfiguration overrideConfig) {
        return inputDetachedWaiter.runAsync(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)),
                inputDetachedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeMultiplexResponse>> waitUntilMultiplexCreated(
            DescribeMultiplexRequest describeMultiplexRequest) {
        return multiplexCreatedWaiter.runAsync(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeMultiplexResponse>> waitUntilMultiplexCreated(
            DescribeMultiplexRequest describeMultiplexRequest, WaiterOverrideConfiguration overrideConfig) {
        return multiplexCreatedWaiter.runAsync(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)),
                multiplexCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeMultiplexResponse>> waitUntilMultiplexDeleted(
            DescribeMultiplexRequest describeMultiplexRequest) {
        return multiplexDeletedWaiter.runAsync(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeMultiplexResponse>> waitUntilMultiplexDeleted(
            DescribeMultiplexRequest describeMultiplexRequest, WaiterOverrideConfiguration overrideConfig) {
        return multiplexDeletedWaiter.runAsync(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)),
                multiplexDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeMultiplexResponse>> waitUntilMultiplexRunning(
            DescribeMultiplexRequest describeMultiplexRequest) {
        return multiplexRunningWaiter.runAsync(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeMultiplexResponse>> waitUntilMultiplexRunning(
            DescribeMultiplexRequest describeMultiplexRequest, WaiterOverrideConfiguration overrideConfig) {
        return multiplexRunningWaiter.runAsync(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)),
                multiplexRunningWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeMultiplexResponse>> waitUntilMultiplexStopped(
            DescribeMultiplexRequest describeMultiplexRequest) {
        return multiplexStoppedWaiter.runAsync(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeMultiplexResponse>> waitUntilMultiplexStopped(
            DescribeMultiplexRequest describeMultiplexRequest, WaiterOverrideConfiguration overrideConfig) {
        return multiplexStoppedWaiter.runAsync(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)),
                multiplexStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNodeResponse>> waitUntilNodeDeregistered(
            DescribeNodeRequest describeNodeRequest) {
        return nodeDeregisteredWaiter.runAsync(() -> client.describeNode(applyWaitersUserAgent(describeNodeRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNodeResponse>> waitUntilNodeDeregistered(
            DescribeNodeRequest describeNodeRequest, WaiterOverrideConfiguration overrideConfig) {
        return nodeDeregisteredWaiter.runAsync(() -> client.describeNode(applyWaitersUserAgent(describeNodeRequest)),
                nodeDeregisteredWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNodeResponse>> waitUntilNodeRegistered(DescribeNodeRequest describeNodeRequest) {
        return nodeRegisteredWaiter.runAsync(() -> client.describeNode(applyWaitersUserAgent(describeNodeRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNodeResponse>> waitUntilNodeRegistered(
            DescribeNodeRequest describeNodeRequest, WaiterOverrideConfiguration overrideConfig) {
        return nodeRegisteredWaiter.runAsync(() -> client.describeNode(applyWaitersUserAgent(describeNodeRequest)),
                nodeRegisteredWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetSignalMapResponse>> waitUntilSignalMapCreated(
            GetSignalMapRequest getSignalMapRequest) {
        return signalMapCreatedWaiter.runAsync(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetSignalMapResponse>> waitUntilSignalMapCreated(
            GetSignalMapRequest getSignalMapRequest, WaiterOverrideConfiguration overrideConfig) {
        return signalMapCreatedWaiter.runAsync(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)),
                signalMapCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetSignalMapResponse>> waitUntilSignalMapMonitorDeleted(
            GetSignalMapRequest getSignalMapRequest) {
        return signalMapMonitorDeletedWaiter.runAsync(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetSignalMapResponse>> waitUntilSignalMapMonitorDeleted(
            GetSignalMapRequest getSignalMapRequest, WaiterOverrideConfiguration overrideConfig) {
        return signalMapMonitorDeletedWaiter.runAsync(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)),
                signalMapMonitorDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetSignalMapResponse>> waitUntilSignalMapMonitorDeployed(
            GetSignalMapRequest getSignalMapRequest) {
        return signalMapMonitorDeployedWaiter.runAsync(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetSignalMapResponse>> waitUntilSignalMapMonitorDeployed(
            GetSignalMapRequest getSignalMapRequest, WaiterOverrideConfiguration overrideConfig) {
        return signalMapMonitorDeployedWaiter.runAsync(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)),
                signalMapMonitorDeployedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetSignalMapResponse>> waitUntilSignalMapUpdated(
            GetSignalMapRequest getSignalMapRequest) {
        return signalMapUpdatedWaiter.runAsync(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetSignalMapResponse>> waitUntilSignalMapUpdated(
            GetSignalMapRequest getSignalMapRequest, WaiterOverrideConfiguration overrideConfig) {
        return signalMapUpdatedWaiter.runAsync(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)),
                signalMapUpdatedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeChannelResponse>> channelCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "IDLE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "CREATING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelResponse>> channelRunningWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "RUNNING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "STARTING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelResponse>> channelStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "IDLE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "STOPPING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelResponse>> channelDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInputResponse>> inputAttachedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ATTACHED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DETACHED");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInputResponse>> inputDetachedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DETACHED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "CREATING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ATTACHED");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInputResponse>> inputDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeMultiplexResponse>> multiplexCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeMultiplexResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "IDLE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "CREATING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeMultiplexResponse>> multiplexRunningWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeMultiplexResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "RUNNING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "STARTING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeMultiplexResponse>> multiplexStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeMultiplexResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "IDLE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "STOPPING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeMultiplexResponse>> multiplexDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeMultiplexResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetSignalMapResponse>> signalMapCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetSignalMapResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "CREATE_COMPLETE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "CREATE_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Status").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetSignalMapResponse>> signalMapMonitorDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetSignalMapResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DELETE_COMPLETE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DELETE_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (MonitorDeployment.Status=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetSignalMapResponse>> signalMapMonitorDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetSignalMapResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DRY_RUN_DEPLOYMENT_COMPLETE");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DEPLOYMENT_COMPLETE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DRY_RUN_DEPLOYMENT_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DEPLOYMENT_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DRY_RUN_DEPLOYMENT_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (MonitorDeployment.Status=DRY_RUN_DEPLOYMENT_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DEPLOYMENT_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (MonitorDeployment.Status=DEPLOYMENT_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetSignalMapResponse>> signalMapUpdatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetSignalMapResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "UPDATE_COMPLETE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "UPDATE_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Status").value(), "UPDATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Status").value(), "UPDATE_REVERTED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=UPDATE_REVERTED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "CREATING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNodeResponse>> nodeRegisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNodeResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "REGISTERING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(404, WaiterState.RETRY));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "REGISTRATION_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=REGISTRATION_FAILED) and transitioned the waiter to failure state"));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNodeResponse>> nodeDeregisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNodeResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DEREGISTERED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DEREGISTERING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DRAINING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> channelPlacementGroupAssignedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ASSIGNED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ASSIGNING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> channelPlacementGroupUnassignedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "UNASSIGNED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "UNASSIGNING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> channelPlacementGroupDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration channelCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelRunningWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(84);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration inputAttachedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration inputDetachedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(84);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration inputDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration multiplexCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration multiplexRunningWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration multiplexStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(28);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration multiplexDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration signalMapCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration signalMapMonitorDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration signalMapMonitorDeployedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration signalMapUpdatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration clusterCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration clusterDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration nodeRegisteredWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration nodeDeregisteredWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelPlacementGroupAssignedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelPlacementGroupUnassignedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelPlacementGroupDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static MediaLiveAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends MediaLiveRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements MediaLiveAsyncWaiter.Builder {
        private MediaLiveAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public MediaLiveAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public MediaLiveAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public MediaLiveAsyncWaiter.Builder client(MediaLiveAsyncClient client) {
            this.client = client;
            return this;
        }

        public MediaLiveAsyncWaiter build() {
            return new DefaultMediaLiveAsyncWaiter(this);
        }
    }
}
