/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.medialive.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.medialive.MediaLiveClient;
import software.amazon.awssdk.services.medialive.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.medialive.model.DescribeChannelPlacementGroupRequest;
import software.amazon.awssdk.services.medialive.model.DescribeChannelPlacementGroupResponse;
import software.amazon.awssdk.services.medialive.model.DescribeChannelRequest;
import software.amazon.awssdk.services.medialive.model.DescribeChannelResponse;
import software.amazon.awssdk.services.medialive.model.DescribeClusterRequest;
import software.amazon.awssdk.services.medialive.model.DescribeClusterResponse;
import software.amazon.awssdk.services.medialive.model.DescribeInputRequest;
import software.amazon.awssdk.services.medialive.model.DescribeInputResponse;
import software.amazon.awssdk.services.medialive.model.DescribeMultiplexRequest;
import software.amazon.awssdk.services.medialive.model.DescribeMultiplexResponse;
import software.amazon.awssdk.services.medialive.model.DescribeNodeRequest;
import software.amazon.awssdk.services.medialive.model.DescribeNodeResponse;
import software.amazon.awssdk.services.medialive.model.GetSignalMapRequest;
import software.amazon.awssdk.services.medialive.model.GetSignalMapResponse;
import software.amazon.awssdk.services.medialive.model.MediaLiveRequest;
import software.amazon.awssdk.services.medialive.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultMediaLiveWaiter implements MediaLiveWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final MediaLiveClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeChannelResponse> channelCreatedWaiter;

    private final Waiter<DescribeChannelResponse> channelRunningWaiter;

    private final Waiter<DescribeChannelResponse> channelStoppedWaiter;

    private final Waiter<DescribeChannelResponse> channelDeletedWaiter;

    private final Waiter<DescribeInputResponse> inputAttachedWaiter;

    private final Waiter<DescribeInputResponse> inputDetachedWaiter;

    private final Waiter<DescribeInputResponse> inputDeletedWaiter;

    private final Waiter<DescribeMultiplexResponse> multiplexCreatedWaiter;

    private final Waiter<DescribeMultiplexResponse> multiplexRunningWaiter;

    private final Waiter<DescribeMultiplexResponse> multiplexStoppedWaiter;

    private final Waiter<DescribeMultiplexResponse> multiplexDeletedWaiter;

    private final Waiter<GetSignalMapResponse> signalMapCreatedWaiter;

    private final Waiter<GetSignalMapResponse> signalMapMonitorDeletedWaiter;

    private final Waiter<GetSignalMapResponse> signalMapMonitorDeployedWaiter;

    private final Waiter<GetSignalMapResponse> signalMapUpdatedWaiter;

    private final Waiter<DescribeClusterResponse> clusterCreatedWaiter;

    private final Waiter<DescribeClusterResponse> clusterDeletedWaiter;

    private final Waiter<DescribeNodeResponse> nodeRegisteredWaiter;

    private final Waiter<DescribeNodeResponse> nodeDeregisteredWaiter;

    private final Waiter<DescribeChannelPlacementGroupResponse> channelPlacementGroupAssignedWaiter;

    private final Waiter<DescribeChannelPlacementGroupResponse> channelPlacementGroupUnassignedWaiter;

    private final Waiter<DescribeChannelPlacementGroupResponse> channelPlacementGroupDeletedWaiter;

    private DefaultMediaLiveWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = MediaLiveClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.channelCreatedWaiter = Waiter.builder(DescribeChannelResponse.class).acceptors(channelCreatedWaiterAcceptors())
                .overrideConfiguration(channelCreatedWaiterConfig(builder.overrideConfiguration)).build();
        this.channelRunningWaiter = Waiter.builder(DescribeChannelResponse.class).acceptors(channelRunningWaiterAcceptors())
                .overrideConfiguration(channelRunningWaiterConfig(builder.overrideConfiguration)).build();
        this.channelStoppedWaiter = Waiter.builder(DescribeChannelResponse.class).acceptors(channelStoppedWaiterAcceptors())
                .overrideConfiguration(channelStoppedWaiterConfig(builder.overrideConfiguration)).build();
        this.channelDeletedWaiter = Waiter.builder(DescribeChannelResponse.class).acceptors(channelDeletedWaiterAcceptors())
                .overrideConfiguration(channelDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.inputAttachedWaiter = Waiter.builder(DescribeInputResponse.class).acceptors(inputAttachedWaiterAcceptors())
                .overrideConfiguration(inputAttachedWaiterConfig(builder.overrideConfiguration)).build();
        this.inputDetachedWaiter = Waiter.builder(DescribeInputResponse.class).acceptors(inputDetachedWaiterAcceptors())
                .overrideConfiguration(inputDetachedWaiterConfig(builder.overrideConfiguration)).build();
        this.inputDeletedWaiter = Waiter.builder(DescribeInputResponse.class).acceptors(inputDeletedWaiterAcceptors())
                .overrideConfiguration(inputDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.multiplexCreatedWaiter = Waiter.builder(DescribeMultiplexResponse.class)
                .acceptors(multiplexCreatedWaiterAcceptors())
                .overrideConfiguration(multiplexCreatedWaiterConfig(builder.overrideConfiguration)).build();
        this.multiplexRunningWaiter = Waiter.builder(DescribeMultiplexResponse.class)
                .acceptors(multiplexRunningWaiterAcceptors())
                .overrideConfiguration(multiplexRunningWaiterConfig(builder.overrideConfiguration)).build();
        this.multiplexStoppedWaiter = Waiter.builder(DescribeMultiplexResponse.class)
                .acceptors(multiplexStoppedWaiterAcceptors())
                .overrideConfiguration(multiplexStoppedWaiterConfig(builder.overrideConfiguration)).build();
        this.multiplexDeletedWaiter = Waiter.builder(DescribeMultiplexResponse.class)
                .acceptors(multiplexDeletedWaiterAcceptors())
                .overrideConfiguration(multiplexDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.signalMapCreatedWaiter = Waiter.builder(GetSignalMapResponse.class).acceptors(signalMapCreatedWaiterAcceptors())
                .overrideConfiguration(signalMapCreatedWaiterConfig(builder.overrideConfiguration)).build();
        this.signalMapMonitorDeletedWaiter = Waiter.builder(GetSignalMapResponse.class)
                .acceptors(signalMapMonitorDeletedWaiterAcceptors())
                .overrideConfiguration(signalMapMonitorDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.signalMapMonitorDeployedWaiter = Waiter.builder(GetSignalMapResponse.class)
                .acceptors(signalMapMonitorDeployedWaiterAcceptors())
                .overrideConfiguration(signalMapMonitorDeployedWaiterConfig(builder.overrideConfiguration)).build();
        this.signalMapUpdatedWaiter = Waiter.builder(GetSignalMapResponse.class).acceptors(signalMapUpdatedWaiterAcceptors())
                .overrideConfiguration(signalMapUpdatedWaiterConfig(builder.overrideConfiguration)).build();
        this.clusterCreatedWaiter = Waiter.builder(DescribeClusterResponse.class).acceptors(clusterCreatedWaiterAcceptors())
                .overrideConfiguration(clusterCreatedWaiterConfig(builder.overrideConfiguration)).build();
        this.clusterDeletedWaiter = Waiter.builder(DescribeClusterResponse.class).acceptors(clusterDeletedWaiterAcceptors())
                .overrideConfiguration(clusterDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.nodeRegisteredWaiter = Waiter.builder(DescribeNodeResponse.class).acceptors(nodeRegisteredWaiterAcceptors())
                .overrideConfiguration(nodeRegisteredWaiterConfig(builder.overrideConfiguration)).build();
        this.nodeDeregisteredWaiter = Waiter.builder(DescribeNodeResponse.class).acceptors(nodeDeregisteredWaiterAcceptors())
                .overrideConfiguration(nodeDeregisteredWaiterConfig(builder.overrideConfiguration)).build();
        this.channelPlacementGroupAssignedWaiter = Waiter.builder(DescribeChannelPlacementGroupResponse.class)
                .acceptors(channelPlacementGroupAssignedWaiterAcceptors())
                .overrideConfiguration(channelPlacementGroupAssignedWaiterConfig(builder.overrideConfiguration)).build();
        this.channelPlacementGroupUnassignedWaiter = Waiter.builder(DescribeChannelPlacementGroupResponse.class)
                .acceptors(channelPlacementGroupUnassignedWaiterAcceptors())
                .overrideConfiguration(channelPlacementGroupUnassignedWaiterConfig(builder.overrideConfiguration)).build();
        this.channelPlacementGroupDeletedWaiter = Waiter.builder(DescribeChannelPlacementGroupResponse.class)
                .acceptors(channelPlacementGroupDeletedWaiterAcceptors())
                .overrideConfiguration(channelPlacementGroupDeletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeChannelResponse> waitUntilChannelCreated(DescribeChannelRequest describeChannelRequest) {
        return channelCreatedWaiter.run(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)));
    }

    @Override
    public WaiterResponse<DescribeChannelResponse> waitUntilChannelCreated(DescribeChannelRequest describeChannelRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return channelCreatedWaiter.run(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)),
                channelCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeChannelResponse> waitUntilChannelDeleted(DescribeChannelRequest describeChannelRequest) {
        return channelDeletedWaiter.run(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)));
    }

    @Override
    public WaiterResponse<DescribeChannelResponse> waitUntilChannelDeleted(DescribeChannelRequest describeChannelRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return channelDeletedWaiter.run(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)),
                channelDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeChannelPlacementGroupResponse> waitUntilChannelPlacementGroupAssigned(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest) {
        return channelPlacementGroupAssignedWaiter.run(() -> client
                .describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)));
    }

    @Override
    public WaiterResponse<DescribeChannelPlacementGroupResponse> waitUntilChannelPlacementGroupAssigned(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest, WaiterOverrideConfiguration overrideConfig) {
        return channelPlacementGroupAssignedWaiter.run(
                () -> client.describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)),
                channelPlacementGroupAssignedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeChannelPlacementGroupResponse> waitUntilChannelPlacementGroupDeleted(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest) {
        return channelPlacementGroupDeletedWaiter.run(() -> client
                .describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)));
    }

    @Override
    public WaiterResponse<DescribeChannelPlacementGroupResponse> waitUntilChannelPlacementGroupDeleted(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest, WaiterOverrideConfiguration overrideConfig) {
        return channelPlacementGroupDeletedWaiter.run(
                () -> client.describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)),
                channelPlacementGroupDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeChannelPlacementGroupResponse> waitUntilChannelPlacementGroupUnassigned(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest) {
        return channelPlacementGroupUnassignedWaiter.run(() -> client
                .describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)));
    }

    @Override
    public WaiterResponse<DescribeChannelPlacementGroupResponse> waitUntilChannelPlacementGroupUnassigned(
            DescribeChannelPlacementGroupRequest describeChannelPlacementGroupRequest, WaiterOverrideConfiguration overrideConfig) {
        return channelPlacementGroupUnassignedWaiter.run(
                () -> client.describeChannelPlacementGroup(applyWaitersUserAgent(describeChannelPlacementGroupRequest)),
                channelPlacementGroupUnassignedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeChannelResponse> waitUntilChannelRunning(DescribeChannelRequest describeChannelRequest) {
        return channelRunningWaiter.run(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)));
    }

    @Override
    public WaiterResponse<DescribeChannelResponse> waitUntilChannelRunning(DescribeChannelRequest describeChannelRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return channelRunningWaiter.run(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)),
                channelRunningWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeChannelResponse> waitUntilChannelStopped(DescribeChannelRequest describeChannelRequest) {
        return channelStoppedWaiter.run(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)));
    }

    @Override
    public WaiterResponse<DescribeChannelResponse> waitUntilChannelStopped(DescribeChannelRequest describeChannelRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return channelStoppedWaiter.run(() -> client.describeChannel(applyWaitersUserAgent(describeChannelRequest)),
                channelStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterCreated(DescribeClusterRequest describeClusterRequest) {
        return clusterCreatedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterCreated(DescribeClusterRequest describeClusterRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return clusterCreatedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterDeleted(DescribeClusterRequest describeClusterRequest) {
        return clusterDeletedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterDeleted(DescribeClusterRequest describeClusterRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return clusterDeletedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeInputResponse> waitUntilInputAttached(DescribeInputRequest describeInputRequest) {
        return inputAttachedWaiter.run(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)));
    }

    @Override
    public WaiterResponse<DescribeInputResponse> waitUntilInputAttached(DescribeInputRequest describeInputRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return inputAttachedWaiter.run(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)),
                inputAttachedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeInputResponse> waitUntilInputDeleted(DescribeInputRequest describeInputRequest) {
        return inputDeletedWaiter.run(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)));
    }

    @Override
    public WaiterResponse<DescribeInputResponse> waitUntilInputDeleted(DescribeInputRequest describeInputRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return inputDeletedWaiter.run(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)),
                inputDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeInputResponse> waitUntilInputDetached(DescribeInputRequest describeInputRequest) {
        return inputDetachedWaiter.run(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)));
    }

    @Override
    public WaiterResponse<DescribeInputResponse> waitUntilInputDetached(DescribeInputRequest describeInputRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return inputDetachedWaiter.run(() -> client.describeInput(applyWaitersUserAgent(describeInputRequest)),
                inputDetachedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexCreated(DescribeMultiplexRequest describeMultiplexRequest) {
        return multiplexCreatedWaiter.run(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)));
    }

    @Override
    public WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexCreated(DescribeMultiplexRequest describeMultiplexRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return multiplexCreatedWaiter.run(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)),
                multiplexCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexDeleted(DescribeMultiplexRequest describeMultiplexRequest) {
        return multiplexDeletedWaiter.run(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)));
    }

    @Override
    public WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexDeleted(DescribeMultiplexRequest describeMultiplexRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return multiplexDeletedWaiter.run(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)),
                multiplexDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexRunning(DescribeMultiplexRequest describeMultiplexRequest) {
        return multiplexRunningWaiter.run(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)));
    }

    @Override
    public WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexRunning(DescribeMultiplexRequest describeMultiplexRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return multiplexRunningWaiter.run(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)),
                multiplexRunningWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexStopped(DescribeMultiplexRequest describeMultiplexRequest) {
        return multiplexStoppedWaiter.run(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)));
    }

    @Override
    public WaiterResponse<DescribeMultiplexResponse> waitUntilMultiplexStopped(DescribeMultiplexRequest describeMultiplexRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return multiplexStoppedWaiter.run(() -> client.describeMultiplex(applyWaitersUserAgent(describeMultiplexRequest)),
                multiplexStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeNodeResponse> waitUntilNodeDeregistered(DescribeNodeRequest describeNodeRequest) {
        return nodeDeregisteredWaiter.run(() -> client.describeNode(applyWaitersUserAgent(describeNodeRequest)));
    }

    @Override
    public WaiterResponse<DescribeNodeResponse> waitUntilNodeDeregistered(DescribeNodeRequest describeNodeRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return nodeDeregisteredWaiter.run(() -> client.describeNode(applyWaitersUserAgent(describeNodeRequest)),
                nodeDeregisteredWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeNodeResponse> waitUntilNodeRegistered(DescribeNodeRequest describeNodeRequest) {
        return nodeRegisteredWaiter.run(() -> client.describeNode(applyWaitersUserAgent(describeNodeRequest)));
    }

    @Override
    public WaiterResponse<DescribeNodeResponse> waitUntilNodeRegistered(DescribeNodeRequest describeNodeRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return nodeRegisteredWaiter.run(() -> client.describeNode(applyWaitersUserAgent(describeNodeRequest)),
                nodeRegisteredWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetSignalMapResponse> waitUntilSignalMapCreated(GetSignalMapRequest getSignalMapRequest) {
        return signalMapCreatedWaiter.run(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)));
    }

    @Override
    public WaiterResponse<GetSignalMapResponse> waitUntilSignalMapCreated(GetSignalMapRequest getSignalMapRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return signalMapCreatedWaiter.run(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)),
                signalMapCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetSignalMapResponse> waitUntilSignalMapMonitorDeleted(GetSignalMapRequest getSignalMapRequest) {
        return signalMapMonitorDeletedWaiter.run(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)));
    }

    @Override
    public WaiterResponse<GetSignalMapResponse> waitUntilSignalMapMonitorDeleted(GetSignalMapRequest getSignalMapRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return signalMapMonitorDeletedWaiter.run(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)),
                signalMapMonitorDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetSignalMapResponse> waitUntilSignalMapMonitorDeployed(GetSignalMapRequest getSignalMapRequest) {
        return signalMapMonitorDeployedWaiter.run(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)));
    }

    @Override
    public WaiterResponse<GetSignalMapResponse> waitUntilSignalMapMonitorDeployed(GetSignalMapRequest getSignalMapRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return signalMapMonitorDeployedWaiter.run(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)),
                signalMapMonitorDeployedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetSignalMapResponse> waitUntilSignalMapUpdated(GetSignalMapRequest getSignalMapRequest) {
        return signalMapUpdatedWaiter.run(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)));
    }

    @Override
    public WaiterResponse<GetSignalMapResponse> waitUntilSignalMapUpdated(GetSignalMapRequest getSignalMapRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return signalMapUpdatedWaiter.run(() -> client.getSignalMap(applyWaitersUserAgent(getSignalMapRequest)),
                signalMapUpdatedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeChannelResponse>> channelCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "IDLE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "CREATING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelResponse>> channelRunningWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "RUNNING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "STARTING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelResponse>> channelStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "IDLE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "STOPPING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelResponse>> channelDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInputResponse>> inputAttachedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ATTACHED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DETACHED");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInputResponse>> inputDetachedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DETACHED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "CREATING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ATTACHED");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInputResponse>> inputDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInputResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeMultiplexResponse>> multiplexCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeMultiplexResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "IDLE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "CREATING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeMultiplexResponse>> multiplexRunningWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeMultiplexResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "RUNNING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "STARTING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeMultiplexResponse>> multiplexStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeMultiplexResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "IDLE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "STOPPING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeMultiplexResponse>> multiplexDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeMultiplexResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetSignalMapResponse>> signalMapCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetSignalMapResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "CREATE_COMPLETE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "CREATE_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Status").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetSignalMapResponse>> signalMapMonitorDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetSignalMapResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DELETE_COMPLETE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DELETE_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (MonitorDeployment.Status=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetSignalMapResponse>> signalMapMonitorDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetSignalMapResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DRY_RUN_DEPLOYMENT_COMPLETE");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DEPLOYMENT_COMPLETE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DRY_RUN_DEPLOYMENT_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DEPLOYMENT_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DRY_RUN_DEPLOYMENT_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (MonitorDeployment.Status=DRY_RUN_DEPLOYMENT_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("MonitorDeployment").field("Status").value(), "DEPLOYMENT_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (MonitorDeployment.Status=DEPLOYMENT_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetSignalMapResponse>> signalMapUpdatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetSignalMapResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "UPDATE_COMPLETE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "UPDATE_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Status").value(), "UPDATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Status").value(), "UPDATE_REVERTED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=UPDATE_REVERTED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "CREATING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNodeResponse>> nodeRegisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNodeResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "REGISTERING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(404, WaiterState.RETRY));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "REGISTRATION_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=REGISTRATION_FAILED) and transitioned the waiter to failure state"));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNodeResponse>> nodeDeregisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNodeResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DEREGISTERED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DEREGISTERING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DRAINING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> channelPlacementGroupAssignedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ASSIGNED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ASSIGNING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> channelPlacementGroupUnassignedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "UNASSIGNED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "UNASSIGNING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> channelPlacementGroupDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChannelPlacementGroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETED");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "DELETING");
        }));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(500, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration channelCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelRunningWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(84);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration inputAttachedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration inputDetachedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(84);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration inputDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration multiplexCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration multiplexRunningWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration multiplexStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(28);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration multiplexDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration signalMapCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration signalMapMonitorDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration signalMapMonitorDeployedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration signalMapUpdatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration clusterCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration clusterDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration nodeRegisteredWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration nodeDeregisteredWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelPlacementGroupAssignedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelPlacementGroupUnassignedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration channelPlacementGroupDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static MediaLiveWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends MediaLiveRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements MediaLiveWaiter.Builder {
        private MediaLiveClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public MediaLiveWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public MediaLiveWaiter.Builder client(MediaLiveClient client) {
            this.client = client;
            return this;
        }

        public MediaLiveWaiter build() {
            return new DefaultMediaLiveWaiter(this);
        }
    }
}
