/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackage;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.protocol.json.AwsJsonProtocol;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolMetadata;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.internal.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.services.mediapackage.model.CreateChannelRequest;
import software.amazon.awssdk.services.mediapackage.model.CreateChannelResponse;
import software.amazon.awssdk.services.mediapackage.model.CreateOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackage.model.CreateOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackage.model.DeleteChannelRequest;
import software.amazon.awssdk.services.mediapackage.model.DeleteChannelResponse;
import software.amazon.awssdk.services.mediapackage.model.DeleteOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackage.model.DeleteOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackage.model.DescribeChannelRequest;
import software.amazon.awssdk.services.mediapackage.model.DescribeChannelResponse;
import software.amazon.awssdk.services.mediapackage.model.DescribeOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackage.model.DescribeOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackage.model.ForbiddenException;
import software.amazon.awssdk.services.mediapackage.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediapackage.model.ListChannelsRequest;
import software.amazon.awssdk.services.mediapackage.model.ListChannelsResponse;
import software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsRequest;
import software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsResponse;
import software.amazon.awssdk.services.mediapackage.model.MediaPackageException;
import software.amazon.awssdk.services.mediapackage.model.MediaPackageRequest;
import software.amazon.awssdk.services.mediapackage.model.NotFoundException;
import software.amazon.awssdk.services.mediapackage.model.RotateChannelCredentialsRequest;
import software.amazon.awssdk.services.mediapackage.model.RotateChannelCredentialsResponse;
import software.amazon.awssdk.services.mediapackage.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mediapackage.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediapackage.model.UnprocessableEntityException;
import software.amazon.awssdk.services.mediapackage.model.UpdateChannelRequest;
import software.amazon.awssdk.services.mediapackage.model.UpdateChannelResponse;
import software.amazon.awssdk.services.mediapackage.model.UpdateOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackage.model.UpdateOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackage.paginators.ListChannelsIterable;
import software.amazon.awssdk.services.mediapackage.paginators.ListOriginEndpointsIterable;
import software.amazon.awssdk.services.mediapackage.transform.CreateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.CreateChannelResponseUnmarshaller;
import software.amazon.awssdk.services.mediapackage.transform.CreateOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.CreateOriginEndpointResponseUnmarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DeleteChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DeleteChannelResponseUnmarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DeleteOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DeleteOriginEndpointResponseUnmarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DescribeChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DescribeChannelResponseUnmarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DescribeOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DescribeOriginEndpointResponseUnmarshaller;
import software.amazon.awssdk.services.mediapackage.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.ListChannelsResponseUnmarshaller;
import software.amazon.awssdk.services.mediapackage.transform.ListOriginEndpointsRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.ListOriginEndpointsResponseUnmarshaller;
import software.amazon.awssdk.services.mediapackage.transform.RotateChannelCredentialsRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.RotateChannelCredentialsResponseUnmarshaller;
import software.amazon.awssdk.services.mediapackage.transform.UpdateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.UpdateChannelResponseUnmarshaller;
import software.amazon.awssdk.services.mediapackage.transform.UpdateOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.UpdateOriginEndpointResponseUnmarshaller;

/**
 * Internal implementation of {@link MediaPackageClient}.
 *
 * @see MediaPackageClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaPackageClient implements MediaPackageClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaPackageClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(false);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Creates a new Channel.
     *
     * @param createChannelRequest
     *        A new Channel configuration.
     * @return Result of the CreateChannel operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.CreateChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/CreateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateChannelResponse createChannel(CreateChannelRequest createChannelRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {

        HttpResponseHandler<CreateChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateChannelResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<CreateChannelRequest, CreateChannelResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(createChannelRequest).withMarshaller(new CreateChannelRequestMarshaller(protocolFactory)));
    }

    /**
     * Creates a new OriginEndpoint record.
     *
     * @param createOriginEndpointRequest
     *        Configuration parameters used to create a new OriginEndpoint.
     * @return Result of the CreateOriginEndpoint operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.CreateOriginEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/CreateOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateOriginEndpointResponse createOriginEndpoint(CreateOriginEndpointRequest createOriginEndpointRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {

        HttpResponseHandler<CreateOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateOriginEndpointResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<CreateOriginEndpointRequest, CreateOriginEndpointResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(createOriginEndpointRequest)
                .withMarshaller(new CreateOriginEndpointRequestMarshaller(protocolFactory)));
    }

    /**
     * Deletes an existing Channel.
     *
     * @param deleteChannelRequest
     * @return Result of the DeleteChannel operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.DeleteChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/DeleteChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteChannelResponse deleteChannel(DeleteChannelRequest deleteChannelRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {

        HttpResponseHandler<DeleteChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteChannelResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DeleteChannelRequest, DeleteChannelResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteChannelRequest).withMarshaller(new DeleteChannelRequestMarshaller(protocolFactory)));
    }

    /**
     * Deletes an existing OriginEndpoint.
     *
     * @param deleteOriginEndpointRequest
     * @return Result of the DeleteOriginEndpoint operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.DeleteOriginEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/DeleteOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteOriginEndpointResponse deleteOriginEndpoint(DeleteOriginEndpointRequest deleteOriginEndpointRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {

        HttpResponseHandler<DeleteOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteOriginEndpointResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DeleteOriginEndpointRequest, DeleteOriginEndpointResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteOriginEndpointRequest)
                .withMarshaller(new DeleteOriginEndpointRequestMarshaller(protocolFactory)));
    }

    /**
     * Gets details about a Channel.
     *
     * @param describeChannelRequest
     * @return Result of the DescribeChannel operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.DescribeChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/DescribeChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeChannelResponse describeChannel(DescribeChannelRequest describeChannelRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {

        HttpResponseHandler<DescribeChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeChannelResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DescribeChannelRequest, DescribeChannelResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeChannelRequest).withMarshaller(new DescribeChannelRequestMarshaller(protocolFactory)));
    }

    /**
     * Gets details about an existing OriginEndpoint.
     *
     * @param describeOriginEndpointRequest
     * @return Result of the DescribeOriginEndpoint operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.DescribeOriginEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/DescribeOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeOriginEndpointResponse describeOriginEndpoint(DescribeOriginEndpointRequest describeOriginEndpointRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {

        HttpResponseHandler<DescribeOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeOriginEndpointResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DescribeOriginEndpointRequest, DescribeOriginEndpointResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeOriginEndpointRequest)
                .withMarshaller(new DescribeOriginEndpointRequestMarshaller(protocolFactory)));
    }

    /**
     * Returns a collection of Channels.
     *
     * @param listChannelsRequest
     * @return Result of the ListChannels operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.ListChannels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListChannelsResponse listChannels(ListChannelsRequest listChannelsRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {

        HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListChannelsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listChannelsRequest).withMarshaller(new ListChannelsRequestMarshaller(protocolFactory)));
    }

    /**
     * Returns a collection of Channels.<br/>
     * <p>
     * This is a variant of
     * {@link #listChannels(software.amazon.awssdk.services.mediapackage.model.ListChannelsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackage.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediapackage.paginators.ListChannelsIterable responses = client
     *             .listChannelsPaginator(request);
     *     for (software.amazon.awssdk.services.mediapackage.model.ListChannelsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackage.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannels(software.amazon.awssdk.services.mediapackage.model.ListChannelsRequest)} operation.</b>
     * </p>
     *
     * @param listChannelsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.ListChannels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListChannelsIterable listChannelsPaginator(ListChannelsRequest listChannelsRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {
        return new ListChannelsIterable(this, applyPaginatorUserAgent(listChannelsRequest));
    }

    /**
     * Returns a collection of OriginEndpoint records.
     *
     * @param listOriginEndpointsRequest
     * @return Result of the ListOriginEndpoints operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.ListOriginEndpoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/ListOriginEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListOriginEndpointsResponse listOriginEndpoints(ListOriginEndpointsRequest listOriginEndpointsRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {

        HttpResponseHandler<ListOriginEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListOriginEndpointsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<ListOriginEndpointsRequest, ListOriginEndpointsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listOriginEndpointsRequest).withMarshaller(new ListOriginEndpointsRequestMarshaller(protocolFactory)));
    }

    /**
     * Returns a collection of OriginEndpoint records.<br/>
     * <p>
     * This is a variant of
     * {@link #listOriginEndpoints(software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackage.paginators.ListOriginEndpointsIterable responses = client.listOriginEndpointsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediapackage.paginators.ListOriginEndpointsIterable responses = client
     *             .listOriginEndpointsPaginator(request);
     *     for (software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackage.paginators.ListOriginEndpointsIterable responses = client.listOriginEndpointsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOriginEndpoints(software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listOriginEndpointsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.ListOriginEndpoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/ListOriginEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListOriginEndpointsIterable listOriginEndpointsPaginator(ListOriginEndpointsRequest listOriginEndpointsRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {
        return new ListOriginEndpointsIterable(this, applyPaginatorUserAgent(listOriginEndpointsRequest));
    }

    /**
     * Changes the Channel ingest username and password.
     *
     * @param rotateChannelCredentialsRequest
     * @return Result of the RotateChannelCredentials operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.RotateChannelCredentials
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/RotateChannelCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RotateChannelCredentialsResponse rotateChannelCredentials(
            RotateChannelCredentialsRequest rotateChannelCredentialsRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {

        HttpResponseHandler<RotateChannelCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new RotateChannelCredentialsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler
                .execute(new ClientExecutionParams<RotateChannelCredentialsRequest, RotateChannelCredentialsResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(rotateChannelCredentialsRequest)
                        .withMarshaller(new RotateChannelCredentialsRequestMarshaller(protocolFactory)));
    }

    /**
     * Updates an existing Channel.
     *
     * @param updateChannelRequest
     *        Configuration parameters used to update the Channel.
     * @return Result of the UpdateChannel operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.UpdateChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/UpdateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateChannelResponse updateChannel(UpdateChannelRequest updateChannelRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {

        HttpResponseHandler<UpdateChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new UpdateChannelResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<UpdateChannelRequest, UpdateChannelResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(updateChannelRequest).withMarshaller(new UpdateChannelRequestMarshaller(protocolFactory)));
    }

    /**
     * Updates an existing OriginEndpoint.
     *
     * @param updateOriginEndpointRequest
     *        Configuration parameters used to update an existing OriginEndpoint.
     * @return Result of the UpdateOriginEndpoint operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageClient.UpdateOriginEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/UpdateOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateOriginEndpointResponse updateOriginEndpoint(UpdateOriginEndpointRequest updateOriginEndpointRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageException {

        HttpResponseHandler<UpdateOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new UpdateOriginEndpointResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<UpdateOriginEndpointRequest, UpdateOriginEndpointResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(updateOriginEndpointRequest)
                .withMarshaller(new UpdateOriginEndpointRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(AwsJsonProtocolFactory protocolFactory) {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }

    private software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory init(boolean supportsCbor) {
        return new AwsJsonProtocolFactory(
                new JsonClientMetadata()
                        .withSupportsCbor(supportsCbor)
                        .withSupportsIon(false)
                        .withBaseServiceExceptionClass(
                                software.amazon.awssdk.services.mediapackage.model.MediaPackageException.class)
                        .withContentTypeOverride("")
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("NotFoundException").withModeledClass(
                                        NotFoundException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("ServiceUnavailableException").withModeledClass(
                                        ServiceUnavailableException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("UnprocessableEntityException").withModeledClass(
                                        UnprocessableEntityException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("ForbiddenException").withModeledClass(
                                        ForbiddenException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("TooManyRequestsException").withModeledClass(
                                        TooManyRequestsException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("InternalServerErrorException").withModeledClass(
                                        InternalServerErrorException.class)), AwsJsonProtocolMetadata.builder()
                        .protocolVersion("1.1").protocol(AwsJsonProtocol.REST_JSON).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MediaPackageRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
