/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackage.model;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.mediapackage.transform.HlsEncryptionMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * An HTTP Live Streaming (HLS) encryption configuration.
 */
@Generated("software.amazon.awssdk:codegen")
public final class HlsEncryption implements StructuredPojo, ToCopyableBuilder<HlsEncryption.Builder, HlsEncryption> {
    private final String constantInitializationVector;

    private final String encryptionMethod;

    private final Integer keyRotationIntervalSeconds;

    private final Boolean repeatExtXKey;

    private final SpekeKeyProvider spekeKeyProvider;

    private HlsEncryption(BuilderImpl builder) {
        this.constantInitializationVector = builder.constantInitializationVector;
        this.encryptionMethod = builder.encryptionMethod;
        this.keyRotationIntervalSeconds = builder.keyRotationIntervalSeconds;
        this.repeatExtXKey = builder.repeatExtXKey;
        this.spekeKeyProvider = builder.spekeKeyProvider;
    }

    /**
     * A constant initialization vector for encryption (optional). When not specified the initialization vector will be
     * periodically rotated.
     *
     * @return A constant initialization vector for encryption (optional). When not specified the initialization vector
     *         will be periodically rotated.
     */
    public String constantInitializationVector() {
        return constantInitializationVector;
    }

    /**
     * The encryption method to use.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionMethod}
     * will return {@link EncryptionMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionMethodAsString}.
     * </p>
     * 
     * @return The encryption method to use.
     * @see EncryptionMethod
     */
    public EncryptionMethod encryptionMethod() {
        return EncryptionMethod.fromValue(encryptionMethod);
    }

    /**
     * The encryption method to use.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionMethod}
     * will return {@link EncryptionMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionMethodAsString}.
     * </p>
     * 
     * @return The encryption method to use.
     * @see EncryptionMethod
     */
    public String encryptionMethodAsString() {
        return encryptionMethod;
    }

    /**
     * Interval (in seconds) between each encryption key rotation.
     * 
     * @return Interval (in seconds) between each encryption key rotation.
     */
    public Integer keyRotationIntervalSeconds() {
        return keyRotationIntervalSeconds;
    }

    /**
     * When enabled, the EXT-X-KEY tag will be repeated in output manifests.
     * 
     * @return When enabled, the EXT-X-KEY tag will be repeated in output manifests.
     */
    public Boolean repeatExtXKey() {
        return repeatExtXKey;
    }

    /**
     * Returns the value of the SpekeKeyProvider property for this object.
     * 
     * @return The value of the SpekeKeyProvider property for this object.
     */
    public SpekeKeyProvider spekeKeyProvider() {
        return spekeKeyProvider;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(constantInitializationVector());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionMethodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyRotationIntervalSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(repeatExtXKey());
        hashCode = 31 * hashCode + Objects.hashCode(spekeKeyProvider());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HlsEncryption)) {
            return false;
        }
        HlsEncryption other = (HlsEncryption) obj;
        return Objects.equals(constantInitializationVector(), other.constantInitializationVector())
                && Objects.equals(encryptionMethodAsString(), other.encryptionMethodAsString())
                && Objects.equals(keyRotationIntervalSeconds(), other.keyRotationIntervalSeconds())
                && Objects.equals(repeatExtXKey(), other.repeatExtXKey())
                && Objects.equals(spekeKeyProvider(), other.spekeKeyProvider());
    }

    @Override
    public String toString() {
        return ToString.builder("HlsEncryption").add("ConstantInitializationVector", constantInitializationVector())
                .add("EncryptionMethod", encryptionMethodAsString())
                .add("KeyRotationIntervalSeconds", keyRotationIntervalSeconds()).add("RepeatExtXKey", repeatExtXKey())
                .add("SpekeKeyProvider", spekeKeyProvider()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConstantInitializationVector":
            return Optional.ofNullable(clazz.cast(constantInitializationVector()));
        case "EncryptionMethod":
            return Optional.ofNullable(clazz.cast(encryptionMethodAsString()));
        case "KeyRotationIntervalSeconds":
            return Optional.ofNullable(clazz.cast(keyRotationIntervalSeconds()));
        case "RepeatExtXKey":
            return Optional.ofNullable(clazz.cast(repeatExtXKey()));
        case "SpekeKeyProvider":
            return Optional.ofNullable(clazz.cast(spekeKeyProvider()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        HlsEncryptionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, HlsEncryption> {
        /**
         * A constant initialization vector for encryption (optional). When not specified the initialization vector will
         * be periodically rotated.
         *
         * @param constantInitializationVector
         *        A constant initialization vector for encryption (optional). When not specified the initialization
         *        vector will be periodically rotated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder constantInitializationVector(String constantInitializationVector);

        /**
         * The encryption method to use.
         * 
         * @param encryptionMethod
         *        The encryption method to use.
         * @see EncryptionMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionMethod
         */
        Builder encryptionMethod(String encryptionMethod);

        /**
         * The encryption method to use.
         * 
         * @param encryptionMethod
         *        The encryption method to use.
         * @see EncryptionMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionMethod
         */
        Builder encryptionMethod(EncryptionMethod encryptionMethod);

        /**
         * Interval (in seconds) between each encryption key rotation.
         * 
         * @param keyRotationIntervalSeconds
         *        Interval (in seconds) between each encryption key rotation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyRotationIntervalSeconds(Integer keyRotationIntervalSeconds);

        /**
         * When enabled, the EXT-X-KEY tag will be repeated in output manifests.
         * 
         * @param repeatExtXKey
         *        When enabled, the EXT-X-KEY tag will be repeated in output manifests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repeatExtXKey(Boolean repeatExtXKey);

        /**
         * Sets the value of the SpekeKeyProvider property for this object.
         *
         * @param spekeKeyProvider
         *        The new value for the SpekeKeyProvider property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spekeKeyProvider(SpekeKeyProvider spekeKeyProvider);

        /**
         * Sets the value of the SpekeKeyProvider property for this object.
         *
         * This is a convenience that creates an instance of the {@link SpekeKeyProvider.Builder} avoiding the need to
         * create one manually via {@link SpekeKeyProvider#builder()}.
         *
         * When the {@link Consumer} completes, {@link SpekeKeyProvider.Builder#build()} is called immediately and its
         * result is passed to {@link #spekeKeyProvider(SpekeKeyProvider)}.
         * 
         * @param spekeKeyProvider
         *        a consumer that will call methods on {@link SpekeKeyProvider.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #spekeKeyProvider(SpekeKeyProvider)
         */
        default Builder spekeKeyProvider(Consumer<SpekeKeyProvider.Builder> spekeKeyProvider) {
            return spekeKeyProvider(SpekeKeyProvider.builder().applyMutation(spekeKeyProvider).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String constantInitializationVector;

        private String encryptionMethod;

        private Integer keyRotationIntervalSeconds;

        private Boolean repeatExtXKey;

        private SpekeKeyProvider spekeKeyProvider;

        private BuilderImpl() {
        }

        private BuilderImpl(HlsEncryption model) {
            constantInitializationVector(model.constantInitializationVector);
            encryptionMethod(model.encryptionMethod);
            keyRotationIntervalSeconds(model.keyRotationIntervalSeconds);
            repeatExtXKey(model.repeatExtXKey);
            spekeKeyProvider(model.spekeKeyProvider);
        }

        public final String getConstantInitializationVector() {
            return constantInitializationVector;
        }

        @Override
        public final Builder constantInitializationVector(String constantInitializationVector) {
            this.constantInitializationVector = constantInitializationVector;
            return this;
        }

        public final void setConstantInitializationVector(String constantInitializationVector) {
            this.constantInitializationVector = constantInitializationVector;
        }

        public final String getEncryptionMethod() {
            return encryptionMethod;
        }

        @Override
        public final Builder encryptionMethod(String encryptionMethod) {
            this.encryptionMethod = encryptionMethod;
            return this;
        }

        @Override
        public final Builder encryptionMethod(EncryptionMethod encryptionMethod) {
            this.encryptionMethod(encryptionMethod.toString());
            return this;
        }

        public final void setEncryptionMethod(String encryptionMethod) {
            this.encryptionMethod = encryptionMethod;
        }

        public final Integer getKeyRotationIntervalSeconds() {
            return keyRotationIntervalSeconds;
        }

        @Override
        public final Builder keyRotationIntervalSeconds(Integer keyRotationIntervalSeconds) {
            this.keyRotationIntervalSeconds = keyRotationIntervalSeconds;
            return this;
        }

        public final void setKeyRotationIntervalSeconds(Integer keyRotationIntervalSeconds) {
            this.keyRotationIntervalSeconds = keyRotationIntervalSeconds;
        }

        public final Boolean getRepeatExtXKey() {
            return repeatExtXKey;
        }

        @Override
        public final Builder repeatExtXKey(Boolean repeatExtXKey) {
            this.repeatExtXKey = repeatExtXKey;
            return this;
        }

        public final void setRepeatExtXKey(Boolean repeatExtXKey) {
            this.repeatExtXKey = repeatExtXKey;
        }

        public final SpekeKeyProvider.Builder getSpekeKeyProvider() {
            return spekeKeyProvider != null ? spekeKeyProvider.toBuilder() : null;
        }

        @Override
        public final Builder spekeKeyProvider(SpekeKeyProvider spekeKeyProvider) {
            this.spekeKeyProvider = spekeKeyProvider;
            return this;
        }

        public final void setSpekeKeyProvider(SpekeKeyProvider.BuilderImpl spekeKeyProvider) {
            this.spekeKeyProvider = spekeKeyProvider != null ? spekeKeyProvider.build() : null;
        }

        @Override
        public HlsEncryption build() {
            return new HlsEncryption(this);
        }
    }
}
