/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackage.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.core.pagination.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.SdkIterable;
import software.amazon.awssdk.core.pagination.SyncPageFetcher;
import software.amazon.awssdk.services.mediapackage.MediaPackageClient;
import software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsRequest;
import software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsResponse;
import software.amazon.awssdk.services.mediapackage.model.OriginEndpoint;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.mediapackage.MediaPackageClient#listOriginEndpointsPaginator(software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.mediapackage.paginators.ListOriginEndpointsIterable responses = client.listOriginEndpointsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.mediapackage.paginators.ListOriginEndpointsIterable responses = client
 *             .listOriginEndpointsPaginator(request);
 *     for (software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.mediapackage.paginators.ListOriginEndpointsIterable responses = client.listOriginEndpointsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listOriginEndpoints(software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListOriginEndpointsIterable implements SdkIterable<ListOriginEndpointsResponse> {
    private final MediaPackageClient client;

    private final ListOriginEndpointsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListOriginEndpointsIterable(final MediaPackageClient client, final ListOriginEndpointsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListOriginEndpointsResponseFetcher();
    }

    @Override
    public Iterator<ListOriginEndpointsResponse> iterator() {
        return new PaginatedResponsesIterator(nextPageFetcher);
    }

    /**
     * Returns an iterable to iterate through the paginated {@link ListOriginEndpointsResponse#originEndpoints()}
     * member. The returned iterable is used to iterate through the results across all response pages and not a single
     * page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<OriginEndpoint> originEndpoints() {
        Function<ListOriginEndpointsResponse, Iterator<OriginEndpoint>> getIterator = response -> {
            if (response != null && response.originEndpoints() != null) {
                return response.originEndpoints().iterator();
            }
            return Collections.emptyIterator();
        };
        return new PaginatedItemsIterable(this, getIterator);
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link ListOriginEndpointsIterable} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    public final ListOriginEndpointsIterable resume(final ListOriginEndpointsResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new ListOriginEndpointsIterable(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken())
                    .build());
        }
        return new ListOriginEndpointsIterable(client, firstRequest) {
            @Override
            public Iterator<ListOriginEndpointsResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class ListOriginEndpointsResponseFetcher implements SyncPageFetcher<ListOriginEndpointsResponse> {
        @Override
        public boolean hasNextPage(ListOriginEndpointsResponse previousPage) {
            return previousPage.nextToken() != null;
        }

        @Override
        public ListOriginEndpointsResponse nextPage(ListOriginEndpointsResponse previousPage) {
            if (previousPage == null) {
                return client.listOriginEndpoints(firstRequest);
            }
            return client.listOriginEndpoints(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
