/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackage;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.mediapackage.internal.MediaPackageServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mediapackage.model.ConfigureLogsRequest;
import software.amazon.awssdk.services.mediapackage.model.ConfigureLogsResponse;
import software.amazon.awssdk.services.mediapackage.model.CreateChannelRequest;
import software.amazon.awssdk.services.mediapackage.model.CreateChannelResponse;
import software.amazon.awssdk.services.mediapackage.model.CreateHarvestJobRequest;
import software.amazon.awssdk.services.mediapackage.model.CreateHarvestJobResponse;
import software.amazon.awssdk.services.mediapackage.model.CreateOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackage.model.CreateOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackage.model.DeleteChannelRequest;
import software.amazon.awssdk.services.mediapackage.model.DeleteChannelResponse;
import software.amazon.awssdk.services.mediapackage.model.DeleteOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackage.model.DeleteOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackage.model.DescribeChannelRequest;
import software.amazon.awssdk.services.mediapackage.model.DescribeChannelResponse;
import software.amazon.awssdk.services.mediapackage.model.DescribeHarvestJobRequest;
import software.amazon.awssdk.services.mediapackage.model.DescribeHarvestJobResponse;
import software.amazon.awssdk.services.mediapackage.model.DescribeOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackage.model.DescribeOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackage.model.ForbiddenException;
import software.amazon.awssdk.services.mediapackage.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediapackage.model.ListChannelsRequest;
import software.amazon.awssdk.services.mediapackage.model.ListChannelsResponse;
import software.amazon.awssdk.services.mediapackage.model.ListHarvestJobsRequest;
import software.amazon.awssdk.services.mediapackage.model.ListHarvestJobsResponse;
import software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsRequest;
import software.amazon.awssdk.services.mediapackage.model.ListOriginEndpointsResponse;
import software.amazon.awssdk.services.mediapackage.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediapackage.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediapackage.model.MediaPackageException;
import software.amazon.awssdk.services.mediapackage.model.NotFoundException;
import software.amazon.awssdk.services.mediapackage.model.RotateIngestEndpointCredentialsRequest;
import software.amazon.awssdk.services.mediapackage.model.RotateIngestEndpointCredentialsResponse;
import software.amazon.awssdk.services.mediapackage.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mediapackage.model.TagResourceRequest;
import software.amazon.awssdk.services.mediapackage.model.TagResourceResponse;
import software.amazon.awssdk.services.mediapackage.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediapackage.model.UnprocessableEntityException;
import software.amazon.awssdk.services.mediapackage.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediapackage.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediapackage.model.UpdateChannelRequest;
import software.amazon.awssdk.services.mediapackage.model.UpdateChannelResponse;
import software.amazon.awssdk.services.mediapackage.model.UpdateOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackage.model.UpdateOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackage.transform.ConfigureLogsRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.CreateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.CreateHarvestJobRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.CreateOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DeleteChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DeleteOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DescribeChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DescribeHarvestJobRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.DescribeOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.ListHarvestJobsRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.ListOriginEndpointsRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.RotateIngestEndpointCredentialsRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.UpdateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackage.transform.UpdateOriginEndpointRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MediaPackageAsyncClient}.
 *
 * @see MediaPackageAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaPackageAsyncClient implements MediaPackageAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMediaPackageAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaPackageAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * Changes the Channel's properities to configure log subscription
     *
     * @param configureLogsRequest
     *        the option to configure log subscription.
     * @return A Java Future containing the result of the ConfigureLogs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.ConfigureLogs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/ConfigureLogs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ConfigureLogsResponse> configureLogs(ConfigureLogsRequest configureLogsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(configureLogsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, configureLogsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ConfigureLogs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ConfigureLogsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ConfigureLogsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ConfigureLogsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ConfigureLogsRequest, ConfigureLogsResponse>()
                            .withOperationName("ConfigureLogs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ConfigureLogsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(configureLogsRequest));
            CompletableFuture<ConfigureLogsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a new Channel.
     *
     * @param createChannelRequest
     *        A new Channel configuration.
     * @return A Java Future containing the result of the CreateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.CreateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/CreateChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChannelResponse> createChannel(CreateChannelRequest createChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChannelRequest, CreateChannelResponse>()
                            .withOperationName("CreateChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createChannelRequest));
            CompletableFuture<CreateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a new HarvestJob record.
     *
     * @param createHarvestJobRequest
     *        Configuration parameters used to create a new HarvestJob.
     * @return A Java Future containing the result of the CreateHarvestJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.CreateHarvestJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/CreateHarvestJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHarvestJobResponse> createHarvestJob(CreateHarvestJobRequest createHarvestJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHarvestJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHarvestJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHarvestJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateHarvestJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateHarvestJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateHarvestJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHarvestJobRequest, CreateHarvestJobResponse>()
                            .withOperationName("CreateHarvestJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateHarvestJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createHarvestJobRequest));
            CompletableFuture<CreateHarvestJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Creates a new OriginEndpoint record.
     *
     * @param createOriginEndpointRequest
     *        Configuration parameters used to create a new OriginEndpoint.
     * @return A Java Future containing the result of the CreateOriginEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.CreateOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/CreateOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateOriginEndpointResponse> createOriginEndpoint(
            CreateOriginEndpointRequest createOriginEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOriginEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateOriginEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateOriginEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateOriginEndpointRequest, CreateOriginEndpointResponse>()
                            .withOperationName("CreateOriginEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateOriginEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createOriginEndpointRequest));
            CompletableFuture<CreateOriginEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes an existing Channel.
     *
     * @param deleteChannelRequest
     * @return A Java Future containing the result of the DeleteChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.DeleteChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/DeleteChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelResponse> deleteChannel(DeleteChannelRequest deleteChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelRequest, DeleteChannelResponse>()
                            .withOperationName("DeleteChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteChannelRequest));
            CompletableFuture<DeleteChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Deletes an existing OriginEndpoint.
     *
     * @param deleteOriginEndpointRequest
     * @return A Java Future containing the result of the DeleteOriginEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.DeleteOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/DeleteOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteOriginEndpointResponse> deleteOriginEndpoint(
            DeleteOriginEndpointRequest deleteOriginEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteOriginEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteOriginEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteOriginEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteOriginEndpointRequest, DeleteOriginEndpointResponse>()
                            .withOperationName("DeleteOriginEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteOriginEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteOriginEndpointRequest));
            CompletableFuture<DeleteOriginEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Gets details about a Channel.
     *
     * @param describeChannelRequest
     * @return A Java Future containing the result of the DescribeChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.DescribeChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/DescribeChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChannelResponse> describeChannel(DescribeChannelRequest describeChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChannelRequest, DescribeChannelResponse>()
                            .withOperationName("DescribeChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeChannelRequest));
            CompletableFuture<DescribeChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Gets details about an existing HarvestJob.
     *
     * @param describeHarvestJobRequest
     * @return A Java Future containing the result of the DescribeHarvestJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.DescribeHarvestJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/DescribeHarvestJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeHarvestJobResponse> describeHarvestJob(DescribeHarvestJobRequest describeHarvestJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeHarvestJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeHarvestJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeHarvestJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeHarvestJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeHarvestJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeHarvestJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeHarvestJobRequest, DescribeHarvestJobResponse>()
                            .withOperationName("DescribeHarvestJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeHarvestJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeHarvestJobRequest));
            CompletableFuture<DescribeHarvestJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Gets details about an existing OriginEndpoint.
     *
     * @param describeOriginEndpointRequest
     * @return A Java Future containing the result of the DescribeOriginEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.DescribeOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/DescribeOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeOriginEndpointResponse> describeOriginEndpoint(
            DescribeOriginEndpointRequest describeOriginEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOriginEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeOriginEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeOriginEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeOriginEndpointRequest, DescribeOriginEndpointResponse>()
                            .withOperationName("DescribeOriginEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeOriginEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeOriginEndpointRequest));
            CompletableFuture<DescribeOriginEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a collection of Channels.
     *
     * @param listChannelsRequest
     * @return A Java Future containing the result of the ListChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelsResponse> listChannels(ListChannelsRequest listChannelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChannelsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                            .withOperationName("ListChannels").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listChannelsRequest));
            CompletableFuture<ListChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a collection of HarvestJob records.
     *
     * @param listHarvestJobsRequest
     * @return A Java Future containing the result of the ListHarvestJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.ListHarvestJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/ListHarvestJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListHarvestJobsResponse> listHarvestJobs(ListHarvestJobsRequest listHarvestJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listHarvestJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHarvestJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHarvestJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListHarvestJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListHarvestJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListHarvestJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListHarvestJobsRequest, ListHarvestJobsResponse>()
                            .withOperationName("ListHarvestJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListHarvestJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listHarvestJobsRequest));
            CompletableFuture<ListHarvestJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Returns a collection of OriginEndpoint records.
     *
     * @param listOriginEndpointsRequest
     * @return A Java Future containing the result of the ListOriginEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.ListOriginEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/ListOriginEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOriginEndpointsResponse> listOriginEndpoints(
            ListOriginEndpointsRequest listOriginEndpointsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOriginEndpointsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOriginEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOriginEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOriginEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOriginEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOriginEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOriginEndpointsRequest, ListOriginEndpointsResponse>()
                            .withOperationName("ListOriginEndpoints").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListOriginEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listOriginEndpointsRequest));
            CompletableFuture<ListOriginEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the ListTagsForResource operation asynchronously.
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Rotate the IngestEndpoint's username and password, as specified by the IngestEndpoint's id.
     *
     * @param rotateIngestEndpointCredentialsRequest
     * @return A Java Future containing the result of the RotateIngestEndpointCredentials operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.RotateIngestEndpointCredentials
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/RotateIngestEndpointCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RotateIngestEndpointCredentialsResponse> rotateIngestEndpointCredentials(
            RotateIngestEndpointCredentialsRequest rotateIngestEndpointCredentialsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rotateIngestEndpointCredentialsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                rotateIngestEndpointCredentialsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RotateIngestEndpointCredentials");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RotateIngestEndpointCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RotateIngestEndpointCredentialsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RotateIngestEndpointCredentialsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RotateIngestEndpointCredentialsRequest, RotateIngestEndpointCredentialsResponse>()
                            .withOperationName("RotateIngestEndpointCredentials").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RotateIngestEndpointCredentialsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(rotateIngestEndpointCredentialsRequest));
            CompletableFuture<RotateIngestEndpointCredentialsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the TagResource operation asynchronously.
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Invokes the UntagResource operation asynchronously.
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates an existing Channel.
     *
     * @param updateChannelRequest
     *        Configuration parameters used to update the Channel.
     * @return A Java Future containing the result of the UpdateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.UpdateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/UpdateChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChannelResponse> updateChannel(UpdateChannelRequest updateChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChannelRequest, UpdateChannelResponse>()
                            .withOperationName("UpdateChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateChannelRequest));
            CompletableFuture<UpdateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Updates an existing OriginEndpoint.
     *
     * @param updateOriginEndpointRequest
     *        Configuration parameters used to update an existing OriginEndpoint.
     * @return A Java Future containing the result of the UpdateOriginEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnprocessableEntityException The parameters sent in the request are not valid.</li>
     *         <li>InternalServerErrorException An unexpected error occurred.</li>
     *         <li>ForbiddenException The client is not authorized to access the requested resource.</li>
     *         <li>NotFoundException The requested resource does not exist.</li>
     *         <li>ServiceUnavailableException An unexpected error occurred.</li>
     *         <li>TooManyRequestsException The client has exceeded their resource or throttling limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaPackageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaPackageAsyncClient.UpdateOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-2017-10-12/UpdateOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateOriginEndpointResponse> updateOriginEndpoint(
            UpdateOriginEndpointRequest updateOriginEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateOriginEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateOriginEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateOriginEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateOriginEndpointRequest, UpdateOriginEndpointResponse>()
                            .withOperationName("UpdateOriginEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateOriginEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateOriginEndpointRequest));
            CompletableFuture<UpdateOriginEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MediaPackageServiceClientConfiguration serviceClientConfiguration() {
        return new MediaPackageServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaPackageException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnprocessableEntityException")
                                .exceptionBuilderSupplier(UnprocessableEntityException::builder).httpStatusCode(422).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MediaPackageServiceClientConfigurationBuilder serviceConfigBuilder = new MediaPackageServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
