/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackagev2;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediapackagev2.internal.MediaPackageV2ServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mediapackagev2.model.AccessDeniedException;
import software.amazon.awssdk.services.mediapackagev2.model.ConflictException;
import software.amazon.awssdk.services.mediapackagev2.model.CreateChannelGroupRequest;
import software.amazon.awssdk.services.mediapackagev2.model.CreateChannelGroupResponse;
import software.amazon.awssdk.services.mediapackagev2.model.CreateChannelRequest;
import software.amazon.awssdk.services.mediapackagev2.model.CreateChannelResponse;
import software.amazon.awssdk.services.mediapackagev2.model.CreateOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackagev2.model.CreateOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelGroupRequest;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelGroupResponse;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelRequest;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteChannelResponse;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteOriginEndpointPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteOriginEndpointPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackagev2.model.DeleteOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelGroupRequest;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelGroupResponse;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelRequest;
import software.amazon.awssdk.services.mediapackagev2.model.GetChannelResponse;
import software.amazon.awssdk.services.mediapackagev2.model.GetOriginEndpointPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.GetOriginEndpointPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.GetOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackagev2.model.GetOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackagev2.model.InternalServerException;
import software.amazon.awssdk.services.mediapackagev2.model.ListChannelGroupsRequest;
import software.amazon.awssdk.services.mediapackagev2.model.ListChannelGroupsResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ListChannelsRequest;
import software.amazon.awssdk.services.mediapackagev2.model.ListChannelsResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ListOriginEndpointsRequest;
import software.amazon.awssdk.services.mediapackagev2.model.ListOriginEndpointsResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediapackagev2.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediapackagev2.model.MediaPackageV2Exception;
import software.amazon.awssdk.services.mediapackagev2.model.PutChannelPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.PutChannelPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.PutOriginEndpointPolicyRequest;
import software.amazon.awssdk.services.mediapackagev2.model.PutOriginEndpointPolicyResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ResourceNotFoundException;
import software.amazon.awssdk.services.mediapackagev2.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.mediapackagev2.model.TagResourceRequest;
import software.amazon.awssdk.services.mediapackagev2.model.TagResourceResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ThrottlingException;
import software.amazon.awssdk.services.mediapackagev2.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediapackagev2.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateChannelGroupRequest;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateChannelGroupResponse;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateChannelRequest;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateChannelResponse;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateOriginEndpointRequest;
import software.amazon.awssdk.services.mediapackagev2.model.UpdateOriginEndpointResponse;
import software.amazon.awssdk.services.mediapackagev2.model.ValidationException;
import software.amazon.awssdk.services.mediapackagev2.transform.CreateChannelGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.CreateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.CreateOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.DeleteChannelGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.DeleteChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.DeleteChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.DeleteOriginEndpointPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.DeleteOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.GetChannelGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.GetChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.GetChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.GetOriginEndpointPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.GetOriginEndpointRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.ListChannelGroupsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.ListOriginEndpointsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.PutChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.PutOriginEndpointPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.UpdateChannelGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.UpdateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediapackagev2.transform.UpdateOriginEndpointRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MediaPackageV2Client}.
 *
 * @see MediaPackageV2Client#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaPackageV2Client implements MediaPackageV2Client {
    private static final Logger log = Logger.loggerFor(DefaultMediaPackageV2Client.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaPackageV2Client(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Create a channel to start receiving content streams. The channel represents the input to MediaPackage for
     * incoming live content from an encoder such as AWS Elemental MediaLive. The channel receives content, and after
     * packaging it, outputs it through an origin endpoint to downstream devices (such as video players or CDNs) that
     * request the content. You can create only one channel with each request. We recommend that you spread out channels
     * between channel groups, such as putting redundant channels in the same AWS Region in different channel groups.
     * </p>
     *
     * @param createChannelRequest
     * @return Result of the CreateChannel operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.CreateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/CreateChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateChannelResponse createChannel(CreateChannelRequest createChannelRequest) throws ThrottlingException,
            ConflictException, InternalServerException, AccessDeniedException, ValidationException, ResourceNotFoundException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannel");

            return clientHandler.execute(new ClientExecutionParams<CreateChannelRequest, CreateChannelResponse>()
                    .withOperationName("CreateChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a channel group to group your channels and origin endpoints. A channel group is the top-level resource
     * that consists of channels and origin endpoints that are associated with it and that provides predictable URLs for
     * stream delivery. All channels and origin endpoints within the channel group are guaranteed to share the DNS. You
     * can create only one channel group with each request.
     * </p>
     *
     * @param createChannelGroupRequest
     * @return Result of the CreateChannelGroup operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.CreateChannelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/CreateChannelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateChannelGroupResponse createChannelGroup(CreateChannelGroupRequest createChannelGroupRequest)
            throws ThrottlingException, ConflictException, InternalServerException, AccessDeniedException, ValidationException,
            ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateChannelGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateChannelGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createChannelGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannelGroup");

            return clientHandler.execute(new ClientExecutionParams<CreateChannelGroupRequest, CreateChannelGroupResponse>()
                    .withOperationName("CreateChannelGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createChannelGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateChannelGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The endpoint is attached to a channel, and represents the output of the live content. You can associate multiple
     * endpoints to a single channel. Each endpoint gives players and downstream CDNs (such as Amazon CloudFront) access
     * to the content for playback. Content can't be served from a channel until it has an endpoint. You can create only
     * one endpoint with each request.
     * </p>
     *
     * @param createOriginEndpointRequest
     * @return Result of the CreateOriginEndpoint operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.CreateOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/CreateOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateOriginEndpointResponse createOriginEndpoint(CreateOriginEndpointRequest createOriginEndpointRequest)
            throws ThrottlingException, ConflictException, InternalServerException, AccessDeniedException, ValidationException,
            ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateOriginEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOriginEndpoint");

            return clientHandler.execute(new ClientExecutionParams<CreateOriginEndpointRequest, CreateOriginEndpointResponse>()
                    .withOperationName("CreateOriginEndpoint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createOriginEndpointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateOriginEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete a channel to stop AWS Elemental MediaPackage from receiving further content. You must delete the channel's
     * origin endpoints before you can delete the channel.
     * </p>
     *
     * @param deleteChannelRequest
     * @return Result of the DeleteChannel operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.DeleteChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/DeleteChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteChannelResponse deleteChannel(DeleteChannelRequest deleteChannelRequest) throws ThrottlingException,
            ConflictException, InternalServerException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannel");

            return clientHandler.execute(new ClientExecutionParams<DeleteChannelRequest, DeleteChannelResponse>()
                    .withOperationName("DeleteChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete a channel group. You must delete the channel group's channels and origin endpoints before you can delete
     * the channel group. If you delete a channel group, you'll lose access to the egress domain and will have to create
     * a new channel group to replace it.
     * </p>
     *
     * @param deleteChannelGroupRequest
     * @return Result of the DeleteChannelGroup operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.DeleteChannelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/DeleteChannelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteChannelGroupResponse deleteChannelGroup(DeleteChannelGroupRequest deleteChannelGroupRequest)
            throws ThrottlingException, ConflictException, InternalServerException, AccessDeniedException, ValidationException,
            AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteChannelGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteChannelGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChannelGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannelGroup");

            return clientHandler.execute(new ClientExecutionParams<DeleteChannelGroupRequest, DeleteChannelGroupResponse>()
                    .withOperationName("DeleteChannelGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteChannelGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteChannelGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete a channel policy.
     * </p>
     *
     * @param deleteChannelPolicyRequest
     * @return Result of the DeleteChannelPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.DeleteChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/DeleteChannelPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteChannelPolicyResponse deleteChannelPolicy(DeleteChannelPolicyRequest deleteChannelPolicyRequest)
            throws ThrottlingException, ConflictException, InternalServerException, AccessDeniedException, ValidationException,
            AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteChannelPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChannelPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannelPolicy");

            return clientHandler.execute(new ClientExecutionParams<DeleteChannelPolicyRequest, DeleteChannelPolicyResponse>()
                    .withOperationName("DeleteChannelPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteChannelPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteChannelPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Origin endpoints can serve content until they're deleted. Delete the endpoint if it should no longer respond to
     * playback requests. You must delete all endpoints from a channel before you can delete the channel.
     * </p>
     *
     * @param deleteOriginEndpointRequest
     * @return Result of the DeleteOriginEndpoint operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.DeleteOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/DeleteOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteOriginEndpointResponse deleteOriginEndpoint(DeleteOriginEndpointRequest deleteOriginEndpointRequest)
            throws ThrottlingException, InternalServerException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteOriginEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteOriginEndpoint");

            return clientHandler.execute(new ClientExecutionParams<DeleteOriginEndpointRequest, DeleteOriginEndpointResponse>()
                    .withOperationName("DeleteOriginEndpoint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteOriginEndpointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteOriginEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an origin endpoint policy.
     * </p>
     *
     * @param deleteOriginEndpointPolicyRequest
     * @return Result of the DeleteOriginEndpointPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.DeleteOriginEndpointPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/DeleteOriginEndpointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteOriginEndpointPolicyResponse deleteOriginEndpointPolicy(
            DeleteOriginEndpointPolicyRequest deleteOriginEndpointPolicyRequest) throws ThrottlingException, ConflictException,
            InternalServerException, AccessDeniedException, ValidationException, AwsServiceException, SdkClientException,
            MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteOriginEndpointPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteOriginEndpointPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteOriginEndpointPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteOriginEndpointPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteOriginEndpointPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteOriginEndpointPolicyRequest, DeleteOriginEndpointPolicyResponse>()
                            .withOperationName("DeleteOriginEndpointPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteOriginEndpointPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteOriginEndpointPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the specified channel that's configured in AWS Elemental MediaPackage, including the origin endpoints
     * that are associated with it.
     * </p>
     *
     * @param getChannelRequest
     * @return Result of the GetChannel operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.GetChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/GetChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetChannelResponse getChannel(GetChannelRequest getChannelRequest) throws ThrottlingException,
            InternalServerException, AccessDeniedException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannel");

            return clientHandler.execute(new ClientExecutionParams<GetChannelRequest, GetChannelResponse>()
                    .withOperationName("GetChannel").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getChannelRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the specified channel group that's configured in AWS Elemental MediaPackage, including the channels and
     * origin endpoints that are associated with it.
     * </p>
     *
     * @param getChannelGroupRequest
     * @return Result of the GetChannelGroup operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.GetChannelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/GetChannelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetChannelGroupResponse getChannelGroup(GetChannelGroupRequest getChannelGroupRequest) throws ThrottlingException,
            InternalServerException, AccessDeniedException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetChannelGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetChannelGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChannelGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannelGroup");

            return clientHandler.execute(new ClientExecutionParams<GetChannelGroupRequest, GetChannelGroupResponse>()
                    .withOperationName("GetChannelGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getChannelGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetChannelGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the specified channel policy that's configured in AWS Elemental MediaPackage. With policies, you can
     * specify who has access to AWS resources and what actions they can perform on those resources.
     * </p>
     *
     * @param getChannelPolicyRequest
     * @return Result of the GetChannelPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.GetChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/GetChannelPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetChannelPolicyResponse getChannelPolicy(GetChannelPolicyRequest getChannelPolicyRequest) throws ThrottlingException,
            InternalServerException, AccessDeniedException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetChannelPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChannelPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannelPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetChannelPolicyRequest, GetChannelPolicyResponse>()
                    .withOperationName("GetChannelPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getChannelPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetChannelPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the specified origin endpoint that's configured in AWS Elemental MediaPackage to obtain its playback
     * URL and to view the packaging settings that it's currently using.
     * </p>
     *
     * @param getOriginEndpointRequest
     * @return Result of the GetOriginEndpoint operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.GetOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/GetOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOriginEndpointResponse getOriginEndpoint(GetOriginEndpointRequest getOriginEndpointRequest)
            throws ThrottlingException, InternalServerException, AccessDeniedException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetOriginEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOriginEndpoint");

            return clientHandler.execute(new ClientExecutionParams<GetOriginEndpointRequest, GetOriginEndpointResponse>()
                    .withOperationName("GetOriginEndpoint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getOriginEndpointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetOriginEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the specified origin endpoint policy that's configured in AWS Elemental MediaPackage.
     * </p>
     *
     * @param getOriginEndpointPolicyRequest
     * @return Result of the GetOriginEndpointPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.GetOriginEndpointPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/GetOriginEndpointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOriginEndpointPolicyResponse getOriginEndpointPolicy(GetOriginEndpointPolicyRequest getOriginEndpointPolicyRequest)
            throws ThrottlingException, InternalServerException, AccessDeniedException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOriginEndpointPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetOriginEndpointPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOriginEndpointPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOriginEndpointPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOriginEndpointPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<GetOriginEndpointPolicyRequest, GetOriginEndpointPolicyResponse>()
                            .withOperationName("GetOriginEndpointPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getOriginEndpointPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetOriginEndpointPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves all channel groups that are configured in AWS Elemental MediaPackage, including the channels and origin
     * endpoints that are associated with it.
     * </p>
     *
     * @param listChannelGroupsRequest
     * @return Result of the ListChannelGroups operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.ListChannelGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/ListChannelGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListChannelGroupsResponse listChannelGroups(ListChannelGroupsRequest listChannelGroupsRequest)
            throws ThrottlingException, InternalServerException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListChannelGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListChannelGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChannelGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannelGroups");

            return clientHandler.execute(new ClientExecutionParams<ListChannelGroupsRequest, ListChannelGroupsResponse>()
                    .withOperationName("ListChannelGroups").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listChannelGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListChannelGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves all channels in a specific channel group that are configured in AWS Elemental MediaPackage, including
     * the origin endpoints that are associated with it.
     * </p>
     *
     * @param listChannelsRequest
     * @return Result of the ListChannels operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/ListChannels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListChannelsResponse listChannels(ListChannelsRequest listChannelsRequest) throws ThrottlingException,
            InternalServerException, AccessDeniedException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListChannelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChannelsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannels");

            return clientHandler.execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                    .withOperationName("ListChannels").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listChannelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListChannelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves all origin endpoints in a specific channel that are configured in AWS Elemental MediaPackage.
     * </p>
     *
     * @param listOriginEndpointsRequest
     * @return Result of the ListOriginEndpoints operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.ListOriginEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/ListOriginEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListOriginEndpointsResponse listOriginEndpoints(ListOriginEndpointsRequest listOriginEndpointsRequest)
            throws ThrottlingException, InternalServerException, AccessDeniedException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOriginEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListOriginEndpointsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOriginEndpointsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOriginEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOriginEndpoints");

            return clientHandler.execute(new ClientExecutionParams<ListOriginEndpointsRequest, ListOriginEndpointsResponse>()
                    .withOperationName("ListOriginEndpoints").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listOriginEndpointsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListOriginEndpointsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags assigned to a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Attaches an IAM policy to the specified channel. With policies, you can specify who has access to AWS resources
     * and what actions they can perform on those resources. You can attach only one policy with each request.
     * </p>
     *
     * @param putChannelPolicyRequest
     * @return Result of the PutChannelPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.PutChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/PutChannelPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutChannelPolicyResponse putChannelPolicy(PutChannelPolicyRequest putChannelPolicyRequest) throws ThrottlingException,
            ConflictException, InternalServerException, AccessDeniedException, ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutChannelPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putChannelPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutChannelPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutChannelPolicyRequest, PutChannelPolicyResponse>()
                    .withOperationName("PutChannelPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putChannelPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutChannelPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Attaches an IAM policy to the specified origin endpoint. You can attach only one policy with each request.
     * </p>
     *
     * @param putOriginEndpointPolicyRequest
     * @return Result of the PutOriginEndpointPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.PutOriginEndpointPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/PutOriginEndpointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutOriginEndpointPolicyResponse putOriginEndpointPolicy(PutOriginEndpointPolicyRequest putOriginEndpointPolicyRequest)
            throws ThrottlingException, ConflictException, InternalServerException, AccessDeniedException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutOriginEndpointPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutOriginEndpointPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putOriginEndpointPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putOriginEndpointPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutOriginEndpointPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<PutOriginEndpointPolicyRequest, PutOriginEndpointPolicyResponse>()
                            .withOperationName("PutOriginEndpointPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putOriginEndpointPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutOriginEndpointPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one of more tags (key-value pairs) to the specified MediaPackage resource.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions, by
     * granting a user permission to access or change only resources with certain tag values. You can use the
     * TagResource operation with a resource that already has tags. If you specify a new tag key for the resource, this
     * tag is appended to the list of tags associated with the resource. If you specify a tag key that is already
     * associated with the resource, the new tag value that you specify replaces the previous value for that tag.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update the specified channel. You can edit if MediaPackage sends ingest or egress access logs to the CloudWatch
     * log group, if content will be encrypted, the description on a channel, and your channel's policy settings. You
     * can't edit the name of the channel or CloudFront distribution details.
     * </p>
     * <p>
     * Any edits you make that impact the video output may not be reflected for a few minutes.
     * </p>
     *
     * @param updateChannelRequest
     * @return Result of the UpdateChannel operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.UpdateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/UpdateChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateChannelResponse updateChannel(UpdateChannelRequest updateChannelRequest) throws ThrottlingException,
            ConflictException, InternalServerException, AccessDeniedException, ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannel");

            return clientHandler.execute(new ClientExecutionParams<UpdateChannelRequest, UpdateChannelResponse>()
                    .withOperationName("UpdateChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update the specified channel group. You can edit the description on a channel group for easier identification
     * later from the AWS Elemental MediaPackage console. You can't edit the name of the channel group.
     * </p>
     * <p>
     * Any edits you make that impact the video output may not be reflected for a few minutes.
     * </p>
     *
     * @param updateChannelGroupRequest
     * @return Result of the UpdateChannelGroup operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.UpdateChannelGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/UpdateChannelGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateChannelGroupResponse updateChannelGroup(UpdateChannelGroupRequest updateChannelGroupRequest)
            throws ThrottlingException, ConflictException, InternalServerException, AccessDeniedException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateChannelGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateChannelGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateChannelGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannelGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdateChannelGroupRequest, UpdateChannelGroupResponse>()
                    .withOperationName("UpdateChannelGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateChannelGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateChannelGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update the specified origin endpoint. Edit the packaging preferences on an endpoint to optimize the viewing
     * experience. You can't edit the name of the endpoint.
     * </p>
     * <p>
     * Any edits you make that impact the video output may not be reflected for a few minutes.
     * </p>
     *
     * @param updateOriginEndpointRequest
     * @return Result of the UpdateOriginEndpoint operation returned by the service.
     * @throws ThrottlingException
     *         The request throughput limit was exceeded.
     * @throws ConflictException
     *         Updating or deleting this resource can cause an inconsistent state.
     * @throws InternalServerException
     *         Indicates that an error from the service occurred while trying to process a request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see Access Management in the IAM User Guide.
     * @throws ValidationException
     *         The input failed to meet the constraints specified by the AWS service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageV2Client.UpdateOriginEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/UpdateOriginEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateOriginEndpointResponse updateOriginEndpoint(UpdateOriginEndpointRequest updateOriginEndpointRequest)
            throws ThrottlingException, ConflictException, InternalServerException, AccessDeniedException, ValidationException,
            ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            MediaPackageV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateOriginEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateOriginEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateOriginEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateOriginEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackageV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateOriginEndpoint");

            return clientHandler.execute(new ClientExecutionParams<UpdateOriginEndpointRequest, UpdateOriginEndpointResponse>()
                    .withOperationName("UpdateOriginEndpoint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateOriginEndpointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateOriginEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MediaPackageV2ServiceClientConfigurationBuilder serviceConfigBuilder = new MediaPackageV2ServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaPackageV2Exception::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final MediaPackageV2ServiceClientConfiguration serviceClientConfiguration() {
        return new MediaPackageV2ServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
