/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackagev2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Create a DASH manifest configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateDashManifestConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CreateDashManifestConfiguration.Builder, CreateDashManifestConfiguration> {
    private static final SdkField<String> MANIFEST_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ManifestName").getter(getter(CreateDashManifestConfiguration::manifestName))
            .setter(setter(Builder::manifestName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManifestName").build()).build();

    private static final SdkField<Integer> MANIFEST_WINDOW_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ManifestWindowSeconds").getter(getter(CreateDashManifestConfiguration::manifestWindowSeconds))
            .setter(setter(Builder::manifestWindowSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManifestWindowSeconds").build())
            .build();

    private static final SdkField<FilterConfiguration> FILTER_CONFIGURATION_FIELD = SdkField
            .<FilterConfiguration> builder(MarshallingType.SDK_POJO).memberName("FilterConfiguration")
            .getter(getter(CreateDashManifestConfiguration::filterConfiguration)).setter(setter(Builder::filterConfiguration))
            .constructor(FilterConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FilterConfiguration").build())
            .build();

    private static final SdkField<Integer> MIN_UPDATE_PERIOD_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinUpdatePeriodSeconds").getter(getter(CreateDashManifestConfiguration::minUpdatePeriodSeconds))
            .setter(setter(Builder::minUpdatePeriodSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinUpdatePeriodSeconds").build())
            .build();

    private static final SdkField<Integer> MIN_BUFFER_TIME_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinBufferTimeSeconds").getter(getter(CreateDashManifestConfiguration::minBufferTimeSeconds))
            .setter(setter(Builder::minBufferTimeSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinBufferTimeSeconds").build())
            .build();

    private static final SdkField<Integer> SUGGESTED_PRESENTATION_DELAY_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("SuggestedPresentationDelaySeconds")
            .getter(getter(CreateDashManifestConfiguration::suggestedPresentationDelaySeconds))
            .setter(setter(Builder::suggestedPresentationDelaySeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SuggestedPresentationDelaySeconds")
                    .build()).build();

    private static final SdkField<String> SEGMENT_TEMPLATE_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SegmentTemplateFormat").getter(getter(CreateDashManifestConfiguration::segmentTemplateFormatAsString))
            .setter(setter(Builder::segmentTemplateFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SegmentTemplateFormat").build())
            .build();

    private static final SdkField<List<String>> PERIOD_TRIGGERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PeriodTriggers")
            .getter(getter(CreateDashManifestConfiguration::periodTriggersAsStrings))
            .setter(setter(Builder::periodTriggersWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PeriodTriggers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ScteDash> SCTE_DASH_FIELD = SdkField.<ScteDash> builder(MarshallingType.SDK_POJO)
            .memberName("ScteDash").getter(getter(CreateDashManifestConfiguration::scteDash)).setter(setter(Builder::scteDash))
            .constructor(ScteDash::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScteDash").build()).build();

    private static final SdkField<String> DRM_SIGNALING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DrmSignaling").getter(getter(CreateDashManifestConfiguration::drmSignalingAsString))
            .setter(setter(Builder::drmSignaling))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DrmSignaling").build()).build();

    private static final SdkField<DashUtcTiming> UTC_TIMING_FIELD = SdkField.<DashUtcTiming> builder(MarshallingType.SDK_POJO)
            .memberName("UtcTiming").getter(getter(CreateDashManifestConfiguration::utcTiming))
            .setter(setter(Builder::utcTiming)).constructor(DashUtcTiming::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UtcTiming").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MANIFEST_NAME_FIELD,
            MANIFEST_WINDOW_SECONDS_FIELD, FILTER_CONFIGURATION_FIELD, MIN_UPDATE_PERIOD_SECONDS_FIELD,
            MIN_BUFFER_TIME_SECONDS_FIELD, SUGGESTED_PRESENTATION_DELAY_SECONDS_FIELD, SEGMENT_TEMPLATE_FORMAT_FIELD,
            PERIOD_TRIGGERS_FIELD, SCTE_DASH_FIELD, DRM_SIGNALING_FIELD, UTC_TIMING_FIELD));

    private static final long serialVersionUID = 1L;

    private final String manifestName;

    private final Integer manifestWindowSeconds;

    private final FilterConfiguration filterConfiguration;

    private final Integer minUpdatePeriodSeconds;

    private final Integer minBufferTimeSeconds;

    private final Integer suggestedPresentationDelaySeconds;

    private final String segmentTemplateFormat;

    private final List<String> periodTriggers;

    private final ScteDash scteDash;

    private final String drmSignaling;

    private final DashUtcTiming utcTiming;

    private CreateDashManifestConfiguration(BuilderImpl builder) {
        this.manifestName = builder.manifestName;
        this.manifestWindowSeconds = builder.manifestWindowSeconds;
        this.filterConfiguration = builder.filterConfiguration;
        this.minUpdatePeriodSeconds = builder.minUpdatePeriodSeconds;
        this.minBufferTimeSeconds = builder.minBufferTimeSeconds;
        this.suggestedPresentationDelaySeconds = builder.suggestedPresentationDelaySeconds;
        this.segmentTemplateFormat = builder.segmentTemplateFormat;
        this.periodTriggers = builder.periodTriggers;
        this.scteDash = builder.scteDash;
        this.drmSignaling = builder.drmSignaling;
        this.utcTiming = builder.utcTiming;
    }

    /**
     * <p>
     * A short string that's appended to the endpoint URL. The child manifest name creates a unique path to this
     * endpoint.
     * </p>
     * 
     * @return A short string that's appended to the endpoint URL. The child manifest name creates a unique path to this
     *         endpoint.
     */
    public final String manifestName() {
        return manifestName;
    }

    /**
     * <p>
     * The total duration (in seconds) of the manifest's content.
     * </p>
     * 
     * @return The total duration (in seconds) of the manifest's content.
     */
    public final Integer manifestWindowSeconds() {
        return manifestWindowSeconds;
    }

    /**
     * Returns the value of the FilterConfiguration property for this object.
     * 
     * @return The value of the FilterConfiguration property for this object.
     */
    public final FilterConfiguration filterConfiguration() {
        return filterConfiguration;
    }

    /**
     * <p>
     * Minimum amount of time (in seconds) that the player should wait before requesting updates to the manifest.
     * </p>
     * 
     * @return Minimum amount of time (in seconds) that the player should wait before requesting updates to the
     *         manifest.
     */
    public final Integer minUpdatePeriodSeconds() {
        return minUpdatePeriodSeconds;
    }

    /**
     * <p>
     * Minimum amount of content (in seconds) that a player must keep available in the buffer.
     * </p>
     * 
     * @return Minimum amount of content (in seconds) that a player must keep available in the buffer.
     */
    public final Integer minBufferTimeSeconds() {
        return minBufferTimeSeconds;
    }

    /**
     * <p>
     * The amount of time (in seconds) that the player should be from the end of the manifest.
     * </p>
     * 
     * @return The amount of time (in seconds) that the player should be from the end of the manifest.
     */
    public final Integer suggestedPresentationDelaySeconds() {
        return suggestedPresentationDelaySeconds;
    }

    /**
     * <p>
     * Determines the type of variable used in the <code>media</code> URL of the <code>SegmentTemplate</code> tag in the
     * manifest. Also specifies if segment timeline information is included in <code>SegmentTimeline</code> or
     * <code>SegmentTemplate</code>.
     * </p>
     * <p>
     * Value description:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NUMBER_WITH_TIMELINE</code> - The <code>$Number$</code> variable is used in the <code>media</code> URL. The
     * value of this variable is the sequential number of the segment. A full <code>SegmentTimeline</code> object is
     * presented in each <code>SegmentTemplate</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #segmentTemplateFormat} will return {@link DashSegmentTemplateFormat#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #segmentTemplateFormatAsString}.
     * </p>
     * 
     * @return Determines the type of variable used in the <code>media</code> URL of the <code>SegmentTemplate</code>
     *         tag in the manifest. Also specifies if segment timeline information is included in
     *         <code>SegmentTimeline</code> or <code>SegmentTemplate</code>.</p>
     *         <p>
     *         Value description:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NUMBER_WITH_TIMELINE</code> - The <code>$Number$</code> variable is used in the <code>media</code>
     *         URL. The value of this variable is the sequential number of the segment. A full
     *         <code>SegmentTimeline</code> object is presented in each <code>SegmentTemplate</code>.
     *         </p>
     *         </li>
     * @see DashSegmentTemplateFormat
     */
    public final DashSegmentTemplateFormat segmentTemplateFormat() {
        return DashSegmentTemplateFormat.fromValue(segmentTemplateFormat);
    }

    /**
     * <p>
     * Determines the type of variable used in the <code>media</code> URL of the <code>SegmentTemplate</code> tag in the
     * manifest. Also specifies if segment timeline information is included in <code>SegmentTimeline</code> or
     * <code>SegmentTemplate</code>.
     * </p>
     * <p>
     * Value description:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NUMBER_WITH_TIMELINE</code> - The <code>$Number$</code> variable is used in the <code>media</code> URL. The
     * value of this variable is the sequential number of the segment. A full <code>SegmentTimeline</code> object is
     * presented in each <code>SegmentTemplate</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #segmentTemplateFormat} will return {@link DashSegmentTemplateFormat#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #segmentTemplateFormatAsString}.
     * </p>
     * 
     * @return Determines the type of variable used in the <code>media</code> URL of the <code>SegmentTemplate</code>
     *         tag in the manifest. Also specifies if segment timeline information is included in
     *         <code>SegmentTimeline</code> or <code>SegmentTemplate</code>.</p>
     *         <p>
     *         Value description:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NUMBER_WITH_TIMELINE</code> - The <code>$Number$</code> variable is used in the <code>media</code>
     *         URL. The value of this variable is the sequential number of the segment. A full
     *         <code>SegmentTimeline</code> object is presented in each <code>SegmentTemplate</code>.
     *         </p>
     *         </li>
     * @see DashSegmentTemplateFormat
     */
    public final String segmentTemplateFormatAsString() {
        return segmentTemplateFormat;
    }

    /**
     * <p>
     * A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into multiple
     * periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create periods in the output
     * manifest that correspond to SCTE-35 ad markers in the input source. Leave this value empty to indicate that the
     * manifest is contained all in one period. For more information about periods in the DASH manifest, see <a
     * href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH in AWS
     * Elemental MediaPackage</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPeriodTriggers} method.
     * </p>
     * 
     * @return A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into
     *         multiple periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create periods
     *         in the output manifest that correspond to SCTE-35 ad markers in the input source. Leave this value empty
     *         to indicate that the manifest is contained all in one period. For more information about periods in the
     *         DASH manifest, see <a
     *         href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH in
     *         AWS Elemental MediaPackage</a>.
     */
    public final List<DashPeriodTrigger> periodTriggers() {
        return DashPeriodTriggersCopier.copyStringToEnum(periodTriggers);
    }

    /**
     * For responses, this returns true if the service returned a value for the PeriodTriggers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPeriodTriggers() {
        return periodTriggers != null && !(periodTriggers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into multiple
     * periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create periods in the output
     * manifest that correspond to SCTE-35 ad markers in the input source. Leave this value empty to indicate that the
     * manifest is contained all in one period. For more information about periods in the DASH manifest, see <a
     * href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH in AWS
     * Elemental MediaPackage</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPeriodTriggers} method.
     * </p>
     * 
     * @return A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into
     *         multiple periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create periods
     *         in the output manifest that correspond to SCTE-35 ad markers in the input source. Leave this value empty
     *         to indicate that the manifest is contained all in one period. For more information about periods in the
     *         DASH manifest, see <a
     *         href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH in
     *         AWS Elemental MediaPackage</a>.
     */
    public final List<String> periodTriggersAsStrings() {
        return periodTriggers;
    }

    /**
     * <p>
     * The SCTE configuration.
     * </p>
     * 
     * @return The SCTE configuration.
     */
    public final ScteDash scteDash() {
        return scteDash;
    }

    /**
     * <p>
     * Determines how the DASH manifest signals the DRM content.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #drmSignaling} will
     * return {@link DashDrmSignaling#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #drmSignalingAsString}.
     * </p>
     * 
     * @return Determines how the DASH manifest signals the DRM content.
     * @see DashDrmSignaling
     */
    public final DashDrmSignaling drmSignaling() {
        return DashDrmSignaling.fromValue(drmSignaling);
    }

    /**
     * <p>
     * Determines how the DASH manifest signals the DRM content.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #drmSignaling} will
     * return {@link DashDrmSignaling#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #drmSignalingAsString}.
     * </p>
     * 
     * @return Determines how the DASH manifest signals the DRM content.
     * @see DashDrmSignaling
     */
    public final String drmSignalingAsString() {
        return drmSignaling;
    }

    /**
     * <p>
     * Determines the type of UTC timing included in the DASH Media Presentation Description (MPD).
     * </p>
     * 
     * @return Determines the type of UTC timing included in the DASH Media Presentation Description (MPD).
     */
    public final DashUtcTiming utcTiming() {
        return utcTiming;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(manifestName());
        hashCode = 31 * hashCode + Objects.hashCode(manifestWindowSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(filterConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(minUpdatePeriodSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(minBufferTimeSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(suggestedPresentationDelaySeconds());
        hashCode = 31 * hashCode + Objects.hashCode(segmentTemplateFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasPeriodTriggers() ? periodTriggersAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(scteDash());
        hashCode = 31 * hashCode + Objects.hashCode(drmSignalingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(utcTiming());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateDashManifestConfiguration)) {
            return false;
        }
        CreateDashManifestConfiguration other = (CreateDashManifestConfiguration) obj;
        return Objects.equals(manifestName(), other.manifestName())
                && Objects.equals(manifestWindowSeconds(), other.manifestWindowSeconds())
                && Objects.equals(filterConfiguration(), other.filterConfiguration())
                && Objects.equals(minUpdatePeriodSeconds(), other.minUpdatePeriodSeconds())
                && Objects.equals(minBufferTimeSeconds(), other.minBufferTimeSeconds())
                && Objects.equals(suggestedPresentationDelaySeconds(), other.suggestedPresentationDelaySeconds())
                && Objects.equals(segmentTemplateFormatAsString(), other.segmentTemplateFormatAsString())
                && hasPeriodTriggers() == other.hasPeriodTriggers()
                && Objects.equals(periodTriggersAsStrings(), other.periodTriggersAsStrings())
                && Objects.equals(scteDash(), other.scteDash())
                && Objects.equals(drmSignalingAsString(), other.drmSignalingAsString())
                && Objects.equals(utcTiming(), other.utcTiming());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateDashManifestConfiguration").add("ManifestName", manifestName())
                .add("ManifestWindowSeconds", manifestWindowSeconds()).add("FilterConfiguration", filterConfiguration())
                .add("MinUpdatePeriodSeconds", minUpdatePeriodSeconds()).add("MinBufferTimeSeconds", minBufferTimeSeconds())
                .add("SuggestedPresentationDelaySeconds", suggestedPresentationDelaySeconds())
                .add("SegmentTemplateFormat", segmentTemplateFormatAsString())
                .add("PeriodTriggers", hasPeriodTriggers() ? periodTriggersAsStrings() : null).add("ScteDash", scteDash())
                .add("DrmSignaling", drmSignalingAsString()).add("UtcTiming", utcTiming()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ManifestName":
            return Optional.ofNullable(clazz.cast(manifestName()));
        case "ManifestWindowSeconds":
            return Optional.ofNullable(clazz.cast(manifestWindowSeconds()));
        case "FilterConfiguration":
            return Optional.ofNullable(clazz.cast(filterConfiguration()));
        case "MinUpdatePeriodSeconds":
            return Optional.ofNullable(clazz.cast(minUpdatePeriodSeconds()));
        case "MinBufferTimeSeconds":
            return Optional.ofNullable(clazz.cast(minBufferTimeSeconds()));
        case "SuggestedPresentationDelaySeconds":
            return Optional.ofNullable(clazz.cast(suggestedPresentationDelaySeconds()));
        case "SegmentTemplateFormat":
            return Optional.ofNullable(clazz.cast(segmentTemplateFormatAsString()));
        case "PeriodTriggers":
            return Optional.ofNullable(clazz.cast(periodTriggersAsStrings()));
        case "ScteDash":
            return Optional.ofNullable(clazz.cast(scteDash()));
        case "DrmSignaling":
            return Optional.ofNullable(clazz.cast(drmSignalingAsString()));
        case "UtcTiming":
            return Optional.ofNullable(clazz.cast(utcTiming()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateDashManifestConfiguration, T> g) {
        return obj -> g.apply((CreateDashManifestConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CreateDashManifestConfiguration> {
        /**
         * <p>
         * A short string that's appended to the endpoint URL. The child manifest name creates a unique path to this
         * endpoint.
         * </p>
         * 
         * @param manifestName
         *        A short string that's appended to the endpoint URL. The child manifest name creates a unique path to
         *        this endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manifestName(String manifestName);

        /**
         * <p>
         * The total duration (in seconds) of the manifest's content.
         * </p>
         * 
         * @param manifestWindowSeconds
         *        The total duration (in seconds) of the manifest's content.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manifestWindowSeconds(Integer manifestWindowSeconds);

        /**
         * Sets the value of the FilterConfiguration property for this object.
         *
         * @param filterConfiguration
         *        The new value for the FilterConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterConfiguration(FilterConfiguration filterConfiguration);

        /**
         * Sets the value of the FilterConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link FilterConfiguration.Builder} avoiding the
         * need to create one manually via {@link FilterConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FilterConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #filterConfiguration(FilterConfiguration)}.
         * 
         * @param filterConfiguration
         *        a consumer that will call methods on {@link FilterConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filterConfiguration(FilterConfiguration)
         */
        default Builder filterConfiguration(Consumer<FilterConfiguration.Builder> filterConfiguration) {
            return filterConfiguration(FilterConfiguration.builder().applyMutation(filterConfiguration).build());
        }

        /**
         * <p>
         * Minimum amount of time (in seconds) that the player should wait before requesting updates to the manifest.
         * </p>
         * 
         * @param minUpdatePeriodSeconds
         *        Minimum amount of time (in seconds) that the player should wait before requesting updates to the
         *        manifest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minUpdatePeriodSeconds(Integer minUpdatePeriodSeconds);

        /**
         * <p>
         * Minimum amount of content (in seconds) that a player must keep available in the buffer.
         * </p>
         * 
         * @param minBufferTimeSeconds
         *        Minimum amount of content (in seconds) that a player must keep available in the buffer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minBufferTimeSeconds(Integer minBufferTimeSeconds);

        /**
         * <p>
         * The amount of time (in seconds) that the player should be from the end of the manifest.
         * </p>
         * 
         * @param suggestedPresentationDelaySeconds
         *        The amount of time (in seconds) that the player should be from the end of the manifest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suggestedPresentationDelaySeconds(Integer suggestedPresentationDelaySeconds);

        /**
         * <p>
         * Determines the type of variable used in the <code>media</code> URL of the <code>SegmentTemplate</code> tag in
         * the manifest. Also specifies if segment timeline information is included in <code>SegmentTimeline</code> or
         * <code>SegmentTemplate</code>.
         * </p>
         * <p>
         * Value description:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NUMBER_WITH_TIMELINE</code> - The <code>$Number$</code> variable is used in the <code>media</code> URL.
         * The value of this variable is the sequential number of the segment. A full <code>SegmentTimeline</code>
         * object is presented in each <code>SegmentTemplate</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param segmentTemplateFormat
         *        Determines the type of variable used in the <code>media</code> URL of the <code>SegmentTemplate</code>
         *        tag in the manifest. Also specifies if segment timeline information is included in
         *        <code>SegmentTimeline</code> or <code>SegmentTemplate</code>.</p>
         *        <p>
         *        Value description:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NUMBER_WITH_TIMELINE</code> - The <code>$Number$</code> variable is used in the
         *        <code>media</code> URL. The value of this variable is the sequential number of the segment. A full
         *        <code>SegmentTimeline</code> object is presented in each <code>SegmentTemplate</code>.
         *        </p>
         *        </li>
         * @see DashSegmentTemplateFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DashSegmentTemplateFormat
         */
        Builder segmentTemplateFormat(String segmentTemplateFormat);

        /**
         * <p>
         * Determines the type of variable used in the <code>media</code> URL of the <code>SegmentTemplate</code> tag in
         * the manifest. Also specifies if segment timeline information is included in <code>SegmentTimeline</code> or
         * <code>SegmentTemplate</code>.
         * </p>
         * <p>
         * Value description:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NUMBER_WITH_TIMELINE</code> - The <code>$Number$</code> variable is used in the <code>media</code> URL.
         * The value of this variable is the sequential number of the segment. A full <code>SegmentTimeline</code>
         * object is presented in each <code>SegmentTemplate</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param segmentTemplateFormat
         *        Determines the type of variable used in the <code>media</code> URL of the <code>SegmentTemplate</code>
         *        tag in the manifest. Also specifies if segment timeline information is included in
         *        <code>SegmentTimeline</code> or <code>SegmentTemplate</code>.</p>
         *        <p>
         *        Value description:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NUMBER_WITH_TIMELINE</code> - The <code>$Number$</code> variable is used in the
         *        <code>media</code> URL. The value of this variable is the sequential number of the segment. A full
         *        <code>SegmentTimeline</code> object is presented in each <code>SegmentTemplate</code>.
         *        </p>
         *        </li>
         * @see DashSegmentTemplateFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DashSegmentTemplateFormat
         */
        Builder segmentTemplateFormat(DashSegmentTemplateFormat segmentTemplateFormat);

        /**
         * <p>
         * A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into
         * multiple periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create periods in
         * the output manifest that correspond to SCTE-35 ad markers in the input source. Leave this value empty to
         * indicate that the manifest is contained all in one period. For more information about periods in the DASH
         * manifest, see <a
         * href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH in AWS
         * Elemental MediaPackage</a>.
         * </p>
         * 
         * @param periodTriggers
         *        A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into
         *        multiple periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create
         *        periods in the output manifest that correspond to SCTE-35 ad markers in the input source. Leave this
         *        value empty to indicate that the manifest is contained all in one period. For more information about
         *        periods in the DASH manifest, see <a
         *        href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH
         *        in AWS Elemental MediaPackage</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder periodTriggersWithStrings(Collection<String> periodTriggers);

        /**
         * <p>
         * A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into
         * multiple periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create periods in
         * the output manifest that correspond to SCTE-35 ad markers in the input source. Leave this value empty to
         * indicate that the manifest is contained all in one period. For more information about periods in the DASH
         * manifest, see <a
         * href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH in AWS
         * Elemental MediaPackage</a>.
         * </p>
         * 
         * @param periodTriggers
         *        A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into
         *        multiple periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create
         *        periods in the output manifest that correspond to SCTE-35 ad markers in the input source. Leave this
         *        value empty to indicate that the manifest is contained all in one period. For more information about
         *        periods in the DASH manifest, see <a
         *        href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH
         *        in AWS Elemental MediaPackage</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder periodTriggersWithStrings(String... periodTriggers);

        /**
         * <p>
         * A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into
         * multiple periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create periods in
         * the output manifest that correspond to SCTE-35 ad markers in the input source. Leave this value empty to
         * indicate that the manifest is contained all in one period. For more information about periods in the DASH
         * manifest, see <a
         * href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH in AWS
         * Elemental MediaPackage</a>.
         * </p>
         * 
         * @param periodTriggers
         *        A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into
         *        multiple periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create
         *        periods in the output manifest that correspond to SCTE-35 ad markers in the input source. Leave this
         *        value empty to indicate that the manifest is contained all in one period. For more information about
         *        periods in the DASH manifest, see <a
         *        href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH
         *        in AWS Elemental MediaPackage</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder periodTriggers(Collection<DashPeriodTrigger> periodTriggers);

        /**
         * <p>
         * A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into
         * multiple periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create periods in
         * the output manifest that correspond to SCTE-35 ad markers in the input source. Leave this value empty to
         * indicate that the manifest is contained all in one period. For more information about periods in the DASH
         * manifest, see <a
         * href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH in AWS
         * Elemental MediaPackage</a>.
         * </p>
         * 
         * @param periodTriggers
         *        A list of triggers that controls when AWS Elemental MediaPackage separates the MPEG-DASH manifest into
         *        multiple periods. Type <code>ADS</code> to indicate that AWS Elemental MediaPackage must create
         *        periods in the output manifest that correspond to SCTE-35 ad markers in the input source. Leave this
         *        value empty to indicate that the manifest is contained all in one period. For more information about
         *        periods in the DASH manifest, see <a
         *        href="https://docs.aws.amazon.com/mediapackage/latest/userguide/multi-period.html">Multi-period DASH
         *        in AWS Elemental MediaPackage</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder periodTriggers(DashPeriodTrigger... periodTriggers);

        /**
         * <p>
         * The SCTE configuration.
         * </p>
         * 
         * @param scteDash
         *        The SCTE configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scteDash(ScteDash scteDash);

        /**
         * <p>
         * The SCTE configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link ScteDash.Builder} avoiding the need to
         * create one manually via {@link ScteDash#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ScteDash.Builder#build()} is called immediately and its result is
         * passed to {@link #scteDash(ScteDash)}.
         * 
         * @param scteDash
         *        a consumer that will call methods on {@link ScteDash.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scteDash(ScteDash)
         */
        default Builder scteDash(Consumer<ScteDash.Builder> scteDash) {
            return scteDash(ScteDash.builder().applyMutation(scteDash).build());
        }

        /**
         * <p>
         * Determines how the DASH manifest signals the DRM content.
         * </p>
         * 
         * @param drmSignaling
         *        Determines how the DASH manifest signals the DRM content.
         * @see DashDrmSignaling
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DashDrmSignaling
         */
        Builder drmSignaling(String drmSignaling);

        /**
         * <p>
         * Determines how the DASH manifest signals the DRM content.
         * </p>
         * 
         * @param drmSignaling
         *        Determines how the DASH manifest signals the DRM content.
         * @see DashDrmSignaling
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DashDrmSignaling
         */
        Builder drmSignaling(DashDrmSignaling drmSignaling);

        /**
         * <p>
         * Determines the type of UTC timing included in the DASH Media Presentation Description (MPD).
         * </p>
         * 
         * @param utcTiming
         *        Determines the type of UTC timing included in the DASH Media Presentation Description (MPD).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utcTiming(DashUtcTiming utcTiming);

        /**
         * <p>
         * Determines the type of UTC timing included in the DASH Media Presentation Description (MPD).
         * </p>
         * This is a convenience method that creates an instance of the {@link DashUtcTiming.Builder} avoiding the need
         * to create one manually via {@link DashUtcTiming#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DashUtcTiming.Builder#build()} is called immediately and its
         * result is passed to {@link #utcTiming(DashUtcTiming)}.
         * 
         * @param utcTiming
         *        a consumer that will call methods on {@link DashUtcTiming.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utcTiming(DashUtcTiming)
         */
        default Builder utcTiming(Consumer<DashUtcTiming.Builder> utcTiming) {
            return utcTiming(DashUtcTiming.builder().applyMutation(utcTiming).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String manifestName;

        private Integer manifestWindowSeconds;

        private FilterConfiguration filterConfiguration;

        private Integer minUpdatePeriodSeconds;

        private Integer minBufferTimeSeconds;

        private Integer suggestedPresentationDelaySeconds;

        private String segmentTemplateFormat;

        private List<String> periodTriggers = DefaultSdkAutoConstructList.getInstance();

        private ScteDash scteDash;

        private String drmSignaling;

        private DashUtcTiming utcTiming;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateDashManifestConfiguration model) {
            manifestName(model.manifestName);
            manifestWindowSeconds(model.manifestWindowSeconds);
            filterConfiguration(model.filterConfiguration);
            minUpdatePeriodSeconds(model.minUpdatePeriodSeconds);
            minBufferTimeSeconds(model.minBufferTimeSeconds);
            suggestedPresentationDelaySeconds(model.suggestedPresentationDelaySeconds);
            segmentTemplateFormat(model.segmentTemplateFormat);
            periodTriggersWithStrings(model.periodTriggers);
            scteDash(model.scteDash);
            drmSignaling(model.drmSignaling);
            utcTiming(model.utcTiming);
        }

        public final String getManifestName() {
            return manifestName;
        }

        public final void setManifestName(String manifestName) {
            this.manifestName = manifestName;
        }

        @Override
        public final Builder manifestName(String manifestName) {
            this.manifestName = manifestName;
            return this;
        }

        public final Integer getManifestWindowSeconds() {
            return manifestWindowSeconds;
        }

        public final void setManifestWindowSeconds(Integer manifestWindowSeconds) {
            this.manifestWindowSeconds = manifestWindowSeconds;
        }

        @Override
        public final Builder manifestWindowSeconds(Integer manifestWindowSeconds) {
            this.manifestWindowSeconds = manifestWindowSeconds;
            return this;
        }

        public final FilterConfiguration.Builder getFilterConfiguration() {
            return filterConfiguration != null ? filterConfiguration.toBuilder() : null;
        }

        public final void setFilterConfiguration(FilterConfiguration.BuilderImpl filterConfiguration) {
            this.filterConfiguration = filterConfiguration != null ? filterConfiguration.build() : null;
        }

        @Override
        public final Builder filterConfiguration(FilterConfiguration filterConfiguration) {
            this.filterConfiguration = filterConfiguration;
            return this;
        }

        public final Integer getMinUpdatePeriodSeconds() {
            return minUpdatePeriodSeconds;
        }

        public final void setMinUpdatePeriodSeconds(Integer minUpdatePeriodSeconds) {
            this.minUpdatePeriodSeconds = minUpdatePeriodSeconds;
        }

        @Override
        public final Builder minUpdatePeriodSeconds(Integer minUpdatePeriodSeconds) {
            this.minUpdatePeriodSeconds = minUpdatePeriodSeconds;
            return this;
        }

        public final Integer getMinBufferTimeSeconds() {
            return minBufferTimeSeconds;
        }

        public final void setMinBufferTimeSeconds(Integer minBufferTimeSeconds) {
            this.minBufferTimeSeconds = minBufferTimeSeconds;
        }

        @Override
        public final Builder minBufferTimeSeconds(Integer minBufferTimeSeconds) {
            this.minBufferTimeSeconds = minBufferTimeSeconds;
            return this;
        }

        public final Integer getSuggestedPresentationDelaySeconds() {
            return suggestedPresentationDelaySeconds;
        }

        public final void setSuggestedPresentationDelaySeconds(Integer suggestedPresentationDelaySeconds) {
            this.suggestedPresentationDelaySeconds = suggestedPresentationDelaySeconds;
        }

        @Override
        public final Builder suggestedPresentationDelaySeconds(Integer suggestedPresentationDelaySeconds) {
            this.suggestedPresentationDelaySeconds = suggestedPresentationDelaySeconds;
            return this;
        }

        public final String getSegmentTemplateFormat() {
            return segmentTemplateFormat;
        }

        public final void setSegmentTemplateFormat(String segmentTemplateFormat) {
            this.segmentTemplateFormat = segmentTemplateFormat;
        }

        @Override
        public final Builder segmentTemplateFormat(String segmentTemplateFormat) {
            this.segmentTemplateFormat = segmentTemplateFormat;
            return this;
        }

        @Override
        public final Builder segmentTemplateFormat(DashSegmentTemplateFormat segmentTemplateFormat) {
            this.segmentTemplateFormat(segmentTemplateFormat == null ? null : segmentTemplateFormat.toString());
            return this;
        }

        public final Collection<String> getPeriodTriggers() {
            if (periodTriggers instanceof SdkAutoConstructList) {
                return null;
            }
            return periodTriggers;
        }

        public final void setPeriodTriggers(Collection<String> periodTriggers) {
            this.periodTriggers = DashPeriodTriggersCopier.copy(periodTriggers);
        }

        @Override
        public final Builder periodTriggersWithStrings(Collection<String> periodTriggers) {
            this.periodTriggers = DashPeriodTriggersCopier.copy(periodTriggers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder periodTriggersWithStrings(String... periodTriggers) {
            periodTriggersWithStrings(Arrays.asList(periodTriggers));
            return this;
        }

        @Override
        public final Builder periodTriggers(Collection<DashPeriodTrigger> periodTriggers) {
            this.periodTriggers = DashPeriodTriggersCopier.copyEnumToString(periodTriggers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder periodTriggers(DashPeriodTrigger... periodTriggers) {
            periodTriggers(Arrays.asList(periodTriggers));
            return this;
        }

        public final ScteDash.Builder getScteDash() {
            return scteDash != null ? scteDash.toBuilder() : null;
        }

        public final void setScteDash(ScteDash.BuilderImpl scteDash) {
            this.scteDash = scteDash != null ? scteDash.build() : null;
        }

        @Override
        public final Builder scteDash(ScteDash scteDash) {
            this.scteDash = scteDash;
            return this;
        }

        public final String getDrmSignaling() {
            return drmSignaling;
        }

        public final void setDrmSignaling(String drmSignaling) {
            this.drmSignaling = drmSignaling;
        }

        @Override
        public final Builder drmSignaling(String drmSignaling) {
            this.drmSignaling = drmSignaling;
            return this;
        }

        @Override
        public final Builder drmSignaling(DashDrmSignaling drmSignaling) {
            this.drmSignaling(drmSignaling == null ? null : drmSignaling.toString());
            return this;
        }

        public final DashUtcTiming.Builder getUtcTiming() {
            return utcTiming != null ? utcTiming.toBuilder() : null;
        }

        public final void setUtcTiming(DashUtcTiming.BuilderImpl utcTiming) {
            this.utcTiming = utcTiming != null ? utcTiming.build() : null;
        }

        @Override
        public final Builder utcTiming(DashUtcTiming utcTiming) {
            this.utcTiming = utcTiming;
            return this;
        }

        @Override
        public CreateDashManifestConfiguration build() {
            return new CreateDashManifestConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
