/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackagevod;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediapackagevod.internal.MediaPackageVodServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mediapackagevod.model.ConfigureLogsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ConfigureLogsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.CreateAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreateAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeleteAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeleteAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribeAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribeAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ForbiddenException;
import software.amazon.awssdk.services.mediapackagevod.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediapackagevod.model.ListAssetsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListAssetsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediapackagevod.model.MediaPackageVodException;
import software.amazon.awssdk.services.mediapackagevod.model.NotFoundException;
import software.amazon.awssdk.services.mediapackagevod.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mediapackagevod.model.TagResourceRequest;
import software.amazon.awssdk.services.mediapackagevod.model.TagResourceResponse;
import software.amazon.awssdk.services.mediapackagevod.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediapackagevod.model.UnprocessableEntityException;
import software.amazon.awssdk.services.mediapackagevod.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediapackagevod.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediapackagevod.model.UpdatePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.UpdatePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.transform.ConfigureLogsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.CreateAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.CreatePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.CreatePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeleteAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeletePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeletePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribeAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListAssetsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListPackagingConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListPackagingGroupsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.UpdatePackagingGroupRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MediaPackageVodClient}.
 *
 * @see MediaPackageVodClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaPackageVodClient implements MediaPackageVodClient {
    private static final Logger log = Logger.loggerFor(DefaultMediaPackageVodClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaPackageVodClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * Changes the packaging group's properities to configure log subscription
     *
     * @param configureLogsRequest
     *        The option to configure log subscription.
     * @return Result of the ConfigureLogs operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.ConfigureLogs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ConfigureLogs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ConfigureLogsResponse configureLogs(ConfigureLogsRequest configureLogsRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ConfigureLogsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ConfigureLogsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(configureLogsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, configureLogsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ConfigureLogs");

            return clientHandler.execute(new ClientExecutionParams<ConfigureLogsRequest, ConfigureLogsResponse>()
                    .withOperationName("ConfigureLogs").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(configureLogsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ConfigureLogsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Creates a new MediaPackage VOD Asset resource.
     *
     * @param createAssetRequest
     *        A new MediaPackage VOD Asset configuration.
     * @return Result of the CreateAsset operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.CreateAsset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreateAsset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateAssetResponse createAsset(CreateAssetRequest createAssetRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAssetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAssetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAsset");

            return clientHandler.execute(new ClientExecutionParams<CreateAssetRequest, CreateAssetResponse>()
                    .withOperationName("CreateAsset").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createAssetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAssetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Creates a new MediaPackage VOD PackagingConfiguration resource.
     *
     * @param createPackagingConfigurationRequest
     *        A new MediaPackage VOD PackagingConfiguration resource configuration.
     * @return Result of the CreatePackagingConfiguration operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.CreatePackagingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreatePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePackagingConfigurationResponse createPackagingConfiguration(
            CreatePackagingConfigurationRequest createPackagingConfigurationRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePackagingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPackagingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPackagingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePackagingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePackagingConfigurationRequest, CreatePackagingConfigurationResponse>()
                            .withOperationName("CreatePackagingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createPackagingConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreatePackagingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Creates a new MediaPackage VOD PackagingGroup resource.
     *
     * @param createPackagingGroupRequest
     *        A new MediaPackage VOD PackagingGroup resource configuration.
     * @return Result of the CreatePackagingGroup operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.CreatePackagingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreatePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePackagingGroupResponse createPackagingGroup(CreatePackagingGroupRequest createPackagingGroupRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePackagingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPackagingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePackagingGroup");

            return clientHandler.execute(new ClientExecutionParams<CreatePackagingGroupRequest, CreatePackagingGroupResponse>()
                    .withOperationName("CreatePackagingGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createPackagingGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreatePackagingGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Deletes an existing MediaPackage VOD Asset resource.
     *
     * @param deleteAssetRequest
     * @return Result of the DeleteAsset operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DeleteAsset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeleteAsset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteAssetResponse deleteAsset(DeleteAssetRequest deleteAssetRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAssetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAssetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAsset");

            return clientHandler.execute(new ClientExecutionParams<DeleteAssetRequest, DeleteAssetResponse>()
                    .withOperationName("DeleteAsset").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteAssetRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAssetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Deletes a MediaPackage VOD PackagingConfiguration resource.
     *
     * @param deletePackagingConfigurationRequest
     * @return Result of the DeletePackagingConfiguration operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DeletePackagingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeletePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeletePackagingConfigurationResponse deletePackagingConfiguration(
            DeletePackagingConfigurationRequest deletePackagingConfigurationRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePackagingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePackagingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePackagingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePackagingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeletePackagingConfigurationRequest, DeletePackagingConfigurationResponse>()
                            .withOperationName("DeletePackagingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deletePackagingConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeletePackagingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Deletes a MediaPackage VOD PackagingGroup resource.
     *
     * @param deletePackagingGroupRequest
     * @return Result of the DeletePackagingGroup operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DeletePackagingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeletePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeletePackagingGroupResponse deletePackagingGroup(DeletePackagingGroupRequest deletePackagingGroupRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePackagingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePackagingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePackagingGroup");

            return clientHandler.execute(new ClientExecutionParams<DeletePackagingGroupRequest, DeletePackagingGroupResponse>()
                    .withOperationName("DeletePackagingGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deletePackagingGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePackagingGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a description of a MediaPackage VOD Asset resource.
     *
     * @param describeAssetRequest
     * @return Result of the DescribeAsset operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DescribeAsset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribeAsset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAssetResponse describeAsset(DescribeAssetRequest describeAssetRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeAssetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAssetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAsset");

            return clientHandler.execute(new ClientExecutionParams<DescribeAssetRequest, DescribeAssetResponse>()
                    .withOperationName("DescribeAsset").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeAssetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAssetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a description of a MediaPackage VOD PackagingConfiguration resource.
     *
     * @param describePackagingConfigurationRequest
     * @return Result of the DescribePackagingConfiguration operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DescribePackagingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribePackagingConfigurationResponse describePackagingConfiguration(
            DescribePackagingConfigurationRequest describePackagingConfigurationRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribePackagingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePackagingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describePackagingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackagingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribePackagingConfigurationRequest, DescribePackagingConfigurationResponse>()
                            .withOperationName("DescribePackagingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describePackagingConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribePackagingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a description of a MediaPackage VOD PackagingGroup resource.
     *
     * @param describePackagingGroupRequest
     * @return Result of the DescribePackagingGroup operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DescribePackagingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribePackagingGroupResponse describePackagingGroup(DescribePackagingGroupRequest describePackagingGroupRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribePackagingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePackagingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackagingGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribePackagingGroupRequest, DescribePackagingGroupResponse>()
                            .withOperationName("DescribePackagingGroup").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describePackagingGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribePackagingGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a collection of MediaPackage VOD Asset resources.
     *
     * @param listAssetsRequest
     * @return Result of the ListAssets operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.ListAssets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListAssets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListAssetsResponse listAssets(ListAssetsRequest listAssetsRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAssetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAssetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAssetsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssets");

            return clientHandler.execute(new ClientExecutionParams<ListAssetsRequest, ListAssetsResponse>()
                    .withOperationName("ListAssets").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listAssetsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAssetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingConfiguration resources.
     *
     * @param listPackagingConfigurationsRequest
     * @return Result of the ListPackagingConfigurations operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.ListPackagingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPackagingConfigurationsResponse listPackagingConfigurations(
            ListPackagingConfigurationsRequest listPackagingConfigurationsRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPackagingConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPackagingConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPackagingConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackagingConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackagingConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListPackagingConfigurationsRequest, ListPackagingConfigurationsResponse>()
                            .withOperationName("ListPackagingConfigurations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listPackagingConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListPackagingConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingGroup resources.
     *
     * @param listPackagingGroupsRequest
     * @return Result of the ListPackagingGroups operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.ListPackagingGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPackagingGroupsResponse listPackagingGroups(ListPackagingGroupsRequest listPackagingGroupsRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPackagingGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPackagingGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPackagingGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackagingGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackagingGroups");

            return clientHandler.execute(new ClientExecutionParams<ListPackagingGroupsRequest, ListPackagingGroupsResponse>()
                    .withOperationName("ListPackagingGroups").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPackagingGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPackagingGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a list of the tags assigned to the specified resource.
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Adds tags to the specified resource. You can specify one or more tags to add.
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws AwsServiceException, SdkClientException,
            MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Removes tags from the specified resource. You can specify one or more tags to remove.
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws AwsServiceException,
            SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Updates a specific packaging group. You can't change the id attribute or any other system-generated attributes.
     *
     * @param updatePackagingGroupRequest
     *        A MediaPackage VOD PackagingGroup resource configuration.
     * @return Result of the UpdatePackagingGroup operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.UpdatePackagingGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/UpdatePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdatePackagingGroupResponse updatePackagingGroup(UpdatePackagingGroupRequest updatePackagingGroupRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdatePackagingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePackagingGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePackagingGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaPackage Vod");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePackagingGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdatePackagingGroupRequest, UpdatePackagingGroupResponse>()
                    .withOperationName("UpdatePackagingGroup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updatePackagingGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdatePackagingGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MediaPackageVodServiceClientConfigurationBuilder serviceConfigBuilder = new MediaPackageVodServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaPackageVodException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnprocessableEntityException")
                                .exceptionBuilderSupplier(UnprocessableEntityException::builder).httpStatusCode(422).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public final MediaPackageVodServiceClientConfiguration serviceClientConfiguration() {
        return new MediaPackageVodServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
