/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediastoredata;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.Aws4UnsignedPayloadSigner;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.protocol.json.AwsJsonProtocol;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolMetadata;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.internal.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.core.runtime.transform.StreamingRequestMarshaller;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.services.mediastoredata.model.ContainerNotFoundException;
import software.amazon.awssdk.services.mediastoredata.model.DeleteObjectRequest;
import software.amazon.awssdk.services.mediastoredata.model.DeleteObjectResponse;
import software.amazon.awssdk.services.mediastoredata.model.DescribeObjectRequest;
import software.amazon.awssdk.services.mediastoredata.model.DescribeObjectResponse;
import software.amazon.awssdk.services.mediastoredata.model.GetObjectRequest;
import software.amazon.awssdk.services.mediastoredata.model.GetObjectResponse;
import software.amazon.awssdk.services.mediastoredata.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediastoredata.model.ListItemsRequest;
import software.amazon.awssdk.services.mediastoredata.model.ListItemsResponse;
import software.amazon.awssdk.services.mediastoredata.model.MediaStoreDataException;
import software.amazon.awssdk.services.mediastoredata.model.MediaStoreDataRequest;
import software.amazon.awssdk.services.mediastoredata.model.ObjectNotFoundException;
import software.amazon.awssdk.services.mediastoredata.model.PutObjectRequest;
import software.amazon.awssdk.services.mediastoredata.model.PutObjectResponse;
import software.amazon.awssdk.services.mediastoredata.model.RequestedRangeNotSatisfiableException;
import software.amazon.awssdk.services.mediastoredata.transform.DeleteObjectRequestMarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.DeleteObjectResponseUnmarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.DescribeObjectRequestMarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.DescribeObjectResponseUnmarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.GetObjectRequestMarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.GetObjectResponseUnmarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.ListItemsRequestMarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.ListItemsResponseUnmarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.PutObjectRequestMarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.PutObjectResponseUnmarshaller;

/**
 * Internal implementation of {@link MediaStoreDataClient}.
 *
 * @see MediaStoreDataClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaStoreDataClient implements MediaStoreDataClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaStoreDataClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(false);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes an object at the specified path.
     * </p>
     *
     * @param deleteObjectRequest
     * @return Result of the DeleteObject operation returned by the service.
     * @throws ContainerNotFoundException
     *         The specified container was not found for the specified account.
     * @throws ObjectNotFoundException
     *         Could not perform an operation on an object that does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreDataClient.DeleteObject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-data-2017-09-01/DeleteObject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteObjectResponse deleteObject(DeleteObjectRequest deleteObjectRequest) throws ContainerNotFoundException,
            ObjectNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaStoreDataException {

        HttpResponseHandler<DeleteObjectResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteObjectResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DeleteObjectRequest, DeleteObjectResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(deleteObjectRequest).withMarshaller(new DeleteObjectRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the headers for an object at the specified path.
     * </p>
     *
     * @param describeObjectRequest
     * @return Result of the DescribeObject operation returned by the service.
     * @throws ContainerNotFoundException
     *         The specified container was not found for the specified account.
     * @throws ObjectNotFoundException
     *         Could not perform an operation on an object that does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreDataClient.DescribeObject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-data-2017-09-01/DescribeObject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeObjectResponse describeObject(DescribeObjectRequest describeObjectRequest) throws ContainerNotFoundException,
            ObjectNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaStoreDataException {

        HttpResponseHandler<DescribeObjectResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeObjectResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<DescribeObjectRequest, DescribeObjectResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeObjectRequest).withMarshaller(new DescribeObjectRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Downloads the object at the specified path.
     * </p>
     *
     * @param getObjectRequest
     * @param streamingHandler
     *        Functional interface for processing the streamed response content. The unmarshalled GetObjectResponse and
     *        an InputStream to the response content are provided as parameters to the callback. The callback may return
     *        a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The bytes of the object.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ContainerNotFoundException
     *         The specified container was not found for the specified account.
     * @throws ObjectNotFoundException
     *         Could not perform an operation on an object that does not exist.
     * @throws RequestedRangeNotSatisfiableException
     *         The requested content range is not valid.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreDataClient.GetObject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-data-2017-09-01/GetObject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT getObject(GetObjectRequest getObjectRequest,
            ResponseTransformer<GetObjectResponse, ReturnT> responseTransformer) throws ContainerNotFoundException,
            ObjectNotFoundException, RequestedRangeNotSatisfiableException, InternalServerErrorException, AwsServiceException,
            SdkClientException, MediaStoreDataException {

        HttpResponseHandler<GetObjectResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(false).withHasStreamingSuccessResponse(true),
                new GetObjectResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(
                new ClientExecutionParams<GetObjectRequest, GetObjectResponse>().withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getObjectRequest)
                        .withMarshaller(new GetObjectRequestMarshaller(protocolFactory)), responseTransformer);
    }

    /**
     * <p>
     * Provides a list of metadata entries about folders and objects in the specified folder.
     * </p>
     *
     * @param listItemsRequest
     * @return Result of the ListItems operation returned by the service.
     * @throws ContainerNotFoundException
     *         The specified container was not found for the specified account.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreDataClient.ListItems
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-data-2017-09-01/ListItems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListItemsResponse listItems(ListItemsRequest listItemsRequest) throws ContainerNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, MediaStoreDataException {

        HttpResponseHandler<ListItemsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListItemsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<ListItemsRequest, ListItemsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listItemsRequest)
                .withMarshaller(new ListItemsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Uploads an object to the specified path. Object sizes are limited to 10 MB.
     * </p>
     *
     * @param putObjectRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows '
     *        <p>
     *        The bytes to be stored.
     *        </p>
     *        '
     * @return Result of the PutObject operation returned by the service.
     * @throws ContainerNotFoundException
     *         The specified container was not found for the specified account.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreDataClient.PutObject
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediastore-data-2017-09-01/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutObjectResponse putObject(PutObjectRequest putObjectRequest, RequestBody requestBody)
            throws ContainerNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaStoreDataException {
        putObjectRequest = applySignerOverride(putObjectRequest, Aws4UnsignedPayloadSigner.create());

        HttpResponseHandler<PutObjectResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new PutObjectResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

        return clientHandler.execute(new ClientExecutionParams<PutObjectRequest, PutObjectResponse>()
                .withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler)
                .withInput(putObjectRequest)
                .withMarshaller(
                        new StreamingRequestMarshaller<PutObjectRequest>(new PutObjectRequestMarshaller(protocolFactory),
                                requestBody)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(AwsJsonProtocolFactory protocolFactory) {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }

    private software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory init(boolean supportsCbor) {
        return new AwsJsonProtocolFactory(
                new JsonClientMetadata()
                        .withSupportsCbor(supportsCbor)
                        .withSupportsIon(false)
                        .withBaseServiceExceptionClass(
                                software.amazon.awssdk.services.mediastoredata.model.MediaStoreDataException.class)
                        .withContentTypeOverride("")
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("RequestedRangeNotSatisfiableException")
                                        .withModeledClass(RequestedRangeNotSatisfiableException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("ObjectNotFoundException").withModeledClass(
                                        ObjectNotFoundException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("InternalServerError").withModeledClass(
                                        InternalServerErrorException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("ContainerNotFoundException").withModeledClass(
                                        ContainerNotFoundException.class)), AwsJsonProtocolMetadata.builder()
                        .protocolVersion("1.1").protocol(AwsJsonProtocol.REST_JSON).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MediaStoreDataRequest> T applySignerOverride(T request, Signer signer) {
        if (request.overrideConfiguration().flatMap(c -> c.signer()).isPresent()) {
            return request;
        }
        Consumer<AwsRequestOverrideConfiguration.Builder> signerOverride = b -> b.signer(signer).build();
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(signerOverride).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(signerOverride).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
