/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediastoredata.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.mediastoredata.transform.ItemMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A metadata entry for a folder or object.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Item implements StructuredPojo, ToCopyableBuilder<Item.Builder, Item> {
    private final String name;

    private final String type;

    private final String eTag;

    private final Instant lastModified;

    private final String contentType;

    private final Long contentLength;

    private Item(BuilderImpl builder) {
        this.name = builder.name;
        this.type = builder.type;
        this.eTag = builder.eTag;
        this.lastModified = builder.lastModified;
        this.contentType = builder.contentType;
        this.contentLength = builder.contentLength;
    }

    /**
     * <p>
     * The name of the item.
     * </p>
     * 
     * @return The name of the item.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The item type (folder or object).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ItemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The item type (folder or object).
     * @see ItemType
     */
    public ItemType type() {
        return ItemType.fromValue(type);
    }

    /**
     * <p>
     * The item type (folder or object).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ItemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The item type (folder or object).
     * @see ItemType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The ETag that represents a unique instance of the item.
     * </p>
     * 
     * @return The ETag that represents a unique instance of the item.
     */
    public String eTag() {
        return eTag;
    }

    /**
     * <p>
     * The date and time that the item was last modified.
     * </p>
     * 
     * @return The date and time that the item was last modified.
     */
    public Instant lastModified() {
        return lastModified;
    }

    /**
     * <p>
     * The content type of the item.
     * </p>
     * 
     * @return The content type of the item.
     */
    public String contentType() {
        return contentType;
    }

    /**
     * <p>
     * The length of the item in bytes.
     * </p>
     * 
     * @return The length of the item in bytes.
     */
    public Long contentLength() {
        return contentLength;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(eTag());
        hashCode = 31 * hashCode + Objects.hashCode(lastModified());
        hashCode = 31 * hashCode + Objects.hashCode(contentType());
        hashCode = 31 * hashCode + Objects.hashCode(contentLength());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Item)) {
            return false;
        }
        Item other = (Item) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(eTag(), other.eTag()) && Objects.equals(lastModified(), other.lastModified())
                && Objects.equals(contentType(), other.contentType()) && Objects.equals(contentLength(), other.contentLength());
    }

    @Override
    public String toString() {
        return ToString.builder("Item").add("Name", name()).add("Type", typeAsString()).add("ETag", eTag())
                .add("LastModified", lastModified()).add("ContentType", contentType()).add("ContentLength", contentLength())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "ETag":
            return Optional.ofNullable(clazz.cast(eTag()));
        case "LastModified":
            return Optional.ofNullable(clazz.cast(lastModified()));
        case "ContentType":
            return Optional.ofNullable(clazz.cast(contentType()));
        case "ContentLength":
            return Optional.ofNullable(clazz.cast(contentLength()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ItemMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Item> {
        /**
         * <p>
         * The name of the item.
         * </p>
         * 
         * @param name
         *        The name of the item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The item type (folder or object).
         * </p>
         * 
         * @param type
         *        The item type (folder or object).
         * @see ItemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ItemType
         */
        Builder type(String type);

        /**
         * <p>
         * The item type (folder or object).
         * </p>
         * 
         * @param type
         *        The item type (folder or object).
         * @see ItemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ItemType
         */
        Builder type(ItemType type);

        /**
         * <p>
         * The ETag that represents a unique instance of the item.
         * </p>
         * 
         * @param eTag
         *        The ETag that represents a unique instance of the item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eTag(String eTag);

        /**
         * <p>
         * The date and time that the item was last modified.
         * </p>
         * 
         * @param lastModified
         *        The date and time that the item was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModified(Instant lastModified);

        /**
         * <p>
         * The content type of the item.
         * </p>
         * 
         * @param contentType
         *        The content type of the item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentType(String contentType);

        /**
         * <p>
         * The length of the item in bytes.
         * </p>
         * 
         * @param contentLength
         *        The length of the item in bytes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentLength(Long contentLength);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String type;

        private String eTag;

        private Instant lastModified;

        private String contentType;

        private Long contentLength;

        private BuilderImpl() {
        }

        private BuilderImpl(Item model) {
            name(model.name);
            type(model.type);
            eTag(model.eTag);
            lastModified(model.lastModified);
            contentType(model.contentType);
            contentLength(model.contentLength);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ItemType type) {
            this.type(type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getETag() {
            return eTag;
        }

        @Override
        public final Builder eTag(String eTag) {
            this.eTag = eTag;
            return this;
        }

        public final void setETag(String eTag) {
            this.eTag = eTag;
        }

        public final Instant getLastModified() {
            return lastModified;
        }

        @Override
        public final Builder lastModified(Instant lastModified) {
            this.lastModified = lastModified;
            return this;
        }

        public final void setLastModified(Instant lastModified) {
            this.lastModified = lastModified;
        }

        public final String getContentType() {
            return contentType;
        }

        @Override
        public final Builder contentType(String contentType) {
            this.contentType = contentType;
            return this;
        }

        public final void setContentType(String contentType) {
            this.contentType = contentType;
        }

        public final Long getContentLength() {
            return contentLength;
        }

        @Override
        public final Builder contentLength(Long contentLength) {
            this.contentLength = contentLength;
            return this;
        }

        public final void setContentLength(Long contentLength) {
            this.contentLength = contentLength;
        }

        @Override
        public Item build() {
            return new Item(this);
        }
    }
}
