/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediastoredata;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.core.runtime.transform.StreamingRequestMarshaller;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.mediastoredata.internal.MediaStoreDataServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mediastoredata.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.mediastoredata.model.ContainerNotFoundException;
import software.amazon.awssdk.services.mediastoredata.model.DeleteObjectRequest;
import software.amazon.awssdk.services.mediastoredata.model.DeleteObjectResponse;
import software.amazon.awssdk.services.mediastoredata.model.DescribeObjectRequest;
import software.amazon.awssdk.services.mediastoredata.model.DescribeObjectResponse;
import software.amazon.awssdk.services.mediastoredata.model.GetObjectRequest;
import software.amazon.awssdk.services.mediastoredata.model.GetObjectResponse;
import software.amazon.awssdk.services.mediastoredata.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediastoredata.model.ListItemsRequest;
import software.amazon.awssdk.services.mediastoredata.model.ListItemsResponse;
import software.amazon.awssdk.services.mediastoredata.model.MediaStoreDataException;
import software.amazon.awssdk.services.mediastoredata.model.ObjectNotFoundException;
import software.amazon.awssdk.services.mediastoredata.model.PutObjectRequest;
import software.amazon.awssdk.services.mediastoredata.model.PutObjectResponse;
import software.amazon.awssdk.services.mediastoredata.model.RequestedRangeNotSatisfiableException;
import software.amazon.awssdk.services.mediastoredata.transform.DeleteObjectRequestMarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.DescribeObjectRequestMarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.GetObjectRequestMarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.ListItemsRequestMarshaller;
import software.amazon.awssdk.services.mediastoredata.transform.PutObjectRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MediaStoreDataClient}.
 *
 * @see MediaStoreDataClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaStoreDataClient implements MediaStoreDataClient {
    private static final Logger log = Logger.loggerFor(DefaultMediaStoreDataClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaStoreDataClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "MediaStore_Data" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes an object at the specified path.
     * </p>
     *
     * @param deleteObjectRequest
     * @return Result of the DeleteObject operation returned by the service.
     * @throws ContainerNotFoundException
     *         The specified container was not found for the specified account.
     * @throws ObjectNotFoundException
     *         Could not perform an operation on an object that does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreDataClient.DeleteObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-data-2017-09-01/DeleteObject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteObjectResponse deleteObject(DeleteObjectRequest deleteObjectRequest) throws ContainerNotFoundException,
            ObjectNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaStoreDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteObjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteObjectResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "RequestedRangeNotSatisfiableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RequestedRangeNotSatisfiableException")
                        .httpStatusCode(416).exceptionBuilderSupplier(RequestedRangeNotSatisfiableException::builder).build());
            case "ObjectNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ObjectNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ObjectNotFoundException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError")
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            case "ContainerNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ContainerNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ContainerNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteObjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteObject");

            return clientHandler.execute(new ClientExecutionParams<DeleteObjectRequest, DeleteObjectResponse>()
                    .withOperationName("DeleteObject").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteObjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteObjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the headers for an object at the specified path.
     * </p>
     *
     * @param describeObjectRequest
     * @return Result of the DescribeObject operation returned by the service.
     * @throws ContainerNotFoundException
     *         The specified container was not found for the specified account.
     * @throws ObjectNotFoundException
     *         Could not perform an operation on an object that does not exist.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreDataClient.DescribeObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-data-2017-09-01/DescribeObject"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeObjectResponse describeObject(DescribeObjectRequest describeObjectRequest) throws ContainerNotFoundException,
            ObjectNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaStoreDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeObjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeObjectResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "RequestedRangeNotSatisfiableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RequestedRangeNotSatisfiableException")
                        .httpStatusCode(416).exceptionBuilderSupplier(RequestedRangeNotSatisfiableException::builder).build());
            case "ObjectNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ObjectNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ObjectNotFoundException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError")
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            case "ContainerNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ContainerNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ContainerNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeObjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeObject");

            return clientHandler.execute(new ClientExecutionParams<DescribeObjectRequest, DescribeObjectResponse>()
                    .withOperationName("DescribeObject").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeObjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeObjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Downloads the object at the specified path. If the object’s upload availability is set to <code>streaming</code>,
     * AWS Elemental MediaStore downloads the object even if it’s still uploading the object.
     * </p>
     *
     * @param getObjectRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled GetObjectResponse and
     *        an InputStream to the response content are provided as parameters to the callback. The callback may return
     *        a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The bytes of the object.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ContainerNotFoundException
     *         The specified container was not found for the specified account.
     * @throws ObjectNotFoundException
     *         Could not perform an operation on an object that does not exist.
     * @throws RequestedRangeNotSatisfiableException
     *         The requested content range is not valid.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreDataClient.GetObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-data-2017-09-01/GetObject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT getObject(GetObjectRequest getObjectRequest,
            ResponseTransformer<GetObjectResponse, ReturnT> responseTransformer) throws ContainerNotFoundException,
            ObjectNotFoundException, RequestedRangeNotSatisfiableException, InternalServerErrorException, AwsServiceException,
            SdkClientException, MediaStoreDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetObjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetObjectResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "RequestedRangeNotSatisfiableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RequestedRangeNotSatisfiableException")
                        .httpStatusCode(416).exceptionBuilderSupplier(RequestedRangeNotSatisfiableException::builder).build());
            case "ObjectNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ObjectNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ObjectNotFoundException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError")
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            case "ContainerNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ContainerNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ContainerNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getObjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetObject");

            return clientHandler.execute(
                    new ClientExecutionParams<GetObjectRequest, GetObjectResponse>().withOperationName("GetObject")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(getObjectRequest).withMetricCollector(apiCallMetricCollector)
                            .withResponseTransformer(responseTransformer)
                            .withMarshaller(new GetObjectRequestMarshaller(protocolFactory)), responseTransformer);
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of metadata entries about folders and objects in the specified folder.
     * </p>
     *
     * @param listItemsRequest
     * @return Result of the ListItems operation returned by the service.
     * @throws ContainerNotFoundException
     *         The specified container was not found for the specified account.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreDataClient.ListItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-data-2017-09-01/ListItems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListItemsResponse listItems(ListItemsRequest listItemsRequest) throws ContainerNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, MediaStoreDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListItemsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListItemsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "RequestedRangeNotSatisfiableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RequestedRangeNotSatisfiableException")
                        .httpStatusCode(416).exceptionBuilderSupplier(RequestedRangeNotSatisfiableException::builder).build());
            case "ObjectNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ObjectNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ObjectNotFoundException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError")
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            case "ContainerNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ContainerNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ContainerNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listItemsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListItems");

            return clientHandler.execute(new ClientExecutionParams<ListItemsRequest, ListItemsResponse>()
                    .withOperationName("ListItems").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listItemsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListItemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Uploads an object to the specified path. Object sizes are limited to 25 MB for standard upload availability and
     * 10 MB for streaming upload availability.
     * </p>
     *
     * @param putObjectRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows '
     *        <p>
     *        The bytes to be stored.
     *        </p>
     *        '
     * @return Result of the PutObject operation returned by the service.
     * @throws ContainerNotFoundException
     *         The specified container was not found for the specified account.
     * @throws InternalServerErrorException
     *         The service is temporarily unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaStoreDataException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaStoreDataClient.PutObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-data-2017-09-01/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutObjectResponse putObject(PutObjectRequest putObjectRequest, RequestBody requestBody)
            throws ContainerNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaStoreDataException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutObjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutObjectResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "RequestedRangeNotSatisfiableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RequestedRangeNotSatisfiableException")
                        .httpStatusCode(416).exceptionBuilderSupplier(RequestedRangeNotSatisfiableException::builder).build());
            case "ObjectNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ObjectNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ObjectNotFoundException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError")
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            case "ContainerNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ContainerNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ContainerNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putObjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutObject");

            return clientHandler.execute(new ClientExecutionParams<PutObjectRequest, PutObjectResponse>()
                    .withOperationName("PutObject")
                    .withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration)
                    .withInput(putObjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withRequestBody(requestBody)
                    .withMarshaller(
                            StreamingRequestMarshaller.builder()
                                    .delegateMarshaller(new PutObjectRequestMarshaller(protocolFactory)).requestBody(requestBody)
                                    .transferEncoding(true).build()));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        MediaStoreDataServiceClientConfigurationBuilder serviceConfigBuilder = new MediaStoreDataServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(MediaStoreDataException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final MediaStoreDataServiceClientConfiguration serviceClientConfiguration() {
        return new MediaStoreDataServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
