/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediatailor.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Creates a playback configuration. For information about MediaTailor configurations, see <a
 * href="https://docs.aws.amazon.com/mediatailor/latest/ug/configurations.html">Working with configurations in AWS
 * Elemental MediaTailor</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PlaybackConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<PlaybackConfiguration.Builder, PlaybackConfiguration> {
    private static final SdkField<String> AD_DECISION_SERVER_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdDecisionServerUrl").getter(getter(PlaybackConfiguration::adDecisionServerUrl))
            .setter(setter(Builder::adDecisionServerUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdDecisionServerUrl").build())
            .build();

    private static final SdkField<AvailSuppression> AVAIL_SUPPRESSION_FIELD = SdkField
            .<AvailSuppression> builder(MarshallingType.SDK_POJO).memberName("AvailSuppression")
            .getter(getter(PlaybackConfiguration::availSuppression)).setter(setter(Builder::availSuppression))
            .constructor(AvailSuppression::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailSuppression").build()).build();

    private static final SdkField<Bumper> BUMPER_FIELD = SdkField.<Bumper> builder(MarshallingType.SDK_POJO).memberName("Bumper")
            .getter(getter(PlaybackConfiguration::bumper)).setter(setter(Builder::bumper)).constructor(Bumper::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bumper").build()).build();

    private static final SdkField<CdnConfiguration> CDN_CONFIGURATION_FIELD = SdkField
            .<CdnConfiguration> builder(MarshallingType.SDK_POJO).memberName("CdnConfiguration")
            .getter(getter(PlaybackConfiguration::cdnConfiguration)).setter(setter(Builder::cdnConfiguration))
            .constructor(CdnConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CdnConfiguration").build()).build();

    private static final SdkField<Map<String, Map<String, String>>> CONFIGURATION_ALIASES_FIELD = SdkField
            .<Map<String, Map<String, String>>> builder(MarshallingType.MAP)
            .memberName("ConfigurationAliases")
            .getter(getter(PlaybackConfiguration::configurationAliases))
            .setter(setter(Builder::configurationAliases))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationAliases").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Map<String, String>> builder(MarshallingType.MAP)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    MapTrait.builder()
                                                            .keyLocationName("key")
                                                            .valueLocationName("value")
                                                            .valueFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("value").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<DashConfiguration> DASH_CONFIGURATION_FIELD = SdkField
            .<DashConfiguration> builder(MarshallingType.SDK_POJO).memberName("DashConfiguration")
            .getter(getter(PlaybackConfiguration::dashConfiguration)).setter(setter(Builder::dashConfiguration))
            .constructor(DashConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DashConfiguration").build()).build();

    private static final SdkField<HlsConfiguration> HLS_CONFIGURATION_FIELD = SdkField
            .<HlsConfiguration> builder(MarshallingType.SDK_POJO).memberName("HlsConfiguration")
            .getter(getter(PlaybackConfiguration::hlsConfiguration)).setter(setter(Builder::hlsConfiguration))
            .constructor(HlsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HlsConfiguration").build()).build();

    private static final SdkField<LivePreRollConfiguration> LIVE_PRE_ROLL_CONFIGURATION_FIELD = SdkField
            .<LivePreRollConfiguration> builder(MarshallingType.SDK_POJO).memberName("LivePreRollConfiguration")
            .getter(getter(PlaybackConfiguration::livePreRollConfiguration)).setter(setter(Builder::livePreRollConfiguration))
            .constructor(LivePreRollConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LivePreRollConfiguration").build())
            .build();

    private static final SdkField<ManifestProcessingRules> MANIFEST_PROCESSING_RULES_FIELD = SdkField
            .<ManifestProcessingRules> builder(MarshallingType.SDK_POJO).memberName("ManifestProcessingRules")
            .getter(getter(PlaybackConfiguration::manifestProcessingRules)).setter(setter(Builder::manifestProcessingRules))
            .constructor(ManifestProcessingRules::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManifestProcessingRules").build())
            .build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(PlaybackConfiguration::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Integer> PERSONALIZATION_THRESHOLD_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("PersonalizationThresholdSeconds")
            .getter(getter(PlaybackConfiguration::personalizationThresholdSeconds))
            .setter(setter(Builder::personalizationThresholdSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PersonalizationThresholdSeconds")
                    .build()).build();

    private static final SdkField<String> PLAYBACK_CONFIGURATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PlaybackConfigurationArn").getter(getter(PlaybackConfiguration::playbackConfigurationArn))
            .setter(setter(Builder::playbackConfigurationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlaybackConfigurationArn").build())
            .build();

    private static final SdkField<String> PLAYBACK_ENDPOINT_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PlaybackEndpointPrefix").getter(getter(PlaybackConfiguration::playbackEndpointPrefix))
            .setter(setter(Builder::playbackEndpointPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlaybackEndpointPrefix").build())
            .build();

    private static final SdkField<String> SESSION_INITIALIZATION_ENDPOINT_PREFIX_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SessionInitializationEndpointPrefix")
            .getter(getter(PlaybackConfiguration::sessionInitializationEndpointPrefix))
            .setter(setter(Builder::sessionInitializationEndpointPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("SessionInitializationEndpointPrefix").build()).build();

    private static final SdkField<String> SLATE_AD_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SlateAdUrl").getter(getter(PlaybackConfiguration::slateAdUrl)).setter(setter(Builder::slateAdUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SlateAdUrl").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(PlaybackConfiguration::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> TRANSCODE_PROFILE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TranscodeProfileName").getter(getter(PlaybackConfiguration::transcodeProfileName))
            .setter(setter(Builder::transcodeProfileName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TranscodeProfileName").build())
            .build();

    private static final SdkField<String> VIDEO_CONTENT_SOURCE_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VideoContentSourceUrl").getter(getter(PlaybackConfiguration::videoContentSourceUrl))
            .setter(setter(Builder::videoContentSourceUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VideoContentSourceUrl").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AD_DECISION_SERVER_URL_FIELD,
            AVAIL_SUPPRESSION_FIELD, BUMPER_FIELD, CDN_CONFIGURATION_FIELD, CONFIGURATION_ALIASES_FIELD,
            DASH_CONFIGURATION_FIELD, HLS_CONFIGURATION_FIELD, LIVE_PRE_ROLL_CONFIGURATION_FIELD,
            MANIFEST_PROCESSING_RULES_FIELD, NAME_FIELD, PERSONALIZATION_THRESHOLD_SECONDS_FIELD,
            PLAYBACK_CONFIGURATION_ARN_FIELD, PLAYBACK_ENDPOINT_PREFIX_FIELD, SESSION_INITIALIZATION_ENDPOINT_PREFIX_FIELD,
            SLATE_AD_URL_FIELD, TAGS_FIELD, TRANSCODE_PROFILE_NAME_FIELD, VIDEO_CONTENT_SOURCE_URL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String adDecisionServerUrl;

    private final AvailSuppression availSuppression;

    private final Bumper bumper;

    private final CdnConfiguration cdnConfiguration;

    private final Map<String, Map<String, String>> configurationAliases;

    private final DashConfiguration dashConfiguration;

    private final HlsConfiguration hlsConfiguration;

    private final LivePreRollConfiguration livePreRollConfiguration;

    private final ManifestProcessingRules manifestProcessingRules;

    private final String name;

    private final Integer personalizationThresholdSeconds;

    private final String playbackConfigurationArn;

    private final String playbackEndpointPrefix;

    private final String sessionInitializationEndpointPrefix;

    private final String slateAdUrl;

    private final Map<String, String> tags;

    private final String transcodeProfileName;

    private final String videoContentSourceUrl;

    private PlaybackConfiguration(BuilderImpl builder) {
        this.adDecisionServerUrl = builder.adDecisionServerUrl;
        this.availSuppression = builder.availSuppression;
        this.bumper = builder.bumper;
        this.cdnConfiguration = builder.cdnConfiguration;
        this.configurationAliases = builder.configurationAliases;
        this.dashConfiguration = builder.dashConfiguration;
        this.hlsConfiguration = builder.hlsConfiguration;
        this.livePreRollConfiguration = builder.livePreRollConfiguration;
        this.manifestProcessingRules = builder.manifestProcessingRules;
        this.name = builder.name;
        this.personalizationThresholdSeconds = builder.personalizationThresholdSeconds;
        this.playbackConfigurationArn = builder.playbackConfigurationArn;
        this.playbackEndpointPrefix = builder.playbackEndpointPrefix;
        this.sessionInitializationEndpointPrefix = builder.sessionInitializationEndpointPrefix;
        this.slateAdUrl = builder.slateAdUrl;
        this.tags = builder.tags;
        this.transcodeProfileName = builder.transcodeProfileName;
        this.videoContentSourceUrl = builder.videoContentSourceUrl;
    }

    /**
     * <p>
     * The URL for the ad decision server (ADS). This includes the specification of static parameters and placeholders
     * for dynamic parameters. AWS Elemental MediaTailor substitutes player-specific and session-specific parameters as
     * needed when calling the ADS. Alternately, for testing you can provide a static VAST URL. The maximum length is
     * 25,000 characters.
     * </p>
     * 
     * @return The URL for the ad decision server (ADS). This includes the specification of static parameters and
     *         placeholders for dynamic parameters. AWS Elemental MediaTailor substitutes player-specific and
     *         session-specific parameters as needed when calling the ADS. Alternately, for testing you can provide a
     *         static VAST URL. The maximum length is 25,000 characters.
     */
    public final String adDecisionServerUrl() {
        return adDecisionServerUrl;
    }

    /**
     * <p>
     * The configuration for avail suppression, also known as ad suppression. For more information about ad suppression,
     * see <a href="https://docs.aws.amazon.com/mediatailor/latest/ug/ad-behavior.html">Ad Suppression</a>.
     * </p>
     * 
     * @return The configuration for avail suppression, also known as ad suppression. For more information about ad
     *         suppression, see <a href="https://docs.aws.amazon.com/mediatailor/latest/ug/ad-behavior.html">Ad
     *         Suppression</a>.
     */
    public final AvailSuppression availSuppression() {
        return availSuppression;
    }

    /**
     * <p>
     * The configuration for bumpers. Bumpers are short audio or video clips that play at the start or before the end of
     * an ad break. To learn more about bumpers, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/bumpers.html">Bumpers</a>.
     * </p>
     * 
     * @return The configuration for bumpers. Bumpers are short audio or video clips that play at the start or before
     *         the end of an ad break. To learn more about bumpers, see <a
     *         href="https://docs.aws.amazon.com/mediatailor/latest/ug/bumpers.html">Bumpers</a>.
     */
    public final Bumper bumper() {
        return bumper;
    }

    /**
     * <p>
     * The configuration for using a content delivery network (CDN), like Amazon CloudFront, for content and ad segment
     * management.
     * </p>
     * 
     * @return The configuration for using a content delivery network (CDN), like Amazon CloudFront, for content and ad
     *         segment management.
     */
    public final CdnConfiguration cdnConfiguration() {
        return cdnConfiguration;
    }

    /**
     * Returns true if the ConfigurationAliases property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasConfigurationAliases() {
        return configurationAliases != null && !(configurationAliases instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The player parameters and aliases used as dynamic variables during session initialization. For more information,
     * see <a href="https://docs.aws.amazon.com/mediatailor/latest/ug/variables-domain.html">Domain Variables</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasConfigurationAliases()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The player parameters and aliases used as dynamic variables during session initialization. For more
     *         information, see <a href="https://docs.aws.amazon.com/mediatailor/latest/ug/variables-domain.html">Domain
     *         Variables</a>.
     */
    public final Map<String, Map<String, String>> configurationAliases() {
        return configurationAliases;
    }

    /**
     * <p>
     * The configuration for a DASH source.
     * </p>
     * 
     * @return The configuration for a DASH source.
     */
    public final DashConfiguration dashConfiguration() {
        return dashConfiguration;
    }

    /**
     * <p>
     * The configuration for HLS content.
     * </p>
     * 
     * @return The configuration for HLS content.
     */
    public final HlsConfiguration hlsConfiguration() {
        return hlsConfiguration;
    }

    /**
     * <p>
     * The configuration for pre-roll ad insertion.
     * </p>
     * 
     * @return The configuration for pre-roll ad insertion.
     */
    public final LivePreRollConfiguration livePreRollConfiguration() {
        return livePreRollConfiguration;
    }

    /**
     * <p>
     * The configuration for manifest processing rules. Manifest processing rules enable customization of the
     * personalized manifests created by MediaTailor.
     * </p>
     * 
     * @return The configuration for manifest processing rules. Manifest processing rules enable customization of the
     *         personalized manifests created by MediaTailor.
     */
    public final ManifestProcessingRules manifestProcessingRules() {
        return manifestProcessingRules;
    }

    /**
     * <p>
     * The identifier for the playback configuration.
     * </p>
     * 
     * @return The identifier for the playback configuration.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Defines the maximum duration of underfilled ad time (in seconds) allowed in an ad break. If the duration of
     * underfilled ad time exceeds the personalization threshold, then the personalization of the ad break is abandoned
     * and the underlying content is shown. This feature applies to <i>ad replacement</i> in live and VOD streams,
     * rather than ad insertion, because it relies on an underlying content stream. For more information about ad break
     * behavior, including ad replacement and insertion, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/ad-behavior.html">Ad Behavior in AWS Elemental
     * MediaTailor</a>.
     * </p>
     * 
     * @return Defines the maximum duration of underfilled ad time (in seconds) allowed in an ad break. If the duration
     *         of underfilled ad time exceeds the personalization threshold, then the personalization of the ad break is
     *         abandoned and the underlying content is shown. This feature applies to <i>ad replacement</i> in live and
     *         VOD streams, rather than ad insertion, because it relies on an underlying content stream. For more
     *         information about ad break behavior, including ad replacement and insertion, see <a
     *         href="https://docs.aws.amazon.com/mediatailor/latest/ug/ad-behavior.html">Ad Behavior in AWS Elemental
     *         MediaTailor</a>.
     */
    public final Integer personalizationThresholdSeconds() {
        return personalizationThresholdSeconds;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the playback configuration.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the playback configuration.
     */
    public final String playbackConfigurationArn() {
        return playbackConfigurationArn;
    }

    /**
     * <p>
     * The URL that the player accesses to get a manifest from AWS Elemental MediaTailor.
     * </p>
     * 
     * @return The URL that the player accesses to get a manifest from AWS Elemental MediaTailor.
     */
    public final String playbackEndpointPrefix() {
        return playbackEndpointPrefix;
    }

    /**
     * <p>
     * The URL that the player uses to initialize a session that uses client-side reporting.
     * </p>
     * 
     * @return The URL that the player uses to initialize a session that uses client-side reporting.
     */
    public final String sessionInitializationEndpointPrefix() {
        return sessionInitializationEndpointPrefix;
    }

    /**
     * <p>
     * The URL for a video asset to transcode and use to fill in time that's not used by ads. AWS Elemental MediaTailor
     * shows the slate to fill in gaps in media content. Configuring the slate is optional for non-VPAID playback
     * configurations. For VPAID, the slate is required because MediaTailor provides it in the slots designated for
     * dynamic ad content. The slate must be a high-quality asset that contains both audio and video.
     * </p>
     * 
     * @return The URL for a video asset to transcode and use to fill in time that's not used by ads. AWS Elemental
     *         MediaTailor shows the slate to fill in gaps in media content. Configuring the slate is optional for
     *         non-VPAID playback configurations. For VPAID, the slate is required because MediaTailor provides it in
     *         the slots designated for dynamic ad content. The slate must be a high-quality asset that contains both
     *         audio and video.
     */
    public final String slateAdUrl() {
        return slateAdUrl;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags to assign to the playback configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tags to assign to the playback configuration.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The name that is used to associate this playback configuration with a custom transcode profile. This overrides
     * the dynamic transcoding defaults of MediaTailor. Use this only if you have already set up custom profiles with
     * the help of AWS Support.
     * </p>
     * 
     * @return The name that is used to associate this playback configuration with a custom transcode profile. This
     *         overrides the dynamic transcoding defaults of MediaTailor. Use this only if you have already set up
     *         custom profiles with the help of AWS Support.
     */
    public final String transcodeProfileName() {
        return transcodeProfileName;
    }

    /**
     * <p>
     * The URL prefix for the parent manifest for the stream, minus the asset ID. The maximum length is 512 characters.
     * </p>
     * 
     * @return The URL prefix for the parent manifest for the stream, minus the asset ID. The maximum length is 512
     *         characters.
     */
    public final String videoContentSourceUrl() {
        return videoContentSourceUrl;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(adDecisionServerUrl());
        hashCode = 31 * hashCode + Objects.hashCode(availSuppression());
        hashCode = 31 * hashCode + Objects.hashCode(bumper());
        hashCode = 31 * hashCode + Objects.hashCode(cdnConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasConfigurationAliases() ? configurationAliases() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dashConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hlsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(livePreRollConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(manifestProcessingRules());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(personalizationThresholdSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(playbackConfigurationArn());
        hashCode = 31 * hashCode + Objects.hashCode(playbackEndpointPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(sessionInitializationEndpointPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(slateAdUrl());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(transcodeProfileName());
        hashCode = 31 * hashCode + Objects.hashCode(videoContentSourceUrl());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PlaybackConfiguration)) {
            return false;
        }
        PlaybackConfiguration other = (PlaybackConfiguration) obj;
        return Objects.equals(adDecisionServerUrl(), other.adDecisionServerUrl())
                && Objects.equals(availSuppression(), other.availSuppression()) && Objects.equals(bumper(), other.bumper())
                && Objects.equals(cdnConfiguration(), other.cdnConfiguration())
                && hasConfigurationAliases() == other.hasConfigurationAliases()
                && Objects.equals(configurationAliases(), other.configurationAliases())
                && Objects.equals(dashConfiguration(), other.dashConfiguration())
                && Objects.equals(hlsConfiguration(), other.hlsConfiguration())
                && Objects.equals(livePreRollConfiguration(), other.livePreRollConfiguration())
                && Objects.equals(manifestProcessingRules(), other.manifestProcessingRules())
                && Objects.equals(name(), other.name())
                && Objects.equals(personalizationThresholdSeconds(), other.personalizationThresholdSeconds())
                && Objects.equals(playbackConfigurationArn(), other.playbackConfigurationArn())
                && Objects.equals(playbackEndpointPrefix(), other.playbackEndpointPrefix())
                && Objects.equals(sessionInitializationEndpointPrefix(), other.sessionInitializationEndpointPrefix())
                && Objects.equals(slateAdUrl(), other.slateAdUrl()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(transcodeProfileName(), other.transcodeProfileName())
                && Objects.equals(videoContentSourceUrl(), other.videoContentSourceUrl());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PlaybackConfiguration").add("AdDecisionServerUrl", adDecisionServerUrl())
                .add("AvailSuppression", availSuppression()).add("Bumper", bumper()).add("CdnConfiguration", cdnConfiguration())
                .add("ConfigurationAliases", hasConfigurationAliases() ? configurationAliases() : null)
                .add("DashConfiguration", dashConfiguration()).add("HlsConfiguration", hlsConfiguration())
                .add("LivePreRollConfiguration", livePreRollConfiguration())
                .add("ManifestProcessingRules", manifestProcessingRules()).add("Name", name())
                .add("PersonalizationThresholdSeconds", personalizationThresholdSeconds())
                .add("PlaybackConfigurationArn", playbackConfigurationArn())
                .add("PlaybackEndpointPrefix", playbackEndpointPrefix())
                .add("SessionInitializationEndpointPrefix", sessionInitializationEndpointPrefix())
                .add("SlateAdUrl", slateAdUrl()).add("Tags", hasTags() ? tags() : null)
                .add("TranscodeProfileName", transcodeProfileName()).add("VideoContentSourceUrl", videoContentSourceUrl())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AdDecisionServerUrl":
            return Optional.ofNullable(clazz.cast(adDecisionServerUrl()));
        case "AvailSuppression":
            return Optional.ofNullable(clazz.cast(availSuppression()));
        case "Bumper":
            return Optional.ofNullable(clazz.cast(bumper()));
        case "CdnConfiguration":
            return Optional.ofNullable(clazz.cast(cdnConfiguration()));
        case "ConfigurationAliases":
            return Optional.ofNullable(clazz.cast(configurationAliases()));
        case "DashConfiguration":
            return Optional.ofNullable(clazz.cast(dashConfiguration()));
        case "HlsConfiguration":
            return Optional.ofNullable(clazz.cast(hlsConfiguration()));
        case "LivePreRollConfiguration":
            return Optional.ofNullable(clazz.cast(livePreRollConfiguration()));
        case "ManifestProcessingRules":
            return Optional.ofNullable(clazz.cast(manifestProcessingRules()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "PersonalizationThresholdSeconds":
            return Optional.ofNullable(clazz.cast(personalizationThresholdSeconds()));
        case "PlaybackConfigurationArn":
            return Optional.ofNullable(clazz.cast(playbackConfigurationArn()));
        case "PlaybackEndpointPrefix":
            return Optional.ofNullable(clazz.cast(playbackEndpointPrefix()));
        case "SessionInitializationEndpointPrefix":
            return Optional.ofNullable(clazz.cast(sessionInitializationEndpointPrefix()));
        case "SlateAdUrl":
            return Optional.ofNullable(clazz.cast(slateAdUrl()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "TranscodeProfileName":
            return Optional.ofNullable(clazz.cast(transcodeProfileName()));
        case "VideoContentSourceUrl":
            return Optional.ofNullable(clazz.cast(videoContentSourceUrl()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PlaybackConfiguration, T> g) {
        return obj -> g.apply((PlaybackConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PlaybackConfiguration> {
        /**
         * <p>
         * The URL for the ad decision server (ADS). This includes the specification of static parameters and
         * placeholders for dynamic parameters. AWS Elemental MediaTailor substitutes player-specific and
         * session-specific parameters as needed when calling the ADS. Alternately, for testing you can provide a static
         * VAST URL. The maximum length is 25,000 characters.
         * </p>
         * 
         * @param adDecisionServerUrl
         *        The URL for the ad decision server (ADS). This includes the specification of static parameters and
         *        placeholders for dynamic parameters. AWS Elemental MediaTailor substitutes player-specific and
         *        session-specific parameters as needed when calling the ADS. Alternately, for testing you can provide a
         *        static VAST URL. The maximum length is 25,000 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adDecisionServerUrl(String adDecisionServerUrl);

        /**
         * <p>
         * The configuration for avail suppression, also known as ad suppression. For more information about ad
         * suppression, see <a href="https://docs.aws.amazon.com/mediatailor/latest/ug/ad-behavior.html">Ad
         * Suppression</a>.
         * </p>
         * 
         * @param availSuppression
         *        The configuration for avail suppression, also known as ad suppression. For more information about ad
         *        suppression, see <a href="https://docs.aws.amazon.com/mediatailor/latest/ug/ad-behavior.html">Ad
         *        Suppression</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availSuppression(AvailSuppression availSuppression);

        /**
         * <p>
         * The configuration for avail suppression, also known as ad suppression. For more information about ad
         * suppression, see <a href="https://docs.aws.amazon.com/mediatailor/latest/ug/ad-behavior.html">Ad
         * Suppression</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link AvailSuppression.Builder} avoiding the need to
         * create one manually via {@link AvailSuppression#builder()}.
         *
         * When the {@link Consumer} completes, {@link AvailSuppression.Builder#build()} is called immediately and its
         * result is passed to {@link #availSuppression(AvailSuppression)}.
         * 
         * @param availSuppression
         *        a consumer that will call methods on {@link AvailSuppression.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #availSuppression(AvailSuppression)
         */
        default Builder availSuppression(Consumer<AvailSuppression.Builder> availSuppression) {
            return availSuppression(AvailSuppression.builder().applyMutation(availSuppression).build());
        }

        /**
         * <p>
         * The configuration for bumpers. Bumpers are short audio or video clips that play at the start or before the
         * end of an ad break. To learn more about bumpers, see <a
         * href="https://docs.aws.amazon.com/mediatailor/latest/ug/bumpers.html">Bumpers</a>.
         * </p>
         * 
         * @param bumper
         *        The configuration for bumpers. Bumpers are short audio or video clips that play at the start or before
         *        the end of an ad break. To learn more about bumpers, see <a
         *        href="https://docs.aws.amazon.com/mediatailor/latest/ug/bumpers.html">Bumpers</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bumper(Bumper bumper);

        /**
         * <p>
         * The configuration for bumpers. Bumpers are short audio or video clips that play at the start or before the
         * end of an ad break. To learn more about bumpers, see <a
         * href="https://docs.aws.amazon.com/mediatailor/latest/ug/bumpers.html">Bumpers</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link Bumper.Builder} avoiding the need to create one
         * manually via {@link Bumper#builder()}.
         *
         * When the {@link Consumer} completes, {@link Bumper.Builder#build()} is called immediately and its result is
         * passed to {@link #bumper(Bumper)}.
         * 
         * @param bumper
         *        a consumer that will call methods on {@link Bumper.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bumper(Bumper)
         */
        default Builder bumper(Consumer<Bumper.Builder> bumper) {
            return bumper(Bumper.builder().applyMutation(bumper).build());
        }

        /**
         * <p>
         * The configuration for using a content delivery network (CDN), like Amazon CloudFront, for content and ad
         * segment management.
         * </p>
         * 
         * @param cdnConfiguration
         *        The configuration for using a content delivery network (CDN), like Amazon CloudFront, for content and
         *        ad segment management.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cdnConfiguration(CdnConfiguration cdnConfiguration);

        /**
         * <p>
         * The configuration for using a content delivery network (CDN), like Amazon CloudFront, for content and ad
         * segment management.
         * </p>
         * This is a convenience that creates an instance of the {@link CdnConfiguration.Builder} avoiding the need to
         * create one manually via {@link CdnConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link CdnConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #cdnConfiguration(CdnConfiguration)}.
         * 
         * @param cdnConfiguration
         *        a consumer that will call methods on {@link CdnConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cdnConfiguration(CdnConfiguration)
         */
        default Builder cdnConfiguration(Consumer<CdnConfiguration.Builder> cdnConfiguration) {
            return cdnConfiguration(CdnConfiguration.builder().applyMutation(cdnConfiguration).build());
        }

        /**
         * <p>
         * The player parameters and aliases used as dynamic variables during session initialization. For more
         * information, see <a href="https://docs.aws.amazon.com/mediatailor/latest/ug/variables-domain.html">Domain
         * Variables</a>.
         * </p>
         * 
         * @param configurationAliases
         *        The player parameters and aliases used as dynamic variables during session initialization. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/mediatailor/latest/ug/variables-domain.html">Domain Variables</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationAliases(Map<String, Map<String, String>> configurationAliases);

        /**
         * <p>
         * The configuration for a DASH source.
         * </p>
         * 
         * @param dashConfiguration
         *        The configuration for a DASH source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dashConfiguration(DashConfiguration dashConfiguration);

        /**
         * <p>
         * The configuration for a DASH source.
         * </p>
         * This is a convenience that creates an instance of the {@link DashConfiguration.Builder} avoiding the need to
         * create one manually via {@link DashConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link DashConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #dashConfiguration(DashConfiguration)}.
         * 
         * @param dashConfiguration
         *        a consumer that will call methods on {@link DashConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dashConfiguration(DashConfiguration)
         */
        default Builder dashConfiguration(Consumer<DashConfiguration.Builder> dashConfiguration) {
            return dashConfiguration(DashConfiguration.builder().applyMutation(dashConfiguration).build());
        }

        /**
         * <p>
         * The configuration for HLS content.
         * </p>
         * 
         * @param hlsConfiguration
         *        The configuration for HLS content.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hlsConfiguration(HlsConfiguration hlsConfiguration);

        /**
         * <p>
         * The configuration for HLS content.
         * </p>
         * This is a convenience that creates an instance of the {@link HlsConfiguration.Builder} avoiding the need to
         * create one manually via {@link HlsConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link HlsConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #hlsConfiguration(HlsConfiguration)}.
         * 
         * @param hlsConfiguration
         *        a consumer that will call methods on {@link HlsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hlsConfiguration(HlsConfiguration)
         */
        default Builder hlsConfiguration(Consumer<HlsConfiguration.Builder> hlsConfiguration) {
            return hlsConfiguration(HlsConfiguration.builder().applyMutation(hlsConfiguration).build());
        }

        /**
         * <p>
         * The configuration for pre-roll ad insertion.
         * </p>
         * 
         * @param livePreRollConfiguration
         *        The configuration for pre-roll ad insertion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder livePreRollConfiguration(LivePreRollConfiguration livePreRollConfiguration);

        /**
         * <p>
         * The configuration for pre-roll ad insertion.
         * </p>
         * This is a convenience that creates an instance of the {@link LivePreRollConfiguration.Builder} avoiding the
         * need to create one manually via {@link LivePreRollConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link LivePreRollConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #livePreRollConfiguration(LivePreRollConfiguration)}.
         * 
         * @param livePreRollConfiguration
         *        a consumer that will call methods on {@link LivePreRollConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #livePreRollConfiguration(LivePreRollConfiguration)
         */
        default Builder livePreRollConfiguration(Consumer<LivePreRollConfiguration.Builder> livePreRollConfiguration) {
            return livePreRollConfiguration(LivePreRollConfiguration.builder().applyMutation(livePreRollConfiguration).build());
        }

        /**
         * <p>
         * The configuration for manifest processing rules. Manifest processing rules enable customization of the
         * personalized manifests created by MediaTailor.
         * </p>
         * 
         * @param manifestProcessingRules
         *        The configuration for manifest processing rules. Manifest processing rules enable customization of the
         *        personalized manifests created by MediaTailor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manifestProcessingRules(ManifestProcessingRules manifestProcessingRules);

        /**
         * <p>
         * The configuration for manifest processing rules. Manifest processing rules enable customization of the
         * personalized manifests created by MediaTailor.
         * </p>
         * This is a convenience that creates an instance of the {@link ManifestProcessingRules.Builder} avoiding the
         * need to create one manually via {@link ManifestProcessingRules#builder()}.
         *
         * When the {@link Consumer} completes, {@link ManifestProcessingRules.Builder#build()} is called immediately
         * and its result is passed to {@link #manifestProcessingRules(ManifestProcessingRules)}.
         * 
         * @param manifestProcessingRules
         *        a consumer that will call methods on {@link ManifestProcessingRules.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #manifestProcessingRules(ManifestProcessingRules)
         */
        default Builder manifestProcessingRules(Consumer<ManifestProcessingRules.Builder> manifestProcessingRules) {
            return manifestProcessingRules(ManifestProcessingRules.builder().applyMutation(manifestProcessingRules).build());
        }

        /**
         * <p>
         * The identifier for the playback configuration.
         * </p>
         * 
         * @param name
         *        The identifier for the playback configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Defines the maximum duration of underfilled ad time (in seconds) allowed in an ad break. If the duration of
         * underfilled ad time exceeds the personalization threshold, then the personalization of the ad break is
         * abandoned and the underlying content is shown. This feature applies to <i>ad replacement</i> in live and VOD
         * streams, rather than ad insertion, because it relies on an underlying content stream. For more information
         * about ad break behavior, including ad replacement and insertion, see <a
         * href="https://docs.aws.amazon.com/mediatailor/latest/ug/ad-behavior.html">Ad Behavior in AWS Elemental
         * MediaTailor</a>.
         * </p>
         * 
         * @param personalizationThresholdSeconds
         *        Defines the maximum duration of underfilled ad time (in seconds) allowed in an ad break. If the
         *        duration of underfilled ad time exceeds the personalization threshold, then the personalization of the
         *        ad break is abandoned and the underlying content is shown. This feature applies to <i>ad
         *        replacement</i> in live and VOD streams, rather than ad insertion, because it relies on an underlying
         *        content stream. For more information about ad break behavior, including ad replacement and insertion,
         *        see <a href="https://docs.aws.amazon.com/mediatailor/latest/ug/ad-behavior.html">Ad Behavior in AWS
         *        Elemental MediaTailor</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder personalizationThresholdSeconds(Integer personalizationThresholdSeconds);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the playback configuration.
         * </p>
         * 
         * @param playbackConfigurationArn
         *        The Amazon Resource Name (ARN) for the playback configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playbackConfigurationArn(String playbackConfigurationArn);

        /**
         * <p>
         * The URL that the player accesses to get a manifest from AWS Elemental MediaTailor.
         * </p>
         * 
         * @param playbackEndpointPrefix
         *        The URL that the player accesses to get a manifest from AWS Elemental MediaTailor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playbackEndpointPrefix(String playbackEndpointPrefix);

        /**
         * <p>
         * The URL that the player uses to initialize a session that uses client-side reporting.
         * </p>
         * 
         * @param sessionInitializationEndpointPrefix
         *        The URL that the player uses to initialize a session that uses client-side reporting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionInitializationEndpointPrefix(String sessionInitializationEndpointPrefix);

        /**
         * <p>
         * The URL for a video asset to transcode and use to fill in time that's not used by ads. AWS Elemental
         * MediaTailor shows the slate to fill in gaps in media content. Configuring the slate is optional for non-VPAID
         * playback configurations. For VPAID, the slate is required because MediaTailor provides it in the slots
         * designated for dynamic ad content. The slate must be a high-quality asset that contains both audio and video.
         * </p>
         * 
         * @param slateAdUrl
         *        The URL for a video asset to transcode and use to fill in time that's not used by ads. AWS Elemental
         *        MediaTailor shows the slate to fill in gaps in media content. Configuring the slate is optional for
         *        non-VPAID playback configurations. For VPAID, the slate is required because MediaTailor provides it in
         *        the slots designated for dynamic ad content. The slate must be a high-quality asset that contains both
         *        audio and video.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slateAdUrl(String slateAdUrl);

        /**
         * <p>
         * The tags to assign to the playback configuration.
         * </p>
         * 
         * @param tags
         *        The tags to assign to the playback configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The name that is used to associate this playback configuration with a custom transcode profile. This
         * overrides the dynamic transcoding defaults of MediaTailor. Use this only if you have already set up custom
         * profiles with the help of AWS Support.
         * </p>
         * 
         * @param transcodeProfileName
         *        The name that is used to associate this playback configuration with a custom transcode profile. This
         *        overrides the dynamic transcoding defaults of MediaTailor. Use this only if you have already set up
         *        custom profiles with the help of AWS Support.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transcodeProfileName(String transcodeProfileName);

        /**
         * <p>
         * The URL prefix for the parent manifest for the stream, minus the asset ID. The maximum length is 512
         * characters.
         * </p>
         * 
         * @param videoContentSourceUrl
         *        The URL prefix for the parent manifest for the stream, minus the asset ID. The maximum length is 512
         *        characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder videoContentSourceUrl(String videoContentSourceUrl);
    }

    static final class BuilderImpl implements Builder {
        private String adDecisionServerUrl;

        private AvailSuppression availSuppression;

        private Bumper bumper;

        private CdnConfiguration cdnConfiguration;

        private Map<String, Map<String, String>> configurationAliases = DefaultSdkAutoConstructMap.getInstance();

        private DashConfiguration dashConfiguration;

        private HlsConfiguration hlsConfiguration;

        private LivePreRollConfiguration livePreRollConfiguration;

        private ManifestProcessingRules manifestProcessingRules;

        private String name;

        private Integer personalizationThresholdSeconds;

        private String playbackConfigurationArn;

        private String playbackEndpointPrefix;

        private String sessionInitializationEndpointPrefix;

        private String slateAdUrl;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String transcodeProfileName;

        private String videoContentSourceUrl;

        private BuilderImpl() {
        }

        private BuilderImpl(PlaybackConfiguration model) {
            adDecisionServerUrl(model.adDecisionServerUrl);
            availSuppression(model.availSuppression);
            bumper(model.bumper);
            cdnConfiguration(model.cdnConfiguration);
            configurationAliases(model.configurationAliases);
            dashConfiguration(model.dashConfiguration);
            hlsConfiguration(model.hlsConfiguration);
            livePreRollConfiguration(model.livePreRollConfiguration);
            manifestProcessingRules(model.manifestProcessingRules);
            name(model.name);
            personalizationThresholdSeconds(model.personalizationThresholdSeconds);
            playbackConfigurationArn(model.playbackConfigurationArn);
            playbackEndpointPrefix(model.playbackEndpointPrefix);
            sessionInitializationEndpointPrefix(model.sessionInitializationEndpointPrefix);
            slateAdUrl(model.slateAdUrl);
            tags(model.tags);
            transcodeProfileName(model.transcodeProfileName);
            videoContentSourceUrl(model.videoContentSourceUrl);
        }

        public final String getAdDecisionServerUrl() {
            return adDecisionServerUrl;
        }

        @Override
        public final Builder adDecisionServerUrl(String adDecisionServerUrl) {
            this.adDecisionServerUrl = adDecisionServerUrl;
            return this;
        }

        public final void setAdDecisionServerUrl(String adDecisionServerUrl) {
            this.adDecisionServerUrl = adDecisionServerUrl;
        }

        public final AvailSuppression.Builder getAvailSuppression() {
            return availSuppression != null ? availSuppression.toBuilder() : null;
        }

        @Override
        public final Builder availSuppression(AvailSuppression availSuppression) {
            this.availSuppression = availSuppression;
            return this;
        }

        public final void setAvailSuppression(AvailSuppression.BuilderImpl availSuppression) {
            this.availSuppression = availSuppression != null ? availSuppression.build() : null;
        }

        public final Bumper.Builder getBumper() {
            return bumper != null ? bumper.toBuilder() : null;
        }

        @Override
        public final Builder bumper(Bumper bumper) {
            this.bumper = bumper;
            return this;
        }

        public final void setBumper(Bumper.BuilderImpl bumper) {
            this.bumper = bumper != null ? bumper.build() : null;
        }

        public final CdnConfiguration.Builder getCdnConfiguration() {
            return cdnConfiguration != null ? cdnConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder cdnConfiguration(CdnConfiguration cdnConfiguration) {
            this.cdnConfiguration = cdnConfiguration;
            return this;
        }

        public final void setCdnConfiguration(CdnConfiguration.BuilderImpl cdnConfiguration) {
            this.cdnConfiguration = cdnConfiguration != null ? cdnConfiguration.build() : null;
        }

        public final Map<String, Map<String, String>> getConfigurationAliases() {
            if (configurationAliases instanceof SdkAutoConstructMap) {
                return null;
            }
            return configurationAliases;
        }

        @Override
        public final Builder configurationAliases(Map<String, Map<String, String>> configurationAliases) {
            this.configurationAliases = ConfigurationAliasesResponseCopier.copy(configurationAliases);
            return this;
        }

        public final void setConfigurationAliases(Map<String, Map<String, String>> configurationAliases) {
            this.configurationAliases = ConfigurationAliasesResponseCopier.copy(configurationAliases);
        }

        public final DashConfiguration.Builder getDashConfiguration() {
            return dashConfiguration != null ? dashConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder dashConfiguration(DashConfiguration dashConfiguration) {
            this.dashConfiguration = dashConfiguration;
            return this;
        }

        public final void setDashConfiguration(DashConfiguration.BuilderImpl dashConfiguration) {
            this.dashConfiguration = dashConfiguration != null ? dashConfiguration.build() : null;
        }

        public final HlsConfiguration.Builder getHlsConfiguration() {
            return hlsConfiguration != null ? hlsConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder hlsConfiguration(HlsConfiguration hlsConfiguration) {
            this.hlsConfiguration = hlsConfiguration;
            return this;
        }

        public final void setHlsConfiguration(HlsConfiguration.BuilderImpl hlsConfiguration) {
            this.hlsConfiguration = hlsConfiguration != null ? hlsConfiguration.build() : null;
        }

        public final LivePreRollConfiguration.Builder getLivePreRollConfiguration() {
            return livePreRollConfiguration != null ? livePreRollConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder livePreRollConfiguration(LivePreRollConfiguration livePreRollConfiguration) {
            this.livePreRollConfiguration = livePreRollConfiguration;
            return this;
        }

        public final void setLivePreRollConfiguration(LivePreRollConfiguration.BuilderImpl livePreRollConfiguration) {
            this.livePreRollConfiguration = livePreRollConfiguration != null ? livePreRollConfiguration.build() : null;
        }

        public final ManifestProcessingRules.Builder getManifestProcessingRules() {
            return manifestProcessingRules != null ? manifestProcessingRules.toBuilder() : null;
        }

        @Override
        public final Builder manifestProcessingRules(ManifestProcessingRules manifestProcessingRules) {
            this.manifestProcessingRules = manifestProcessingRules;
            return this;
        }

        public final void setManifestProcessingRules(ManifestProcessingRules.BuilderImpl manifestProcessingRules) {
            this.manifestProcessingRules = manifestProcessingRules != null ? manifestProcessingRules.build() : null;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Integer getPersonalizationThresholdSeconds() {
            return personalizationThresholdSeconds;
        }

        @Override
        public final Builder personalizationThresholdSeconds(Integer personalizationThresholdSeconds) {
            this.personalizationThresholdSeconds = personalizationThresholdSeconds;
            return this;
        }

        public final void setPersonalizationThresholdSeconds(Integer personalizationThresholdSeconds) {
            this.personalizationThresholdSeconds = personalizationThresholdSeconds;
        }

        public final String getPlaybackConfigurationArn() {
            return playbackConfigurationArn;
        }

        @Override
        public final Builder playbackConfigurationArn(String playbackConfigurationArn) {
            this.playbackConfigurationArn = playbackConfigurationArn;
            return this;
        }

        public final void setPlaybackConfigurationArn(String playbackConfigurationArn) {
            this.playbackConfigurationArn = playbackConfigurationArn;
        }

        public final String getPlaybackEndpointPrefix() {
            return playbackEndpointPrefix;
        }

        @Override
        public final Builder playbackEndpointPrefix(String playbackEndpointPrefix) {
            this.playbackEndpointPrefix = playbackEndpointPrefix;
            return this;
        }

        public final void setPlaybackEndpointPrefix(String playbackEndpointPrefix) {
            this.playbackEndpointPrefix = playbackEndpointPrefix;
        }

        public final String getSessionInitializationEndpointPrefix() {
            return sessionInitializationEndpointPrefix;
        }

        @Override
        public final Builder sessionInitializationEndpointPrefix(String sessionInitializationEndpointPrefix) {
            this.sessionInitializationEndpointPrefix = sessionInitializationEndpointPrefix;
            return this;
        }

        public final void setSessionInitializationEndpointPrefix(String sessionInitializationEndpointPrefix) {
            this.sessionInitializationEndpointPrefix = sessionInitializationEndpointPrefix;
        }

        public final String getSlateAdUrl() {
            return slateAdUrl;
        }

        @Override
        public final Builder slateAdUrl(String slateAdUrl) {
            this.slateAdUrl = slateAdUrl;
            return this;
        }

        public final void setSlateAdUrl(String slateAdUrl) {
            this.slateAdUrl = slateAdUrl;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = ___mapOf__stringCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = ___mapOf__stringCopier.copy(tags);
        }

        public final String getTranscodeProfileName() {
            return transcodeProfileName;
        }

        @Override
        public final Builder transcodeProfileName(String transcodeProfileName) {
            this.transcodeProfileName = transcodeProfileName;
            return this;
        }

        public final void setTranscodeProfileName(String transcodeProfileName) {
            this.transcodeProfileName = transcodeProfileName;
        }

        public final String getVideoContentSourceUrl() {
            return videoContentSourceUrl;
        }

        @Override
        public final Builder videoContentSourceUrl(String videoContentSourceUrl) {
            this.videoContentSourceUrl = videoContentSourceUrl;
            return this;
        }

        public final void setVideoContentSourceUrl(String videoContentSourceUrl) {
            this.videoContentSourceUrl = videoContentSourceUrl;
        }

        @Override
        public PlaybackConfiguration build() {
            return new PlaybackConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
