/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediatailor;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediatailor.model.BadRequestException;
import software.amazon.awssdk.services.mediatailor.model.CreateChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.CreateChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.CreateProgramRequest;
import software.amazon.awssdk.services.mediatailor.model.CreateProgramResponse;
import software.amazon.awssdk.services.mediatailor.model.CreateSourceLocationRequest;
import software.amazon.awssdk.services.mediatailor.model.CreateSourceLocationResponse;
import software.amazon.awssdk.services.mediatailor.model.CreateVodSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.CreateVodSourceResponse;
import software.amazon.awssdk.services.mediatailor.model.DeleteChannelPolicyRequest;
import software.amazon.awssdk.services.mediatailor.model.DeleteChannelPolicyResponse;
import software.amazon.awssdk.services.mediatailor.model.DeleteChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.DeleteChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.DeletePlaybackConfigurationRequest;
import software.amazon.awssdk.services.mediatailor.model.DeletePlaybackConfigurationResponse;
import software.amazon.awssdk.services.mediatailor.model.DeleteProgramRequest;
import software.amazon.awssdk.services.mediatailor.model.DeleteProgramResponse;
import software.amazon.awssdk.services.mediatailor.model.DeleteSourceLocationRequest;
import software.amazon.awssdk.services.mediatailor.model.DeleteSourceLocationResponse;
import software.amazon.awssdk.services.mediatailor.model.DeleteVodSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.DeleteVodSourceResponse;
import software.amazon.awssdk.services.mediatailor.model.DescribeChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.DescribeChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.DescribeProgramRequest;
import software.amazon.awssdk.services.mediatailor.model.DescribeProgramResponse;
import software.amazon.awssdk.services.mediatailor.model.DescribeSourceLocationRequest;
import software.amazon.awssdk.services.mediatailor.model.DescribeSourceLocationResponse;
import software.amazon.awssdk.services.mediatailor.model.DescribeVodSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.DescribeVodSourceResponse;
import software.amazon.awssdk.services.mediatailor.model.GetChannelPolicyRequest;
import software.amazon.awssdk.services.mediatailor.model.GetChannelPolicyResponse;
import software.amazon.awssdk.services.mediatailor.model.GetChannelScheduleRequest;
import software.amazon.awssdk.services.mediatailor.model.GetChannelScheduleResponse;
import software.amazon.awssdk.services.mediatailor.model.GetPlaybackConfigurationRequest;
import software.amazon.awssdk.services.mediatailor.model.GetPlaybackConfigurationResponse;
import software.amazon.awssdk.services.mediatailor.model.ListChannelsRequest;
import software.amazon.awssdk.services.mediatailor.model.ListChannelsResponse;
import software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsRequest;
import software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsResponse;
import software.amazon.awssdk.services.mediatailor.model.ListSourceLocationsRequest;
import software.amazon.awssdk.services.mediatailor.model.ListSourceLocationsResponse;
import software.amazon.awssdk.services.mediatailor.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediatailor.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediatailor.model.ListVodSourcesRequest;
import software.amazon.awssdk.services.mediatailor.model.ListVodSourcesResponse;
import software.amazon.awssdk.services.mediatailor.model.MediaTailorException;
import software.amazon.awssdk.services.mediatailor.model.MediaTailorRequest;
import software.amazon.awssdk.services.mediatailor.model.PutChannelPolicyRequest;
import software.amazon.awssdk.services.mediatailor.model.PutChannelPolicyResponse;
import software.amazon.awssdk.services.mediatailor.model.PutPlaybackConfigurationRequest;
import software.amazon.awssdk.services.mediatailor.model.PutPlaybackConfigurationResponse;
import software.amazon.awssdk.services.mediatailor.model.StartChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.StartChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.StopChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.StopChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.TagResourceRequest;
import software.amazon.awssdk.services.mediatailor.model.TagResourceResponse;
import software.amazon.awssdk.services.mediatailor.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediatailor.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediatailor.model.UpdateChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.UpdateChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.UpdateSourceLocationRequest;
import software.amazon.awssdk.services.mediatailor.model.UpdateSourceLocationResponse;
import software.amazon.awssdk.services.mediatailor.model.UpdateVodSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.UpdateVodSourceResponse;
import software.amazon.awssdk.services.mediatailor.paginators.GetChannelScheduleIterable;
import software.amazon.awssdk.services.mediatailor.paginators.ListChannelsIterable;
import software.amazon.awssdk.services.mediatailor.paginators.ListPlaybackConfigurationsIterable;
import software.amazon.awssdk.services.mediatailor.paginators.ListSourceLocationsIterable;
import software.amazon.awssdk.services.mediatailor.paginators.ListVodSourcesIterable;
import software.amazon.awssdk.services.mediatailor.transform.CreateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.CreateProgramRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.CreateSourceLocationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.CreateVodSourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeleteChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeleteChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeletePlaybackConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeleteProgramRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeleteSourceLocationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeleteVodSourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DescribeChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DescribeProgramRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DescribeSourceLocationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DescribeVodSourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.GetChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.GetChannelScheduleRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.GetPlaybackConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListPlaybackConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListSourceLocationsRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListVodSourcesRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.PutChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.PutPlaybackConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.StartChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.StopChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.UpdateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.UpdateSourceLocationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.UpdateVodSourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MediaTailorClient}.
 *
 * @see MediaTailorClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaTailorClient implements MediaTailorClient {
    private static final Logger log = Logger.loggerFor(DefaultMediaTailorClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaTailorClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a channel.
     * </p>
     *
     * @param createChannelRequest
     * @return Result of the CreateChannel operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.CreateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/CreateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateChannelResponse createChannel(CreateChannelRequest createChannelRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannel");

            return clientHandler.execute(new ClientExecutionParams<CreateChannelRequest, CreateChannelResponse>()
                    .withOperationName("CreateChannel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a program.
     * </p>
     *
     * @param createProgramRequest
     * @return Result of the CreateProgram operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.CreateProgram
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/CreateProgram" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateProgramResponse createProgram(CreateProgramRequest createProgramRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProgramResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProgramResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProgramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProgram");

            return clientHandler.execute(new ClientExecutionParams<CreateProgramRequest, CreateProgramResponse>()
                    .withOperationName("CreateProgram").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createProgramRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateProgramRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a source location on a specific channel.
     * </p>
     *
     * @param createSourceLocationRequest
     * @return Result of the CreateSourceLocation operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.CreateSourceLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/CreateSourceLocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSourceLocationResponse createSourceLocation(CreateSourceLocationRequest createSourceLocationRequest)
            throws AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSourceLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSourceLocationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSourceLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSourceLocation");

            return clientHandler.execute(new ClientExecutionParams<CreateSourceLocationRequest, CreateSourceLocationResponse>()
                    .withOperationName("CreateSourceLocation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSourceLocationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSourceLocationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates name for a specific VOD source in a source location.
     * </p>
     *
     * @param createVodSourceRequest
     * @return Result of the CreateVodSource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.CreateVodSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/CreateVodSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateVodSourceResponse createVodSource(CreateVodSourceRequest createVodSourceRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVodSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateVodSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVodSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVodSource");

            return clientHandler.execute(new ClientExecutionParams<CreateVodSourceRequest, CreateVodSourceResponse>()
                    .withOperationName("CreateVodSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createVodSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVodSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a channel. You must stop the channel before it can be deleted.
     * </p>
     *
     * @param deleteChannelRequest
     * @return Result of the DeleteChannel operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.DeleteChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeleteChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteChannelResponse deleteChannel(DeleteChannelRequest deleteChannelRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannel");

            return clientHandler.execute(new ClientExecutionParams<DeleteChannelRequest, DeleteChannelResponse>()
                    .withOperationName("DeleteChannel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a channel's IAM policy.
     * </p>
     *
     * @param deleteChannelPolicyRequest
     * @return Result of the DeleteChannelPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.DeleteChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeleteChannelPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteChannelPolicyResponse deleteChannelPolicy(DeleteChannelPolicyRequest deleteChannelPolicyRequest)
            throws AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteChannelPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannelPolicy");

            return clientHandler.execute(new ClientExecutionParams<DeleteChannelPolicyRequest, DeleteChannelPolicyResponse>()
                    .withOperationName("DeleteChannelPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteChannelPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteChannelPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the playback configuration for the specified name.
     * </p>
     *
     * @param deletePlaybackConfigurationRequest
     * @return Result of the DeletePlaybackConfiguration operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.DeletePlaybackConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeletePlaybackConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeletePlaybackConfigurationResponse deletePlaybackConfiguration(
            DeletePlaybackConfigurationRequest deletePlaybackConfigurationRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePlaybackConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePlaybackConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePlaybackConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePlaybackConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeletePlaybackConfigurationRequest, DeletePlaybackConfigurationResponse>()
                            .withOperationName("DeletePlaybackConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deletePlaybackConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeletePlaybackConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specific program on a specific channel.
     * </p>
     *
     * @param deleteProgramRequest
     * @return Result of the DeleteProgram operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.DeleteProgram
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeleteProgram" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteProgramResponse deleteProgram(DeleteProgramRequest deleteProgramRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProgramResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteProgramResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProgramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProgram");

            return clientHandler.execute(new ClientExecutionParams<DeleteProgramRequest, DeleteProgramResponse>()
                    .withOperationName("DeleteProgram").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteProgramRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteProgramRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a source location on a specific channel.
     * </p>
     *
     * @param deleteSourceLocationRequest
     * @return Result of the DeleteSourceLocation operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.DeleteSourceLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeleteSourceLocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSourceLocationResponse deleteSourceLocation(DeleteSourceLocationRequest deleteSourceLocationRequest)
            throws AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSourceLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSourceLocationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSourceLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSourceLocation");

            return clientHandler.execute(new ClientExecutionParams<DeleteSourceLocationRequest, DeleteSourceLocationResponse>()
                    .withOperationName("DeleteSourceLocation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSourceLocationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSourceLocationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specific VOD source in a specific source location.
     * </p>
     *
     * @param deleteVodSourceRequest
     * @return Result of the DeleteVodSource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.DeleteVodSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeleteVodSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteVodSourceResponse deleteVodSource(DeleteVodSourceRequest deleteVodSourceRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVodSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteVodSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVodSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVodSource");

            return clientHandler.execute(new ClientExecutionParams<DeleteVodSourceRequest, DeleteVodSourceResponse>()
                    .withOperationName("DeleteVodSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteVodSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVodSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the properties of a specific channel.
     * </p>
     *
     * @param describeChannelRequest
     * @return Result of the DescribeChannel operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.DescribeChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DescribeChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeChannelResponse describeChannel(DescribeChannelRequest describeChannelRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannel");

            return clientHandler.execute(new ClientExecutionParams<DescribeChannelRequest, DescribeChannelResponse>()
                    .withOperationName("DescribeChannel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the properties of the requested program.
     * </p>
     *
     * @param describeProgramRequest
     * @return Result of the DescribeProgram operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.DescribeProgram
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DescribeProgram" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeProgramResponse describeProgram(DescribeProgramRequest describeProgramRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeProgramResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeProgramResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProgramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProgram");

            return clientHandler.execute(new ClientExecutionParams<DescribeProgramRequest, DescribeProgramResponse>()
                    .withOperationName("DescribeProgram").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeProgramRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeProgramRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the properties of the requested source location.
     * </p>
     *
     * @param describeSourceLocationRequest
     * @return Result of the DescribeSourceLocation operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.DescribeSourceLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DescribeSourceLocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeSourceLocationResponse describeSourceLocation(DescribeSourceLocationRequest describeSourceLocationRequest)
            throws AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSourceLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSourceLocationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSourceLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSourceLocation");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeSourceLocationRequest, DescribeSourceLocationResponse>()
                            .withOperationName("DescribeSourceLocation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeSourceLocationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeSourceLocationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides details about a specific VOD source in a specific source location.
     * </p>
     *
     * @param describeVodSourceRequest
     * @return Result of the DescribeVodSource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.DescribeVodSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DescribeVodSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeVodSourceResponse describeVodSource(DescribeVodSourceRequest describeVodSourceRequest)
            throws AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeVodSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeVodSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVodSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVodSource");

            return clientHandler.execute(new ClientExecutionParams<DescribeVodSourceRequest, DescribeVodSourceResponse>()
                    .withOperationName("DescribeVodSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeVodSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeVodSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a channel's IAM policy.
     * </p>
     *
     * @param getChannelPolicyRequest
     * @return Result of the GetChannelPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.GetChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/GetChannelPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetChannelPolicyResponse getChannelPolicy(GetChannelPolicyRequest getChannelPolicyRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetChannelPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannelPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetChannelPolicyRequest, GetChannelPolicyResponse>()
                    .withOperationName("GetChannelPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getChannelPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetChannelPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about your channel's schedule.
     * </p>
     *
     * @param getChannelScheduleRequest
     * @return Result of the GetChannelSchedule operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.GetChannelSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/GetChannelSchedule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetChannelScheduleResponse getChannelSchedule(GetChannelScheduleRequest getChannelScheduleRequest)
            throws AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetChannelScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetChannelScheduleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannelSchedule");

            return clientHandler.execute(new ClientExecutionParams<GetChannelScheduleRequest, GetChannelScheduleResponse>()
                    .withOperationName("GetChannelSchedule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getChannelScheduleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetChannelScheduleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about your channel's schedule.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getChannelSchedule(software.amazon.awssdk.services.mediatailor.model.GetChannelScheduleRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.GetChannelScheduleIterable responses = client.getChannelSchedulePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediatailor.paginators.GetChannelScheduleIterable responses = client
     *             .getChannelSchedulePaginator(request);
     *     for (software.amazon.awssdk.services.mediatailor.model.GetChannelScheduleResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.GetChannelScheduleIterable responses = client.getChannelSchedulePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getChannelSchedule(software.amazon.awssdk.services.mediatailor.model.GetChannelScheduleRequest)}
     * operation.</b>
     * </p>
     *
     * @param getChannelScheduleRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.GetChannelSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/GetChannelSchedule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetChannelScheduleIterable getChannelSchedulePaginator(GetChannelScheduleRequest getChannelScheduleRequest)
            throws AwsServiceException, SdkClientException, MediaTailorException {
        return new GetChannelScheduleIterable(this, applyPaginatorUserAgent(getChannelScheduleRequest));
    }

    /**
     * <p>
     * Returns the playback configuration for the specified name.
     * </p>
     *
     * @param getPlaybackConfigurationRequest
     * @return Result of the GetPlaybackConfiguration operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.GetPlaybackConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/GetPlaybackConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPlaybackConfigurationResponse getPlaybackConfiguration(
            GetPlaybackConfigurationRequest getPlaybackConfigurationRequest) throws AwsServiceException, SdkClientException,
            MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPlaybackConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPlaybackConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlaybackConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlaybackConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetPlaybackConfigurationRequest, GetPlaybackConfigurationResponse>()
                            .withOperationName("GetPlaybackConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getPlaybackConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetPlaybackConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of channels that are associated with this account.
     * </p>
     *
     * @param listChannelsRequest
     * @return Result of the ListChannels operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListChannelsResponse listChannels(ListChannelsRequest listChannelsRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListChannelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannels");

            return clientHandler.execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                    .withOperationName("ListChannels").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listChannelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListChannelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of channels that are associated with this account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listChannels(software.amazon.awssdk.services.mediatailor.model.ListChannelsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediatailor.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     *     for (software.amazon.awssdk.services.mediatailor.model.ListChannelsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannels(software.amazon.awssdk.services.mediatailor.model.ListChannelsRequest)} operation.</b>
     * </p>
     *
     * @param listChannelsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListChannelsIterable listChannelsPaginator(ListChannelsRequest listChannelsRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        return new ListChannelsIterable(this, applyPaginatorUserAgent(listChannelsRequest));
    }

    /**
     * <p>
     * Returns a list of the playback configurations defined in AWS Elemental MediaTailor. You can specify a maximum
     * number of configurations to return at a time. The default maximum is 50. Results are returned in pagefuls. If
     * MediaTailor has more configurations than the specified maximum, it provides parameters in the response that you
     * can use to retrieve the next pageful.
     * </p>
     *
     * @param listPlaybackConfigurationsRequest
     * @return Result of the ListPlaybackConfigurations operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.ListPlaybackConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListPlaybackConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPlaybackConfigurationsResponse listPlaybackConfigurations(
            ListPlaybackConfigurationsRequest listPlaybackConfigurationsRequest) throws AwsServiceException, SdkClientException,
            MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPlaybackConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPlaybackConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPlaybackConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPlaybackConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListPlaybackConfigurationsRequest, ListPlaybackConfigurationsResponse>()
                            .withOperationName("ListPlaybackConfigurations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listPlaybackConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListPlaybackConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the playback configurations defined in AWS Elemental MediaTailor. You can specify a maximum
     * number of configurations to return at a time. The default maximum is 50. Results are returned in pagefuls. If
     * MediaTailor has more configurations than the specified maximum, it provides parameters in the response that you
     * can use to retrieve the next pageful.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPlaybackConfigurations(software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListPlaybackConfigurationsIterable responses = client.listPlaybackConfigurationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediatailor.paginators.ListPlaybackConfigurationsIterable responses = client
     *             .listPlaybackConfigurationsPaginator(request);
     *     for (software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListPlaybackConfigurationsIterable responses = client.listPlaybackConfigurationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlaybackConfigurations(software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPlaybackConfigurationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.ListPlaybackConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListPlaybackConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPlaybackConfigurationsIterable listPlaybackConfigurationsPaginator(
            ListPlaybackConfigurationsRequest listPlaybackConfigurationsRequest) throws AwsServiceException, SdkClientException,
            MediaTailorException {
        return new ListPlaybackConfigurationsIterable(this, applyPaginatorUserAgent(listPlaybackConfigurationsRequest));
    }

    /**
     * <p>
     * Retrieves a list of source locations.
     * </p>
     *
     * @param listSourceLocationsRequest
     * @return Result of the ListSourceLocations operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.ListSourceLocations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListSourceLocations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSourceLocationsResponse listSourceLocations(ListSourceLocationsRequest listSourceLocationsRequest)
            throws AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSourceLocationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSourceLocationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSourceLocationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSourceLocations");

            return clientHandler.execute(new ClientExecutionParams<ListSourceLocationsRequest, ListSourceLocationsResponse>()
                    .withOperationName("ListSourceLocations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSourceLocationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSourceLocationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of source locations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSourceLocations(software.amazon.awssdk.services.mediatailor.model.ListSourceLocationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListSourceLocationsIterable responses = client.listSourceLocationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediatailor.paginators.ListSourceLocationsIterable responses = client
     *             .listSourceLocationsPaginator(request);
     *     for (software.amazon.awssdk.services.mediatailor.model.ListSourceLocationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListSourceLocationsIterable responses = client.listSourceLocationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSourceLocations(software.amazon.awssdk.services.mediatailor.model.ListSourceLocationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSourceLocationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.ListSourceLocations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListSourceLocations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSourceLocationsIterable listSourceLocationsPaginator(ListSourceLocationsRequest listSourceLocationsRequest)
            throws AwsServiceException, SdkClientException, MediaTailorException {
        return new ListSourceLocationsIterable(this, applyPaginatorUserAgent(listSourceLocationsRequest));
    }

    /**
     * <p>
     * Returns a list of the tags assigned to the specified playback configuration resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws BadRequestException
     *         Invalid request parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the VOD sources in a source location.
     * </p>
     *
     * @param listVodSourcesRequest
     * @return Result of the ListVodSources operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.ListVodSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListVodSources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVodSourcesResponse listVodSources(ListVodSourcesRequest listVodSourcesRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVodSourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListVodSourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVodSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVodSources");

            return clientHandler.execute(new ClientExecutionParams<ListVodSourcesRequest, ListVodSourcesResponse>()
                    .withOperationName("ListVodSources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listVodSourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVodSourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the VOD sources in a source location.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVodSources(software.amazon.awssdk.services.mediatailor.model.ListVodSourcesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListVodSourcesIterable responses = client.listVodSourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediatailor.paginators.ListVodSourcesIterable responses = client
     *             .listVodSourcesPaginator(request);
     *     for (software.amazon.awssdk.services.mediatailor.model.ListVodSourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListVodSourcesIterable responses = client.listVodSourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVodSources(software.amazon.awssdk.services.mediatailor.model.ListVodSourcesRequest)} operation.</b>
     * </p>
     *
     * @param listVodSourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.ListVodSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListVodSources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListVodSourcesIterable listVodSourcesPaginator(ListVodSourcesRequest listVodSourcesRequest)
            throws AwsServiceException, SdkClientException, MediaTailorException {
        return new ListVodSourcesIterable(this, applyPaginatorUserAgent(listVodSourcesRequest));
    }

    /**
     * <p>
     * Creates an IAM policy for the channel.
     * </p>
     *
     * @param putChannelPolicyRequest
     * @return Result of the PutChannelPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.PutChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/PutChannelPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutChannelPolicyResponse putChannelPolicy(PutChannelPolicyRequest putChannelPolicyRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutChannelPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutChannelPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutChannelPolicyRequest, PutChannelPolicyResponse>()
                    .withOperationName("PutChannelPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putChannelPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutChannelPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a new playback configuration to AWS Elemental MediaTailor.
     * </p>
     *
     * @param putPlaybackConfigurationRequest
     * @return Result of the PutPlaybackConfiguration operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.PutPlaybackConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/PutPlaybackConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutPlaybackConfigurationResponse putPlaybackConfiguration(
            PutPlaybackConfigurationRequest putPlaybackConfigurationRequest) throws AwsServiceException, SdkClientException,
            MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutPlaybackConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutPlaybackConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPlaybackConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPlaybackConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<PutPlaybackConfigurationRequest, PutPlaybackConfigurationResponse>()
                            .withOperationName("PutPlaybackConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putPlaybackConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutPlaybackConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a specific channel.
     * </p>
     *
     * @param startChannelRequest
     * @return Result of the StartChannel operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.StartChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/StartChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartChannelResponse startChannel(StartChannelRequest startChannelRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartChannel");

            return clientHandler.execute(new ClientExecutionParams<StartChannelRequest, StartChannelResponse>()
                    .withOperationName("StartChannel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a specific channel.
     * </p>
     *
     * @param stopChannelRequest
     * @return Result of the StopChannel operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.StopChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/StopChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopChannelResponse stopChannel(StopChannelRequest stopChannelRequest) throws AwsServiceException, SdkClientException,
            MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopChannel");

            return clientHandler.execute(new ClientExecutionParams<StopChannelRequest, StopChannelResponse>()
                    .withOperationName("StopChannel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to the specified playback configuration resource. You can specify one or more tags to add.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws BadRequestException
     *         Invalid request parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws BadRequestException,
            AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the specified playback configuration resource. You can specify one or more tags to remove.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws BadRequestException
     *         Invalid request parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws BadRequestException,
            AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing channel.
     * </p>
     *
     * @param updateChannelRequest
     * @return Result of the UpdateChannel operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.UpdateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/UpdateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateChannelResponse updateChannel(UpdateChannelRequest updateChannelRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannel");

            return clientHandler.execute(new ClientExecutionParams<UpdateChannelRequest, UpdateChannelResponse>()
                    .withOperationName("UpdateChannel").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a source location on a specific channel.
     * </p>
     *
     * @param updateSourceLocationRequest
     * @return Result of the UpdateSourceLocation operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.UpdateSourceLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/UpdateSourceLocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateSourceLocationResponse updateSourceLocation(UpdateSourceLocationRequest updateSourceLocationRequest)
            throws AwsServiceException, SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSourceLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSourceLocationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSourceLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSourceLocation");

            return clientHandler.execute(new ClientExecutionParams<UpdateSourceLocationRequest, UpdateSourceLocationResponse>()
                    .withOperationName("UpdateSourceLocation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateSourceLocationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSourceLocationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a specific VOD source in a specific source location.
     * </p>
     *
     * @param updateVodSourceRequest
     * @return Result of the UpdateVodSource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaTailorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaTailorClient.UpdateVodSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/UpdateVodSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateVodSourceResponse updateVodSource(UpdateVodSourceRequest updateVodSourceRequest) throws AwsServiceException,
            SdkClientException, MediaTailorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateVodSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateVodSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVodSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVodSource");

            return clientHandler.execute(new ClientExecutionParams<UpdateVodSourceRequest, UpdateVodSourceResponse>()
                    .withOperationName("UpdateVodSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateVodSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateVodSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaTailorException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MediaTailorRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
