/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediatailor;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediatailor.model.BadRequestException;
import software.amazon.awssdk.services.mediatailor.model.ConfigureLogsForChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.ConfigureLogsForChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.ConfigureLogsForPlaybackConfigurationRequest;
import software.amazon.awssdk.services.mediatailor.model.ConfigureLogsForPlaybackConfigurationResponse;
import software.amazon.awssdk.services.mediatailor.model.CreateChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.CreateChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.CreateLiveSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.CreateLiveSourceResponse;
import software.amazon.awssdk.services.mediatailor.model.CreatePrefetchScheduleRequest;
import software.amazon.awssdk.services.mediatailor.model.CreatePrefetchScheduleResponse;
import software.amazon.awssdk.services.mediatailor.model.CreateProgramRequest;
import software.amazon.awssdk.services.mediatailor.model.CreateProgramResponse;
import software.amazon.awssdk.services.mediatailor.model.CreateSourceLocationRequest;
import software.amazon.awssdk.services.mediatailor.model.CreateSourceLocationResponse;
import software.amazon.awssdk.services.mediatailor.model.CreateVodSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.CreateVodSourceResponse;
import software.amazon.awssdk.services.mediatailor.model.DeleteChannelPolicyRequest;
import software.amazon.awssdk.services.mediatailor.model.DeleteChannelPolicyResponse;
import software.amazon.awssdk.services.mediatailor.model.DeleteChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.DeleteChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.DeleteLiveSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.DeleteLiveSourceResponse;
import software.amazon.awssdk.services.mediatailor.model.DeletePlaybackConfigurationRequest;
import software.amazon.awssdk.services.mediatailor.model.DeletePlaybackConfigurationResponse;
import software.amazon.awssdk.services.mediatailor.model.DeletePrefetchScheduleRequest;
import software.amazon.awssdk.services.mediatailor.model.DeletePrefetchScheduleResponse;
import software.amazon.awssdk.services.mediatailor.model.DeleteProgramRequest;
import software.amazon.awssdk.services.mediatailor.model.DeleteProgramResponse;
import software.amazon.awssdk.services.mediatailor.model.DeleteSourceLocationRequest;
import software.amazon.awssdk.services.mediatailor.model.DeleteSourceLocationResponse;
import software.amazon.awssdk.services.mediatailor.model.DeleteVodSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.DeleteVodSourceResponse;
import software.amazon.awssdk.services.mediatailor.model.DescribeChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.DescribeChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.DescribeLiveSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.DescribeLiveSourceResponse;
import software.amazon.awssdk.services.mediatailor.model.DescribeProgramRequest;
import software.amazon.awssdk.services.mediatailor.model.DescribeProgramResponse;
import software.amazon.awssdk.services.mediatailor.model.DescribeSourceLocationRequest;
import software.amazon.awssdk.services.mediatailor.model.DescribeSourceLocationResponse;
import software.amazon.awssdk.services.mediatailor.model.DescribeVodSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.DescribeVodSourceResponse;
import software.amazon.awssdk.services.mediatailor.model.GetChannelPolicyRequest;
import software.amazon.awssdk.services.mediatailor.model.GetChannelPolicyResponse;
import software.amazon.awssdk.services.mediatailor.model.GetChannelScheduleRequest;
import software.amazon.awssdk.services.mediatailor.model.GetChannelScheduleResponse;
import software.amazon.awssdk.services.mediatailor.model.GetPlaybackConfigurationRequest;
import software.amazon.awssdk.services.mediatailor.model.GetPlaybackConfigurationResponse;
import software.amazon.awssdk.services.mediatailor.model.GetPrefetchScheduleRequest;
import software.amazon.awssdk.services.mediatailor.model.GetPrefetchScheduleResponse;
import software.amazon.awssdk.services.mediatailor.model.ListAlertsRequest;
import software.amazon.awssdk.services.mediatailor.model.ListAlertsResponse;
import software.amazon.awssdk.services.mediatailor.model.ListChannelsRequest;
import software.amazon.awssdk.services.mediatailor.model.ListChannelsResponse;
import software.amazon.awssdk.services.mediatailor.model.ListLiveSourcesRequest;
import software.amazon.awssdk.services.mediatailor.model.ListLiveSourcesResponse;
import software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsRequest;
import software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsResponse;
import software.amazon.awssdk.services.mediatailor.model.ListPrefetchSchedulesRequest;
import software.amazon.awssdk.services.mediatailor.model.ListPrefetchSchedulesResponse;
import software.amazon.awssdk.services.mediatailor.model.ListSourceLocationsRequest;
import software.amazon.awssdk.services.mediatailor.model.ListSourceLocationsResponse;
import software.amazon.awssdk.services.mediatailor.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediatailor.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediatailor.model.ListVodSourcesRequest;
import software.amazon.awssdk.services.mediatailor.model.ListVodSourcesResponse;
import software.amazon.awssdk.services.mediatailor.model.MediaTailorException;
import software.amazon.awssdk.services.mediatailor.model.MediaTailorRequest;
import software.amazon.awssdk.services.mediatailor.model.PutChannelPolicyRequest;
import software.amazon.awssdk.services.mediatailor.model.PutChannelPolicyResponse;
import software.amazon.awssdk.services.mediatailor.model.PutPlaybackConfigurationRequest;
import software.amazon.awssdk.services.mediatailor.model.PutPlaybackConfigurationResponse;
import software.amazon.awssdk.services.mediatailor.model.StartChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.StartChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.StopChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.StopChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.TagResourceRequest;
import software.amazon.awssdk.services.mediatailor.model.TagResourceResponse;
import software.amazon.awssdk.services.mediatailor.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediatailor.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediatailor.model.UpdateChannelRequest;
import software.amazon.awssdk.services.mediatailor.model.UpdateChannelResponse;
import software.amazon.awssdk.services.mediatailor.model.UpdateLiveSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.UpdateLiveSourceResponse;
import software.amazon.awssdk.services.mediatailor.model.UpdateProgramRequest;
import software.amazon.awssdk.services.mediatailor.model.UpdateProgramResponse;
import software.amazon.awssdk.services.mediatailor.model.UpdateSourceLocationRequest;
import software.amazon.awssdk.services.mediatailor.model.UpdateSourceLocationResponse;
import software.amazon.awssdk.services.mediatailor.model.UpdateVodSourceRequest;
import software.amazon.awssdk.services.mediatailor.model.UpdateVodSourceResponse;
import software.amazon.awssdk.services.mediatailor.paginators.GetChannelSchedulePublisher;
import software.amazon.awssdk.services.mediatailor.paginators.ListAlertsPublisher;
import software.amazon.awssdk.services.mediatailor.paginators.ListChannelsPublisher;
import software.amazon.awssdk.services.mediatailor.paginators.ListLiveSourcesPublisher;
import software.amazon.awssdk.services.mediatailor.paginators.ListPlaybackConfigurationsPublisher;
import software.amazon.awssdk.services.mediatailor.paginators.ListPrefetchSchedulesPublisher;
import software.amazon.awssdk.services.mediatailor.paginators.ListSourceLocationsPublisher;
import software.amazon.awssdk.services.mediatailor.paginators.ListVodSourcesPublisher;
import software.amazon.awssdk.services.mediatailor.transform.ConfigureLogsForChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ConfigureLogsForPlaybackConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.CreateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.CreateLiveSourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.CreatePrefetchScheduleRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.CreateProgramRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.CreateSourceLocationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.CreateVodSourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeleteChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeleteChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeleteLiveSourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeletePlaybackConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeletePrefetchScheduleRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeleteProgramRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeleteSourceLocationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DeleteVodSourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DescribeChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DescribeLiveSourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DescribeProgramRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DescribeSourceLocationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.DescribeVodSourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.GetChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.GetChannelScheduleRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.GetPlaybackConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.GetPrefetchScheduleRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListAlertsRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListLiveSourcesRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListPlaybackConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListPrefetchSchedulesRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListSourceLocationsRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.ListVodSourcesRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.PutChannelPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.PutPlaybackConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.StartChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.StopChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.UpdateChannelRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.UpdateLiveSourceRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.UpdateProgramRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.UpdateSourceLocationRequestMarshaller;
import software.amazon.awssdk.services.mediatailor.transform.UpdateVodSourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MediaTailorAsyncClient}.
 *
 * @see MediaTailorAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaTailorAsyncClient implements MediaTailorAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMediaTailorAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaTailorAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Configures Amazon CloudWatch log settings for a channel.
     * </p>
     *
     * @param configureLogsForChannelRequest
     * @return A Java Future containing the result of the ConfigureLogsForChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ConfigureLogsForChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ConfigureLogsForChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ConfigureLogsForChannelResponse> configureLogsForChannel(
            ConfigureLogsForChannelRequest configureLogsForChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, configureLogsForChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ConfigureLogsForChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ConfigureLogsForChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ConfigureLogsForChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ConfigureLogsForChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ConfigureLogsForChannelRequest, ConfigureLogsForChannelResponse>()
                            .withOperationName("ConfigureLogsForChannel")
                            .withMarshaller(new ConfigureLogsForChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(configureLogsForChannelRequest));
            CompletableFuture<ConfigureLogsForChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Amazon CloudWatch log settings for a playback configuration.
     * </p>
     *
     * @param configureLogsForPlaybackConfigurationRequest
     *        Configures Amazon CloudWatch log settings for a playback configuration.
     * @return A Java Future containing the result of the ConfigureLogsForPlaybackConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ConfigureLogsForPlaybackConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ConfigureLogsForPlaybackConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ConfigureLogsForPlaybackConfigurationResponse> configureLogsForPlaybackConfiguration(
            ConfigureLogsForPlaybackConfigurationRequest configureLogsForPlaybackConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                configureLogsForPlaybackConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ConfigureLogsForPlaybackConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ConfigureLogsForPlaybackConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ConfigureLogsForPlaybackConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ConfigureLogsForPlaybackConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ConfigureLogsForPlaybackConfigurationRequest, ConfigureLogsForPlaybackConfigurationResponse>()
                            .withOperationName("ConfigureLogsForPlaybackConfiguration")
                            .withMarshaller(new ConfigureLogsForPlaybackConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(configureLogsForPlaybackConfigurationRequest));
            CompletableFuture<ConfigureLogsForPlaybackConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a channel. For information about MediaTailor channels, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-channels.html">Working with channels</a>
     * in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param createChannelRequest
     * @return A Java Future containing the result of the CreateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.CreateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/CreateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChannelResponse> createChannel(CreateChannelRequest createChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChannelRequest, CreateChannelResponse>()
                            .withOperationName("CreateChannel")
                            .withMarshaller(new CreateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createChannelRequest));
            CompletableFuture<CreateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The live source configuration.
     * </p>
     *
     * @param createLiveSourceRequest
     * @return A Java Future containing the result of the CreateLiveSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.CreateLiveSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/CreateLiveSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLiveSourceResponse> createLiveSource(CreateLiveSourceRequest createLiveSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLiveSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLiveSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLiveSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLiveSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLiveSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLiveSourceRequest, CreateLiveSourceResponse>()
                            .withOperationName("CreateLiveSource")
                            .withMarshaller(new CreateLiveSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLiveSourceRequest));
            CompletableFuture<CreateLiveSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a prefetch schedule for a playback configuration. A prefetch schedule allows you to tell MediaTailor to
     * fetch and prepare certain ads before an ad break happens. For more information about ad prefetching, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/prefetching-ads.html">Using ad prefetching</a> in the
     * <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param createPrefetchScheduleRequest
     * @return A Java Future containing the result of the CreatePrefetchSchedule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.CreatePrefetchSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/CreatePrefetchSchedule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePrefetchScheduleResponse> createPrefetchSchedule(
            CreatePrefetchScheduleRequest createPrefetchScheduleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPrefetchScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePrefetchSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePrefetchScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePrefetchScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePrefetchScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePrefetchScheduleRequest, CreatePrefetchScheduleResponse>()
                            .withOperationName("CreatePrefetchSchedule")
                            .withMarshaller(new CreatePrefetchScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPrefetchScheduleRequest));
            CompletableFuture<CreatePrefetchScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a program within a channel. For information about programs, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-programs.html">Working with programs</a>
     * in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param createProgramRequest
     * @return A Java Future containing the result of the CreateProgram operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.CreateProgram
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/CreateProgram" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProgramResponse> createProgram(CreateProgramRequest createProgramRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProgramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProgram");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProgramResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProgramResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProgramResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProgramRequest, CreateProgramResponse>()
                            .withOperationName("CreateProgram")
                            .withMarshaller(new CreateProgramRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createProgramRequest));
            CompletableFuture<CreateProgramResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a source location. A source location is a container for sources. For more information about source
     * locations, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-source-locations.html">Working with
     * source locations</a> in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param createSourceLocationRequest
     * @return A Java Future containing the result of the CreateSourceLocation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.CreateSourceLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/CreateSourceLocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSourceLocationResponse> createSourceLocation(
            CreateSourceLocationRequest createSourceLocationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSourceLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSourceLocation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSourceLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSourceLocationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSourceLocationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSourceLocationRequest, CreateSourceLocationResponse>()
                            .withOperationName("CreateSourceLocation")
                            .withMarshaller(new CreateSourceLocationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSourceLocationRequest));
            CompletableFuture<CreateSourceLocationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The VOD source configuration parameters.
     * </p>
     *
     * @param createVodSourceRequest
     * @return A Java Future containing the result of the CreateVodSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.CreateVodSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/CreateVodSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVodSourceResponse> createVodSource(CreateVodSourceRequest createVodSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVodSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVodSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVodSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVodSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVodSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVodSourceRequest, CreateVodSourceResponse>()
                            .withOperationName("CreateVodSource")
                            .withMarshaller(new CreateVodSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createVodSourceRequest));
            CompletableFuture<CreateVodSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a channel. For information about MediaTailor channels, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-channels.html">Working with channels</a>
     * in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param deleteChannelRequest
     * @return A Java Future containing the result of the DeleteChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DeleteChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeleteChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelResponse> deleteChannel(DeleteChannelRequest deleteChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelRequest, DeleteChannelResponse>()
                            .withOperationName("DeleteChannel")
                            .withMarshaller(new DeleteChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteChannelRequest));
            CompletableFuture<DeleteChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The channel policy to delete.
     * </p>
     *
     * @param deleteChannelPolicyRequest
     * @return A Java Future containing the result of the DeleteChannelPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DeleteChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeleteChannelPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelPolicyResponse> deleteChannelPolicy(
            DeleteChannelPolicyRequest deleteChannelPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannelPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteChannelPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelPolicyRequest, DeleteChannelPolicyResponse>()
                            .withOperationName("DeleteChannelPolicy")
                            .withMarshaller(new DeleteChannelPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteChannelPolicyRequest));
            CompletableFuture<DeleteChannelPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The live source to delete.
     * </p>
     *
     * @param deleteLiveSourceRequest
     * @return A Java Future containing the result of the DeleteLiveSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DeleteLiveSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeleteLiveSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLiveSourceResponse> deleteLiveSource(DeleteLiveSourceRequest deleteLiveSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLiveSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLiveSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLiveSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLiveSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLiveSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLiveSourceRequest, DeleteLiveSourceResponse>()
                            .withOperationName("DeleteLiveSource")
                            .withMarshaller(new DeleteLiveSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLiveSourceRequest));
            CompletableFuture<DeleteLiveSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a playback configuration. For information about MediaTailor configurations, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/configurations.html">Working with configurations in AWS
     * Elemental MediaTailor</a>.
     * </p>
     *
     * @param deletePlaybackConfigurationRequest
     * @return A Java Future containing the result of the DeletePlaybackConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DeletePlaybackConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeletePlaybackConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePlaybackConfigurationResponse> deletePlaybackConfiguration(
            DeletePlaybackConfigurationRequest deletePlaybackConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePlaybackConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePlaybackConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePlaybackConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePlaybackConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePlaybackConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePlaybackConfigurationRequest, DeletePlaybackConfigurationResponse>()
                            .withOperationName("DeletePlaybackConfiguration")
                            .withMarshaller(new DeletePlaybackConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePlaybackConfigurationRequest));
            CompletableFuture<DeletePlaybackConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a prefetch schedule for a specific playback configuration. If you call
     * <code>DeletePrefetchSchedule</code> on an expired prefetch schedule, MediaTailor returns an HTTP 404 status code.
     * For more information about ad prefetching, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/prefetching-ads.html">Using ad prefetching</a> in the
     * <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param deletePrefetchScheduleRequest
     * @return A Java Future containing the result of the DeletePrefetchSchedule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DeletePrefetchSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeletePrefetchSchedule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePrefetchScheduleResponse> deletePrefetchSchedule(
            DeletePrefetchScheduleRequest deletePrefetchScheduleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePrefetchScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePrefetchSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePrefetchScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePrefetchScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePrefetchScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePrefetchScheduleRequest, DeletePrefetchScheduleResponse>()
                            .withOperationName("DeletePrefetchSchedule")
                            .withMarshaller(new DeletePrefetchScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePrefetchScheduleRequest));
            CompletableFuture<DeletePrefetchScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a program within a channel. For information about programs, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-programs.html">Working with programs</a>
     * in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param deleteProgramRequest
     * @return A Java Future containing the result of the DeleteProgram operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DeleteProgram
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeleteProgram" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProgramResponse> deleteProgram(DeleteProgramRequest deleteProgramRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProgramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProgram");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProgramResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProgramResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProgramResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProgramRequest, DeleteProgramResponse>()
                            .withOperationName("DeleteProgram")
                            .withMarshaller(new DeleteProgramRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProgramRequest));
            CompletableFuture<DeleteProgramResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a source location. A source location is a container for sources. For more information about source
     * locations, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-source-locations.html">Working with
     * source locations</a> in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param deleteSourceLocationRequest
     * @return A Java Future containing the result of the DeleteSourceLocation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DeleteSourceLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeleteSourceLocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSourceLocationResponse> deleteSourceLocation(
            DeleteSourceLocationRequest deleteSourceLocationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSourceLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSourceLocation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSourceLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSourceLocationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSourceLocationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSourceLocationRequest, DeleteSourceLocationResponse>()
                            .withOperationName("DeleteSourceLocation")
                            .withMarshaller(new DeleteSourceLocationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSourceLocationRequest));
            CompletableFuture<DeleteSourceLocationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The video on demand (VOD) source to delete.
     * </p>
     *
     * @param deleteVodSourceRequest
     * @return A Java Future containing the result of the DeleteVodSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DeleteVodSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DeleteVodSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVodSourceResponse> deleteVodSource(DeleteVodSourceRequest deleteVodSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVodSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVodSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVodSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVodSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVodSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVodSourceRequest, DeleteVodSourceResponse>()
                            .withOperationName("DeleteVodSource")
                            .withMarshaller(new DeleteVodSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteVodSourceRequest));
            CompletableFuture<DeleteVodSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a channel. For information about MediaTailor channels, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-channels.html">Working with channels</a>
     * in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param describeChannelRequest
     * @return A Java Future containing the result of the DescribeChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DescribeChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DescribeChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChannelResponse> describeChannel(DescribeChannelRequest describeChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChannelRequest, DescribeChannelResponse>()
                            .withOperationName("DescribeChannel")
                            .withMarshaller(new DescribeChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeChannelRequest));
            CompletableFuture<DescribeChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The live source to describe.
     * </p>
     *
     * @param describeLiveSourceRequest
     * @return A Java Future containing the result of the DescribeLiveSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DescribeLiveSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DescribeLiveSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLiveSourceResponse> describeLiveSource(DescribeLiveSourceRequest describeLiveSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLiveSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLiveSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLiveSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLiveSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLiveSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLiveSourceRequest, DescribeLiveSourceResponse>()
                            .withOperationName("DescribeLiveSource")
                            .withMarshaller(new DescribeLiveSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLiveSourceRequest));
            CompletableFuture<DescribeLiveSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a program within a channel. For information about programs, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-programs.html">Working with programs</a>
     * in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param describeProgramRequest
     * @return A Java Future containing the result of the DescribeProgram operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DescribeProgram
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DescribeProgram" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProgramResponse> describeProgram(DescribeProgramRequest describeProgramRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProgramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProgram");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProgramResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProgramResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProgramResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProgramRequest, DescribeProgramResponse>()
                            .withOperationName("DescribeProgram")
                            .withMarshaller(new DescribeProgramRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeProgramRequest));
            CompletableFuture<DescribeProgramResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a source location. A source location is a container for sources. For more information about source
     * locations, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-source-locations.html">Working with
     * source locations</a> in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param describeSourceLocationRequest
     * @return A Java Future containing the result of the DescribeSourceLocation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DescribeSourceLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DescribeSourceLocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSourceLocationResponse> describeSourceLocation(
            DescribeSourceLocationRequest describeSourceLocationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSourceLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSourceLocation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSourceLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSourceLocationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSourceLocationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSourceLocationRequest, DescribeSourceLocationResponse>()
                            .withOperationName("DescribeSourceLocation")
                            .withMarshaller(new DescribeSourceLocationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSourceLocationRequest));
            CompletableFuture<DescribeSourceLocationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides details about a specific video on demand (VOD) source in a specific source location.
     * </p>
     *
     * @param describeVodSourceRequest
     * @return A Java Future containing the result of the DescribeVodSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.DescribeVodSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/DescribeVodSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVodSourceResponse> describeVodSource(DescribeVodSourceRequest describeVodSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVodSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVodSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVodSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVodSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVodSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVodSourceRequest, DescribeVodSourceResponse>()
                            .withOperationName("DescribeVodSource")
                            .withMarshaller(new DescribeVodSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeVodSourceRequest));
            CompletableFuture<DescribeVodSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the channel's IAM policy. IAM policies are used to control access to your channel.
     * </p>
     *
     * @param getChannelPolicyRequest
     * @return A Java Future containing the result of the GetChannelPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.GetChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/GetChannelPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetChannelPolicyResponse> getChannelPolicy(GetChannelPolicyRequest getChannelPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannelPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetChannelPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetChannelPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetChannelPolicyRequest, GetChannelPolicyResponse>()
                            .withOperationName("GetChannelPolicy")
                            .withMarshaller(new GetChannelPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getChannelPolicyRequest));
            CompletableFuture<GetChannelPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about your channel's schedule.
     * </p>
     *
     * @param getChannelScheduleRequest
     * @return A Java Future containing the result of the GetChannelSchedule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.GetChannelSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/GetChannelSchedule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetChannelScheduleResponse> getChannelSchedule(GetChannelScheduleRequest getChannelScheduleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannelSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetChannelScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetChannelScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetChannelScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetChannelScheduleRequest, GetChannelScheduleResponse>()
                            .withOperationName("GetChannelSchedule")
                            .withMarshaller(new GetChannelScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getChannelScheduleRequest));
            CompletableFuture<GetChannelScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about your channel's schedule.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getChannelSchedule(software.amazon.awssdk.services.mediatailor.model.GetChannelScheduleRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.GetChannelSchedulePublisher publisher = client.getChannelSchedulePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.GetChannelSchedulePublisher publisher = client.getChannelSchedulePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediatailor.model.GetChannelScheduleResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediatailor.model.GetChannelScheduleResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getChannelSchedule(software.amazon.awssdk.services.mediatailor.model.GetChannelScheduleRequest)}
     * operation.</b>
     * </p>
     *
     * @param getChannelScheduleRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.GetChannelSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/GetChannelSchedule"
     *      target="_top">AWS API Documentation</a>
     */
    public GetChannelSchedulePublisher getChannelSchedulePaginator(GetChannelScheduleRequest getChannelScheduleRequest) {
        return new GetChannelSchedulePublisher(this, applyPaginatorUserAgent(getChannelScheduleRequest));
    }

    /**
     * <p>
     * Retrieves a playback configuration. For information about MediaTailor configurations, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/configurations.html">Working with configurations in AWS
     * Elemental MediaTailor</a>.
     * </p>
     *
     * @param getPlaybackConfigurationRequest
     * @return A Java Future containing the result of the GetPlaybackConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.GetPlaybackConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/GetPlaybackConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPlaybackConfigurationResponse> getPlaybackConfiguration(
            GetPlaybackConfigurationRequest getPlaybackConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlaybackConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlaybackConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPlaybackConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPlaybackConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPlaybackConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPlaybackConfigurationRequest, GetPlaybackConfigurationResponse>()
                            .withOperationName("GetPlaybackConfiguration")
                            .withMarshaller(new GetPlaybackConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPlaybackConfigurationRequest));
            CompletableFuture<GetPlaybackConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a prefetch schedule for a playback configuration. A prefetch schedule allows you to tell MediaTailor to
     * fetch and prepare certain ads before an ad break happens. For more information about ad prefetching, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/prefetching-ads.html">Using ad prefetching</a> in the
     * <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param getPrefetchScheduleRequest
     * @return A Java Future containing the result of the GetPrefetchSchedule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.GetPrefetchSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/GetPrefetchSchedule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPrefetchScheduleResponse> getPrefetchSchedule(
            GetPrefetchScheduleRequest getPrefetchScheduleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPrefetchScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPrefetchSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPrefetchScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPrefetchScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPrefetchScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPrefetchScheduleRequest, GetPrefetchScheduleResponse>()
                            .withOperationName("GetPrefetchSchedule")
                            .withMarshaller(new GetPrefetchScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPrefetchScheduleRequest));
            CompletableFuture<GetPrefetchScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the alerts that are associated with a MediaTailor channel assembly resource.
     * </p>
     *
     * @param listAlertsRequest
     * @return A Java Future containing the result of the ListAlerts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListAlerts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListAlerts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAlertsResponse> listAlerts(ListAlertsRequest listAlertsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAlertsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAlerts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAlertsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAlertsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAlertsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAlertsRequest, ListAlertsResponse>().withOperationName("ListAlerts")
                            .withMarshaller(new ListAlertsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAlertsRequest));
            CompletableFuture<ListAlertsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the alerts that are associated with a MediaTailor channel assembly resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAlerts(software.amazon.awssdk.services.mediatailor.model.ListAlertsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListAlertsPublisher publisher = client.listAlertsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListAlertsPublisher publisher = client.listAlertsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediatailor.model.ListAlertsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediatailor.model.ListAlertsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAlerts(software.amazon.awssdk.services.mediatailor.model.ListAlertsRequest)} operation.</b>
     * </p>
     *
     * @param listAlertsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListAlerts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListAlerts" target="_top">AWS API
     *      Documentation</a>
     */
    public ListAlertsPublisher listAlertsPaginator(ListAlertsRequest listAlertsRequest) {
        return new ListAlertsPublisher(this, applyPaginatorUserAgent(listAlertsRequest));
    }

    /**
     * <p>
     * Retrieves information about the channels that are associated with the current AWS account.
     * </p>
     *
     * @param listChannelsRequest
     * @return A Java Future containing the result of the ListChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelsResponse> listChannels(ListChannelsRequest listChannelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                            .withOperationName("ListChannels").withMarshaller(new ListChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listChannelsRequest));
            CompletableFuture<ListChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the channels that are associated with the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listChannels(software.amazon.awssdk.services.mediatailor.model.ListChannelsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListChannelsPublisher publisher = client.listChannelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListChannelsPublisher publisher = client.listChannelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediatailor.model.ListChannelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediatailor.model.ListChannelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannels(software.amazon.awssdk.services.mediatailor.model.ListChannelsRequest)} operation.</b>
     * </p>
     *
     * @param listChannelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    public ListChannelsPublisher listChannelsPaginator(ListChannelsRequest listChannelsRequest) {
        return new ListChannelsPublisher(this, applyPaginatorUserAgent(listChannelsRequest));
    }

    /**
     * <p>
     * Lists the live sources contained in a source location. A source represents a piece of content.
     * </p>
     *
     * @param listLiveSourcesRequest
     * @return A Java Future containing the result of the ListLiveSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListLiveSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListLiveSources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLiveSourcesResponse> listLiveSources(ListLiveSourcesRequest listLiveSourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLiveSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLiveSources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLiveSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLiveSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLiveSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLiveSourcesRequest, ListLiveSourcesResponse>()
                            .withOperationName("ListLiveSources")
                            .withMarshaller(new ListLiveSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLiveSourcesRequest));
            CompletableFuture<ListLiveSourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the live sources contained in a source location. A source represents a piece of content.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLiveSources(software.amazon.awssdk.services.mediatailor.model.ListLiveSourcesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListLiveSourcesPublisher publisher = client.listLiveSourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListLiveSourcesPublisher publisher = client.listLiveSourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediatailor.model.ListLiveSourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediatailor.model.ListLiveSourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLiveSources(software.amazon.awssdk.services.mediatailor.model.ListLiveSourcesRequest)} operation.</b>
     * </p>
     *
     * @param listLiveSourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListLiveSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListLiveSources" target="_top">AWS
     *      API Documentation</a>
     */
    public ListLiveSourcesPublisher listLiveSourcesPaginator(ListLiveSourcesRequest listLiveSourcesRequest) {
        return new ListLiveSourcesPublisher(this, applyPaginatorUserAgent(listLiveSourcesRequest));
    }

    /**
     * <p>
     * Retrieves existing playback configurations. For information about MediaTailor configurations, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/configurations.html">Working with Configurations in AWS
     * Elemental MediaTailor</a>.
     * </p>
     *
     * @param listPlaybackConfigurationsRequest
     * @return A Java Future containing the result of the ListPlaybackConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListPlaybackConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListPlaybackConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPlaybackConfigurationsResponse> listPlaybackConfigurations(
            ListPlaybackConfigurationsRequest listPlaybackConfigurationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPlaybackConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPlaybackConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPlaybackConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPlaybackConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPlaybackConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPlaybackConfigurationsRequest, ListPlaybackConfigurationsResponse>()
                            .withOperationName("ListPlaybackConfigurations")
                            .withMarshaller(new ListPlaybackConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPlaybackConfigurationsRequest));
            CompletableFuture<ListPlaybackConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves existing playback configurations. For information about MediaTailor configurations, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/configurations.html">Working with Configurations in AWS
     * Elemental MediaTailor</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPlaybackConfigurations(software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListPlaybackConfigurationsPublisher publisher = client.listPlaybackConfigurationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListPlaybackConfigurationsPublisher publisher = client.listPlaybackConfigurationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlaybackConfigurations(software.amazon.awssdk.services.mediatailor.model.ListPlaybackConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPlaybackConfigurationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListPlaybackConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListPlaybackConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPlaybackConfigurationsPublisher listPlaybackConfigurationsPaginator(
            ListPlaybackConfigurationsRequest listPlaybackConfigurationsRequest) {
        return new ListPlaybackConfigurationsPublisher(this, applyPaginatorUserAgent(listPlaybackConfigurationsRequest));
    }

    /**
     * <p>
     * Lists the prefetch schedules for a playback configuration.
     * </p>
     *
     * @param listPrefetchSchedulesRequest
     * @return A Java Future containing the result of the ListPrefetchSchedules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListPrefetchSchedules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListPrefetchSchedules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPrefetchSchedulesResponse> listPrefetchSchedules(
            ListPrefetchSchedulesRequest listPrefetchSchedulesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPrefetchSchedulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPrefetchSchedules");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPrefetchSchedulesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPrefetchSchedulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPrefetchSchedulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPrefetchSchedulesRequest, ListPrefetchSchedulesResponse>()
                            .withOperationName("ListPrefetchSchedules")
                            .withMarshaller(new ListPrefetchSchedulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPrefetchSchedulesRequest));
            CompletableFuture<ListPrefetchSchedulesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the prefetch schedules for a playback configuration.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPrefetchSchedules(software.amazon.awssdk.services.mediatailor.model.ListPrefetchSchedulesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListPrefetchSchedulesPublisher publisher = client.listPrefetchSchedulesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListPrefetchSchedulesPublisher publisher = client.listPrefetchSchedulesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediatailor.model.ListPrefetchSchedulesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediatailor.model.ListPrefetchSchedulesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPrefetchSchedules(software.amazon.awssdk.services.mediatailor.model.ListPrefetchSchedulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPrefetchSchedulesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListPrefetchSchedules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListPrefetchSchedules"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPrefetchSchedulesPublisher listPrefetchSchedulesPaginator(ListPrefetchSchedulesRequest listPrefetchSchedulesRequest) {
        return new ListPrefetchSchedulesPublisher(this, applyPaginatorUserAgent(listPrefetchSchedulesRequest));
    }

    /**
     * <p>
     * Lists the source locations for a channel. A source location defines the host server URL, and contains a list of
     * sources.
     * </p>
     *
     * @param listSourceLocationsRequest
     * @return A Java Future containing the result of the ListSourceLocations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListSourceLocations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListSourceLocations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSourceLocationsResponse> listSourceLocations(
            ListSourceLocationsRequest listSourceLocationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSourceLocationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSourceLocations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSourceLocationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSourceLocationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSourceLocationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSourceLocationsRequest, ListSourceLocationsResponse>()
                            .withOperationName("ListSourceLocations")
                            .withMarshaller(new ListSourceLocationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSourceLocationsRequest));
            CompletableFuture<ListSourceLocationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the source locations for a channel. A source location defines the host server URL, and contains a list of
     * sources.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSourceLocations(software.amazon.awssdk.services.mediatailor.model.ListSourceLocationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListSourceLocationsPublisher publisher = client.listSourceLocationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListSourceLocationsPublisher publisher = client.listSourceLocationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediatailor.model.ListSourceLocationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediatailor.model.ListSourceLocationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSourceLocations(software.amazon.awssdk.services.mediatailor.model.ListSourceLocationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSourceLocationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListSourceLocations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListSourceLocations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListSourceLocationsPublisher listSourceLocationsPaginator(ListSourceLocationsRequest listSourceLocationsRequest) {
        return new ListSourceLocationsPublisher(this, applyPaginatorUserAgent(listSourceLocationsRequest));
    }

    /**
     * <p>
     * A list of tags that are associated with this resource. Tags are key-value pairs that you can associate with
     * Amazon resources to help with organization, access control, and cost tracking. For more information, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/tagging.html">Tagging AWS Elemental MediaTailor
     * Resources</a>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the VOD sources contained in a source location. A source represents a piece of content.
     * </p>
     *
     * @param listVodSourcesRequest
     * @return A Java Future containing the result of the ListVodSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListVodSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListVodSources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVodSourcesResponse> listVodSources(ListVodSourcesRequest listVodSourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVodSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVodSources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVodSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVodSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVodSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVodSourcesRequest, ListVodSourcesResponse>()
                            .withOperationName("ListVodSources")
                            .withMarshaller(new ListVodSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listVodSourcesRequest));
            CompletableFuture<ListVodSourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the VOD sources contained in a source location. A source represents a piece of content.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVodSources(software.amazon.awssdk.services.mediatailor.model.ListVodSourcesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListVodSourcesPublisher publisher = client.listVodSourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediatailor.paginators.ListVodSourcesPublisher publisher = client.listVodSourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.mediatailor.model.ListVodSourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.mediatailor.model.ListVodSourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVodSources(software.amazon.awssdk.services.mediatailor.model.ListVodSourcesRequest)} operation.</b>
     * </p>
     *
     * @param listVodSourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.ListVodSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/ListVodSources" target="_top">AWS
     *      API Documentation</a>
     */
    public ListVodSourcesPublisher listVodSourcesPaginator(ListVodSourcesRequest listVodSourcesRequest) {
        return new ListVodSourcesPublisher(this, applyPaginatorUserAgent(listVodSourcesRequest));
    }

    /**
     * <p>
     * Creates an IAM policy for the channel. IAM policies are used to control access to your channel.
     * </p>
     *
     * @param putChannelPolicyRequest
     * @return A Java Future containing the result of the PutChannelPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.PutChannelPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/PutChannelPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutChannelPolicyResponse> putChannelPolicy(PutChannelPolicyRequest putChannelPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putChannelPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutChannelPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutChannelPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutChannelPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutChannelPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutChannelPolicyRequest, PutChannelPolicyResponse>()
                            .withOperationName("PutChannelPolicy")
                            .withMarshaller(new PutChannelPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putChannelPolicyRequest));
            CompletableFuture<PutChannelPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a playback configuration. For information about MediaTailor configurations, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/configurations.html">Working with configurations in AWS
     * Elemental MediaTailor</a>.
     * </p>
     *
     * @param putPlaybackConfigurationRequest
     * @return A Java Future containing the result of the PutPlaybackConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.PutPlaybackConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/PutPlaybackConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutPlaybackConfigurationResponse> putPlaybackConfiguration(
            PutPlaybackConfigurationRequest putPlaybackConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPlaybackConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPlaybackConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPlaybackConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutPlaybackConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPlaybackConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPlaybackConfigurationRequest, PutPlaybackConfigurationResponse>()
                            .withOperationName("PutPlaybackConfiguration")
                            .withMarshaller(new PutPlaybackConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putPlaybackConfigurationRequest));
            CompletableFuture<PutPlaybackConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a channel. For information about MediaTailor channels, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-channels.html">Working with channels</a>
     * in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param startChannelRequest
     * @return A Java Future containing the result of the StartChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.StartChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/StartChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartChannelResponse> startChannel(StartChannelRequest startChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartChannelRequest, StartChannelResponse>()
                            .withOperationName("StartChannel").withMarshaller(new StartChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startChannelRequest));
            CompletableFuture<StartChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a channel. For information about MediaTailor channels, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-channels.html">Working with channels</a>
     * in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param stopChannelRequest
     * @return A Java Future containing the result of the StopChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.StopChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/StopChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopChannelResponse> stopChannel(StopChannelRequest stopChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopChannelRequest, StopChannelResponse>()
                            .withOperationName("StopChannel").withMarshaller(new StopChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopChannelRequest));
            CompletableFuture<StopChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The resource to tag. Tags are key-value pairs that you can associate with Amazon resources to help with
     * organization, access control, and cost tracking. For more information, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/tagging.html">Tagging AWS Elemental MediaTailor
     * Resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The resource to untag.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a channel. For information about MediaTailor channels, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-channels.html">Working with channels</a>
     * in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param updateChannelRequest
     * @return A Java Future containing the result of the UpdateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.UpdateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/UpdateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChannelResponse> updateChannel(UpdateChannelRequest updateChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChannelRequest, UpdateChannelResponse>()
                            .withOperationName("UpdateChannel")
                            .withMarshaller(new UpdateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateChannelRequest));
            CompletableFuture<UpdateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a live source's configuration.
     * </p>
     *
     * @param updateLiveSourceRequest
     * @return A Java Future containing the result of the UpdateLiveSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.UpdateLiveSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/UpdateLiveSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLiveSourceResponse> updateLiveSource(UpdateLiveSourceRequest updateLiveSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLiveSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLiveSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLiveSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLiveSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLiveSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLiveSourceRequest, UpdateLiveSourceResponse>()
                            .withOperationName("UpdateLiveSource")
                            .withMarshaller(new UpdateLiveSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLiveSourceRequest));
            CompletableFuture<UpdateLiveSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a program within a channel.
     * </p>
     *
     * @param updateProgramRequest
     * @return A Java Future containing the result of the UpdateProgram operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.UpdateProgram
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/UpdateProgram" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProgramResponse> updateProgram(UpdateProgramRequest updateProgramRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProgramRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProgram");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProgramResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateProgramResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProgramResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProgramRequest, UpdateProgramResponse>()
                            .withOperationName("UpdateProgram")
                            .withMarshaller(new UpdateProgramRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateProgramRequest));
            CompletableFuture<UpdateProgramResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a source location. A source location is a container for sources. For more information about source
     * locations, see <a
     * href="https://docs.aws.amazon.com/mediatailor/latest/ug/channel-assembly-source-locations.html">Working with
     * source locations</a> in the <i>MediaTailor User Guide</i>.
     * </p>
     *
     * @param updateSourceLocationRequest
     * @return A Java Future containing the result of the UpdateSourceLocation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.UpdateSourceLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/UpdateSourceLocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSourceLocationResponse> updateSourceLocation(
            UpdateSourceLocationRequest updateSourceLocationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSourceLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSourceLocation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSourceLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSourceLocationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSourceLocationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSourceLocationRequest, UpdateSourceLocationResponse>()
                            .withOperationName("UpdateSourceLocation")
                            .withMarshaller(new UpdateSourceLocationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSourceLocationRequest));
            CompletableFuture<UpdateSourceLocationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a VOD source's configuration.
     * </p>
     *
     * @param updateVodSourceRequest
     * @return A Java Future containing the result of the UpdateVodSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaTailorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaTailorAsyncClient.UpdateVodSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediatailor-2018-04-23/UpdateVodSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVodSourceResponse> updateVodSource(UpdateVodSourceRequest updateVodSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVodSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaTailor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVodSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVodSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVodSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVodSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVodSourceRequest, UpdateVodSourceResponse>()
                            .withOperationName("UpdateVodSource")
                            .withMarshaller(new UpdateVodSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateVodSourceRequest));
            CompletableFuture<UpdateVodSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaTailorException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends MediaTailorRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
