/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediatailor.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Alert configuration parameters.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Alert implements SdkPojo, Serializable, ToCopyableBuilder<Alert.Builder, Alert> {
    private static final SdkField<String> ALERT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlertCode").getter(getter(Alert::alertCode)).setter(setter(Builder::alertCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlertCode").build()).build();

    private static final SdkField<String> ALERT_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlertMessage").getter(getter(Alert::alertMessage)).setter(setter(Builder::alertMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlertMessage").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime")
            .getter(getter(Alert::lastModifiedTime))
            .setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.UNIX_TIMESTAMP)).build();

    private static final SdkField<List<String>> RELATED_RESOURCE_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RelatedResourceArns")
            .getter(getter(Alert::relatedResourceArns))
            .setter(setter(Builder::relatedResourceArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedResourceArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceArn").getter(getter(Alert::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALERT_CODE_FIELD,
            ALERT_MESSAGE_FIELD, LAST_MODIFIED_TIME_FIELD, RELATED_RESOURCE_ARNS_FIELD, RESOURCE_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String alertCode;

    private final String alertMessage;

    private final Instant lastModifiedTime;

    private final List<String> relatedResourceArns;

    private final String resourceArn;

    private Alert(BuilderImpl builder) {
        this.alertCode = builder.alertCode;
        this.alertMessage = builder.alertMessage;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.relatedResourceArns = builder.relatedResourceArns;
        this.resourceArn = builder.resourceArn;
    }

    /**
     * <p>
     * The code for the alert. For example, <code>NOT_PROCESSED</code>.
     * </p>
     * 
     * @return The code for the alert. For example, <code>NOT_PROCESSED</code>.
     */
    public final String alertCode() {
        return alertCode;
    }

    /**
     * <p>
     * If an alert is generated for a resource, an explanation of the reason for the alert.
     * </p>
     * 
     * @return If an alert is generated for a resource, an explanation of the reason for the alert.
     */
    public final String alertMessage() {
        return alertMessage;
    }

    /**
     * <p>
     * The timestamp when the alert was last modified.
     * </p>
     * 
     * @return The timestamp when the alert was last modified.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the RelatedResourceArns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRelatedResourceArns() {
        return relatedResourceArns != null && !(relatedResourceArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Names (ARNs) related to this alert.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelatedResourceArns} method.
     * </p>
     * 
     * @return The Amazon Resource Names (ARNs) related to this alert.
     */
    public final List<String> relatedResourceArns() {
        return relatedResourceArns;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the resource.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the resource.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(alertCode());
        hashCode = 31 * hashCode + Objects.hashCode(alertMessage());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasRelatedResourceArns() ? relatedResourceArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Alert)) {
            return false;
        }
        Alert other = (Alert) obj;
        return Objects.equals(alertCode(), other.alertCode()) && Objects.equals(alertMessage(), other.alertMessage())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && hasRelatedResourceArns() == other.hasRelatedResourceArns()
                && Objects.equals(relatedResourceArns(), other.relatedResourceArns())
                && Objects.equals(resourceArn(), other.resourceArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Alert").add("AlertCode", alertCode()).add("AlertMessage", alertMessage())
                .add("LastModifiedTime", lastModifiedTime())
                .add("RelatedResourceArns", hasRelatedResourceArns() ? relatedResourceArns() : null)
                .add("ResourceArn", resourceArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AlertCode":
            return Optional.ofNullable(clazz.cast(alertCode()));
        case "AlertMessage":
            return Optional.ofNullable(clazz.cast(alertMessage()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "RelatedResourceArns":
            return Optional.ofNullable(clazz.cast(relatedResourceArns()));
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Alert, T> g) {
        return obj -> g.apply((Alert) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Alert> {
        /**
         * <p>
         * The code for the alert. For example, <code>NOT_PROCESSED</code>.
         * </p>
         * 
         * @param alertCode
         *        The code for the alert. For example, <code>NOT_PROCESSED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alertCode(String alertCode);

        /**
         * <p>
         * If an alert is generated for a resource, an explanation of the reason for the alert.
         * </p>
         * 
         * @param alertMessage
         *        If an alert is generated for a resource, an explanation of the reason for the alert.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alertMessage(String alertMessage);

        /**
         * <p>
         * The timestamp when the alert was last modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        The timestamp when the alert was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) related to this alert.
         * </p>
         * 
         * @param relatedResourceArns
         *        The Amazon Resource Names (ARNs) related to this alert.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedResourceArns(Collection<String> relatedResourceArns);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) related to this alert.
         * </p>
         * 
         * @param relatedResourceArns
         *        The Amazon Resource Names (ARNs) related to this alert.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedResourceArns(String... relatedResourceArns);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the resource.
         * </p>
         * 
         * @param resourceArn
         *        The Amazon Resource Name (ARN) of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);
    }

    static final class BuilderImpl implements Builder {
        private String alertCode;

        private String alertMessage;

        private Instant lastModifiedTime;

        private List<String> relatedResourceArns = DefaultSdkAutoConstructList.getInstance();

        private String resourceArn;

        private BuilderImpl() {
        }

        private BuilderImpl(Alert model) {
            alertCode(model.alertCode);
            alertMessage(model.alertMessage);
            lastModifiedTime(model.lastModifiedTime);
            relatedResourceArns(model.relatedResourceArns);
            resourceArn(model.resourceArn);
        }

        public final String getAlertCode() {
            return alertCode;
        }

        public final void setAlertCode(String alertCode) {
            this.alertCode = alertCode;
        }

        @Override
        public final Builder alertCode(String alertCode) {
            this.alertCode = alertCode;
            return this;
        }

        public final String getAlertMessage() {
            return alertMessage;
        }

        public final void setAlertMessage(String alertMessage) {
            this.alertMessage = alertMessage;
        }

        @Override
        public final Builder alertMessage(String alertMessage) {
            this.alertMessage = alertMessage;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final Collection<String> getRelatedResourceArns() {
            if (relatedResourceArns instanceof SdkAutoConstructList) {
                return null;
            }
            return relatedResourceArns;
        }

        public final void setRelatedResourceArns(Collection<String> relatedResourceArns) {
            this.relatedResourceArns = ___listOf__stringCopier.copy(relatedResourceArns);
        }

        @Override
        public final Builder relatedResourceArns(Collection<String> relatedResourceArns) {
            this.relatedResourceArns = ___listOf__stringCopier.copy(relatedResourceArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedResourceArns(String... relatedResourceArns) {
            relatedResourceArns(Arrays.asList(relatedResourceArns));
            return this;
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        @Override
        public Alert build() {
            return new Alert(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
