/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediatailor.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * MediaTailor only places (consumes) prefetched ads if the ad break meets the criteria defined by the dynamic
 * variables. This gives you granular control over which ad break to place the prefetched ads into.
 * </p>
 * <p>
 * As an example, let's say that you set <code>DynamicVariable</code> to <code>scte.event_id</code> and
 * <code>Operator</code> to <code>EQUALS</code>, and your playback configuration has an ADS URL of
 * <code>https://my.ads.server.com/path?&amp;podId=[scte.avail_num]&amp;event=[scte.event_id]&amp;duration=[session.avail_duration_secs]</code>
 * . And the prefetch request to the ADS contains these values
 * <code>https://my.ads.server.com/path?&amp;podId=3&amp;event=my-awesome-event&amp;duration=30</code>. MediaTailor will
 * only insert the prefetched ads into the ad break if has a SCTE marker with an event id of
 * <code>my-awesome-event</code>, since it must match the event id that MediaTailor uses to query the ADS.
 * </p>
 * <p>
 * You can specify up to five <code>AvailMatchingCriteria</code>. If you specify multiple
 * <code>AvailMatchingCriteria</code>, MediaTailor combines them to match using a logical <code>AND</code>. You can
 * model logical <code>OR</code> combinations by creating multiple prefetch schedules.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AvailMatchingCriteria implements SdkPojo, Serializable,
        ToCopyableBuilder<AvailMatchingCriteria.Builder, AvailMatchingCriteria> {
    private static final SdkField<String> DYNAMIC_VARIABLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DynamicVariable").getter(getter(AvailMatchingCriteria::dynamicVariable))
            .setter(setter(Builder::dynamicVariable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DynamicVariable").build()).build();

    private static final SdkField<String> OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Operator").getter(getter(AvailMatchingCriteria::operatorAsString)).setter(setter(Builder::operator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operator").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DYNAMIC_VARIABLE_FIELD,
            OPERATOR_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String dynamicVariable;

    private final String operator;

    private AvailMatchingCriteria(BuilderImpl builder) {
        this.dynamicVariable = builder.dynamicVariable;
        this.operator = builder.operator;
    }

    /**
     * <p>
     * The dynamic variable(s) that MediaTailor should use as avail matching criteria. MediaTailor only places the
     * prefetched ads into the avail if the avail matches the criteria defined by the dynamic variable. For information
     * about dynamic variables, see <a href="https://docs.aws.amazon.com/mediatailor/latest/ug/variables.html">Using
     * dynamic ad variables</a> in the <i>MediaTailor User Guide</i>.
     * </p>
     * <p>
     * You can include up to 100 dynamic variables.
     * </p>
     * 
     * @return The dynamic variable(s) that MediaTailor should use as avail matching criteria. MediaTailor only places
     *         the prefetched ads into the avail if the avail matches the criteria defined by the dynamic variable. For
     *         information about dynamic variables, see <a
     *         href="https://docs.aws.amazon.com/mediatailor/latest/ug/variables.html">Using dynamic ad variables</a> in
     *         the <i>MediaTailor User Guide</i>.</p>
     *         <p>
     *         You can include up to 100 dynamic variables.
     */
    public final String dynamicVariable() {
        return dynamicVariable;
    }

    /**
     * <p>
     * For the <code>DynamicVariable</code> specified in <code>AvailMatchingCriteria</code>, the Operator that is used
     * for the comparison.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link Operator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operatorAsString}.
     * </p>
     * 
     * @return For the <code>DynamicVariable</code> specified in <code>AvailMatchingCriteria</code>, the Operator that
     *         is used for the comparison.
     * @see Operator
     */
    public final Operator operator() {
        return Operator.fromValue(operator);
    }

    /**
     * <p>
     * For the <code>DynamicVariable</code> specified in <code>AvailMatchingCriteria</code>, the Operator that is used
     * for the comparison.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link Operator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operatorAsString}.
     * </p>
     * 
     * @return For the <code>DynamicVariable</code> specified in <code>AvailMatchingCriteria</code>, the Operator that
     *         is used for the comparison.
     * @see Operator
     */
    public final String operatorAsString() {
        return operator;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dynamicVariable());
        hashCode = 31 * hashCode + Objects.hashCode(operatorAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AvailMatchingCriteria)) {
            return false;
        }
        AvailMatchingCriteria other = (AvailMatchingCriteria) obj;
        return Objects.equals(dynamicVariable(), other.dynamicVariable())
                && Objects.equals(operatorAsString(), other.operatorAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AvailMatchingCriteria").add("DynamicVariable", dynamicVariable())
                .add("Operator", operatorAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DynamicVariable":
            return Optional.ofNullable(clazz.cast(dynamicVariable()));
        case "Operator":
            return Optional.ofNullable(clazz.cast(operatorAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("DynamicVariable", DYNAMIC_VARIABLE_FIELD);
        map.put("Operator", OPERATOR_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AvailMatchingCriteria, T> g) {
        return obj -> g.apply((AvailMatchingCriteria) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AvailMatchingCriteria> {
        /**
         * <p>
         * The dynamic variable(s) that MediaTailor should use as avail matching criteria. MediaTailor only places the
         * prefetched ads into the avail if the avail matches the criteria defined by the dynamic variable. For
         * information about dynamic variables, see <a
         * href="https://docs.aws.amazon.com/mediatailor/latest/ug/variables.html">Using dynamic ad variables</a> in the
         * <i>MediaTailor User Guide</i>.
         * </p>
         * <p>
         * You can include up to 100 dynamic variables.
         * </p>
         * 
         * @param dynamicVariable
         *        The dynamic variable(s) that MediaTailor should use as avail matching criteria. MediaTailor only
         *        places the prefetched ads into the avail if the avail matches the criteria defined by the dynamic
         *        variable. For information about dynamic variables, see <a
         *        href="https://docs.aws.amazon.com/mediatailor/latest/ug/variables.html">Using dynamic ad variables</a>
         *        in the <i>MediaTailor User Guide</i>.</p>
         *        <p>
         *        You can include up to 100 dynamic variables.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dynamicVariable(String dynamicVariable);

        /**
         * <p>
         * For the <code>DynamicVariable</code> specified in <code>AvailMatchingCriteria</code>, the Operator that is
         * used for the comparison.
         * </p>
         * 
         * @param operator
         *        For the <code>DynamicVariable</code> specified in <code>AvailMatchingCriteria</code>, the Operator
         *        that is used for the comparison.
         * @see Operator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Operator
         */
        Builder operator(String operator);

        /**
         * <p>
         * For the <code>DynamicVariable</code> specified in <code>AvailMatchingCriteria</code>, the Operator that is
         * used for the comparison.
         * </p>
         * 
         * @param operator
         *        For the <code>DynamicVariable</code> specified in <code>AvailMatchingCriteria</code>, the Operator
         *        that is used for the comparison.
         * @see Operator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Operator
         */
        Builder operator(Operator operator);
    }

    static final class BuilderImpl implements Builder {
        private String dynamicVariable;

        private String operator;

        private BuilderImpl() {
        }

        private BuilderImpl(AvailMatchingCriteria model) {
            dynamicVariable(model.dynamicVariable);
            operator(model.operator);
        }

        public final String getDynamicVariable() {
            return dynamicVariable;
        }

        public final void setDynamicVariable(String dynamicVariable) {
            this.dynamicVariable = dynamicVariable;
        }

        @Override
        public final Builder dynamicVariable(String dynamicVariable) {
            this.dynamicVariable = dynamicVariable;
            return this;
        }

        public final String getOperator() {
            return operator;
        }

        public final void setOperator(String operator) {
            this.operator = operator;
        }

        @Override
        public final Builder operator(String operator) {
            this.operator = operator;
            return this;
        }

        @Override
        public final Builder operator(Operator operator) {
            this.operator(operator == null ? null : operator.toString());
            return this;
        }

        @Override
        public AvailMatchingCriteria build() {
            return new AvailMatchingCriteria(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
