/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediatailor.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration that defines how MediaTailor performs recurring prefetch.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecurringPrefetchConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<RecurringPrefetchConfiguration.Builder, RecurringPrefetchConfiguration> {
    private static final SdkField<Instant> START_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime")
            .getter(getter(RecurringPrefetchConfiguration::startTime))
            .setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.UNIX_TIMESTAMP)).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime")
            .getter(getter(RecurringPrefetchConfiguration::endTime))
            .setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.UNIX_TIMESTAMP)).build();

    private static final SdkField<RecurringConsumption> RECURRING_CONSUMPTION_FIELD = SdkField
            .<RecurringConsumption> builder(MarshallingType.SDK_POJO).memberName("RecurringConsumption")
            .getter(getter(RecurringPrefetchConfiguration::recurringConsumption)).setter(setter(Builder::recurringConsumption))
            .constructor(RecurringConsumption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecurringConsumption").build())
            .build();

    private static final SdkField<RecurringRetrieval> RECURRING_RETRIEVAL_FIELD = SdkField
            .<RecurringRetrieval> builder(MarshallingType.SDK_POJO).memberName("RecurringRetrieval")
            .getter(getter(RecurringPrefetchConfiguration::recurringRetrieval)).setter(setter(Builder::recurringRetrieval))
            .constructor(RecurringRetrieval::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecurringRetrieval").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(START_TIME_FIELD,
            END_TIME_FIELD, RECURRING_CONSUMPTION_FIELD, RECURRING_RETRIEVAL_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Instant startTime;

    private final Instant endTime;

    private final RecurringConsumption recurringConsumption;

    private final RecurringRetrieval recurringRetrieval;

    private RecurringPrefetchConfiguration(BuilderImpl builder) {
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.recurringConsumption = builder.recurringConsumption;
        this.recurringRetrieval = builder.recurringRetrieval;
    }

    /**
     * <p>
     * The start time for the window that MediaTailor prefetches and inserts ads in a live event.
     * </p>
     * 
     * @return The start time for the window that MediaTailor prefetches and inserts ads in a live event.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The end time for the window that MediaTailor prefetches and inserts ads in a live event.
     * </p>
     * 
     * @return The end time for the window that MediaTailor prefetches and inserts ads in a live event.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The settings that determine how and when MediaTailor places prefetched ads into upcoming ad breaks for recurring
     * prefetch scedules.
     * </p>
     * 
     * @return The settings that determine how and when MediaTailor places prefetched ads into upcoming ad breaks for
     *         recurring prefetch scedules.
     */
    public final RecurringConsumption recurringConsumption() {
        return recurringConsumption;
    }

    /**
     * <p>
     * The configuration for prefetch ad retrieval from the ADS.
     * </p>
     * 
     * @return The configuration for prefetch ad retrieval from the ADS.
     */
    public final RecurringRetrieval recurringRetrieval() {
        return recurringRetrieval;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(recurringConsumption());
        hashCode = 31 * hashCode + Objects.hashCode(recurringRetrieval());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecurringPrefetchConfiguration)) {
            return false;
        }
        RecurringPrefetchConfiguration other = (RecurringPrefetchConfiguration) obj;
        return Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(recurringConsumption(), other.recurringConsumption())
                && Objects.equals(recurringRetrieval(), other.recurringRetrieval());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecurringPrefetchConfiguration").add("StartTime", startTime()).add("EndTime", endTime())
                .add("RecurringConsumption", recurringConsumption()).add("RecurringRetrieval", recurringRetrieval()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "RecurringConsumption":
            return Optional.ofNullable(clazz.cast(recurringConsumption()));
        case "RecurringRetrieval":
            return Optional.ofNullable(clazz.cast(recurringRetrieval()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("StartTime", START_TIME_FIELD);
        map.put("EndTime", END_TIME_FIELD);
        map.put("RecurringConsumption", RECURRING_CONSUMPTION_FIELD);
        map.put("RecurringRetrieval", RECURRING_RETRIEVAL_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RecurringPrefetchConfiguration, T> g) {
        return obj -> g.apply((RecurringPrefetchConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecurringPrefetchConfiguration> {
        /**
         * <p>
         * The start time for the window that MediaTailor prefetches and inserts ads in a live event.
         * </p>
         * 
         * @param startTime
         *        The start time for the window that MediaTailor prefetches and inserts ads in a live event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The end time for the window that MediaTailor prefetches and inserts ads in a live event.
         * </p>
         * 
         * @param endTime
         *        The end time for the window that MediaTailor prefetches and inserts ads in a live event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The settings that determine how and when MediaTailor places prefetched ads into upcoming ad breaks for
         * recurring prefetch scedules.
         * </p>
         * 
         * @param recurringConsumption
         *        The settings that determine how and when MediaTailor places prefetched ads into upcoming ad breaks for
         *        recurring prefetch scedules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recurringConsumption(RecurringConsumption recurringConsumption);

        /**
         * <p>
         * The settings that determine how and when MediaTailor places prefetched ads into upcoming ad breaks for
         * recurring prefetch scedules.
         * </p>
         * This is a convenience method that creates an instance of the {@link RecurringConsumption.Builder} avoiding
         * the need to create one manually via {@link RecurringConsumption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RecurringConsumption.Builder#build()} is called immediately and
         * its result is passed to {@link #recurringConsumption(RecurringConsumption)}.
         * 
         * @param recurringConsumption
         *        a consumer that will call methods on {@link RecurringConsumption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recurringConsumption(RecurringConsumption)
         */
        default Builder recurringConsumption(Consumer<RecurringConsumption.Builder> recurringConsumption) {
            return recurringConsumption(RecurringConsumption.builder().applyMutation(recurringConsumption).build());
        }

        /**
         * <p>
         * The configuration for prefetch ad retrieval from the ADS.
         * </p>
         * 
         * @param recurringRetrieval
         *        The configuration for prefetch ad retrieval from the ADS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recurringRetrieval(RecurringRetrieval recurringRetrieval);

        /**
         * <p>
         * The configuration for prefetch ad retrieval from the ADS.
         * </p>
         * This is a convenience method that creates an instance of the {@link RecurringRetrieval.Builder} avoiding the
         * need to create one manually via {@link RecurringRetrieval#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RecurringRetrieval.Builder#build()} is called immediately and its
         * result is passed to {@link #recurringRetrieval(RecurringRetrieval)}.
         * 
         * @param recurringRetrieval
         *        a consumer that will call methods on {@link RecurringRetrieval.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recurringRetrieval(RecurringRetrieval)
         */
        default Builder recurringRetrieval(Consumer<RecurringRetrieval.Builder> recurringRetrieval) {
            return recurringRetrieval(RecurringRetrieval.builder().applyMutation(recurringRetrieval).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Instant startTime;

        private Instant endTime;

        private RecurringConsumption recurringConsumption;

        private RecurringRetrieval recurringRetrieval;

        private BuilderImpl() {
        }

        private BuilderImpl(RecurringPrefetchConfiguration model) {
            startTime(model.startTime);
            endTime(model.endTime);
            recurringConsumption(model.recurringConsumption);
            recurringRetrieval(model.recurringRetrieval);
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final RecurringConsumption.Builder getRecurringConsumption() {
            return recurringConsumption != null ? recurringConsumption.toBuilder() : null;
        }

        public final void setRecurringConsumption(RecurringConsumption.BuilderImpl recurringConsumption) {
            this.recurringConsumption = recurringConsumption != null ? recurringConsumption.build() : null;
        }

        @Override
        public final Builder recurringConsumption(RecurringConsumption recurringConsumption) {
            this.recurringConsumption = recurringConsumption;
            return this;
        }

        public final RecurringRetrieval.Builder getRecurringRetrieval() {
            return recurringRetrieval != null ? recurringRetrieval.toBuilder() : null;
        }

        public final void setRecurringRetrieval(RecurringRetrieval.BuilderImpl recurringRetrieval) {
            this.recurringRetrieval = recurringRetrieval != null ? recurringRetrieval.build() : null;
        }

        @Override
        public final Builder recurringRetrieval(RecurringRetrieval recurringRetrieval) {
            this.recurringRetrieval = recurringRetrieval;
            return this;
        }

        @Override
        public RecurringPrefetchConfiguration build() {
            return new RecurringPrefetchConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
