/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.memorydb;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.memorydb.model.AclAlreadyExistsException;
import software.amazon.awssdk.services.memorydb.model.AclNotFoundException;
import software.amazon.awssdk.services.memorydb.model.AclQuotaExceededException;
import software.amazon.awssdk.services.memorydb.model.ApiCallRateForCustomerExceededException;
import software.amazon.awssdk.services.memorydb.model.BatchUpdateClusterRequest;
import software.amazon.awssdk.services.memorydb.model.BatchUpdateClusterResponse;
import software.amazon.awssdk.services.memorydb.model.ClusterAlreadyExistsException;
import software.amazon.awssdk.services.memorydb.model.ClusterNotFoundException;
import software.amazon.awssdk.services.memorydb.model.ClusterQuotaForCustomerExceededException;
import software.amazon.awssdk.services.memorydb.model.CopySnapshotRequest;
import software.amazon.awssdk.services.memorydb.model.CopySnapshotResponse;
import software.amazon.awssdk.services.memorydb.model.CreateAclRequest;
import software.amazon.awssdk.services.memorydb.model.CreateAclResponse;
import software.amazon.awssdk.services.memorydb.model.CreateClusterRequest;
import software.amazon.awssdk.services.memorydb.model.CreateClusterResponse;
import software.amazon.awssdk.services.memorydb.model.CreateParameterGroupRequest;
import software.amazon.awssdk.services.memorydb.model.CreateParameterGroupResponse;
import software.amazon.awssdk.services.memorydb.model.CreateSnapshotRequest;
import software.amazon.awssdk.services.memorydb.model.CreateSnapshotResponse;
import software.amazon.awssdk.services.memorydb.model.CreateSubnetGroupRequest;
import software.amazon.awssdk.services.memorydb.model.CreateSubnetGroupResponse;
import software.amazon.awssdk.services.memorydb.model.CreateUserRequest;
import software.amazon.awssdk.services.memorydb.model.CreateUserResponse;
import software.amazon.awssdk.services.memorydb.model.DefaultUserRequiredException;
import software.amazon.awssdk.services.memorydb.model.DeleteAclRequest;
import software.amazon.awssdk.services.memorydb.model.DeleteAclResponse;
import software.amazon.awssdk.services.memorydb.model.DeleteClusterRequest;
import software.amazon.awssdk.services.memorydb.model.DeleteClusterResponse;
import software.amazon.awssdk.services.memorydb.model.DeleteParameterGroupRequest;
import software.amazon.awssdk.services.memorydb.model.DeleteParameterGroupResponse;
import software.amazon.awssdk.services.memorydb.model.DeleteSnapshotRequest;
import software.amazon.awssdk.services.memorydb.model.DeleteSnapshotResponse;
import software.amazon.awssdk.services.memorydb.model.DeleteSubnetGroupRequest;
import software.amazon.awssdk.services.memorydb.model.DeleteSubnetGroupResponse;
import software.amazon.awssdk.services.memorydb.model.DeleteUserRequest;
import software.amazon.awssdk.services.memorydb.model.DeleteUserResponse;
import software.amazon.awssdk.services.memorydb.model.DescribeAcLsRequest;
import software.amazon.awssdk.services.memorydb.model.DescribeAcLsResponse;
import software.amazon.awssdk.services.memorydb.model.DescribeClustersRequest;
import software.amazon.awssdk.services.memorydb.model.DescribeClustersResponse;
import software.amazon.awssdk.services.memorydb.model.DescribeEngineVersionsRequest;
import software.amazon.awssdk.services.memorydb.model.DescribeEngineVersionsResponse;
import software.amazon.awssdk.services.memorydb.model.DescribeEventsRequest;
import software.amazon.awssdk.services.memorydb.model.DescribeEventsResponse;
import software.amazon.awssdk.services.memorydb.model.DescribeParameterGroupsRequest;
import software.amazon.awssdk.services.memorydb.model.DescribeParameterGroupsResponse;
import software.amazon.awssdk.services.memorydb.model.DescribeParametersRequest;
import software.amazon.awssdk.services.memorydb.model.DescribeParametersResponse;
import software.amazon.awssdk.services.memorydb.model.DescribeServiceUpdatesRequest;
import software.amazon.awssdk.services.memorydb.model.DescribeServiceUpdatesResponse;
import software.amazon.awssdk.services.memorydb.model.DescribeSnapshotsRequest;
import software.amazon.awssdk.services.memorydb.model.DescribeSnapshotsResponse;
import software.amazon.awssdk.services.memorydb.model.DescribeSubnetGroupsRequest;
import software.amazon.awssdk.services.memorydb.model.DescribeSubnetGroupsResponse;
import software.amazon.awssdk.services.memorydb.model.DescribeUsersRequest;
import software.amazon.awssdk.services.memorydb.model.DescribeUsersResponse;
import software.amazon.awssdk.services.memorydb.model.DuplicateUserNameException;
import software.amazon.awssdk.services.memorydb.model.FailoverShardRequest;
import software.amazon.awssdk.services.memorydb.model.FailoverShardResponse;
import software.amazon.awssdk.services.memorydb.model.InsufficientClusterCapacityException;
import software.amazon.awssdk.services.memorydb.model.InvalidAclStateException;
import software.amazon.awssdk.services.memorydb.model.InvalidArnException;
import software.amazon.awssdk.services.memorydb.model.InvalidClusterStateException;
import software.amazon.awssdk.services.memorydb.model.InvalidCredentialsException;
import software.amazon.awssdk.services.memorydb.model.InvalidKmsKeyException;
import software.amazon.awssdk.services.memorydb.model.InvalidNodeStateException;
import software.amazon.awssdk.services.memorydb.model.InvalidParameterCombinationException;
import software.amazon.awssdk.services.memorydb.model.InvalidParameterGroupStateException;
import software.amazon.awssdk.services.memorydb.model.InvalidParameterValueException;
import software.amazon.awssdk.services.memorydb.model.InvalidSnapshotStateException;
import software.amazon.awssdk.services.memorydb.model.InvalidSubnetException;
import software.amazon.awssdk.services.memorydb.model.InvalidUserStateException;
import software.amazon.awssdk.services.memorydb.model.InvalidVpcNetworkStateException;
import software.amazon.awssdk.services.memorydb.model.ListAllowedNodeTypeUpdatesRequest;
import software.amazon.awssdk.services.memorydb.model.ListAllowedNodeTypeUpdatesResponse;
import software.amazon.awssdk.services.memorydb.model.ListTagsRequest;
import software.amazon.awssdk.services.memorydb.model.ListTagsResponse;
import software.amazon.awssdk.services.memorydb.model.MemoryDbException;
import software.amazon.awssdk.services.memorydb.model.NoOperationException;
import software.amazon.awssdk.services.memorydb.model.NodeQuotaForClusterExceededException;
import software.amazon.awssdk.services.memorydb.model.NodeQuotaForCustomerExceededException;
import software.amazon.awssdk.services.memorydb.model.ParameterGroupAlreadyExistsException;
import software.amazon.awssdk.services.memorydb.model.ParameterGroupNotFoundException;
import software.amazon.awssdk.services.memorydb.model.ParameterGroupQuotaExceededException;
import software.amazon.awssdk.services.memorydb.model.ResetParameterGroupRequest;
import software.amazon.awssdk.services.memorydb.model.ResetParameterGroupResponse;
import software.amazon.awssdk.services.memorydb.model.ServiceLinkedRoleNotFoundException;
import software.amazon.awssdk.services.memorydb.model.ServiceUpdateNotFoundException;
import software.amazon.awssdk.services.memorydb.model.ShardNotFoundException;
import software.amazon.awssdk.services.memorydb.model.ShardsPerClusterQuotaExceededException;
import software.amazon.awssdk.services.memorydb.model.SnapshotAlreadyExistsException;
import software.amazon.awssdk.services.memorydb.model.SnapshotNotFoundException;
import software.amazon.awssdk.services.memorydb.model.SnapshotQuotaExceededException;
import software.amazon.awssdk.services.memorydb.model.SubnetGroupAlreadyExistsException;
import software.amazon.awssdk.services.memorydb.model.SubnetGroupInUseException;
import software.amazon.awssdk.services.memorydb.model.SubnetGroupNotFoundException;
import software.amazon.awssdk.services.memorydb.model.SubnetGroupQuotaExceededException;
import software.amazon.awssdk.services.memorydb.model.SubnetInUseException;
import software.amazon.awssdk.services.memorydb.model.SubnetNotAllowedException;
import software.amazon.awssdk.services.memorydb.model.SubnetQuotaExceededException;
import software.amazon.awssdk.services.memorydb.model.TagNotFoundException;
import software.amazon.awssdk.services.memorydb.model.TagQuotaPerResourceExceededException;
import software.amazon.awssdk.services.memorydb.model.TagResourceRequest;
import software.amazon.awssdk.services.memorydb.model.TagResourceResponse;
import software.amazon.awssdk.services.memorydb.model.TestFailoverNotAvailableException;
import software.amazon.awssdk.services.memorydb.model.UntagResourceRequest;
import software.amazon.awssdk.services.memorydb.model.UntagResourceResponse;
import software.amazon.awssdk.services.memorydb.model.UpdateAclRequest;
import software.amazon.awssdk.services.memorydb.model.UpdateAclResponse;
import software.amazon.awssdk.services.memorydb.model.UpdateClusterRequest;
import software.amazon.awssdk.services.memorydb.model.UpdateClusterResponse;
import software.amazon.awssdk.services.memorydb.model.UpdateParameterGroupRequest;
import software.amazon.awssdk.services.memorydb.model.UpdateParameterGroupResponse;
import software.amazon.awssdk.services.memorydb.model.UpdateSubnetGroupRequest;
import software.amazon.awssdk.services.memorydb.model.UpdateSubnetGroupResponse;
import software.amazon.awssdk.services.memorydb.model.UpdateUserRequest;
import software.amazon.awssdk.services.memorydb.model.UpdateUserResponse;
import software.amazon.awssdk.services.memorydb.model.UserAlreadyExistsException;
import software.amazon.awssdk.services.memorydb.model.UserNotFoundException;
import software.amazon.awssdk.services.memorydb.model.UserQuotaExceededException;
import software.amazon.awssdk.services.memorydb.transform.BatchUpdateClusterRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.CopySnapshotRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.CreateAclRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.CreateParameterGroupRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.CreateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.CreateSubnetGroupRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DeleteAclRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DeleteParameterGroupRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DeleteSnapshotRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DeleteSubnetGroupRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DescribeAcLsRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DescribeClustersRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DescribeEngineVersionsRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DescribeEventsRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DescribeParameterGroupsRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DescribeParametersRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DescribeServiceUpdatesRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DescribeSnapshotsRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DescribeSubnetGroupsRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.DescribeUsersRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.FailoverShardRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.ListAllowedNodeTypeUpdatesRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.ResetParameterGroupRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.UpdateAclRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.UpdateClusterRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.UpdateParameterGroupRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.UpdateSubnetGroupRequestMarshaller;
import software.amazon.awssdk.services.memorydb.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MemoryDbAsyncClient}.
 *
 * @see MemoryDbAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMemoryDbAsyncClient implements MemoryDbAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMemoryDbAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMemoryDbAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Apply the service update to a list of clusters supplied. For more information on service updates and applying
     * them, see <a
     * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/managing-updates.html#applying-updates">Applying the
     * service updates</a>.
     * </p>
     *
     * @param batchUpdateClusterRequest
     * @return A Java Future containing the result of the BatchUpdateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUpdateNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.BatchUpdateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/BatchUpdateCluster" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUpdateClusterResponse> batchUpdateCluster(BatchUpdateClusterRequest batchUpdateClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdateClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUpdateClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchUpdateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchUpdateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUpdateClusterRequest, BatchUpdateClusterResponse>()
                            .withOperationName("BatchUpdateCluster")
                            .withMarshaller(new BatchUpdateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchUpdateClusterRequest));
            CompletableFuture<BatchUpdateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Makes a copy of an existing snapshot.
     * </p>
     *
     * @param copySnapshotRequest
     * @return A Java Future containing the result of the CopySnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SnapshotAlreadyExistsException</li>
     *         <li>SnapshotNotFoundException</li>
     *         <li>SnapshotQuotaExceededException</li>
     *         <li>InvalidSnapshotStateException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>TagQuotaPerResourceExceededException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.CopySnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/CopySnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CopySnapshotResponse> copySnapshot(CopySnapshotRequest copySnapshotRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copySnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopySnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CopySnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CopySnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CopySnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CopySnapshotRequest, CopySnapshotResponse>()
                            .withOperationName("CopySnapshot").withMarshaller(new CopySnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(copySnapshotRequest));
            CompletableFuture<CopySnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Access Control List. For more information, see <a
     * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/clusters.acls.html">Authenticating users with Access
     * Contol Lists (ACLs)</a>.
     * </p>
     *
     * @param createAclRequest
     * @return A Java Future containing the result of the CreateACL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UserNotFoundException</li>
     *         <li>DuplicateUserNameException</li>
     *         <li>AclAlreadyExistsException</li>
     *         <li>DefaultUserRequiredException</li>
     *         <li>AclQuotaExceededException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>TagQuotaPerResourceExceededException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.CreateACL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/CreateACL" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAclResponse> createACL(CreateAclRequest createAclRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAclRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateACL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAclResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAclResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAclResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAclRequest, CreateAclResponse>().withOperationName("CreateACL")
                            .withMarshaller(new CreateAclRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAclRequest));
            CompletableFuture<CreateAclResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a cluster. All nodes in the cluster run the same protocol-compliant engine software.
     * </p>
     *
     * @param createClusterRequest
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterAlreadyExistsException</li>
     *         <li>SubnetGroupNotFoundException</li>
     *         <li>ClusterQuotaForCustomerExceededException</li>
     *         <li>NodeQuotaForClusterExceededException</li>
     *         <li>NodeQuotaForCustomerExceededException</li>
     *         <li>ParameterGroupNotFoundException</li>
     *         <li>InsufficientClusterCapacityException</li>
     *         <li>InvalidVpcNetworkStateException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>ShardsPerClusterQuotaExceededException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>InvalidCredentialsException</li>
     *         <li>TagQuotaPerResourceExceededException</li>
     *         <li>AclNotFoundException</li>
     *         <li>InvalidAclStateException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateClusterResponse> createCluster(CreateClusterRequest createClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                            .withOperationName("CreateCluster")
                            .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createClusterRequest));
            CompletableFuture<CreateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new MemoryDB parameter group. A parameter group is a collection of parameters and their values that are
     * applied to all of the nodes in any cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/parametergroups.html">Configuring engine parameters
     * using parameter groups</a>.
     * </p>
     *
     * @param createParameterGroupRequest
     * @return A Java Future containing the result of the CreateParameterGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ParameterGroupQuotaExceededException</li>
     *         <li>ParameterGroupAlreadyExistsException</li>
     *         <li>InvalidParameterGroupStateException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>TagQuotaPerResourceExceededException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.CreateParameterGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/CreateParameterGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateParameterGroupResponse> createParameterGroup(
            CreateParameterGroupRequest createParameterGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createParameterGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateParameterGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateParameterGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateParameterGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateParameterGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateParameterGroupRequest, CreateParameterGroupResponse>()
                            .withOperationName("CreateParameterGroup")
                            .withMarshaller(new CreateParameterGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createParameterGroupRequest));
            CompletableFuture<CreateParameterGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a copy of an entire cluster at a specific moment in time.
     * </p>
     *
     * @param createSnapshotRequest
     * @return A Java Future containing the result of the CreateSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SnapshotAlreadyExistsException</li>
     *         <li>ClusterNotFoundException</li>
     *         <li>InvalidClusterStateException</li>
     *         <li>SnapshotQuotaExceededException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>TagQuotaPerResourceExceededException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.CreateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/CreateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSnapshotResponse> createSnapshot(CreateSnapshotRequest createSnapshotRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSnapshotRequest, CreateSnapshotResponse>()
                            .withOperationName("CreateSnapshot")
                            .withMarshaller(new CreateSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSnapshotRequest));
            CompletableFuture<CreateSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a subnet group. A subnet group is a collection of subnets (typically private) that you can designate for
     * your clusters running in an Amazon Virtual Private Cloud (VPC) environment. When you create a cluster in an
     * Amazon VPC, you must specify a subnet group. MemoryDB uses that subnet group to choose a subnet and IP addresses
     * within that subnet to associate with your nodes. For more information, see <a
     * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/subnetgroups.html">Subnets and subnet groups</a>.
     * </p>
     *
     * @param createSubnetGroupRequest
     * @return A Java Future containing the result of the CreateSubnetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubnetGroupAlreadyExistsException</li>
     *         <li>SubnetGroupQuotaExceededException</li>
     *         <li>SubnetQuotaExceededException</li>
     *         <li>InvalidSubnetException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>SubnetNotAllowedException</li>
     *         <li>TagQuotaPerResourceExceededException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.CreateSubnetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/CreateSubnetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSubnetGroupResponse> createSubnetGroup(CreateSubnetGroupRequest createSubnetGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSubnetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubnetGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSubnetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSubnetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSubnetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSubnetGroupRequest, CreateSubnetGroupResponse>()
                            .withOperationName("CreateSubnetGroup")
                            .withMarshaller(new CreateSubnetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSubnetGroupRequest));
            CompletableFuture<CreateSubnetGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a MemoryDB user. For more information, see <a
     * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/clusters.acls.html">Authenticating users with Access
     * Contol Lists (ACLs)</a>.
     * </p>
     *
     * @param createUserRequest
     * @return A Java Future containing the result of the CreateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UserAlreadyExistsException</li>
     *         <li>UserQuotaExceededException</li>
     *         <li>DuplicateUserNameException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>TagQuotaPerResourceExceededException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.CreateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateUserResponse> createUser(CreateUserRequest createUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>().withOperationName("CreateUser")
                            .withMarshaller(new CreateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createUserRequest));
            CompletableFuture<CreateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Access Control List. The ACL must first be disassociated from the cluster before it can be deleted.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/clusters.acls.html">Authenticating users with Access
     * Contol Lists (ACLs)</a>.
     * </p>
     *
     * @param deleteAclRequest
     * @return A Java Future containing the result of the DeleteACL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AclNotFoundException</li>
     *         <li>InvalidAclStateException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DeleteACL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DeleteACL" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAclResponse> deleteACL(DeleteAclRequest deleteAclRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAclRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteACL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAclResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAclResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAclResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAclRequest, DeleteAclResponse>().withOperationName("DeleteACL")
                            .withMarshaller(new DeleteAclRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAclRequest));
            CompletableFuture<DeleteAclResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a cluster. It also deletes all associated nodes and node endpoints
     * </p>
     *
     * @param deleteClusterRequest
     * @return A Java Future containing the result of the DeleteCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException</li>
     *         <li>InvalidClusterStateException</li>
     *         <li>SnapshotAlreadyExistsException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteClusterResponse> deleteCluster(DeleteClusterRequest deleteClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                            .withOperationName("DeleteCluster")
                            .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteClusterRequest));
            CompletableFuture<DeleteClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified parameter group. You cannot delete a parameter group if it is associated with any clusters.
     * You cannot delete the default parameter groups in your account.
     * </p>
     *
     * @param deleteParameterGroupRequest
     * @return A Java Future containing the result of the DeleteParameterGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterGroupStateException</li>
     *         <li>ParameterGroupNotFoundException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DeleteParameterGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DeleteParameterGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteParameterGroupResponse> deleteParameterGroup(
            DeleteParameterGroupRequest deleteParameterGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteParameterGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteParameterGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteParameterGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteParameterGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteParameterGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteParameterGroupRequest, DeleteParameterGroupResponse>()
                            .withOperationName("DeleteParameterGroup")
                            .withMarshaller(new DeleteParameterGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteParameterGroupRequest));
            CompletableFuture<DeleteParameterGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing snapshot. When you receive a successful response from this operation, MemoryDB immediately
     * begins deleting the snapshot; you cannot cancel or revert this operation.
     * </p>
     *
     * @param deleteSnapshotRequest
     * @return A Java Future containing the result of the DeleteSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SnapshotNotFoundException</li>
     *         <li>InvalidSnapshotStateException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DeleteSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DeleteSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSnapshotResponse> deleteSnapshot(DeleteSnapshotRequest deleteSnapshotRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSnapshotRequest, DeleteSnapshotResponse>()
                            .withOperationName("DeleteSnapshot")
                            .withMarshaller(new DeleteSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSnapshotRequest));
            CompletableFuture<DeleteSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a subnet group. You cannot delete a default subnet group or one that is associated with any clusters.
     * </p>
     *
     * @param deleteSubnetGroupRequest
     * @return A Java Future containing the result of the DeleteSubnetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubnetGroupInUseException</li>
     *         <li>SubnetGroupNotFoundException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DeleteSubnetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DeleteSubnetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSubnetGroupResponse> deleteSubnetGroup(DeleteSubnetGroupRequest deleteSubnetGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSubnetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSubnetGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSubnetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSubnetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSubnetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSubnetGroupRequest, DeleteSubnetGroupResponse>()
                            .withOperationName("DeleteSubnetGroup")
                            .withMarshaller(new DeleteSubnetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSubnetGroupRequest));
            CompletableFuture<DeleteSubnetGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a user. The user will be removed from all ACLs and in turn removed from all clusters.
     * </p>
     *
     * @param deleteUserRequest
     * @return A Java Future containing the result of the DeleteUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidUserStateException</li>
     *         <li>UserNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DeleteUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteUserResponse> deleteUser(DeleteUserRequest deleteUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>().withOperationName("DeleteUser")
                            .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteUserRequest));
            CompletableFuture<DeleteUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of ACLs
     * </p>
     *
     * @param describeAcLsRequest
     * @return A Java Future containing the result of the DescribeACLs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AclNotFoundException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DescribeACLs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DescribeACLs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAcLsResponse> describeACLs(DescribeAcLsRequest describeAcLsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAcLsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeACLs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAcLsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeAcLsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAcLsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAcLsRequest, DescribeAcLsResponse>()
                            .withOperationName("DescribeACLs").withMarshaller(new DescribeAcLsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAcLsRequest));
            CompletableFuture<DescribeAcLsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about all provisioned clusters if no cluster identifier is specified, or about a specific
     * cluster if a cluster name is supplied.
     * </p>
     *
     * @param describeClustersRequest
     * @return A Java Future containing the result of the DescribeClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DescribeClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DescribeClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeClustersResponse> describeClusters(DescribeClustersRequest describeClustersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeClusters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeClustersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeClustersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeClustersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeClustersRequest, DescribeClustersResponse>()
                            .withOperationName("DescribeClusters")
                            .withMarshaller(new DescribeClustersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeClustersRequest));
            CompletableFuture<DescribeClustersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the available Redis engine versions.
     * </p>
     *
     * @param describeEngineVersionsRequest
     * @return A Java Future containing the result of the DescribeEngineVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DescribeEngineVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DescribeEngineVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEngineVersionsResponse> describeEngineVersions(
            DescribeEngineVersionsRequest describeEngineVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEngineVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEngineVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEngineVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEngineVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEngineVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEngineVersionsRequest, DescribeEngineVersionsResponse>()
                            .withOperationName("DescribeEngineVersions")
                            .withMarshaller(new DescribeEngineVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEngineVersionsRequest));
            CompletableFuture<DescribeEngineVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns events related to clusters, security groups, and parameter groups. You can obtain events specific to a
     * particular cluster, security group, or parameter group by providing the name as a parameter. By default, only the
     * events occurring within the last hour are returned; however, you can retrieve up to 14 days' worth of events if
     * necessary.
     * </p>
     *
     * @param describeEventsRequest
     * @return A Java Future containing the result of the DescribeEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DescribeEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventsResponse> describeEvents(DescribeEventsRequest describeEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventsRequest, DescribeEventsResponse>()
                            .withOperationName("DescribeEvents")
                            .withMarshaller(new DescribeEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEventsRequest));
            CompletableFuture<DescribeEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of parameter group descriptions. If a parameter group name is specified, the list contains only
     * the descriptions for that group.
     * </p>
     *
     * @param describeParameterGroupsRequest
     * @return A Java Future containing the result of the DescribeParameterGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ParameterGroupNotFoundException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DescribeParameterGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DescribeParameterGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeParameterGroupsResponse> describeParameterGroups(
            DescribeParameterGroupsRequest describeParameterGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeParameterGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeParameterGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeParameterGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeParameterGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeParameterGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeParameterGroupsRequest, DescribeParameterGroupsResponse>()
                            .withOperationName("DescribeParameterGroups")
                            .withMarshaller(new DescribeParameterGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeParameterGroupsRequest));
            CompletableFuture<DescribeParameterGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the detailed parameter list for a particular parameter group.
     * </p>
     *
     * @param describeParametersRequest
     * @return A Java Future containing the result of the DescribeParameters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ParameterGroupNotFoundException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DescribeParameters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DescribeParameters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeParametersResponse> describeParameters(DescribeParametersRequest describeParametersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeParametersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeParameters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeParametersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeParametersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeParametersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeParametersRequest, DescribeParametersResponse>()
                            .withOperationName("DescribeParameters")
                            .withMarshaller(new DescribeParametersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeParametersRequest));
            CompletableFuture<DescribeParametersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details of the service updates
     * </p>
     *
     * @param describeServiceUpdatesRequest
     * @return A Java Future containing the result of the DescribeServiceUpdates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DescribeServiceUpdates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DescribeServiceUpdates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeServiceUpdatesResponse> describeServiceUpdates(
            DescribeServiceUpdatesRequest describeServiceUpdatesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServiceUpdatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServiceUpdates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeServiceUpdatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeServiceUpdatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeServiceUpdatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeServiceUpdatesRequest, DescribeServiceUpdatesResponse>()
                            .withOperationName("DescribeServiceUpdates")
                            .withMarshaller(new DescribeServiceUpdatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeServiceUpdatesRequest));
            CompletableFuture<DescribeServiceUpdatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about cluster snapshots. By default, DescribeSnapshots lists all of your snapshots; it can
     * optionally describe a single snapshot, or just the snapshots associated with a particular cluster.
     * </p>
     *
     * @param describeSnapshotsRequest
     * @return A Java Future containing the result of the DescribeSnapshots operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SnapshotNotFoundException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DescribeSnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DescribeSnapshots" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSnapshotsResponse> describeSnapshots(DescribeSnapshotsRequest describeSnapshotsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSnapshotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSnapshots");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSnapshotsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSnapshotsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSnapshotsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSnapshotsRequest, DescribeSnapshotsResponse>()
                            .withOperationName("DescribeSnapshots")
                            .withMarshaller(new DescribeSnapshotsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSnapshotsRequest));
            CompletableFuture<DescribeSnapshotsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of subnet group descriptions. If a subnet group name is specified, the list contains only the
     * description of that group.
     * </p>
     *
     * @param describeSubnetGroupsRequest
     * @return A Java Future containing the result of the DescribeSubnetGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubnetGroupNotFoundException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DescribeSubnetGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DescribeSubnetGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSubnetGroupsResponse> describeSubnetGroups(
            DescribeSubnetGroupsRequest describeSubnetGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSubnetGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSubnetGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSubnetGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSubnetGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSubnetGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSubnetGroupsRequest, DescribeSubnetGroupsResponse>()
                            .withOperationName("DescribeSubnetGroups")
                            .withMarshaller(new DescribeSubnetGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSubnetGroupsRequest));
            CompletableFuture<DescribeSubnetGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of users.
     * </p>
     *
     * @param describeUsersRequest
     * @return A Java Future containing the result of the DescribeUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UserNotFoundException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.DescribeUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/DescribeUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeUsersResponse> describeUsers(DescribeUsersRequest describeUsersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeUsersRequest, DescribeUsersResponse>()
                            .withOperationName("DescribeUsers")
                            .withMarshaller(new DescribeUsersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeUsersRequest));
            CompletableFuture<DescribeUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to failover a shard
     * </p>
     *
     * @param failoverShardRequest
     * @return A Java Future containing the result of the FailoverShard operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ApiCallRateForCustomerExceededException</li>
     *         <li>InvalidClusterStateException</li>
     *         <li>ShardNotFoundException</li>
     *         <li>ClusterNotFoundException</li>
     *         <li>TestFailoverNotAvailableException</li>
     *         <li>InvalidKmsKeyException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.FailoverShard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/FailoverShard" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<FailoverShardResponse> failoverShard(FailoverShardRequest failoverShardRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, failoverShardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "FailoverShard");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<FailoverShardResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    FailoverShardResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<FailoverShardResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<FailoverShardRequest, FailoverShardResponse>()
                            .withOperationName("FailoverShard")
                            .withMarshaller(new FailoverShardRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(failoverShardRequest));
            CompletableFuture<FailoverShardResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all available node types that you can scale to from your cluster's current node type. When you use the
     * UpdateCluster operation to scale your cluster, the value of the NodeType parameter must be one of the node types
     * returned by this operation.
     * </p>
     *
     * @param listAllowedNodeTypeUpdatesRequest
     * @return A Java Future containing the result of the ListAllowedNodeTypeUpdates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.ListAllowedNodeTypeUpdates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/ListAllowedNodeTypeUpdates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAllowedNodeTypeUpdatesResponse> listAllowedNodeTypeUpdates(
            ListAllowedNodeTypeUpdatesRequest listAllowedNodeTypeUpdatesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAllowedNodeTypeUpdatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAllowedNodeTypeUpdates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAllowedNodeTypeUpdatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAllowedNodeTypeUpdatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAllowedNodeTypeUpdatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAllowedNodeTypeUpdatesRequest, ListAllowedNodeTypeUpdatesResponse>()
                            .withOperationName("ListAllowedNodeTypeUpdates")
                            .withMarshaller(new ListAllowedNodeTypeUpdatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAllowedNodeTypeUpdatesRequest));
            CompletableFuture<ListAllowedNodeTypeUpdatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all tags currently on a named resource. A tag is a key-value pair where the key and value are
     * case-sensitive. You can use tags to categorize and track your MemoryDB resources. For more information, see <a
     * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/Tagging-Resources.html">Tagging your MemoryDB
     * resources</a>
     * </p>
     *
     * @param listTagsRequest
     * @return A Java Future containing the result of the ListTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException</li>
     *         <li>InvalidClusterStateException</li>
     *         <li>ParameterGroupNotFoundException</li>
     *         <li>SubnetGroupNotFoundException</li>
     *         <li>SnapshotNotFoundException</li>
     *         <li>InvalidArnException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>UserNotFoundException</li>
     *         <li>AclNotFoundException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsResponse> listTags(ListTagsRequest listTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>().withOperationName("ListTags")
                            .withMarshaller(new ListTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsRequest));
            CompletableFuture<ListTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the parameters of a parameter group to the engine or system default value. You can reset specific
     * parameters by submitting a list of parameter names. To reset the entire parameter group, specify the
     * AllParameters and ParameterGroupName parameters.
     * </p>
     *
     * @param resetParameterGroupRequest
     * @return A Java Future containing the result of the ResetParameterGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterGroupStateException</li>
     *         <li>ParameterGroupNotFoundException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.ResetParameterGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/ResetParameterGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ResetParameterGroupResponse> resetParameterGroup(
            ResetParameterGroupRequest resetParameterGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetParameterGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetParameterGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResetParameterGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ResetParameterGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResetParameterGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResetParameterGroupRequest, ResetParameterGroupResponse>()
                            .withOperationName("ResetParameterGroup")
                            .withMarshaller(new ResetParameterGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(resetParameterGroupRequest));
            CompletableFuture<ResetParameterGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A tag is a key-value pair where the key and value are case-sensitive. You can use tags to categorize and track
     * all your MemoryDB resources. When you add or remove tags on clusters, those actions will be replicated to all
     * nodes in the cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/iam.resourcelevelpermissions.html">Resource-level
     * permissions</a>.
     * </p>
     * <p>
     * For example, you can use cost-allocation tags to your MemoryDB resources, Amazon generates a cost allocation
     * report as a comma-separated value (CSV) file with your usage and costs aggregated by your tags. You can apply
     * tags that represent business categories (such as cost centers, application names, or owners) to organize your
     * costs across multiple services. For more information, see <a
     * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/tagging.html">Using Cost Allocation Tags</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException</li>
     *         <li>ParameterGroupNotFoundException</li>
     *         <li>SubnetGroupNotFoundException</li>
     *         <li>InvalidClusterStateException</li>
     *         <li>SnapshotNotFoundException</li>
     *         <li>UserNotFoundException</li>
     *         <li>AclNotFoundException</li>
     *         <li>TagQuotaPerResourceExceededException</li>
     *         <li>InvalidArnException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to remove tags on a resource
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException</li>
     *         <li>InvalidClusterStateException</li>
     *         <li>ParameterGroupNotFoundException</li>
     *         <li>SubnetGroupNotFoundException</li>
     *         <li>SnapshotNotFoundException</li>
     *         <li>InvalidArnException</li>
     *         <li>TagNotFoundException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>UserNotFoundException</li>
     *         <li>AclNotFoundException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the list of users that belong to the Access Control List.
     * </p>
     *
     * @param updateAclRequest
     * @return A Java Future containing the result of the UpdateACL operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AclNotFoundException</li>
     *         <li>UserNotFoundException</li>
     *         <li>DuplicateUserNameException</li>
     *         <li>DefaultUserRequiredException</li>
     *         <li>InvalidAclStateException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.UpdateACL
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/UpdateACL" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAclResponse> updateACL(UpdateAclRequest updateAclRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAclRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateACL");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAclResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAclResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAclResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAclRequest, UpdateAclResponse>().withOperationName("UpdateACL")
                            .withMarshaller(new UpdateAclRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAclRequest));
            CompletableFuture<UpdateAclResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the settings for a cluster. You can use this operation to change one or more cluster configuration
     * settings by specifying the settings and the new values.
     * </p>
     *
     * @param updateClusterRequest
     * @return A Java Future containing the result of the UpdateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException</li>
     *         <li>InvalidClusterStateException</li>
     *         <li>InvalidNodeStateException</li>
     *         <li>ParameterGroupNotFoundException</li>
     *         <li>InvalidVpcNetworkStateException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidKmsKeyException</li>
     *         <li>NodeQuotaForClusterExceededException</li>
     *         <li>ClusterQuotaForCustomerExceededException</li>
     *         <li>ShardsPerClusterQuotaExceededException</li>
     *         <li>NodeQuotaForCustomerExceededException</li>
     *         <li>NoOperationException</li>
     *         <li>InvalidAclStateException</li>
     *         <li>AclNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.UpdateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateClusterResponse> updateCluster(UpdateClusterRequest updateClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateClusterRequest, UpdateClusterResponse>()
                            .withOperationName("UpdateCluster")
                            .withMarshaller(new UpdateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateClusterRequest));
            CompletableFuture<UpdateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the parameters of a parameter group. You can modify up to 20 parameters in a single request by submitting
     * a list parameter name and value pairs.
     * </p>
     *
     * @param updateParameterGroupRequest
     * @return A Java Future containing the result of the UpdateParameterGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ParameterGroupNotFoundException</li>
     *         <li>InvalidParameterGroupStateException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.UpdateParameterGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/UpdateParameterGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateParameterGroupResponse> updateParameterGroup(
            UpdateParameterGroupRequest updateParameterGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateParameterGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateParameterGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateParameterGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateParameterGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateParameterGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateParameterGroupRequest, UpdateParameterGroupResponse>()
                            .withOperationName("UpdateParameterGroup")
                            .withMarshaller(new UpdateParameterGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateParameterGroupRequest));
            CompletableFuture<UpdateParameterGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a subnet group. For more information, see <a
     * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/ubnetGroups.Modifying.html">Updating a subnet
     * group</a>
     * </p>
     *
     * @param updateSubnetGroupRequest
     * @return A Java Future containing the result of the UpdateSubnetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubnetGroupNotFoundException</li>
     *         <li>SubnetQuotaExceededException</li>
     *         <li>SubnetInUseException</li>
     *         <li>InvalidSubnetException</li>
     *         <li>ServiceLinkedRoleNotFoundException</li>
     *         <li>SubnetNotAllowedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.UpdateSubnetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/UpdateSubnetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSubnetGroupResponse> updateSubnetGroup(UpdateSubnetGroupRequest updateSubnetGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSubnetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubnetGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSubnetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSubnetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSubnetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSubnetGroupRequest, UpdateSubnetGroupResponse>()
                            .withOperationName("UpdateSubnetGroup")
                            .withMarshaller(new UpdateSubnetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSubnetGroupRequest));
            CompletableFuture<UpdateSubnetGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes user password(s) and/or access string.
     * </p>
     *
     * @param updateUserRequest
     * @return A Java Future containing the result of the UpdateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UserNotFoundException</li>
     *         <li>InvalidUserStateException</li>
     *         <li>InvalidParameterValueException</li>
     *         <li>InvalidParameterCombinationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MemoryDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MemoryDbAsyncClient.UpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/memorydb-2021-01-01/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateUserResponse> updateUser(UpdateUserRequest updateUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MemoryDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>().withOperationName("UpdateUser")
                            .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateUserRequest));
            CompletableFuture<UpdateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MemoryDbException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ACLNotFoundFault")
                                .exceptionBuilderSupplier(AclNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetGroupNotFoundFault")
                                .exceptionBuilderSupplier(SubnetGroupNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetGroupQuotaExceededFault")
                                .exceptionBuilderSupplier(SubnetGroupQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParameterGroupAlreadyExistsFault")
                                .exceptionBuilderSupplier(ParameterGroupAlreadyExistsException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetInUse")
                                .exceptionBuilderSupplier(SubnetInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SnapshotAlreadyExistsFault")
                                .exceptionBuilderSupplier(SnapshotAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SnapshotQuotaExceededFault")
                                .exceptionBuilderSupplier(SnapshotQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidARNFault")
                                .exceptionBuilderSupplier(InvalidArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TestFailoverNotAvailableFault")
                                .exceptionBuilderSupplier(TestFailoverNotAvailableException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ACLQuotaExceededFault")
                                .exceptionBuilderSupplier(AclQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DefaultUserRequired")
                                .exceptionBuilderSupplier(DefaultUserRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NodeQuotaForCustomerExceededFault")
                                .exceptionBuilderSupplier(NodeQuotaForCustomerExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UserAlreadyExistsFault")
                                .exceptionBuilderSupplier(UserAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidKMSKeyFault")
                                .exceptionBuilderSupplier(InvalidKmsKeyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateUserNameFault")
                                .exceptionBuilderSupplier(DuplicateUserNameException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidACLStateFault")
                                .exceptionBuilderSupplier(InvalidAclStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetNotAllowedFault")
                                .exceptionBuilderSupplier(SubnetNotAllowedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagNotFoundFault")
                                .exceptionBuilderSupplier(TagNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidVPCNetworkStateFault")
                                .exceptionBuilderSupplier(InvalidVpcNetworkStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClusterQuotaForCustomerExceededFault")
                                .exceptionBuilderSupplier(ClusterQuotaForCustomerExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterGroupStateFault")
                                .exceptionBuilderSupplier(InvalidParameterGroupStateException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParameterGroupNotFoundFault")
                                .exceptionBuilderSupplier(ParameterGroupNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLinkedRoleNotFoundFault")
                                .exceptionBuilderSupplier(ServiceLinkedRoleNotFoundException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("APICallRateForCustomerExceededFault")
                                .exceptionBuilderSupplier(ApiCallRateForCustomerExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NodeQuotaForClusterExceededFault")
                                .exceptionBuilderSupplier(NodeQuotaForClusterExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParameterGroupQuotaExceededFault")
                                .exceptionBuilderSupplier(ParameterGroupQuotaExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUpdateNotFoundFault")
                                .exceptionBuilderSupplier(ServiceUpdateNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UserNotFoundFault")
                                .exceptionBuilderSupplier(UserNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetQuotaExceededFault")
                                .exceptionBuilderSupplier(SubnetQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClusterNotFoundFault")
                                .exceptionBuilderSupplier(ClusterNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSnapshotStateFault")
                                .exceptionBuilderSupplier(InvalidSnapshotStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNodeStateFault")
                                .exceptionBuilderSupplier(InvalidNodeStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ShardsPerClusterQuotaExceededFault")
                                .exceptionBuilderSupplier(ShardsPerClusterQuotaExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCredentialsException")
                                .exceptionBuilderSupplier(InvalidCredentialsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterCombinationException")
                                .exceptionBuilderSupplier(InvalidParameterCombinationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidUserStateFault")
                                .exceptionBuilderSupplier(InvalidUserStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SnapshotNotFoundFault")
                                .exceptionBuilderSupplier(SnapshotNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClusterStateFault")
                                .exceptionBuilderSupplier(InvalidClusterStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ACLAlreadyExistsFault")
                                .exceptionBuilderSupplier(AclAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientClusterCapacityFault")
                                .exceptionBuilderSupplier(InsufficientClusterCapacityException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UserQuotaExceededFault")
                                .exceptionBuilderSupplier(UserQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagQuotaPerResourceExceeded")
                                .exceptionBuilderSupplier(TagQuotaPerResourceExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetGroupInUseFault")
                                .exceptionBuilderSupplier(SubnetGroupInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSubnet")
                                .exceptionBuilderSupplier(InvalidSubnetException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClusterAlreadyExistsFault")
                                .exceptionBuilderSupplier(ClusterAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoOperationFault")
                                .exceptionBuilderSupplier(NoOperationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ShardNotFoundFault")
                                .exceptionBuilderSupplier(ShardNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetGroupAlreadyExistsFault")
                                .exceptionBuilderSupplier(SubnetGroupAlreadyExistsException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
