/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.migrationhub;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.migrationhub.model.AccessDeniedException;
import software.amazon.awssdk.services.migrationhub.model.AssociateCreatedArtifactRequest;
import software.amazon.awssdk.services.migrationhub.model.AssociateCreatedArtifactResponse;
import software.amazon.awssdk.services.migrationhub.model.AssociateDiscoveredResourceRequest;
import software.amazon.awssdk.services.migrationhub.model.AssociateDiscoveredResourceResponse;
import software.amazon.awssdk.services.migrationhub.model.CreateProgressUpdateStreamRequest;
import software.amazon.awssdk.services.migrationhub.model.CreateProgressUpdateStreamResponse;
import software.amazon.awssdk.services.migrationhub.model.DeleteProgressUpdateStreamRequest;
import software.amazon.awssdk.services.migrationhub.model.DeleteProgressUpdateStreamResponse;
import software.amazon.awssdk.services.migrationhub.model.DescribeApplicationStateRequest;
import software.amazon.awssdk.services.migrationhub.model.DescribeApplicationStateResponse;
import software.amazon.awssdk.services.migrationhub.model.DescribeMigrationTaskRequest;
import software.amazon.awssdk.services.migrationhub.model.DescribeMigrationTaskResponse;
import software.amazon.awssdk.services.migrationhub.model.DisassociateCreatedArtifactRequest;
import software.amazon.awssdk.services.migrationhub.model.DisassociateCreatedArtifactResponse;
import software.amazon.awssdk.services.migrationhub.model.DisassociateDiscoveredResourceRequest;
import software.amazon.awssdk.services.migrationhub.model.DisassociateDiscoveredResourceResponse;
import software.amazon.awssdk.services.migrationhub.model.DryRunOperationException;
import software.amazon.awssdk.services.migrationhub.model.HomeRegionNotSetException;
import software.amazon.awssdk.services.migrationhub.model.ImportMigrationTaskRequest;
import software.amazon.awssdk.services.migrationhub.model.ImportMigrationTaskResponse;
import software.amazon.awssdk.services.migrationhub.model.InternalServerErrorException;
import software.amazon.awssdk.services.migrationhub.model.InvalidInputException;
import software.amazon.awssdk.services.migrationhub.model.ListApplicationStatesRequest;
import software.amazon.awssdk.services.migrationhub.model.ListApplicationStatesResponse;
import software.amazon.awssdk.services.migrationhub.model.ListCreatedArtifactsRequest;
import software.amazon.awssdk.services.migrationhub.model.ListCreatedArtifactsResponse;
import software.amazon.awssdk.services.migrationhub.model.ListDiscoveredResourcesRequest;
import software.amazon.awssdk.services.migrationhub.model.ListDiscoveredResourcesResponse;
import software.amazon.awssdk.services.migrationhub.model.ListMigrationTasksRequest;
import software.amazon.awssdk.services.migrationhub.model.ListMigrationTasksResponse;
import software.amazon.awssdk.services.migrationhub.model.ListProgressUpdateStreamsRequest;
import software.amazon.awssdk.services.migrationhub.model.ListProgressUpdateStreamsResponse;
import software.amazon.awssdk.services.migrationhub.model.MigrationHubException;
import software.amazon.awssdk.services.migrationhub.model.MigrationHubRequest;
import software.amazon.awssdk.services.migrationhub.model.NotifyApplicationStateRequest;
import software.amazon.awssdk.services.migrationhub.model.NotifyApplicationStateResponse;
import software.amazon.awssdk.services.migrationhub.model.NotifyMigrationTaskStateRequest;
import software.amazon.awssdk.services.migrationhub.model.NotifyMigrationTaskStateResponse;
import software.amazon.awssdk.services.migrationhub.model.PolicyErrorException;
import software.amazon.awssdk.services.migrationhub.model.PutResourceAttributesRequest;
import software.amazon.awssdk.services.migrationhub.model.PutResourceAttributesResponse;
import software.amazon.awssdk.services.migrationhub.model.ResourceNotFoundException;
import software.amazon.awssdk.services.migrationhub.model.ServiceUnavailableException;
import software.amazon.awssdk.services.migrationhub.model.ThrottlingException;
import software.amazon.awssdk.services.migrationhub.model.UnauthorizedOperationException;
import software.amazon.awssdk.services.migrationhub.paginators.ListApplicationStatesIterable;
import software.amazon.awssdk.services.migrationhub.paginators.ListCreatedArtifactsIterable;
import software.amazon.awssdk.services.migrationhub.paginators.ListDiscoveredResourcesIterable;
import software.amazon.awssdk.services.migrationhub.paginators.ListMigrationTasksIterable;
import software.amazon.awssdk.services.migrationhub.paginators.ListProgressUpdateStreamsIterable;
import software.amazon.awssdk.services.migrationhub.transform.AssociateCreatedArtifactRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.AssociateDiscoveredResourceRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.CreateProgressUpdateStreamRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.DeleteProgressUpdateStreamRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.DescribeApplicationStateRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.DescribeMigrationTaskRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.DisassociateCreatedArtifactRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.DisassociateDiscoveredResourceRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ImportMigrationTaskRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ListApplicationStatesRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ListCreatedArtifactsRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ListDiscoveredResourcesRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ListMigrationTasksRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.ListProgressUpdateStreamsRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.NotifyApplicationStateRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.NotifyMigrationTaskStateRequestMarshaller;
import software.amazon.awssdk.services.migrationhub.transform.PutResourceAttributesRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MigrationHubClient}.
 *
 * @see MigrationHubClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMigrationHubClient implements MigrationHubClient {
    private static final Logger log = Logger.loggerFor(DefaultMigrationHubClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final MigrationHubServiceClientConfiguration serviceClientConfiguration;

    protected DefaultMigrationHubClient(MigrationHubServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a created artifact of an AWS cloud resource, the target receiving the migration, with the migration
     * task performed by a migration tool. This API has the following traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Migration tools can call the <code>AssociateCreatedArtifact</code> operation to indicate which AWS artifact is
     * associated with a migration task.
     * </p>
     * </li>
     * <li>
     * <p>
     * The created artifact name must be provided in ARN (Amazon Resource Name) format which will contain information
     * about type and region; for example: <code>arn:aws:ec2:us-east-1:488216288981:image/ami-6d0ba87b</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Examples of the AWS resource behind the created artifact are, AMI's, EC2 instance, or DMS endpoint, etc.
     * </p>
     * </li>
     * </ul>
     *
     * @param associateCreatedArtifactRequest
     * @return Result of the AssociateCreatedArtifact operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws DryRunOperationException
     *         Exception raised to indicate a successfully authorized action when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws UnauthorizedOperationException
     *         Exception raised to indicate a request was not authorized when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.AssociateCreatedArtifact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/AssociateCreatedArtifact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateCreatedArtifactResponse associateCreatedArtifact(
            AssociateCreatedArtifactRequest associateCreatedArtifactRequest) throws AccessDeniedException, ThrottlingException,
            InternalServerErrorException, ServiceUnavailableException, DryRunOperationException, UnauthorizedOperationException,
            InvalidInputException, ResourceNotFoundException, HomeRegionNotSetException, AwsServiceException, SdkClientException,
            MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateCreatedArtifactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateCreatedArtifactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateCreatedArtifactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateCreatedArtifact");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateCreatedArtifactRequest, AssociateCreatedArtifactResponse>()
                            .withOperationName("AssociateCreatedArtifact").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateCreatedArtifactRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateCreatedArtifactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates a discovered resource ID from Application Discovery Service with a migration task.
     * </p>
     *
     * @param associateDiscoveredResourceRequest
     * @return Result of the AssociateDiscoveredResource operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws DryRunOperationException
     *         Exception raised to indicate a successfully authorized action when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws UnauthorizedOperationException
     *         Exception raised to indicate a request was not authorized when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws PolicyErrorException
     *         Exception raised when there are problems accessing Application Discovery Service (Application Discovery
     *         Service); most likely due to a misconfigured policy or the <code>migrationhub-discovery</code> role is
     *         missing or not configured correctly.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.AssociateDiscoveredResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/AssociateDiscoveredResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateDiscoveredResourceResponse associateDiscoveredResource(
            AssociateDiscoveredResourceRequest associateDiscoveredResourceRequest) throws AccessDeniedException,
            ThrottlingException, InternalServerErrorException, ServiceUnavailableException, DryRunOperationException,
            UnauthorizedOperationException, InvalidInputException, PolicyErrorException, ResourceNotFoundException,
            HomeRegionNotSetException, AwsServiceException, SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateDiscoveredResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateDiscoveredResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateDiscoveredResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateDiscoveredResource");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateDiscoveredResourceRequest, AssociateDiscoveredResourceResponse>()
                            .withOperationName("AssociateDiscoveredResource").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateDiscoveredResourceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateDiscoveredResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a progress update stream which is an AWS resource used for access control as well as a namespace for
     * migration task names that is implicitly linked to your AWS account. It must uniquely identify the migration tool
     * as it is used for all updates made by the tool; however, it does not need to be unique for each AWS account
     * because it is scoped to the AWS account.
     * </p>
     *
     * @param createProgressUpdateStreamRequest
     * @return Result of the CreateProgressUpdateStream operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws DryRunOperationException
     *         Exception raised to indicate a successfully authorized action when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws UnauthorizedOperationException
     *         Exception raised to indicate a request was not authorized when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.CreateProgressUpdateStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/CreateProgressUpdateStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateProgressUpdateStreamResponse createProgressUpdateStream(
            CreateProgressUpdateStreamRequest createProgressUpdateStreamRequest) throws AccessDeniedException,
            ThrottlingException, InternalServerErrorException, ServiceUnavailableException, DryRunOperationException,
            UnauthorizedOperationException, InvalidInputException, HomeRegionNotSetException, AwsServiceException,
            SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProgressUpdateStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateProgressUpdateStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProgressUpdateStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProgressUpdateStream");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateProgressUpdateStreamRequest, CreateProgressUpdateStreamResponse>()
                            .withOperationName("CreateProgressUpdateStream").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createProgressUpdateStreamRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateProgressUpdateStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a progress update stream, including all of its tasks, which was previously created as an AWS resource
     * used for access control. This API has the following traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The only parameter needed for <code>DeleteProgressUpdateStream</code> is the stream name (same as a
     * <code>CreateProgressUpdateStream</code> call).
     * </p>
     * </li>
     * <li>
     * <p>
     * The call will return, and a background process will asynchronously delete the stream and all of its resources
     * (tasks, associated resources, resource attributes, created artifacts).
     * </p>
     * </li>
     * <li>
     * <p>
     * If the stream takes time to be deleted, it might still show up on a <code>ListProgressUpdateStreams</code> call.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreateProgressUpdateStream</code>, <code>ImportMigrationTask</code>, <code>NotifyMigrationTaskState</code>,
     * and all Associate[*] APIs related to the tasks belonging to the stream will throw "InvalidInputException" if the
     * stream of the same name is in the process of being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * Once the stream and all of its resources are deleted, <code>CreateProgressUpdateStream</code> for a stream of the
     * same name will succeed, and that stream will be an entirely new logical resource (without any resources
     * associated with the old stream).
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteProgressUpdateStreamRequest
     * @return Result of the DeleteProgressUpdateStream operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws DryRunOperationException
     *         Exception raised to indicate a successfully authorized action when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws UnauthorizedOperationException
     *         Exception raised to indicate a request was not authorized when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.DeleteProgressUpdateStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/DeleteProgressUpdateStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteProgressUpdateStreamResponse deleteProgressUpdateStream(
            DeleteProgressUpdateStreamRequest deleteProgressUpdateStreamRequest) throws AccessDeniedException,
            ThrottlingException, InternalServerErrorException, ServiceUnavailableException, DryRunOperationException,
            UnauthorizedOperationException, InvalidInputException, ResourceNotFoundException, HomeRegionNotSetException,
            AwsServiceException, SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProgressUpdateStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteProgressUpdateStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProgressUpdateStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProgressUpdateStream");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteProgressUpdateStreamRequest, DeleteProgressUpdateStreamResponse>()
                            .withOperationName("DeleteProgressUpdateStream").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteProgressUpdateStreamRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteProgressUpdateStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the migration status of an application.
     * </p>
     *
     * @param describeApplicationStateRequest
     * @return Result of the DescribeApplicationState operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws PolicyErrorException
     *         Exception raised when there are problems accessing Application Discovery Service (Application Discovery
     *         Service); most likely due to a misconfigured policy or the <code>migrationhub-discovery</code> role is
     *         missing or not configured correctly.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.DescribeApplicationState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/DescribeApplicationState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeApplicationStateResponse describeApplicationState(
            DescribeApplicationStateRequest describeApplicationStateRequest) throws AccessDeniedException, ThrottlingException,
            InternalServerErrorException, ServiceUnavailableException, InvalidInputException, PolicyErrorException,
            ResourceNotFoundException, HomeRegionNotSetException, AwsServiceException, SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeApplicationStateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeApplicationStateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeApplicationStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeApplicationState");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeApplicationStateRequest, DescribeApplicationStateResponse>()
                            .withOperationName("DescribeApplicationState").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeApplicationStateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeApplicationStateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all attributes associated with a specific migration task.
     * </p>
     *
     * @param describeMigrationTaskRequest
     * @return Result of the DescribeMigrationTask operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.DescribeMigrationTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/DescribeMigrationTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeMigrationTaskResponse describeMigrationTask(DescribeMigrationTaskRequest describeMigrationTaskRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerErrorException, ServiceUnavailableException,
            InvalidInputException, ResourceNotFoundException, HomeRegionNotSetException, AwsServiceException, SdkClientException,
            MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeMigrationTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeMigrationTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMigrationTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMigrationTask");

            return clientHandler.execute(new ClientExecutionParams<DescribeMigrationTaskRequest, DescribeMigrationTaskResponse>()
                    .withOperationName("DescribeMigrationTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeMigrationTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeMigrationTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates a created artifact of an AWS resource with a migration task performed by a migration tool that was
     * previously associated. This API has the following traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A migration user can call the <code>DisassociateCreatedArtifacts</code> operation to disassociate a created AWS
     * Artifact from a migration task.
     * </p>
     * </li>
     * <li>
     * <p>
     * The created artifact name must be provided in ARN (Amazon Resource Name) format which will contain information
     * about type and region; for example: <code>arn:aws:ec2:us-east-1:488216288981:image/ami-6d0ba87b</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Examples of the AWS resource behind the created artifact are, AMI's, EC2 instance, or RDS instance, etc.
     * </p>
     * </li>
     * </ul>
     *
     * @param disassociateCreatedArtifactRequest
     * @return Result of the DisassociateCreatedArtifact operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws DryRunOperationException
     *         Exception raised to indicate a successfully authorized action when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws UnauthorizedOperationException
     *         Exception raised to indicate a request was not authorized when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.DisassociateCreatedArtifact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/DisassociateCreatedArtifact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateCreatedArtifactResponse disassociateCreatedArtifact(
            DisassociateCreatedArtifactRequest disassociateCreatedArtifactRequest) throws AccessDeniedException,
            ThrottlingException, InternalServerErrorException, ServiceUnavailableException, DryRunOperationException,
            UnauthorizedOperationException, InvalidInputException, ResourceNotFoundException, HomeRegionNotSetException,
            AwsServiceException, SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateCreatedArtifactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateCreatedArtifactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateCreatedArtifactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateCreatedArtifact");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateCreatedArtifactRequest, DisassociateCreatedArtifactResponse>()
                            .withOperationName("DisassociateCreatedArtifact").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateCreatedArtifactRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateCreatedArtifactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociate an Application Discovery Service discovered resource from a migration task.
     * </p>
     *
     * @param disassociateDiscoveredResourceRequest
     * @return Result of the DisassociateDiscoveredResource operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws DryRunOperationException
     *         Exception raised to indicate a successfully authorized action when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws UnauthorizedOperationException
     *         Exception raised to indicate a request was not authorized when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.DisassociateDiscoveredResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/DisassociateDiscoveredResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateDiscoveredResourceResponse disassociateDiscoveredResource(
            DisassociateDiscoveredResourceRequest disassociateDiscoveredResourceRequest) throws AccessDeniedException,
            ThrottlingException, InternalServerErrorException, ServiceUnavailableException, DryRunOperationException,
            UnauthorizedOperationException, InvalidInputException, ResourceNotFoundException, HomeRegionNotSetException,
            AwsServiceException, SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateDiscoveredResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateDiscoveredResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateDiscoveredResourceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateDiscoveredResource");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateDiscoveredResourceRequest, DisassociateDiscoveredResourceResponse>()
                            .withOperationName("DisassociateDiscoveredResource").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateDiscoveredResourceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateDiscoveredResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Registers a new migration task which represents a server, database, etc., being migrated to AWS by a migration
     * tool.
     * </p>
     * <p>
     * This API is a prerequisite to calling the <code>NotifyMigrationTaskState</code> API as the migration tool must
     * first register the migration task with Migration Hub.
     * </p>
     *
     * @param importMigrationTaskRequest
     * @return Result of the ImportMigrationTask operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws DryRunOperationException
     *         Exception raised to indicate a successfully authorized action when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws UnauthorizedOperationException
     *         Exception raised to indicate a request was not authorized when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.ImportMigrationTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ImportMigrationTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ImportMigrationTaskResponse importMigrationTask(ImportMigrationTaskRequest importMigrationTaskRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerErrorException, ServiceUnavailableException,
            DryRunOperationException, UnauthorizedOperationException, InvalidInputException, ResourceNotFoundException,
            HomeRegionNotSetException, AwsServiceException, SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportMigrationTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ImportMigrationTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importMigrationTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportMigrationTask");

            return clientHandler.execute(new ClientExecutionParams<ImportMigrationTaskRequest, ImportMigrationTaskResponse>()
                    .withOperationName("ImportMigrationTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(importMigrationTaskRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ImportMigrationTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the migration statuses for your applications. If you use the optional <code>ApplicationIds</code>
     * parameter, only the migration statuses for those applications will be returned.
     * </p>
     *
     * @param listApplicationStatesRequest
     * @return Result of the ListApplicationStates operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.ListApplicationStates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListApplicationStates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationStatesResponse listApplicationStates(ListApplicationStatesRequest listApplicationStatesRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerErrorException, ServiceUnavailableException,
            InvalidInputException, HomeRegionNotSetException, AwsServiceException, SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationStatesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListApplicationStatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationStatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationStates");

            return clientHandler.execute(new ClientExecutionParams<ListApplicationStatesRequest, ListApplicationStatesResponse>()
                    .withOperationName("ListApplicationStates").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listApplicationStatesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListApplicationStatesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the migration statuses for your applications. If you use the optional <code>ApplicationIds</code>
     * parameter, only the migration statuses for those applications will be returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplicationStates(software.amazon.awssdk.services.migrationhub.model.ListApplicationStatesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhub.paginators.ListApplicationStatesIterable responses = client.listApplicationStatesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.migrationhub.paginators.ListApplicationStatesIterable responses = client
     *             .listApplicationStatesPaginator(request);
     *     for (software.amazon.awssdk.services.migrationhub.model.ListApplicationStatesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhub.paginators.ListApplicationStatesIterable responses = client.listApplicationStatesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplicationStates(software.amazon.awssdk.services.migrationhub.model.ListApplicationStatesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listApplicationStatesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.ListApplicationStates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListApplicationStates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationStatesIterable listApplicationStatesPaginator(ListApplicationStatesRequest listApplicationStatesRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerErrorException, ServiceUnavailableException,
            InvalidInputException, HomeRegionNotSetException, AwsServiceException, SdkClientException, MigrationHubException {
        return new ListApplicationStatesIterable(this, applyPaginatorUserAgent(listApplicationStatesRequest));
    }

    /**
     * <p>
     * Lists the created artifacts attached to a given migration task in an update stream. This API has the following
     * traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Gets the list of the created artifacts while migration is taking place.
     * </p>
     * </li>
     * <li>
     * <p>
     * Shows the artifacts created by the migration tool that was associated by the
     * <code>AssociateCreatedArtifact</code> API.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lists created artifacts in a paginated interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param listCreatedArtifactsRequest
     * @return Result of the ListCreatedArtifacts operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.ListCreatedArtifacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListCreatedArtifacts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCreatedArtifactsResponse listCreatedArtifacts(ListCreatedArtifactsRequest listCreatedArtifactsRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerErrorException, ServiceUnavailableException,
            InvalidInputException, ResourceNotFoundException, HomeRegionNotSetException, AwsServiceException, SdkClientException,
            MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCreatedArtifactsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListCreatedArtifactsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCreatedArtifactsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCreatedArtifacts");

            return clientHandler.execute(new ClientExecutionParams<ListCreatedArtifactsRequest, ListCreatedArtifactsResponse>()
                    .withOperationName("ListCreatedArtifacts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listCreatedArtifactsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCreatedArtifactsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the created artifacts attached to a given migration task in an update stream. This API has the following
     * traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Gets the list of the created artifacts while migration is taking place.
     * </p>
     * </li>
     * <li>
     * <p>
     * Shows the artifacts created by the migration tool that was associated by the
     * <code>AssociateCreatedArtifact</code> API.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lists created artifacts in a paginated interface.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCreatedArtifacts(software.amazon.awssdk.services.migrationhub.model.ListCreatedArtifactsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhub.paginators.ListCreatedArtifactsIterable responses = client.listCreatedArtifactsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.migrationhub.paginators.ListCreatedArtifactsIterable responses = client
     *             .listCreatedArtifactsPaginator(request);
     *     for (software.amazon.awssdk.services.migrationhub.model.ListCreatedArtifactsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhub.paginators.ListCreatedArtifactsIterable responses = client.listCreatedArtifactsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCreatedArtifacts(software.amazon.awssdk.services.migrationhub.model.ListCreatedArtifactsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCreatedArtifactsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.ListCreatedArtifacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListCreatedArtifacts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCreatedArtifactsIterable listCreatedArtifactsPaginator(ListCreatedArtifactsRequest listCreatedArtifactsRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerErrorException, ServiceUnavailableException,
            InvalidInputException, ResourceNotFoundException, HomeRegionNotSetException, AwsServiceException, SdkClientException,
            MigrationHubException {
        return new ListCreatedArtifactsIterable(this, applyPaginatorUserAgent(listCreatedArtifactsRequest));
    }

    /**
     * <p>
     * Lists discovered resources associated with the given <code>MigrationTask</code>.
     * </p>
     *
     * @param listDiscoveredResourcesRequest
     * @return Result of the ListDiscoveredResources operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.ListDiscoveredResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListDiscoveredResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDiscoveredResourcesResponse listDiscoveredResources(ListDiscoveredResourcesRequest listDiscoveredResourcesRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerErrorException, ServiceUnavailableException,
            InvalidInputException, ResourceNotFoundException, HomeRegionNotSetException, AwsServiceException, SdkClientException,
            MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDiscoveredResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDiscoveredResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDiscoveredResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDiscoveredResources");

            return clientHandler
                    .execute(new ClientExecutionParams<ListDiscoveredResourcesRequest, ListDiscoveredResourcesResponse>()
                            .withOperationName("ListDiscoveredResources").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listDiscoveredResourcesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListDiscoveredResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists discovered resources associated with the given <code>MigrationTask</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDiscoveredResources(software.amazon.awssdk.services.migrationhub.model.ListDiscoveredResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhub.paginators.ListDiscoveredResourcesIterable responses = client.listDiscoveredResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.migrationhub.paginators.ListDiscoveredResourcesIterable responses = client
     *             .listDiscoveredResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.migrationhub.model.ListDiscoveredResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhub.paginators.ListDiscoveredResourcesIterable responses = client.listDiscoveredResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDiscoveredResources(software.amazon.awssdk.services.migrationhub.model.ListDiscoveredResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDiscoveredResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.ListDiscoveredResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListDiscoveredResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDiscoveredResourcesIterable listDiscoveredResourcesPaginator(
            ListDiscoveredResourcesRequest listDiscoveredResourcesRequest) throws AccessDeniedException, ThrottlingException,
            InternalServerErrorException, ServiceUnavailableException, InvalidInputException, ResourceNotFoundException,
            HomeRegionNotSetException, AwsServiceException, SdkClientException, MigrationHubException {
        return new ListDiscoveredResourcesIterable(this, applyPaginatorUserAgent(listDiscoveredResourcesRequest));
    }

    /**
     * <p>
     * Lists all, or filtered by resource name, migration tasks associated with the user account making this call. This
     * API has the following traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Can show a summary list of the most recent migration tasks.
     * </p>
     * </li>
     * <li>
     * <p>
     * Can show a summary list of migration tasks associated with a given discovered resource.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lists migration tasks in a paginated interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param listMigrationTasksRequest
     * @return Result of the ListMigrationTasks operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws PolicyErrorException
     *         Exception raised when there are problems accessing Application Discovery Service (Application Discovery
     *         Service); most likely due to a misconfigured policy or the <code>migrationhub-discovery</code> role is
     *         missing or not configured correctly.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.ListMigrationTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListMigrationTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMigrationTasksResponse listMigrationTasks(ListMigrationTasksRequest listMigrationTasksRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerErrorException, ServiceUnavailableException,
            InvalidInputException, PolicyErrorException, ResourceNotFoundException, HomeRegionNotSetException,
            AwsServiceException, SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMigrationTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMigrationTasksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMigrationTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMigrationTasks");

            return clientHandler.execute(new ClientExecutionParams<ListMigrationTasksRequest, ListMigrationTasksResponse>()
                    .withOperationName("ListMigrationTasks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listMigrationTasksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMigrationTasksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all, or filtered by resource name, migration tasks associated with the user account making this call. This
     * API has the following traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Can show a summary list of the most recent migration tasks.
     * </p>
     * </li>
     * <li>
     * <p>
     * Can show a summary list of migration tasks associated with a given discovered resource.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lists migration tasks in a paginated interface.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMigrationTasks(software.amazon.awssdk.services.migrationhub.model.ListMigrationTasksRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhub.paginators.ListMigrationTasksIterable responses = client.listMigrationTasksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.migrationhub.paginators.ListMigrationTasksIterable responses = client
     *             .listMigrationTasksPaginator(request);
     *     for (software.amazon.awssdk.services.migrationhub.model.ListMigrationTasksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhub.paginators.ListMigrationTasksIterable responses = client.listMigrationTasksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMigrationTasks(software.amazon.awssdk.services.migrationhub.model.ListMigrationTasksRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMigrationTasksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws PolicyErrorException
     *         Exception raised when there are problems accessing Application Discovery Service (Application Discovery
     *         Service); most likely due to a misconfigured policy or the <code>migrationhub-discovery</code> role is
     *         missing or not configured correctly.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.ListMigrationTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListMigrationTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMigrationTasksIterable listMigrationTasksPaginator(ListMigrationTasksRequest listMigrationTasksRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerErrorException, ServiceUnavailableException,
            InvalidInputException, PolicyErrorException, ResourceNotFoundException, HomeRegionNotSetException,
            AwsServiceException, SdkClientException, MigrationHubException {
        return new ListMigrationTasksIterable(this, applyPaginatorUserAgent(listMigrationTasksRequest));
    }

    /**
     * <p>
     * Lists progress update streams associated with the user account making this call.
     * </p>
     *
     * @param listProgressUpdateStreamsRequest
     * @return Result of the ListProgressUpdateStreams operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.ListProgressUpdateStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListProgressUpdateStreams"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProgressUpdateStreamsResponse listProgressUpdateStreams(
            ListProgressUpdateStreamsRequest listProgressUpdateStreamsRequest) throws AccessDeniedException, ThrottlingException,
            InternalServerErrorException, ServiceUnavailableException, InvalidInputException, HomeRegionNotSetException,
            AwsServiceException, SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProgressUpdateStreamsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProgressUpdateStreamsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProgressUpdateStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProgressUpdateStreams");

            return clientHandler
                    .execute(new ClientExecutionParams<ListProgressUpdateStreamsRequest, ListProgressUpdateStreamsResponse>()
                            .withOperationName("ListProgressUpdateStreams").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listProgressUpdateStreamsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListProgressUpdateStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists progress update streams associated with the user account making this call.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProgressUpdateStreams(software.amazon.awssdk.services.migrationhub.model.ListProgressUpdateStreamsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhub.paginators.ListProgressUpdateStreamsIterable responses = client.listProgressUpdateStreamsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.migrationhub.paginators.ListProgressUpdateStreamsIterable responses = client
     *             .listProgressUpdateStreamsPaginator(request);
     *     for (software.amazon.awssdk.services.migrationhub.model.ListProgressUpdateStreamsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhub.paginators.ListProgressUpdateStreamsIterable responses = client.listProgressUpdateStreamsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProgressUpdateStreams(software.amazon.awssdk.services.migrationhub.model.ListProgressUpdateStreamsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProgressUpdateStreamsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.ListProgressUpdateStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/ListProgressUpdateStreams"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProgressUpdateStreamsIterable listProgressUpdateStreamsPaginator(
            ListProgressUpdateStreamsRequest listProgressUpdateStreamsRequest) throws AccessDeniedException, ThrottlingException,
            InternalServerErrorException, ServiceUnavailableException, InvalidInputException, HomeRegionNotSetException,
            AwsServiceException, SdkClientException, MigrationHubException {
        return new ListProgressUpdateStreamsIterable(this, applyPaginatorUserAgent(listProgressUpdateStreamsRequest));
    }

    /**
     * <p>
     * Sets the migration state of an application. For a given application identified by the value passed to
     * <code>ApplicationId</code>, its status is set or updated by passing one of three values to <code>Status</code>:
     * <code>NOT_STARTED | IN_PROGRESS | COMPLETED</code>.
     * </p>
     *
     * @param notifyApplicationStateRequest
     * @return Result of the NotifyApplicationState operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws DryRunOperationException
     *         Exception raised to indicate a successfully authorized action when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws UnauthorizedOperationException
     *         Exception raised to indicate a request was not authorized when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws PolicyErrorException
     *         Exception raised when there are problems accessing Application Discovery Service (Application Discovery
     *         Service); most likely due to a misconfigured policy or the <code>migrationhub-discovery</code> role is
     *         missing or not configured correctly.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.NotifyApplicationState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/NotifyApplicationState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public NotifyApplicationStateResponse notifyApplicationState(NotifyApplicationStateRequest notifyApplicationStateRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerErrorException, ServiceUnavailableException,
            DryRunOperationException, UnauthorizedOperationException, InvalidInputException, PolicyErrorException,
            ResourceNotFoundException, HomeRegionNotSetException, AwsServiceException, SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<NotifyApplicationStateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, NotifyApplicationStateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, notifyApplicationStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "NotifyApplicationState");

            return clientHandler
                    .execute(new ClientExecutionParams<NotifyApplicationStateRequest, NotifyApplicationStateResponse>()
                            .withOperationName("NotifyApplicationState").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(notifyApplicationStateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new NotifyApplicationStateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Notifies Migration Hub of the current status, progress, or other detail regarding a migration task. This API has
     * the following traits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Migration tools will call the <code>NotifyMigrationTaskState</code> API to share the latest progress and status.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MigrationTaskName</code> is used for addressing updates to the correct target.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ProgressUpdateStream</code> is used for access control and to provide a namespace for each migration tool.
     * </p>
     * </li>
     * </ul>
     *
     * @param notifyMigrationTaskStateRequest
     * @return Result of the NotifyMigrationTaskState operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws DryRunOperationException
     *         Exception raised to indicate a successfully authorized action when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws UnauthorizedOperationException
     *         Exception raised to indicate a request was not authorized when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.NotifyMigrationTaskState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/NotifyMigrationTaskState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public NotifyMigrationTaskStateResponse notifyMigrationTaskState(
            NotifyMigrationTaskStateRequest notifyMigrationTaskStateRequest) throws AccessDeniedException, ThrottlingException,
            InternalServerErrorException, ServiceUnavailableException, DryRunOperationException, UnauthorizedOperationException,
            InvalidInputException, ResourceNotFoundException, HomeRegionNotSetException, AwsServiceException, SdkClientException,
            MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<NotifyMigrationTaskStateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, NotifyMigrationTaskStateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, notifyMigrationTaskStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "NotifyMigrationTaskState");

            return clientHandler
                    .execute(new ClientExecutionParams<NotifyMigrationTaskStateRequest, NotifyMigrationTaskStateResponse>()
                            .withOperationName("NotifyMigrationTaskState").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(notifyMigrationTaskStateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new NotifyMigrationTaskStateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides identifying details of the resource being migrated so that it can be associated in the Application
     * Discovery Service repository. This association occurs asynchronously after <code>PutResourceAttributes</code>
     * returns.
     * </p>
     * <important>
     * <ul>
     * <li>
     * <p>
     * Keep in mind that subsequent calls to PutResourceAttributes will override previously stored attributes. For
     * example, if it is first called with a MAC address, but later, it is desired to <i>add</i> an IP address, it will
     * then be required to call it with <i>both</i> the IP and MAC addresses to prevent overriding the MAC address.
     * </p>
     * </li>
     * <li>
     * <p>
     * Note the instructions regarding the special use case of the <a href=
     * "https://docs.aws.amazon.com/migrationhub/latest/ug/API_PutResourceAttributes.html#migrationhub-PutResourceAttributes-request-ResourceAttributeList"
     * > <code>ResourceAttributeList</code> </a> parameter when specifying any "VM" related value.
     * </p>
     * </li>
     * </ul>
     * </important> <note>
     * <p>
     * Because this is an asynchronous call, it will always return 200, whether an association occurs or not. To confirm
     * if an association was found based on the provided details, call <code>ListDiscoveredResources</code>.
     * </p>
     * </note>
     *
     * @param putResourceAttributesRequest
     * @return Result of the PutResourceAttributes operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when there is an internal, configuration, or dependency error encountered.
     * @throws DryRunOperationException
     *         Exception raised to indicate a successfully authorized action when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws UnauthorizedOperationException
     *         Exception raised to indicate a request was not authorized when the <code>DryRun</code> flag is set to
     *         "true".
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws ResourceNotFoundException
     *         Exception raised when the request references a resource (Application Discovery Service configuration,
     *         update stream, migration task, etc.) that does not exist in Application Discovery Service (Application
     *         Discovery Service) or in Migration Hub's repository.
     * @throws HomeRegionNotSetException
     *         The home region is not set. Set the home region to continue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubClient.PutResourceAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSMigrationHub-2017-05-31/PutResourceAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutResourceAttributesResponse putResourceAttributes(PutResourceAttributesRequest putResourceAttributesRequest)
            throws AccessDeniedException, ThrottlingException, InternalServerErrorException, ServiceUnavailableException,
            DryRunOperationException, UnauthorizedOperationException, InvalidInputException, ResourceNotFoundException,
            HomeRegionNotSetException, AwsServiceException, SdkClientException, MigrationHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResourceAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutResourceAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourceAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Migration Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourceAttributes");

            return clientHandler.execute(new ClientExecutionParams<PutResourceAttributesRequest, PutResourceAttributesResponse>()
                    .withOperationName("PutResourceAttributes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putResourceAttributesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutResourceAttributesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends MigrationHubRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MigrationHubException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedOperation")
                                .exceptionBuilderSupplier(UnauthorizedOperationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyErrorException")
                                .exceptionBuilderSupplier(PolicyErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("HomeRegionNotSetException")
                                .exceptionBuilderSupplier(HomeRegionNotSetException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DryRunOperation")
                                .exceptionBuilderSupplier(DryRunOperationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final MigrationHubServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
