/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.migrationhubconfig;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.migrationhubconfig.model.AccessDeniedException;
import software.amazon.awssdk.services.migrationhubconfig.model.CreateHomeRegionControlRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.CreateHomeRegionControlResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.DryRunOperationException;
import software.amazon.awssdk.services.migrationhubconfig.model.GetHomeRegionRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.GetHomeRegionResponse;
import software.amazon.awssdk.services.migrationhubconfig.model.InternalServerErrorException;
import software.amazon.awssdk.services.migrationhubconfig.model.InvalidInputException;
import software.amazon.awssdk.services.migrationhubconfig.model.MigrationHubConfigException;
import software.amazon.awssdk.services.migrationhubconfig.model.MigrationHubConfigRequest;
import software.amazon.awssdk.services.migrationhubconfig.model.ServiceUnavailableException;
import software.amazon.awssdk.services.migrationhubconfig.model.ThrottlingException;
import software.amazon.awssdk.services.migrationhubconfig.paginators.DescribeHomeRegionControlsIterable;
import software.amazon.awssdk.services.migrationhubconfig.transform.CreateHomeRegionControlRequestMarshaller;
import software.amazon.awssdk.services.migrationhubconfig.transform.DescribeHomeRegionControlsRequestMarshaller;
import software.amazon.awssdk.services.migrationhubconfig.transform.GetHomeRegionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MigrationHubConfigClient}.
 *
 * @see MigrationHubConfigClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMigrationHubConfigClient implements MigrationHubConfigClient {
    private static final Logger log = Logger.loggerFor(DefaultMigrationHubConfigClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMigrationHubConfigClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * This API sets up the home region for the calling account only.
     * </p>
     *
     * @param createHomeRegionControlRequest
     * @return Result of the CreateHomeRegionControl operation returned by the service.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when a request fails due to temporary unavailability of the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws DryRunOperationException
     *         Exception raised to indicate that authorization of an action was successful, when the <code>DryRun</code>
     *         flag is set to true.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubConfigClient.CreateHomeRegionControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/CreateHomeRegionControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateHomeRegionControlResponse createHomeRegionControl(CreateHomeRegionControlRequest createHomeRegionControlRequest)
            throws InternalServerErrorException, ServiceUnavailableException, AccessDeniedException, ThrottlingException,
            DryRunOperationException, InvalidInputException, AwsServiceException, SdkClientException, MigrationHubConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHomeRegionControlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateHomeRegionControlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHomeRegionControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHomeRegionControl");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateHomeRegionControlRequest, CreateHomeRegionControlResponse>()
                            .withOperationName("CreateHomeRegionControl").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createHomeRegionControlRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateHomeRegionControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This API permits filtering on the <code>ControlId</code> and <code>HomeRegion</code> fields.
     * </p>
     *
     * @param describeHomeRegionControlsRequest
     * @return Result of the DescribeHomeRegionControls operation returned by the service.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when a request fails due to temporary unavailability of the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubConfigClient.DescribeHomeRegionControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/DescribeHomeRegionControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeHomeRegionControlsResponse describeHomeRegionControls(
            DescribeHomeRegionControlsRequest describeHomeRegionControlsRequest) throws InternalServerErrorException,
            ServiceUnavailableException, AccessDeniedException, ThrottlingException, InvalidInputException, AwsServiceException,
            SdkClientException, MigrationHubConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeHomeRegionControlsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeHomeRegionControlsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeHomeRegionControlsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeHomeRegionControls");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeHomeRegionControlsRequest, DescribeHomeRegionControlsResponse>()
                            .withOperationName("DescribeHomeRegionControls").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeHomeRegionControlsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeHomeRegionControlsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This API permits filtering on the <code>ControlId</code> and <code>HomeRegion</code> fields.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeHomeRegionControls(software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubconfig.paginators.DescribeHomeRegionControlsIterable responses = client.describeHomeRegionControlsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.migrationhubconfig.paginators.DescribeHomeRegionControlsIterable responses = client
     *             .describeHomeRegionControlsPaginator(request);
     *     for (software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.migrationhubconfig.paginators.DescribeHomeRegionControlsIterable responses = client.describeHomeRegionControlsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeHomeRegionControls(software.amazon.awssdk.services.migrationhubconfig.model.DescribeHomeRegionControlsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeHomeRegionControlsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when a request fails due to temporary unavailability of the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubConfigClient.DescribeHomeRegionControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/DescribeHomeRegionControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeHomeRegionControlsIterable describeHomeRegionControlsPaginator(
            DescribeHomeRegionControlsRequest describeHomeRegionControlsRequest) throws InternalServerErrorException,
            ServiceUnavailableException, AccessDeniedException, ThrottlingException, InvalidInputException, AwsServiceException,
            SdkClientException, MigrationHubConfigException {
        return new DescribeHomeRegionControlsIterable(this, applyPaginatorUserAgent(describeHomeRegionControlsRequest));
    }

    /**
     * <p>
     * Returns the calling account’s home region, if configured. This API is used by other AWS services to determine the
     * regional endpoint for calling AWS Application Discovery Service and Migration Hub. You must call
     * <code>GetHomeRegion</code> at least once before you call any other AWS Application Discovery Service and AWS
     * Migration Hub APIs, to obtain the account's Migration Hub home region.
     * </p>
     *
     * @param getHomeRegionRequest
     * @return Result of the GetHomeRegion operation returned by the service.
     * @throws InternalServerErrorException
     *         Exception raised when an internal, configuration, or dependency error is encountered.
     * @throws ServiceUnavailableException
     *         Exception raised when a request fails due to temporary unavailability of the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InvalidInputException
     *         Exception raised when the provided input violates a policy constraint or is entered in the wrong format
     *         or data type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MigrationHubConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MigrationHubConfigClient.GetHomeRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/migrationhub-config-2019-06-30/GetHomeRegion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetHomeRegionResponse getHomeRegion(GetHomeRegionRequest getHomeRegionRequest) throws InternalServerErrorException,
            ServiceUnavailableException, AccessDeniedException, ThrottlingException, InvalidInputException, AwsServiceException,
            SdkClientException, MigrationHubConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetHomeRegionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetHomeRegionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHomeRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MigrationHub Config");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHomeRegion");

            return clientHandler.execute(new ClientExecutionParams<GetHomeRegionRequest, GetHomeRegionResponse>()
                    .withOperationName("GetHomeRegion").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getHomeRegionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetHomeRegionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MigrationHubConfigException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DryRunOperation")
                                .exceptionBuilderSupplier(DryRunOperationException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MigrationHubConfigRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
